# -*- coding: utf-8 -*-
"""
/***************************************************************************
 opeNoise

 opeNoise allows to compute the noise level generated by road traffic
 at fixed receiver points and buildings.

                             -------------------
        begin                : March 2014
        copyright            : (C) 2014 by Arpa Piemonte
        email                : s.masera@arpa.piemonte.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from processing.core.VectorWriter import VectorWriter
import fTools
import os, imp
import traceback

from math import *
from datetime import datetime
from ui_CreateReceiverPoints import Ui_CreateReceiverPoints_window

path = os.path.dirname(fTools.__file__)
ftools_utils = imp.load_source('ftools_utils', os.path.join(path,'tools','ftools_utils.py'))

class Dialog(QDialog,Ui_CreateReceiverPoints_window):
   
    def __init__(self, iface):
        QDialog.__init__(self, iface.mainWindow())
        self.iface = iface
        # Set up the user interface from Designer.
        self.setupUi(self)
        
        self.progressBar.setValue(0)
        self.populateLayers()
        QObject.connect(self.receiver_layer_pushButton, SIGNAL("clicked()"), self.outFile)        
        self.buttonBox = self.buttonBox.button( QDialogButtonBox.Ok )
        
    def populateLayers( self ):
        layers = ftools_utils.getLayerNames([QGis.Polygon])
        self.buildings_layer_comboBox.clear()
        self.buildings_layer_comboBox.addItems(layers)
        
    def outFile(self):
        self.receiver_layer_lineEdit.clear()
        ( self.shapefileName, self.encoding ) = ftools_utils.saveDialog( self )
        if self.shapefileName is None or self.encoding is None:
            return
        self.receiver_layer_lineEdit.setText( self.shapefileName )    

    def log_start(self):
        
        # creates 2 log file: settings and errors
        global log_errors, log_settings,log_settings_path_name,log_errors_path_name
        path = os.path.abspath(__file__)
        dir_path = os.path.dirname(path)        
        log_settings_path_name = os.path.join(dir_path,"log_CreateReceiverPoints_settings.txt")
        log_errors_path_name = os.path.join(dir_path,"log_CreateReceiverPoints_errors.txt")
        log_settings = open(log_settings_path_name,"w")
        log_errors = open(log_errors_path_name,"w")
        log_settings.write("opeNoise - Create Receiver Points Settings" + "\n\n")
        log_errors.write("opeNoise - Create Receiver Points Errors" + "\n\n")        
        
    def log_end(self):

        log_settings.close()        
        log_errors.close()    
            
    def accept(self):
        
        self.buttonBox.setEnabled( False )
        
        if self.buildings_layer_comboBox.currentText() == "":
            QMessageBox.information(self, self.tr("opeNoise - Create Receiver Points"), self.tr("Please specify input polygon vector layer"))
            self.buttonBox.setEnabled( True )
            return
        elif self.receiver_layer_lineEdit.text() == "":
            QMessageBox.information(self, self.tr("opeNoise - Create Receiver Points"), self.tr("Please specify output shapefile"))
            self.buttonBox.setEnabled( True )
            return
        else:
            
            
            buildings_layer = ftools_utils.getVectorLayerByName(self.buildings_layer_comboBox.currentText())
            receiver_points_layer_path = self.receiver_layer_lineEdit.text()
            
            # writes the settings log file
            self.log_start()
            
            log_settings.write("Buildings layer:\n" + buildings_layer.source() + "\n\n")
            log_settings.write("Receiver points layer:\n" + str(receiver_points_layer_path) + "\n\n")
            
            self.time_start = datetime.now()
            
            # CreateReceiverPoints
            try: 
                self.CreateReceiverPoints(buildings_layer,receiver_points_layer_path)
                run = 1
            except:
                error= traceback.format_exc()
                log_errors.write(error)
                run = 0
                
            self.time_end = datetime.now()

            if run == 1:
                log_errors.write("No errors." + "\n\n") 
                result_string = "Receiver points created with success." + "\n\n" +\
                                 "View and rename the settings file to keep it:" + "\n" +\
                                 str(log_settings_path_name) + "\n\n" + str(self.duration())
                QMessageBox.information(self, self.tr("opeNoise - Create Receiver Points"), self.tr(result_string))
                self.iface.messageBar().pushMessage("opeNoise - Create Receiver Points", "Process complete")
            else:
                result_string = "Sorry, process not complete." + "\n\n" +\
                                "View the log file to understand the problem:" + "\n" +\
                                str(log_errors_path_name) + "\n\n" + str(self.duration())
                QMessageBox.information(self, self.tr("opeNoise - Create Receiver Points"), self.tr(result_string))
                self.iface.messageBar().pushMessage("opeNoise - Create Receiver Points", "Process not complete")

            
            log_settings.write("\n\n=======================================================\n")
            log_settings.write(result_string)
            self.log_end()

        self.progressBar.setValue(0)
        self.buttonBox.setEnabled( True )

        self.iface.mainWindow().statusBar().clearMessage()
        self.iface.mapCanvas().refresh() 
        self.close()
        
    def duration(self):
        duration = self.time_end - self.time_start
        duration_h = duration.seconds/3600
        duration_m = (duration.seconds - duration_h*3600)/60
        duration_s = duration.seconds - duration_m*60 - duration_h*3600
        duration_string = "Starting time: " + self.time_start.strftime("%a %d/%b/%Y %H:%M:%S.%f") + "\n" +\
                          "Ending time: " + self.time_end.strftime("%a %d/%b/%Y %H:%M:%S.%f") + "\n"+\
                          "Execution time: " + str(duration_h) + " hours, " + str(duration_m) + \
                          " minutes, " + str(duration_s) + "." + str(duration.microseconds) + " seconds."
        return duration_string

    
    def CreateReceiverPoints(self, buildings_layer,receiver_points_layer_path):
       
        # defines emission_points layer
        receiver_points_fields = [QgsField("id_pt", QVariant.Int), QgsField("id_bui", QVariant.Int)]
        receiver_points_writer = VectorWriter(receiver_points_layer_path, None, receiver_points_fields, 0, buildings_layer.crs())
        
        # gets features from layer
        buildings_feat_all = buildings_layer.dataProvider().getFeatures()    
        
        # creates SpatialIndex
        buildings_spIndex = QgsSpatialIndex()
        buildings_feat_all_dict = {}
        for buildings_feat in buildings_feat_all:
            buildings_spIndex.insertFeature(buildings_feat)
            buildings_feat_all_dict[buildings_feat.id()] = buildings_feat
        
        # defines distanze_point
        distance_point = 0.1
        
        # re-gets features from layer
        buildings_feat_all = buildings_layer.dataProvider().getFeatures()    
        buildings_feat_total = buildings_layer.dataProvider().featureCount()
        
        pt_id = 0
        buildings_feat_number = 0
        for buildings_feat in buildings_feat_all:
            
            buildings_feat_number = buildings_feat_number + 1
            bar = buildings_feat_number/float(buildings_feat_total)*100
            self.progressBar.setValue(bar)
            
            buildings_pt = buildings_feat.geometry().asPolygon()
            
            # creates the search rectangle
            rect = QgsRectangle()
            rect.setXMinimum( buildings_feat.geometry().boundingBox().xMinimum() - distance_point )
            rect.setXMaximum( buildings_feat.geometry().boundingBox().xMaximum() + distance_point )
            rect.setYMinimum( buildings_feat.geometry().boundingBox().yMinimum() - distance_point )
            rect.setYMaximum( buildings_feat.geometry().boundingBox().yMaximum() + distance_point )
        
            buildings_selection = buildings_spIndex.intersects(rect)
        
            for i in range(0,len(buildings_pt)):
        
                for ii in range(0,len(buildings_pt[i])-1):
                    
                    x1 = buildings_pt[i][ii][0]
                    x2 = buildings_pt[i][ii+1][0]
                    y1 = buildings_pt[i][ii][1]
                    y2 = buildings_pt[i][ii+1][1]
                    
                    xm = ( x1 + x2 )/2
                    ym = ( y1 + y2 )/2
                
                    if y2 == y1:
                        dx = 0
                        dy = distance_point
                    elif x2 == x1:
                        dx = distance_point
                        dy = 0
                    else:
                        m = ( y2 - y1 )/ ( x2 - x1 )
                        m_p = -1/m
                        dx = sqrt((distance_point**2)/(1 + m_p**2))
                        dy = sqrt(((distance_point**2)*(m_p**2))/(1 + m_p**2))
        
                    if (x2 >= x1 and y2 >= y1) or (x2 < x1 and y2 < y1):
                        pt1 = QgsPoint(xm + dx, ym - dy) 
                        pt2 = QgsPoint(xm - dx, ym + dy) 
                    if (x2 >= x1 and y2 < y1) or (x2 < x1 and y2 >= y1):
                        pt1 = QgsPoint(xm + dx, ym + dy) 
                        pt2 = QgsPoint(xm - dx, ym - dy) 
                    
                    pt = QgsFeature()
                    
                    # pt1
                    pt.setGeometry(QgsGeometry.fromPoint(pt1))            
                    intersect = 0
                    for buildings_id in buildings_selection:
                        if buildings_feat_all_dict[buildings_id].geometry().intersects(pt.geometry()) == 1:
                            intersect = 1
                            break 
                    
                    if intersect == 0:
                        pt.setAttributes([pt_id, buildings_feat.id()])
                        receiver_points_writer.addFeature(pt)
                        pt_id = pt_id + 1
                    
                    # pt2
                    pt.setGeometry(QgsGeometry.fromPoint(pt2))            
                    intersect = 0
                    for buildings_id in buildings_selection:
                        if buildings_feat_all_dict[buildings_id].geometry().intersects(pt.geometry()) == 1:
                            intersect = 1
                            break 
                    
                    if intersect == 0:
                        pt.setAttributes([pt_id, buildings_feat.id()])
                        receiver_points_writer.addFeature(pt)
                        pt_id = pt_id + 1                
        
        del receiver_points_writer
        
        receiver_points_layer = QgsVectorLayer(receiver_points_layer_path, unicode(ftools_utils.getShapefileName( receiver_points_layer_path )), "ogr")        

        QgsMapLayerRegistry.instance().addMapLayers([receiver_points_layer])

    