# -*- coding: utf-8 -*-
"""
/***************************************************************************
 opeNoise

 opeNoise allows to compute the noise level generated by road traffic
 at fixed receiver points and buildings.

                             -------------------
        begin                : March 2014
        copyright            : (C) 2014 by Arpa Piemonte
        email                : s.masera@arpa.piemonte.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from processing.core.VectorWriter import VectorWriter
import fTools
import os, imp
import traceback
from multiprocessing import Pool,cpu_count

from math import *
from datetime import datetime
from ui_CalculateNoiseLevels import Ui_CalculateNoiseLevels_window

from symbols import render

from nmpb import nmpb

path = os.path.dirname(fTools.__file__)
ftools_utils = imp.load_source('ftools_utils', os.path.join(path,'tools','ftools_utils.py'))

class Dialog(QDialog,Ui_CalculateNoiseLevels_window):
   
    def __init__(self, iface):
        QDialog.__init__(self, iface.mainWindow())
        self.iface = iface
        # Set up the user interface from Designer.
        self.setupUi(self)
        
        self.populate_comboBox()
        
        self.progressBar.setValue(0)

        self.update_field_roads_layer()
        
        self.diu_v_check_value = 0
        self.nig_v_check_value = 1
        self.day_v_check_value = 1
        self.eve_v_check_value = 1
        self.diu_v_check()
        self.nig_v_check()
        self.day_v_check()
        self.eve_v_check()

        self.diu_p_check_value = 0
        self.nig_p_check_value = 1
        self.day_p_check_value = 1
        self.eve_p_check_value = 1
        self.diu_p_check()
        self.nig_p_check()
        self.day_p_check()
        self.eve_p_check()
        
        self.vehicle_true()
        
        self.obstacles_check_value = 0
        self.emission_points_check_value = 0
        self.rays_check_value = 0
        
        QObject.connect(self.roads_layer_comboBox, SIGNAL("currentIndexChanged(QString)"), self.update_field_roads_layer)
        QObject.connect(self.vehicles_radioButton, SIGNAL("toggled(bool)"), self.vehicle_true)
        QObject.connect(self.power_radioButton, SIGNAL("toggled(bool)"), self.vehicle_false)

        QObject.connect(self.L_diu_v_checkBox, SIGNAL("toggled(bool)"), self.diu_v_check)
        QObject.connect(self.L_nig_v_checkBox, SIGNAL("toggled(bool)"), self.nig_v_check)
        QObject.connect(self.L_day_v_checkBox, SIGNAL("toggled(bool)"), self.day_v_check)
        QObject.connect(self.L_eve_v_checkBox, SIGNAL("toggled(bool)"), self.eve_v_check)
        
        QObject.connect(self.L_diu_p_checkBox, SIGNAL("toggled(bool)"), self.diu_p_check)
        QObject.connect(self.L_nig_p_checkBox, SIGNAL("toggled(bool)"), self.nig_p_check)
        QObject.connect(self.L_day_p_checkBox, SIGNAL("toggled(bool)"), self.day_p_check)
        QObject.connect(self.L_eve_p_checkBox, SIGNAL("toggled(bool)"), self.eve_p_check)
        
        QObject.connect(self.obstacles_layer_checkBox, SIGNAL("toggled(bool)"), self.obstacles_check)
        QObject.connect(self.emission_points_layer_checkBox, SIGNAL("toggled(bool)"), self.emission_points_check)
        QObject.connect(self.rays_layer_checkBox, SIGNAL("toggled(bool)"), self.rays_check)
        
        #QObject.connect(self.L_nig_v_checkBox, SIGNAL("toggled(bool)"), self.diu_false)
        
        QObject.connect(self.emission_points_layer_pushButton, SIGNAL("clicked()"), self.outFile_emission_points)
        QObject.connect(self.rays_layer_pushButton, SIGNAL("clicked()"), self.outFile_rays)
        
        self.run_buttonBox.button( QDialogButtonBox.Ok )

        
    def populate_comboBox( self ):
        receiver_points_layers = ftools_utils.getLayerNames([QGis.Point])
        self.receiver_points_layer_comboBox.clear()
        self.receiver_points_layer_comboBox.addItems(receiver_points_layers)
 
        roads_layers = ftools_utils.getLayerNames([QGis.Line])
        self.roads_layer_comboBox.clear()
        self.roads_layer_comboBox.addItems(roads_layers)
        
        research_ray = ['50','100','200','500','1000']        
        self.research_ray_comboBox.clear()
        for distance in research_ray:
            self.research_ray_comboBox.addItem(distance)
        
        obstacles_layers = ftools_utils.getLayerNames([QGis.Polygon])
        self.obstacles_layer_comboBox.clear()
        self.obstacles_layer_comboBox.addItems(obstacles_layers)
        
    def update_field_roads_layer(self):
        
        if unicode(self.roads_layer_comboBox.currentText()) == "":
            return

        roads_layer = ftools_utils.getVectorLayerByName(unicode(self.roads_layer_comboBox.currentText()))
        roads_layer_fields = ftools_utils.getFieldList(roads_layer)

        self.surface_comboBox.clear()        
        self.slope_comboBox.clear()  
        self.L_diu_l_n_comboBox.clear()
        self.L_diu_h_n_comboBox.clear()
        self.L_diu_l_s_comboBox.clear()
        self.L_diu_h_s_comboBox.clear()
        self.L_diu_type_comboBox.clear()        
        self.L_diu_p_comboBox.clear() 

        self.L_nig_l_n_comboBox.clear()
        self.L_nig_h_n_comboBox.clear()
        self.L_nig_l_s_comboBox.clear()
        self.L_nig_h_s_comboBox.clear()
        self.L_nig_type_comboBox.clear()        
        self.L_nig_p_comboBox.clear() 

        self.L_day_l_n_comboBox.clear()
        self.L_day_h_n_comboBox.clear()
        self.L_day_l_s_comboBox.clear()
        self.L_day_h_s_comboBox.clear()
        self.L_day_type_comboBox.clear()        
        self.L_day_p_comboBox.clear() 
        
        self.L_eve_l_n_comboBox.clear()
        self.L_eve_h_n_comboBox.clear()
        self.L_eve_l_s_comboBox.clear()
        self.L_eve_h_s_comboBox.clear()
        self.L_eve_type_comboBox.clear()        
        self.L_eve_p_comboBox.clear() 
        
        roads_layer_fields_number = ["choose field"]
        roads_layer_fields_string = ["choose field"]
        
        for f in roads_layer_fields:
            if f.type() == QVariant.Int or f.type() == QVariant.Double:         
                roads_layer_fields_number.append(unicode(f.name()))
            elif f.type() == QVariant.String:
                roads_layer_fields_string.append(unicode(f.name()))

        for f_label in roads_layer_fields_number:
            self.L_diu_l_n_comboBox.addItem(f_label)
            self.L_diu_h_n_comboBox.addItem(f_label)
            self.L_diu_l_s_comboBox.addItem(f_label)
            self.L_diu_h_s_comboBox.addItem(f_label)
            self.L_diu_p_comboBox.addItem(f_label)

            self.L_nig_l_n_comboBox.addItem(f_label)
            self.L_nig_h_n_comboBox.addItem(f_label)
            self.L_nig_l_s_comboBox.addItem(f_label)
            self.L_nig_h_s_comboBox.addItem(f_label)                
            self.L_nig_p_comboBox.addItem(f_label) 
        
            self.L_day_l_n_comboBox.addItem(f_label)
            self.L_day_h_n_comboBox.addItem(f_label)
            self.L_day_l_s_comboBox.addItem(f_label)
            self.L_day_h_s_comboBox.addItem(f_label)                
            self.L_day_p_comboBox.addItem(f_label) 

            self.L_eve_l_n_comboBox.addItem(f_label)
            self.L_eve_h_n_comboBox.addItem(f_label)
            self.L_eve_l_s_comboBox.addItem(f_label)
            self.L_eve_h_s_comboBox.addItem(f_label)                
            self.L_eve_p_comboBox.addItem(f_label) 
            
        for f_label in roads_layer_fields_string:        
                self.surface_comboBox.addItem(f_label)
                self.slope_comboBox.addItem(f_label)
                self.L_diu_type_comboBox.addItem(f_label)  
                self.L_nig_type_comboBox.addItem(f_label)
                self.L_day_type_comboBox.addItem(f_label)  
                self.L_eve_type_comboBox.addItem(f_label)
                
        surface_item_default = ['========','smooth','porous','stones','cement','corrugated']
        slope_item_default = ['========','flat','down','up']
        type_item_default = ['========','continuos','pulsed accelerated','pulsed decelerated','non-differentiated pulsed']
        
        for item_default in surface_item_default:
            self.surface_comboBox.addItem(item_default) 
        for item_default in slope_item_default:
            self.slope_comboBox.addItem(item_default)             
        for item_default in type_item_default:
            self.L_diu_type_comboBox.addItem(item_default)  
            self.L_nig_type_comboBox.addItem(item_default)            
            self.L_day_type_comboBox.addItem(item_default)  
            self.L_eve_type_comboBox.addItem(item_default) 
            
    def vehicle_false(self):

        self.surface_label.setEnabled( False )
        self.slope_label.setEnabled( False )
        self.n_l_label.setEnabled( False )
        self.n_p_label.setEnabled( False )
        self.v_l_label.setEnabled( False )
        self.v_p_label.setEnabled( False )
        self.type_label.setEnabled( False )
        
        self.surface_comboBox.setEnabled( False )
        self.slope_comboBox.setEnabled( False )
        
        self.L_diu_v_checkBox.setEnabled( False )
        self.L_diu_l_n_comboBox.setEnabled( False ) 
        self.L_diu_h_n_comboBox.setEnabled( False )
        self.L_diu_l_s_comboBox.setEnabled( False )
        self.L_diu_h_s_comboBox.setEnabled( False )
        self.L_diu_type_comboBox.setEnabled( False )

        self.L_nig_v_checkBox.setEnabled( False )
        self.L_nig_l_n_comboBox.setEnabled( False ) 
        self.L_nig_h_n_comboBox.setEnabled( False )
        self.L_nig_l_s_comboBox.setEnabled( False )
        self.L_nig_h_s_comboBox.setEnabled( False )
        self.L_nig_type_comboBox.setEnabled( False )

        self.L_day_v_checkBox.setEnabled( False )
        self.L_day_l_n_comboBox.setEnabled( False ) 
        self.L_day_h_n_comboBox.setEnabled( False )
        self.L_day_l_s_comboBox.setEnabled( False )
        self.L_day_h_s_comboBox.setEnabled( False )
        self.L_day_type_comboBox.setEnabled( False )

        self.L_eve_v_checkBox.setEnabled( False )
        self.L_eve_l_n_comboBox.setEnabled( False ) 
        self.L_eve_h_n_comboBox.setEnabled( False )
        self.L_eve_l_s_comboBox.setEnabled( False )
        self.L_eve_h_s_comboBox.setEnabled( False )
        self.L_eve_type_comboBox.setEnabled( False )

        self.L_diu_p_checkBox.setEnabled( True )
        self.L_nig_p_checkBox.setEnabled( True )
        self.L_day_p_checkBox.setEnabled( True )
        self.L_eve_p_checkBox.setEnabled( True )


        self.power_label.setEnabled( True )
        
        if self.diu_p_check_value == 1: 
            self.L_diu_p_comboBox.setEnabled( True )
        if self.nig_p_check_value == 1: 
            self.L_nig_p_comboBox.setEnabled( True )
        if self.day_p_check_value == 1: 
            self.L_day_p_comboBox.setEnabled( True )
        if self.eve_p_check_value == 1: 
            self.L_eve_p_comboBox.setEnabled( True )        

    def vehicle_true(self):
        
        self.surface_label.setEnabled( True )
        self.slope_label.setEnabled( True )
        self.n_l_label.setEnabled( True )
        self.n_p_label.setEnabled( True )
        self.v_l_label.setEnabled( True )
        self.v_p_label.setEnabled( True )
        self.type_label.setEnabled( True )

        self.surface_comboBox.setEnabled( True ) 
        self.slope_comboBox.setEnabled( True )
        self.L_diu_v_checkBox.setEnabled( True )        
        self.L_nig_v_checkBox.setEnabled( True )
        self.L_day_v_checkBox.setEnabled( True )
        self.L_eve_v_checkBox.setEnabled( True )
        
        if self.diu_v_check_value == 1:  
            self.L_diu_l_n_comboBox.setEnabled( True ) 
            self.L_diu_h_n_comboBox.setEnabled( True )
            self.L_diu_l_s_comboBox.setEnabled( True )
            self.L_diu_h_s_comboBox.setEnabled( True )
            self.L_diu_type_comboBox.setEnabled( True )

        if self.nig_v_check_value == 1:
            self.L_nig_l_n_comboBox.setEnabled( True ) 
            self.L_nig_h_n_comboBox.setEnabled( True )
            self.L_nig_l_s_comboBox.setEnabled( True )
            self.L_nig_h_s_comboBox.setEnabled( True )
            self.L_nig_type_comboBox.setEnabled( True )

        if self.day_v_check_value == 1: 
            self.L_day_l_n_comboBox.setEnabled( True ) 
            self.L_day_h_n_comboBox.setEnabled( True )
            self.L_day_l_s_comboBox.setEnabled( True )
            self.L_day_h_s_comboBox.setEnabled( True )
            self.L_day_type_comboBox.setEnabled( True )

        if self.eve_v_check_value == 1: 
            self.L_eve_l_n_comboBox.setEnabled( True ) 
            self.L_eve_h_n_comboBox.setEnabled( True )
            self.L_eve_l_s_comboBox.setEnabled( True )
            self.L_eve_h_s_comboBox.setEnabled( True )
            self.L_eve_type_comboBox.setEnabled( True )
        
        self.power_label.setEnabled( False )        
        
        self.L_diu_p_checkBox.setEnabled( False )
        self.L_nig_p_checkBox.setEnabled( False )
        self.L_day_p_checkBox.setEnabled( False )
        self.L_eve_p_checkBox.setEnabled( False )
        
        self.L_diu_p_comboBox.setEnabled( False )
        self.L_nig_p_comboBox.setEnabled( False )
        self.L_day_p_comboBox.setEnabled( False )
        self.L_eve_p_comboBox.setEnabled( False )    
    
    def diu_v_check(self):
        
        if self.diu_v_check_value == 0:        
            self.L_diu_l_n_comboBox.setEnabled( True ) 
            self.L_diu_h_n_comboBox.setEnabled( True )
            self.L_diu_l_s_comboBox.setEnabled( True )
            self.L_diu_h_s_comboBox.setEnabled( True )
            self.L_diu_type_comboBox.setEnabled( True )
            self. diu_v_check_value = 1
        else:
            self.L_diu_l_n_comboBox.setEnabled( False ) 
            self.L_diu_h_n_comboBox.setEnabled( False )
            self.L_diu_l_s_comboBox.setEnabled( False )
            self.L_diu_h_s_comboBox.setEnabled( False )
            self.L_diu_type_comboBox.setEnabled( False )  
            self.L_diu_p_comboBox.setEnabled( False )
            self.diu_v_check_value = 0

    def nig_v_check(self):
        
        if self.nig_v_check_value == 0:        
            self.L_nig_l_n_comboBox.setEnabled( True ) 
            self.L_nig_h_n_comboBox.setEnabled( True )
            self.L_nig_l_s_comboBox.setEnabled( True )
            self.L_nig_h_s_comboBox.setEnabled( True )
            self.L_nig_type_comboBox.setEnabled( True )
            self. nig_v_check_value = 1
        else:
            self.L_nig_l_n_comboBox.setEnabled( False ) 
            self.L_nig_h_n_comboBox.setEnabled( False )
            self.L_nig_l_s_comboBox.setEnabled( False )
            self.L_nig_h_s_comboBox.setEnabled( False )
            self.L_nig_type_comboBox.setEnabled( False )
            self.nig_v_check_value = 0

    def day_v_check(self):
        
        if self.day_v_check_value == 0:        
            self.L_day_l_n_comboBox.setEnabled( True ) 
            self.L_day_h_n_comboBox.setEnabled( True )
            self.L_day_l_s_comboBox.setEnabled( True )
            self.L_day_h_s_comboBox.setEnabled( True )
            self.L_day_type_comboBox.setEnabled( True )
            self. day_v_check_value = 1
        else:
            self.L_day_l_n_comboBox.setEnabled( False ) 
            self.L_day_h_n_comboBox.setEnabled( False )
            self.L_day_l_s_comboBox.setEnabled( False )
            self.L_day_h_s_comboBox.setEnabled( False )
            self.L_day_type_comboBox.setEnabled( False )
            self.day_v_check_value = 0

    def eve_v_check(self):
        
        if self.eve_v_check_value == 0:        
            self.L_eve_l_n_comboBox.setEnabled( True ) 
            self.L_eve_h_n_comboBox.setEnabled( True )
            self.L_eve_l_s_comboBox.setEnabled( True )
            self.L_eve_h_s_comboBox.setEnabled( True )
            self.L_eve_type_comboBox.setEnabled( True )
            self.eve_v_check_value = 1
        else:
            self.L_eve_l_n_comboBox.setEnabled( False ) 
            self.L_eve_h_n_comboBox.setEnabled( False )
            self.L_eve_l_s_comboBox.setEnabled( False )
            self.L_eve_h_s_comboBox.setEnabled( False )
            self.L_eve_type_comboBox.setEnabled( False )
            self.eve_v_check_value = 0

    def diu_p_check(self):
        
        if self.diu_p_check_value == 0:        
            self.L_diu_p_comboBox.setEnabled( True ) 
            self.diu_p_check_value = 1
        else:
            self.L_diu_p_comboBox.setEnabled( False ) 
            self.diu_p_check_value = 0

    def nig_p_check(self):
        
        if self.nig_p_check_value == 0:        
            self.L_nig_p_comboBox.setEnabled( True ) 
            self.nig_p_check_value = 1
        else:
            self.L_nig_p_comboBox.setEnabled( False ) 
            self.nig_p_check_value = 0

    def day_p_check(self):
        
        if self.day_p_check_value == 0:        
            self.L_day_p_comboBox.setEnabled( True ) 
            self.day_p_check_value = 1
        else:
            self.L_day_p_comboBox.setEnabled( False ) 
            self.day_p_check_value = 0

    def eve_p_check(self):
        
        if self.eve_p_check_value == 0:        
            self.L_eve_p_comboBox.setEnabled( True ) 
            self.eve_p_check_value = 1
        else:
            self.L_eve_p_comboBox.setEnabled( False ) 
            self.eve_p_check_value = 0
            
    def obstacles_check(self):
        
        if self.obstacles_check_value == 0:        
            self.obstacles_layer_comboBox.setEnabled( True ) 
            self.obstacles_check_value = 1
        else:
            self.obstacles_layer_comboBox.setEnabled( False ) 
            self.obstacles_check_value = 0            

    def emission_points_check(self):
        
        if self.emission_points_check_value == 0:        
            self.emission_points_layer_lineEdit.setEnabled( True ) 
            self.emission_points_layer_pushButton.setEnabled( True ) 
            self.emission_points_check_value = 1
        else:
            self.emission_points_layer_lineEdit.setEnabled( False ) 
            self.emission_points_layer_pushButton.setEnabled( False ) 
            self.emission_points_check_value = 0  

    def rays_check(self):
        
        if self.rays_check_value == 0:        
            self.rays_layer_lineEdit.setEnabled( True ) 
            self.rays_layer_pushButton.setEnabled( True ) 
            self.rays_check_value = 1
        else:
            self.rays_layer_lineEdit.setEnabled( False ) 
            self.rays_layer_pushButton.setEnabled( False ) 
            self.rays_check_value = 0              
            
       
    def outFile_emission_points(self):
        self.emission_points_layer_lineEdit.clear()
        ( self.shapefileName, self.encoding ) = ftools_utils.saveDialog( self )
        if self.shapefileName is None or self.encoding is None:
            return
        self.emission_points_layer_lineEdit.setText( self.shapefileName )    

    def outFile_rays(self):
        self.rays_layer_lineEdit.clear()
        ( self.shapefileName, self.encoding ) = ftools_utils.saveDialog( self )
        if self.shapefileName is None or self.encoding is None:
            return
        self.rays_layer_lineEdit.setText( self.shapefileName ) 

    def controls(self):
        self.run_buttonBox.setEnabled( False )
        if self.receiver_points_layer_comboBox.currentText() == "":
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the receiver points vector layer."))
            return 0
        
        if self.roads_layer_comboBox.currentText() == "":
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the roads line vector layer."))
            return 0

        if self.vehicles_radioButton.isChecked():
            if self.L_diu_v_checkBox.isChecked():
                if self.surface_comboBox.currentText() == "choose field" or self.surface_comboBox.currentText() == "========"\
                  or self.slope_comboBox.currentText() == "choose field" or self.slope_comboBox.currentText() == "========"\
                  or (self.L_diu_l_n_comboBox.currentText() == "choose field" and self.L_diu_h_n_comboBox.currentText() == "choose field")\
                  or self.L_diu_type_comboBox.currentText() == "choose field" or self.L_diu_type_comboBox.currentText() == "========"\
                  or (self.L_diu_l_n_comboBox.currentText() <> "choose field" and self.L_diu_l_s_comboBox.currentText() == "choose field")\
                  or (self.L_diu_l_n_comboBox.currentText() == "choose field" and self.L_diu_l_s_comboBox.currentText() <> "choose field")\
                  or (self.L_diu_h_n_comboBox.currentText() <> "choose field" and self.L_diu_h_s_comboBox.currentText() == "choose field")\
                  or (self.L_diu_h_n_comboBox.currentText() == "choose field" and self.L_diu_h_s_comboBox.currentText() <> "choose field"):  
                    message = "Please specify the next fields for each reference period to calculate:" + "\n" +\
                              "- Road surface" + "\n" +\
                              "- Road slope" + "\n" +\
                              "- Light vehicles number and speed or heavy vehicles number and speed" + "\n" +\
                              "- Traffic type"
                    QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr(message))
                    return 0
            if self.L_nig_v_checkBox.isChecked():
                if self.surface_comboBox.currentText() == "choose field" or self.surface_comboBox.currentText() == "========"\
                  or self.slope_comboBox.currentText() == "choose field" or self.slope_comboBox.currentText() == "========"\
                  or (self.L_nig_l_n_comboBox.currentText() == "choose field" and self.L_nig_h_n_comboBox.currentText() == "choose field")\
                  or self.L_nig_type_comboBox.currentText() == "choose field" or self.L_nig_type_comboBox.currentText() == "========"\
                  or (self.L_nig_l_n_comboBox.currentText() <> "choose field" and self.L_nig_l_s_comboBox.currentText() == "choose field")\
                  or (self.L_nig_l_n_comboBox.currentText() == "choose field" and self.L_nig_l_s_comboBox.currentText() <> "choose field")\
                  or (self.L_nig_h_n_comboBox.currentText() <> "choose field" and self.L_nig_h_s_comboBox.currentText() == "choose field")\
                  or (self.L_nig_h_n_comboBox.currentText() == "choose field" and self.L_nig_h_s_comboBox.currentText() <> "choose field"):  
                    message = "Please specify the next fields for each reference period to calculate:" + "\n" +\
                              "- Road surface" + "\n" +\
                              "- Road slope" + "\n" +\
                              "- Light vehicles number and speed or heavy vehicles number and speed" + "\n" +\
                              "- Traffic type"
                    QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr(message))
                    return 0
            if self.L_day_v_checkBox.isChecked():
                if self.surface_comboBox.currentText() == "choose field" or self.surface_comboBox.currentText() == "========"\
                  or self.slope_comboBox.currentText() == "choose field" or self.slope_comboBox.currentText() == "========"\
                  or (self.L_day_l_n_comboBox.currentText() == "choose field" and self.L_day_h_n_comboBox.currentText() == "choose field")\
                  or self.L_day_type_comboBox.currentText() == "choose field" or self.L_day_type_comboBox.currentText() == "========"\
                  or (self.L_day_l_n_comboBox.currentText() <> "choose field" and self.L_day_l_s_comboBox.currentText() == "choose field")\
                  or (self.L_day_l_n_comboBox.currentText() == "choose field" and self.L_day_l_s_comboBox.currentText() <> "choose field")\
                  or (self.L_day_h_n_comboBox.currentText() <> "choose field" and self.L_day_h_s_comboBox.currentText() == "choose field")\
                  or (self.L_day_h_n_comboBox.currentText() == "choose field" and self.L_day_h_s_comboBox.currentText() <> "choose field"):  
                    message = "Please specify the next fields for each reference period to calculate:" + "\n" +\
                              "- Road surface" + "\n" +\
                              "- Road slope" + "\n" +\
                              "- Light vehicles number and speed or heavy vehicles number and speed" + "\n" +\
                              "- Traffic type"
                    QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr(message))
                    return 0  
            if self.L_eve_v_checkBox.isChecked():
                if self.surface_comboBox.currentText() == "choose field" or self.surface_comboBox.currentText() == "========"\
                  or self.slope_comboBox.currentText() == "choose field" or self.slope_comboBox.currentText() == "========"\
                  or (self.L_eve_l_n_comboBox.currentText() == "choose field" and self.L_eve_h_n_comboBox.currentText() == "choose field")\
                  or self.L_eve_type_comboBox.currentText() == "choose field" or self.L_eve_type_comboBox.currentText() == "========"\
                  or (self.L_eve_l_n_comboBox.currentText() <> "choose field" and self.L_eve_l_s_comboBox.currentText() == "choose field")\
                  or (self.L_eve_l_n_comboBox.currentText() == "choose field" and self.L_eve_l_s_comboBox.currentText() <> "choose field")\
                  or (self.L_eve_h_n_comboBox.currentText() <> "choose field" and self.L_eve_h_s_comboBox.currentText() == "choose field")\
                  or (self.L_eve_h_n_comboBox.currentText() == "choose field" and self.L_eve_h_s_comboBox.currentText() <> "choose field"):  
                    message = "Please specify the next fields for each reference period to calculate:" + "\n" +\
                              "- Road surface" + "\n" +\
                              "- Road slope" + "\n" +\
                              "- Light vehicles number and speed or heavy vehicles number and speed" + "\n" +\
                              "- Traffic type"
                    QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr(message))
                    return 0 
            if self.L_diu_v_checkBox.isChecked() == False and self.L_nig_v_checkBox.isChecked() == False\
                and self.L_day_v_checkBox.isChecked() == False and self.L_eve_v_checkBox.isChecked() == False:
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify at least a reference period."))
                return 0
            if  self.L_den_checkBox.isChecked() and self.L_nig_v_checkBox.isChecked() == False\
                and self.L_day_v_checkBox.isChecked() == False and self.L_eve_v_checkBox.isChecked() == False:
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify at least a reference period among Lday, Leve or Lnig in order to calculate Lden."))
                return 0

        elif self.power_radioButton.isChecked():
            if self.L_diu_p_checkBox.isChecked() and self.L_diu_p_comboBox.currentText() == "choose field":
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the power level field for each reference period to calculate."))
                return 0
            if self.L_nig_p_checkBox.isChecked() and self.L_nig_p_comboBox.currentText() == "choose field":
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the power level field for each reference period to calculate."))
                return 0
            if self.L_day_p_checkBox.isChecked() and self.L_day_p_comboBox.currentText() == "choose field":
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the power level field for each reference period to calculate."))
                return 0
            if self.L_eve_p_checkBox.isChecked() and self.L_eve_p_comboBox.currentText() == "choose field":
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the power level field for each reference period to calculate."))
                return 0
            if self.L_diu_p_checkBox.isChecked() == False and self.L_nig_p_checkBox.isChecked() == False\
                and self.L_day_p_checkBox.isChecked() == False and self.L_eve_p_checkBox.isChecked() == False:
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify at least a reference period."))
                return 0                    
            if self.L_den_checkBox.isChecked() and self.L_nig_p_checkBox.isChecked() == False\
                and self.L_day_p_checkBox.isChecked() == False and self.L_eve_p_checkBox.isChecked() == False:
                QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify at least a reference period among Lday, Leve or Lnig in order to calculate Lden."))
                return 0                    
        
        if self.obstacles_layer_checkBox.isChecked() and self.obstacles_layer_comboBox.currentText() == "":
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify the obstacles polygon vector layer."))
            return 0

        if self.emission_points_layer_checkBox.isChecked() and self.emission_points_layer_lineEdit.text() == "":
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify output shapefile name for the emission points."))          
            return 0
        
        if self.rays_layer_checkBox.isChecked() and self.rays_layer_lineEdit.text() == "":
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr("Please specify output shapefile name for the sound rays."))          
            return 0
        
        return 1

    def populate_roads_fields(self):

        roads_dict = {}

        if self.vehicles_radioButton.isChecked():  
            roads_dict['implementation'] = "vehicles"
            roads_dict['surface'] = self.surface_comboBox.currentText()
            roads_dict['slope'] = self.slope_comboBox.currentText()
            
            if self.L_diu_v_checkBox.isChecked():
                roads_dict['diu'] = True
                if self.L_diu_l_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_diu_l_n'] = 'none'
                else:
                    roads_dict['L_diu_l_n'] = self.L_diu_l_n_comboBox.currentText()
                if self.L_diu_h_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_diu_h_n'] = 'none'
                else:
                    roads_dict['L_diu_h_n'] = self.L_diu_h_n_comboBox.currentText()
                if self.L_diu_l_s_comboBox.currentText() == 'choose field':
                    roads_dict['L_diu_l_s'] = 'none'
                else:
                    roads_dict['L_diu_l_s'] = self.L_diu_l_s_comboBox.currentText()
                if self.L_diu_h_s_comboBox.currentText()  == 'choose field':
                    roads_dict['L_diu_h_s'] = 'none'
                else:
                    roads_dict['L_diu_h_s'] = self.L_diu_h_s_comboBox.currentText()
                roads_dict['L_diu_type'] = self.L_diu_type_comboBox.currentText()
            else:
                roads_dict['diu'] = False
                
            if self.L_nig_v_checkBox.isChecked():
                roads_dict['nig'] = True
                if self.L_nig_l_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_nig_l_n'] = 'none'
                else:
                    roads_dict['L_nig_l_n'] = self.L_nig_l_n_comboBox.currentText()
                if self.L_nig_h_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_nig_h_n'] = 'none'
                else:
                    roads_dict['L_nig_h_n'] = self.L_nig_h_n_comboBox.currentText()
                if self.L_nig_l_s_comboBox.currentText() == 'choose field':
                    roads_dict['L_nig_l_s'] = 'none'
                else:
                    roads_dict['L_nig_l_s'] = self.L_nig_l_s_comboBox.currentText()
                if self.L_nig_h_s_comboBox.currentText()  == 'choose field':
                    roads_dict['L_nig_h_s'] = 'none'
                else:
                    roads_dict['L_nig_h_s'] = self.L_nig_h_s_comboBox.currentText()
                roads_dict['L_nig_type'] = self.L_nig_type_comboBox.currentText()
            else:
                roads_dict['nig'] = False
                
            if self.L_day_v_checkBox.isChecked():
                roads_dict['day'] = True
                if self.L_day_l_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_day_l_n'] = 'none'
                else:
                    roads_dict['L_day_l_n'] = self.L_day_l_n_comboBox.currentText()
                if self.L_day_h_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_day_h_n'] = 'none'
                else:
                    roads_dict['L_day_h_n'] = self.L_day_h_n_comboBox.currentText()
                if self.L_day_l_s_comboBox.currentText() == 'choose field':
                    roads_dict['L_day_l_s'] = 'none'
                else:
                    roads_dict['L_day_l_s'] = self.L_day_l_s_comboBox.currentText()
                if self.L_day_h_s_comboBox.currentText()  == 'choose field':
                    roads_dict['L_day_h_s'] = 'none'
                else:
                    roads_dict['L_day_h_s'] = self.L_day_h_s_comboBox.currentText()
                roads_dict['L_day_type'] = self.L_day_type_comboBox.currentText()
            else:
                roads_dict['day'] = False
                
            if self.L_eve_v_checkBox.isChecked():
                roads_dict['eve'] = True
                if self.L_day_l_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_eve_l_n'] = 'none'
                else:
                    roads_dict['L_eve_l_n'] = self.L_eve_l_n_comboBox.currentText()
                if self.L_eve_h_n_comboBox.currentText() == 'choose field':
                    roads_dict['L_eve_h_n'] = 'none'
                else:
                    roads_dict['L_eve_h_n'] = self.L_eve_h_n_comboBox.currentText()
                if self.L_eve_l_s_comboBox.currentText() == 'choose field':
                    roads_dict['L_eve_l_s'] = 'none'
                else:
                    roads_dict['L_eve_l_s'] = self.L_eve_l_s_comboBox.currentText()
                if self.L_eve_h_s_comboBox.currentText()  == 'choose field':
                    roads_dict['L_eve_h_s'] = 'none'
                else:
                    roads_dict['L_eve_h_s'] = self.L_eve_h_s_comboBox.currentText()
                roads_dict['L_eve_type'] = self.L_eve_type_comboBox.currentText()
            else:
                roads_dict['eve'] = False
                
        # power implementation data
        elif self.power_radioButton.isChecked():
            roads_dict['implementation'] = "power"
            
            if self.L_diu_p_checkBox.isChecked():
                roads_dict['diu'] = True
                roads_dict['L_diu_p'] = self.L_diu_p_comboBox.currentText()
            else:
                roads_dict['diu'] = False
                
            if self.L_nig_p_checkBox.isChecked():
                roads_dict['nig'] = True
                roads_dict['L_nig_p'] = self.L_nig_p_comboBox.currentText()
            else:
                roads_dict['nig'] = False
                
            if self.L_day_p_checkBox.isChecked():
                roads_dict['day'] = True
                roads_dict['L_day_p'] = self.L_day_p_comboBox.currentText()
            else:
                roads_dict['day'] = False
                
            if self.L_eve_p_checkBox.isChecked():
                roads_dict['eve'] = True
                roads_dict['L_eve_p'] = self.L_eve_p_comboBox.currentText()
            else:
                roads_dict['eve'] = False
                
        return roads_dict
    
    def accept(self):
        self.run_buttonBox.setEnabled( False )
        
        if self.controls() == 0:
            self.run_buttonBox.setEnabled( True )
            return

        receiver_points_layer = ftools_utils.getVectorLayerByName(self.receiver_points_layer_comboBox.currentText())
        roads_layer = ftools_utils.getVectorLayerByName(self.roads_layer_comboBox.currentText())
        roads_layer_details = self.populate_roads_fields()
        
        parameters = {}
        parameters['research_ray'] = int(self.research_ray_comboBox.currentText())
        parameters['L_den'] = self.L_den_checkBox.isChecked()
        
        if self.obstacles_layer_checkBox.isChecked():
            obstacles_layer = ftools_utils.getVectorLayerByName(self.obstacles_layer_comboBox.currentText())
        else:
            obstacles_layer = ""
        if self.emission_points_layer_checkBox.isChecked():
            emission_points_layer_path = self.emission_points_layer_lineEdit.text()
        else:
            emission_points_layer_path = ""
        if self.rays_layer_checkBox.isChecked():
            rays_layer_path = self.rays_layer_lineEdit.text()
        else:
            rays_layer_path = ""
        
        # writes the settings log file
        self.log_start()  
        log_settings.write("Reveicer points layer:\n" + receiver_points_layer.source() + "\n\n")
        log_settings.write("Roads layer:\n" + roads_layer.source() + "\n\n")
        log_settings.write("Roads layer details:\n" + str(roads_layer_details) + "\n\n")
        log_settings.write("Parameters:\n" + str(parameters) + "\n\n")
        if obstacles_layer == "":
            log_settings.write("Obstacles layer:\n" + "None" + "\n\n")
        else:
            log_settings.write("Obstacles layer:\n" + obstacles_layer.source() + "\n\n")
        if emission_points_layer_path == "":
            log_settings.write("Emission points layer:\n" + "None" + "\n\n")
        else:
            log_settings.write("Emission points layer:\n" + str(emission_points_layer_path) + "\n\n")            
        if rays_layer_path == "":
            log_settings.write("Rays layer:\n" + "None" + "\n\n")
        else:
            log_settings.write("Rays layer:\n" + str(rays_layer_path) + "\n\n")            

        self.time_start = datetime.now()
        
        # Run
        try:    
            self.run(receiver_points_layer,roads_layer,roads_layer_details,parameters,obstacles_layer,emission_points_layer_path,rays_layer_path)
            run = 1
        except:
            error= traceback.format_exc()
            log_errors.write(error)
            run = 0
            
        self.time_end = datetime.now()

        if run == 1:
            log_errors.write("No errors." + "\n\n") 
            result_string = "Noise levels calculated with success." + "\n\n" +\
                             "View and rename the settings file to keep it:" + "\n" +\
                             str(log_settings_path_name) + "\n\n" + str(self.duration())
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr(result_string))
            self.iface.messageBar().pushMessage("opeNoise - Calculate Noise Levels", "Process complete")
        else:
            result_string = "Sorry, process not complete." + "\n\n" +\
                            "View the log file to understand the problem:" + "\n" +\
                            str(log_errors_path_name) + "\n\n" + str(self.duration())
            QMessageBox.information(self, self.tr("opeNoise - Calculate Noise Levels"), self.tr(result_string))
            self.iface.messageBar().pushMessage("opeNoise - Calculate Noise Levels", "Process not complete")              

        log_settings.write("\n\n=======================================================\n")
        log_settings.write(result_string)
        self.log_end()
        
        self.run_buttonBox.setEnabled( True )

        self.iface.mainWindow().statusBar().clearMessage()
        self.iface.mapCanvas().refresh() 
        self.close()  


    def duration(self):
        duration = self.time_end - self.time_start
        duration_h = duration.seconds/3600
        duration_m = (duration.seconds - duration_h*3600)/60
        duration_s = duration.seconds - duration_m*60 - duration_h*3600
        duration_string = "Starting time: " + self.time_start.strftime("%a %d/%b/%Y %H:%M:%S.%f") + "\n" +\
                          "Ending time: " + self.time_end.strftime("%a %d/%b/%Y %H:%M:%S.%f") + "\n"+\
                          "Execution time: " + str(duration_h) + " hours, " + str(duration_m) + \
                          " minutes, " + str(duration_s) + "." + str(duration.microseconds) + " seconds."
        return duration_string
    
    def log_start(self):
        
        # creates 2 log file: settings and errors
        global log_errors, log_settings,log_settings_path_name,log_errors_path_name
        path = os.path.abspath(__file__)
        dir_path = os.path.dirname(path)        
        log_settings_path_name = os.path.join(dir_path,"log_CalculateNoiseLevels_settings.txt")
        log_errors_path_name = os.path.join(dir_path,"log_CalculateNoiseLevels_errors.txt")
        log_settings = open(log_settings_path_name,"w")
        log_errors = open(log_errors_path_name,"w")
        log_settings.write("opeNoise - Calculate Noise Level Settings" + "\n\n")
        log_errors.write("opeNoise - Calculate Noise Level Errors" + "\n\n")    
        
    def log_end(self):

        log_settings.close()        
        log_errors.close()                

    # computes distance (input two QgsPoints, return a float)    
    def compute_distance(self,QgsPoint1,QgsPoint2):
        return sqrt((QgsPoint1.x()-QgsPoint2.x())**2+(QgsPoint1.y()-QgsPoint2.y())**2)
    
    # computes mid point (input two QgsPoints, return a QgsPoint)     
    def compute_mid_point(self,QgsPoint1,QgsPoint2):
        return QgsPoint((QgsPoint1.x()+QgsPoint2.x())/2,(QgsPoint1.y()+QgsPoint2.y())/2)
    
    # adds a point to a layer (input writer, QgsPoint, Attributes)
    def add_point_to_layer(self,writer,point,attributes):
        geometry = QgsGeometry.fromPoint(point)
        feature = QgsFeature()
        feature.setAttributes(attributes)
        feature.setGeometry(geometry)
        writer.addFeature(feature)   

    # adds a point to a memory layer (input writer, QgsPoint, Attributes)
    def add_point_to_memory_layer(self,memory_layer,point,attributes):
        geometry = QgsGeometry.fromPoint(point)
        feature = QgsFeature()
        feature.setAttributes(attributes)
        feature.setGeometry(geometry)
        memory_layer.dataProvider().addFeatures([feature])   
        memory_layer.updateExtents()

    # adds a point to a memory layer (input writer, QgsPoint, Attributes)
    def emission_points_creation(self,roads_layer,roads_layer_details,research_ray,receiver_points_feat_all_dict,receiver_points_spIndex,emission_points_layer_path,emission_points_writer,emission_points_memory_layer):

        # roads
        roads_feat_all = roads_layer.dataProvider().getFeatures()
        roads_feat_total = roads_layer.dataProvider().featureCount()
        
        # gets fields index from roads layer for implementation with vehicles and power
        if roads_layer_details['implementation'] == 'vehicles':
            vehicles_field_index = {}
            if roads_layer_details['surface'] == 'smooth' or\
               roads_layer_details['surface'] == 'porous' or\
               roads_layer_details['surface'] == 'stones' or\
               roads_layer_details['surface'] == 'cement' or\
               roads_layer_details['surface'] == 'corrugated':
                surface = roads_layer_details['surface']
            else:
                surface = ""
                vehicles_field_index['surface'] = roads_layer.fieldNameIndex(roads_layer_details['surface'])

            if roads_layer_details['slope'] == 'flat' or\
               roads_layer_details['slope'] == 'down' or\
               roads_layer_details['slope'] == 'up':
                slope = roads_layer_details['slope']
            else:
                slope = ""
                vehicles_field_index['slope'] = roads_layer.fieldNameIndex(roads_layer_details['slope'])                                 

            if roads_layer_details['diu'] == True:
                if roads_layer_details['L_diu_l_n'] == 'none':
                    L_diu_l_n = 0
                else:
                    vehicles_field_index['L_diu_l_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_diu_l_n'])
                if roads_layer_details['L_diu_h_n'] == 'none':
                    L_diu_h_n = 0
                else:
                    vehicles_field_index['L_diu_h_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_diu_h_n'])
                if roads_layer_details['L_diu_l_s'] == 'none':
                    L_diu_l_s = 0
                else:
                    vehicles_field_index['L_diu_l_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_diu_l_s'])
                if roads_layer_details['L_diu_h_s'] == 'none':
                    L_diu_h_s = 0
                else:
                    vehicles_field_index['L_diu_h_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_diu_h_s'])
                if roads_layer_details['L_diu_type'] == 'continuos' or\
                   roads_layer_details['L_diu_type'] == 'pulsed accelerated' or\
                   roads_layer_details['L_diu_type'] == 'pulsed decelerated' or\
                   roads_layer_details['L_diu_type'] == 'non-differentiated pulsed':
                    L_diu_type = roads_layer_details['L_diu_type']
                else:
                    L_diu_type = ""
                    vehicles_field_index['L_diu_type'] = roads_layer.fieldNameIndex(roads_layer_details['L_diu_type'])
                
            if roads_layer_details['nig'] == True:
                if roads_layer_details['L_nig_l_n'] == 'none':
                    L_nig_l_n = 0
                else:
                    vehicles_field_index['L_nig_l_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_nig_l_n'])
                if roads_layer_details['L_nig_h_n'] == 'none':
                    L_nig_h_n = 0
                else:
                    vehicles_field_index['L_nig_h_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_nig_h_n'])
                if roads_layer_details['L_nig_l_s'] == 'none':
                    L_nig_l_s = 0
                else:
                    vehicles_field_index['L_nig_l_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_nig_l_s'])
                if roads_layer_details['L_nig_h_s'] == 'none':
                    L_nig_h_s = 0
                else:
                    vehicles_field_index['L_nig_h_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_nig_h_s'])
                if roads_layer_details['L_nig_type'] == 'continuos' or\
                   roads_layer_details['L_nig_type'] == 'pulsed accelerated' or\
                   roads_layer_details['L_nig_type'] == 'pulsed decelerated' or\
                   roads_layer_details['L_nig_type'] == 'non-differentiated pulsed':
                    L_nig_type = roads_layer_details['L_nig_type']
                else:
                    L_nig_type = ""
                    vehicles_field_index['L_nig_type'] = roads_layer.fieldNameIndex(roads_layer_details['L_nig_type'])
                    
            if roads_layer_details['day'] == True:
                if roads_layer_details['L_day_l_n'] == 'none':
                    L_day_l_n = 0
                else:
                    vehicles_field_index['L_day_l_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_day_l_n'])
                if roads_layer_details['L_day_h_n'] == 'none':
                    L_day_h_n = 0
                else:
                    vehicles_field_index['L_day_h_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_day_h_n'])
                if roads_layer_details['L_day_l_s'] == 'none':
                    L_day_l_s = 0
                else:
                    vehicles_field_index['L_day_l_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_day_l_s'])
                if roads_layer_details['L_day_h_s'] == 'none':
                    L_day_h_s = 0
                else:
                    vehicles_field_index['L_day_h_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_day_h_s'])
                if roads_layer_details['L_day_type'] == 'continuos' or\
                   roads_layer_details['L_day_type'] == 'pulsed accelerated' or\
                   roads_layer_details['L_day_type'] == 'pulsed decelerated' or\
                   roads_layer_details['L_day_type'] == 'non-differentiated pulsed':
                    L_day_type = roads_layer_details['L_day_type']
                else:
                    L_day_type = ""
                    vehicles_field_index['L_day_type'] = roads_layer.fieldNameIndex(roads_layer_details['L_day_type'])
                    
            if roads_layer_details['eve'] == True:
                if roads_layer_details['L_eve_l_n'] == 'none':
                    L_eve_l_n = 0
                else:
                    vehicles_field_index['L_eve_l_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_eve_l_n'])
                if roads_layer_details['L_eve_h_n'] == 'none':
                    L_eve_h_n = 0
                else:
                    vehicles_field_index['L_eve_h_n'] = roads_layer.fieldNameIndex(roads_layer_details['L_eve_h_n'])
                if roads_layer_details['L_eve_l_s'] == 'none':
                    L_eve_l_s = 0
                else:
                    vehicles_field_index['L_eve_l_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_eve_l_s'])
                if roads_layer_details['L_eve_h_s'] == 'none':
                    L_eve_h_s = 0
                else:
                    vehicles_field_index['L_eve_h_s'] = roads_layer.fieldNameIndex(roads_layer_details['L_eve_h_s'])                
                if roads_layer_details['L_eve_type'] == 'continuos' or\
                   roads_layer_details['L_eve_type'] == 'pulsed accelerated' or\
                   roads_layer_details['L_eve_type'] == 'pulsed decelerated' or\
                   roads_layer_details['L_eve_type'] == 'non-differentiated pulsed':
                    L_eve_type = roads_layer_details['L_eve_type']
                else:
                    L_eve_type = ""
                    vehicles_field_index['L_eve_type'] = roads_layer.fieldNameIndex(roads_layer_details['L_eve_type'])
                
        if roads_layer_details['implementation'] == 'power':
            power_field_index = {}
            if roads_layer_details['diu'] == True:            
                power_field_index['diu'] = roads_layer.fieldNameIndex(roads_layer_details['L_diu_p'])
            if roads_layer_details['nig'] == True:            
                power_field_index['nig'] = roads_layer.fieldNameIndex(roads_layer_details['L_nig_p'])
            if roads_layer_details['day'] == True:            
                power_field_index['day'] = roads_layer.fieldNameIndex(roads_layer_details['L_day_p'])
            if roads_layer_details['eve'] == True:            
                power_field_index['eve'] = roads_layer.fieldNameIndex(roads_layer_details['L_eve_p'])
        
        # initializes ray and emission point id
        emission_point_id = 0

        roads_feat_number = 0
        
        roads_powers = {}
        
        for roads_feat in roads_feat_all:

            roads_feat_number = roads_feat_number + 1
            bar = roads_feat_number/float(roads_feat_total)*100
            self.progressBar.setValue(bar)

            power = {}
            
            # implementation with vehicles: calculates the road power using the nmpb module
            if roads_layer_details['implementation'] == 'vehicles':
                
                if surface == "":
                    surface = roads_feat.attributes()[vehicles_field_index['surface']]
                if slope == "":
                    slope = roads_feat.attributes()[vehicles_field_index['slope']]

                if roads_layer_details['diu'] == True:
                    if roads_layer_details['L_diu_l_n'] <> 'none':
                        L_diu_l_n = roads_feat.attributes()[vehicles_field_index['L_diu_l_n']]
                    if roads_layer_details['L_diu_h_n'] <> 'none':
                        L_diu_h_n = roads_feat.attributes()[vehicles_field_index['L_diu_h_n']]
                    if roads_layer_details['L_diu_l_s'] <> 'none':
                        L_diu_l_s = roads_feat.attributes()[vehicles_field_index['L_diu_l_s']]
                    if roads_layer_details['L_diu_h_s'] <> 'none':
                        L_diu_h_s = roads_feat.attributes()[vehicles_field_index['L_diu_h_s']]
                    if L_diu_type == "":
                        roads_feat.attributes()[vehicles_field_index['L_diu_type']]
                    #log_errors.write('\n'+str(L_diu_l_n)+'\n'+str(L_diu_h_n)+'\n'+str(L_diu_l_s)+'\n'+str(L_diu_h_s)+'\n'+L_diu_type+'\n'+surface+'\n'+slope)
                    power['diu'] = nmpb(L_diu_l_n,L_diu_h_n,L_diu_l_s,L_diu_h_s,L_diu_type,surface,slope).power()
                                 
                else:
                    power['diu'] = 0
                    
                if roads_layer_details['nig'] == True:
                    if roads_layer_details['L_nig_l_n'] <> 'none':
                        L_nig_l_n = roads_feat.attributes()[vehicles_field_index['L_nig_l_n']]
                    if roads_layer_details['L_nig_h_n'] <> 'none':
                        L_nig_h_n = roads_feat.attributes()[vehicles_field_index['L_nig_h_n']]
                    if roads_layer_details['L_nig_l_s'] <> 'none':
                        L_nig_l_s = roads_feat.attributes()[vehicles_field_index['L_nig_l_s']]
                    if roads_layer_details['L_nig_h_s'] <> 'none':
                        L_nig_h_s = roads_feat.attributes()[vehicles_field_index['L_nig_h_s']]
                    if L_nig_type == "":
                        roads_feat.attributes()[vehicles_field_index['L_nig_type']]
                    #log_errors.write('\n'+str(L_nig_l_n)+'\n'+str(L_nig_h_n)+'\n'+str(L_nig_l_s)+'\n'+str(L_nig_h_s)+'\n'+L_nig_type+'\n'+surface+'\n'+slope)
                    power['nig'] = nmpb(L_nig_l_n,L_nig_h_n,L_nig_l_s,L_nig_h_s,L_nig_type,surface,slope).power()
                else:
                    power['nig'] = 0   
                    
                if roads_layer_details['day'] == True:
                    if roads_layer_details['L_day_l_n'] <> 'none':
                        L_day_l_n = roads_feat.attributes()[vehicles_field_index['L_day_l_n']]
                    if roads_layer_details['L_day_h_n'] <> 'none':
                        L_day_h_n = roads_feat.attributes()[vehicles_field_index['L_day_h_n']]
                    if roads_layer_details['L_day_l_s'] <> 'none':
                        L_day_l_s = roads_feat.attributes()[vehicles_field_index['L_day_l_s']]
                    if roads_layer_details['L_day_h_s'] <> 'none':
                        L_day_h_s = roads_feat.attributes()[vehicles_field_index['L_day_h_s']]
                    if L_day_type == "":
                        roads_feat.attributes()[vehicles_field_index['L_day_type']]
                    power['day'] = nmpb(L_day_l_n,L_day_h_n,L_day_l_s,L_day_h_s,L_day_type,surface,slope).power()
                else:
                    power['day'] = 0 
                    
                if roads_layer_details['eve'] == True:
                    if roads_layer_details['L_eve_l_n'] <> 'none':
                        L_eve_l_n = roads_feat.attributes()[vehicles_field_index['L_eve_l_n']]
                    if roads_layer_details['L_eve_h_n'] <> 'none':
                        L_eve_h_n = roads_feat.attributes()[vehicles_field_index['L_eve_h_n']]
                    if roads_layer_details['L_eve_l_s'] <> 'none':
                        L_eve_l_s = roads_feat.attributes()[vehicles_field_index['L_eve_l_s']]
                    if roads_layer_details['L_eve_h_s'] <> 'none':
                        L_eve_h_s = roads_feat.attributes()[vehicles_field_index['L_eve_h_s']]
                    if L_eve_type == "":
                        roads_feat.attributes()[vehicles_field_index['L_eve_type']]
                    power['eve'] = nmpb(L_eve_l_n,L_eve_h_n,L_eve_l_s,L_eve_h_s,L_eve_type,surface,slope).power()
                else:
                    power['eve'] = 0 
            
            # implementation with power: gets road power from the fields        
            if roads_layer_details['implementation'] == 'power':
                if roads_layer_details['diu'] == True and roads_feat.attributes()[power_field_index['diu']] != None:
                    power['diu'] = roads_feat.attributes()[power_field_index['diu']]
                else:
                    power['diu'] = 0
                if roads_layer_details['nig'] == True and roads_feat.attributes()[power_field_index['nig']] != None:
                    power['nig'] = roads_feat.attributes()[power_field_index['nig']]
                else:
                    power['nig'] = 0
                if roads_layer_details['day'] == True and roads_feat.attributes()[power_field_index['day']] != None:
                    power['day'] = roads_feat.attributes()[power_field_index['day']]
                else:
                    power['day'] = 0
                if roads_layer_details['eve'] == True and roads_feat.attributes()[power_field_index['eve']] != None:
                    power['eve'] = roads_feat.attributes()[power_field_index['eve']]
                else:
                    power['eve'] = 0
            
            if power['diu'] == 0 and power['nig'] == 0 and power['day'] == 0 and power['eve'] == 0:
                continue
            
            # researches the receiver points in a rectangle created by the research_ray
            # creates the search rectangle
            rect = QgsRectangle()
            rect.setXMinimum( roads_feat.geometry().boundingBox().xMinimum() - research_ray )
            rect.setXMaximum( roads_feat.geometry().boundingBox().xMaximum() + research_ray )
            rect.setYMinimum( roads_feat.geometry().boundingBox().yMinimum() - research_ray )
            rect.setYMaximum( roads_feat.geometry().boundingBox().yMaximum() + research_ray )
        
            receiver_points_request = receiver_points_spIndex.intersects(rect)
            
            distance_min = []
            for receiver_points_id in receiver_points_request:
                receiver_points_feat = receiver_points_feat_all_dict[receiver_points_id]
                result = roads_feat.geometry().closestSegmentWithContext(receiver_points_feat.geometry().asPoint())
                distance_min_tmp = sqrt(result[0])
                
                if distance_min_tmp <= research_ray:
                    distance_min.append(distance_min_tmp)
        
            # defines segment max length
            if len(distance_min) >= 1:
                segment_max = min(distance_min)/2
                if segment_max < 2:
                    segment_max = 2
            else:
                continue
            
            # splits the road in emission points at a fix distance (minimum distance/2) and create the emission point layer
            # gets vertex
            roads_vertex_pt_all = roads_feat.geometry().asPolyline()
            emission_point_id_road = 0
            for i in range(0,len(roads_vertex_pt_all)):
                pt1 = QgsPoint(roads_vertex_pt_all[i])
                if emission_points_layer_path <> "":
                    self.add_point_to_layer(emission_points_writer,pt1,[emission_point_id, emission_point_id_road, roads_feat.id()])
                else:
                    self.add_point_to_memory_layer(emission_points_memory_layer,pt1,[emission_point_id, emission_point_id_road, roads_feat.id()])
                
                roads_powers[roads_feat.id()] = [power,segment_max]
                
                emission_point_id = emission_point_id + 1
                emission_point_id_road = emission_point_id_road + 1
                
                if i < len(roads_vertex_pt_all)-1:
                    
                    pt2 = QgsPoint(roads_vertex_pt_all[i+1])
        
                    x1 = pt1.x()
                    y1 = pt1.y()
                    x2 = pt2.x()
                    y2 = pt2.y()
                    
                    if y2 == y1:
                        dx = segment_max
                        dy = 0
                        m = 0
                    elif x2 == x1:
                        dx = 0
                        dy = segment_max
                    else:
                        m = ( y2 - y1 )/ ( x2 - x1 )
                        dx = sqrt((segment_max**2)/(1 + m**2))
                        dy = sqrt(((segment_max**2)*(m**2))/(1 + m**2))
                    
                    pt = pt1
                    
                    while self.compute_distance(pt,pt2) > segment_max:
                        x_temp = pt.x()
                        y_temp = pt.y()
                        if x_temp < x2:
                            if m > 0:
                                pt = QgsPoint(x_temp + dx, y_temp + dy) 
                            elif m < 0:
                                pt = QgsPoint(x_temp + dx, y_temp - dy)
                            elif m == 0:
                                pt = QgsPoint(x_temp + dx, y_temp)
                        elif x_temp > x2:
                            if m > 0:
                                pt = QgsPoint(x_temp - dx, y_temp - dy) 
                            elif m < 0:
                                pt = QgsPoint(x_temp - dx, y_temp + dy)                   
                            elif m == 0:
                                pt = QgsPoint(x_temp - dx, y_temp)   
                        elif x_temp == x2:
                            if y2 > y_temp:
                                pt = QgsPoint(x_temp, y_temp + dy) 
                            else:
                                pt = QgsPoint(x_temp, y_temp - dy) 
            
                        if emission_points_layer_path <> "":
                            self.add_point_to_layer(emission_points_writer,pt,[emission_point_id,emission_point_id_road, roads_feat.id()])
                        else:
                            self.add_point_to_memory_layer(emission_points_memory_layer,pt,[emission_point_id,emission_point_id_road, roads_feat.id()])                            
                        
                        emission_point_id = emission_point_id + 1
                        emission_point_id_road = emission_point_id_road + 1        
        
        
        return roads_powers        



    def noise_levels_computation(self,receiver_points_layer,roads_layer_details,emission_points_road_id_field_index,parameters,obstacles_layer,obstacles_spIndex,obstacles_feat_all_dict,emission_points_spIndex,emission_points_feat_all_dict,rays_layer_path,rays_writer,research_ray,roads_powers,level_field_index):
        
        self.step_label.setText("  STEP 2 / 2  ")
        self.progressBar.setValue(0)
        
        ray_id = 0
        # calculates levels fr each receiver point
        receiver_points_feat_all = receiver_points_layer.dataProvider().getFeatures()
        receiver_points_feat_all_number = 0
        
        receiver_point_field_level =  {}
        
        for receiver_points_feat in receiver_points_feat_all:

            receiver_points_feat_all_number = receiver_points_feat_all_number + 1
            bar = receiver_points_feat_all_number/float(receiver_points_feat_total)*100
            self.progressBar.setValue(bar)

            # initializes the receiver point field level
            field_level = {}
            
            # initializes the receiver point lin level
            receiver_point_lin_level = {}
            receiver_point_lin_level['diu'] = 0
            receiver_point_lin_level['nig'] = 0
            receiver_point_lin_level['day'] = 0
            receiver_point_lin_level['eve'] = 0
            receiver_point_lin_level['den'] = 0            
            
            # researches the emission points in a rectangle created by the research_ray
            # creates the search rectangle
            rect = QgsRectangle()
            rect.setXMinimum( receiver_points_feat.geometry().asPoint().x() - research_ray )
            rect.setXMaximum( receiver_points_feat.geometry().asPoint().x() + research_ray )
            rect.setYMinimum( receiver_points_feat.geometry().asPoint().y() - research_ray )
            rect.setYMaximum( receiver_points_feat.geometry().asPoint().y() + research_ray )
            
            emission_points_request = emission_points_spIndex.intersects(rect)
            #log_errors.write(str(roads_powers.keys())+'\n')
            for emission_points_id in emission_points_request:
                
                emission_points_feat = emission_points_feat_all_dict[emission_points_id] 
                
                road_id = emission_points_feat.attributes()[emission_points_road_id_field_index]
                
                #log_errors.write(str(receiver_points_feat.id()) + '\t' + str(emission_points_feat.id()) + '\t' + str(road_id)+'\n')
                power = roads_powers[road_id][0]
                segment_max = roads_powers[road_id][1]
                
                ray_to_test_length = self.compute_distance(receiver_points_feat.geometry().asPoint(),emission_points_feat.geometry().asPoint())

                if ray_to_test_length <= research_ray:

                    ray_to_test = QgsGeometry.fromPolyline( [ receiver_points_feat.geometry().asPoint() , emission_points_feat.geometry().asPoint() ] ) 

                    intersect = 0
                
                    if obstacles_layer <> "":
                        obstacles_request = obstacles_spIndex.intersects(ray_to_test.boundingBox())
                        for obstacles_id in obstacles_request:
                            if obstacles_feat_all_dict[obstacles_id].geometry().intersects(ray_to_test) ==1:
                                intersect = 1
                                break
                    
                    # if the ray is free from intersections it's added in the ray layer with the the sound level computation
                    if intersect == 0:

                        level = {}
                        level_lin = {}
                        
                        # lenght with receiver points height fixed to 4 m
                        ray_to_test_length_4m = sqrt(ray_to_test_length**2 + 16)
                        
                        if roads_layer_details['diu'] == True:
                            if power['diu'] > 0:
                                level['diu'] = power['diu'] + 20 + 10*log10(segment_max) - (20*log10(ray_to_test_length_4m)+11) + 3
                                level_lin['diu'] = 10**(level['diu']/10)
                            else:
                                level['diu'] = 0
                                level_lin['diu'] = 0
                            receiver_point_lin_level['diu'] = receiver_point_lin_level['diu'] + level_lin['diu']
                        if roads_layer_details['nig'] == True:
                            if power['nig'] > 0:
                                level['nig'] = power['nig'] + 20 + 10*log10(segment_max) - (20*log10(ray_to_test_length_4m)+11) + 3
                                level_lin['nig'] = 10**(level['nig']/10)
                            else:
                                level['nig'] = 0
                                level_lin['nig'] = 0
                            receiver_point_lin_level['nig'] = receiver_point_lin_level['nig'] + level_lin['nig']
                        if roads_layer_details['day'] == True:
                            if power['day'] > 0:
                                level['day'] = power['day'] + 20 + 10*log10(segment_max) - (20*log10(ray_to_test_length_4m)+11) + 3
                                level_lin['day'] = 10**(level['day']/10)
                            else:
                                level['day'] = 0
                                level_lin['day'] = 0
                            receiver_point_lin_level['day'] = receiver_point_lin_level['day'] + level_lin['day']
                        if roads_layer_details['eve'] == True:
                            if power['eve'] > 0:
                                 level['eve'] = power['eve'] + 20 + 10*log10(segment_max) - (20*log10(ray_to_test_length_4m)+11) + 3
                                 level_lin['eve'] = 10**(level['eve']/10)
                            else:
                                level['eve'] = 0
                                level_lin['eve'] = 0                                 
                            receiver_point_lin_level['eve'] = receiver_point_lin_level['eve'] + level_lin['eve']
                        if parameters['L_den'] == True:
                            if level.has_key('day') and level['day'] > 0:
                                day_part_ray = 10**(level['day'] / 10)
                            else:
                                day_part_ray = 0
                            if level.has_key('eve') and level['eve'] > 0:
                                eve_part_ray = 10**((level['eve'] + 5) / 10)
                            else:
                                eve_part_ray = 0
                            if level.has_key('nig') and level['nig'] > 0:
                                nig_part_ray = 10**((level['nig'] + 10)/ 10)
                            else:
                                nig_part_ray = 0
                            if day_part_ray == 0 and eve_part_ray == 0 and nig_part_ray == 0:
                                level['den'] = 0
                            else:
                                level['den'] = 10*log10(1/24.0*(14*day_part_ray + 2*eve_part_ray + 8*nig_part_ray))


                        if rays_layer_path <> "":
                            ray = QgsFeature()
                            ray.setGeometry(ray_to_test)
                            attributes = [ray_id, receiver_points_feat.id(), road_id, ray_to_test_length]
                            if roads_layer_details['diu'] == True:
                                attributes.append(power['diu'])
                                attributes.append(level['diu'])
                                attributes.append(level_lin['diu'])
                            if roads_layer_details['nig'] == True:
                                attributes.append(power['nig'])
                                attributes.append(level['nig'])
                                attributes.append(level_lin['nig'])                                
                            if roads_layer_details['day'] == True:
                                attributes.append(power['day'])
                                attributes.append(level['day'])
                                attributes.append(level_lin['day'])
                            if roads_layer_details['eve'] == True:
                                attributes.append(power['eve'])
                                attributes.append(level['eve'])
                                attributes.append(level_lin['eve'])
                            if parameters['L_den'] == True:  
                                attributes.append(level['den'])
                                
                            ray.setAttributes(attributes)
                            rays_writer.addFeature(ray)

                        ray_id = ray_id + 1
            
            
            if roads_layer_details['diu'] == True:
                if receiver_point_lin_level['diu'] > 0:
                    field_level[level_field_index['diu']] = 10*log10(receiver_point_lin_level['diu'])                
                else:
                    field_level[level_field_index['diu']] = -1
            if roads_layer_details['nig'] == True:
                if receiver_point_lin_level['nig'] > 0:
                    field_level[level_field_index['nig']] = 10*log10(receiver_point_lin_level['nig'])           
                else:
                    field_level[level_field_index['nig']] = -1                
            if roads_layer_details['day'] == True:
                if receiver_point_lin_level['day'] > 0:
                    field_level[level_field_index['day']] = 10*log10(receiver_point_lin_level['day'])
                else:
                    field_level[level_field_index['day']] = -1
            if roads_layer_details['eve'] == True:
                if receiver_point_lin_level['eve'] > 0:
                    field_level[level_field_index['eve']] = 10*log10(receiver_point_lin_level['eve'])
                else:
                    field_level[level_field_index['eve']] = -1

            if parameters['L_den'] == True:
                if receiver_point_lin_level['day'] > 0:
                    day_part = 10**(field_level[level_field_index['day']] / 10)
                else:
                    day_part = 0
                if receiver_point_lin_level['eve'] > 0:
                    eve_part = 10**((field_level[level_field_index['eve']] + 5) / 10)
                else:
                    eve_part = 0
                if receiver_point_lin_level['nig'] > 0:
                    nig_part = 10**((field_level[level_field_index['nig']] + 10)/ 10)
                else:
                    nig_part = 0
                if day_part == 0 and eve_part == 0 and nig_part == 0:
                    field_level[level_field_index['den']] = -1
                else:
                    field_level[level_field_index['den']] = 10*log10(1/24.0*(14*day_part + 2*eve_part + 8*nig_part))

            #if receiver_point_lin_level['diu'] > 0 or receiver_point_lin_level['nig'] > 0 or receiver_point_lin_level['day'] > 0 or receiver_point_lin_level['eve'] > 0:
            #    receiver_point_field_level[receiver_points_feat.id()] = field_level
            receiver_point_field_level[receiver_points_feat.id()] = field_level    

        if rays_layer_path <> "":
            del rays_writer
            rays_layer_name = unicode(ftools_utils.getShapefileName( rays_layer_path ))
            rays_layer = QgsVectorLayer(rays_layer_path, rays_layer_name, "ogr")
            QgsMapLayerRegistry.instance().addMapLayers([rays_layer])        
            
        return receiver_point_field_level            
            
        
    def run(self,receiver_points_layer,roads_layer,roads_layer_details,parameters,obstacles_layer,emission_points_layer_path,rays_layer_path):
        
        # gets vector layers, features amd creates SpatialIndex if need
        # receiver points        
        global receiver_points_feat_total 
        receiver_points_feat_total = receiver_points_layer.dataProvider().featureCount()
        receiver_points_feat_all = receiver_points_layer.dataProvider().getFeatures()
        receiver_points_spIndex = QgsSpatialIndex()
        receiver_points_feat_all_dict = {}
        for receiver_points_feat in receiver_points_feat_all:
            receiver_points_spIndex.insertFeature(receiver_points_feat)
            receiver_points_feat_all_dict[receiver_points_feat.id()] = receiver_points_feat        
        
        #research ray
        research_ray = parameters['research_ray']
        
        # obstacles
        if obstacles_layer <> "":
            obstacles_feat_all = obstacles_layer.dataProvider().getFeatures()
            obstacles_spIndex = QgsSpatialIndex()
            obstacles_feat_all_dict = {}
            for obstacles_feat in obstacles_feat_all:
                obstacles_spIndex.insertFeature(obstacles_feat)
                obstacles_feat_all_dict[obstacles_feat.id()] = obstacles_feat
        else:
            obstacles_spIndex = ""
            obstacles_feat_all_dict = ""
        
        # defines emission_points layer
        emission_points_fields = [QgsField("id_emi", QVariant.Int), QgsField("id_emi_road", QVariant.Int), QgsField("id_road", QVariant.Int)]
        if emission_points_layer_path <> "":
            emission_points_writer = VectorWriter(emission_points_layer_path, None, emission_points_fields, 0, receiver_points_layer.crs())
            emission_points_memory_layer = ""
        else:
            emission_points_writer = ""
            emission_points_memory_layer = QgsVectorLayer("Point?crs=" + str(receiver_points_layer.crs().authid()), "emission_points_temp", "memory")
            emission_points_memory_layer.dataProvider().addAttributes(emission_points_fields)
            
        # defines rays layer
        if rays_layer_path <> "":
            rays_fields = [QgsField("id_ray", QVariant.Int), QgsField("id_rec", QVariant.Int),
                           QgsField("id_road", QVariant.Int), QgsField("distance", QVariant.Double,len=10,prec=2)]
            if roads_layer_details['diu'] == True:
                rays_fields.append(QgsField("diu_p", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("diu", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("diu_lin", QVariant.Double,len=10,prec=1))
            if roads_layer_details['nig'] == True:
                rays_fields.append(QgsField("nig_p", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("nig", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("nig_lin", QVariant.Double,len=10,prec=1))
            if roads_layer_details['day'] == True:
                rays_fields.append(QgsField("day_p", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("day", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("day_lin", QVariant.Double,len=10,prec=1))
            if roads_layer_details['eve'] == True:
                rays_fields.append(QgsField("eve_p", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("eve", QVariant.Double,len=5,prec=1))
                rays_fields.append(QgsField("eve_lin", QVariant.Double,len=10,prec=1))
            if parameters['L_den'] == True:  
                rays_fields.append(QgsField("den", QVariant.Double,len=5,prec=1))
                 
            rays_writer = VectorWriter(rays_layer_path, None, rays_fields, 2, receiver_points_layer.crs())
        else:
            rays_writer = ""
        
        # gets fields from point layer and initializes the final receiver_point_field_level to populate the receiver points layer attribute table
        level_field_index = {}
        fields_number = int(receiver_points_layer.dataProvider().fields().count())
        if roads_layer_details['diu'] == True:            
            level_field_index['diu'] = fields_number
            fields_number = fields_number + 1
        if roads_layer_details['nig'] == True:            
            level_field_index['nig'] = fields_number
            fields_number = fields_number + 1
        if roads_layer_details['day'] == True:            
            level_field_index['day'] = fields_number
            fields_number = fields_number + 1
        if roads_layer_details['eve'] == True:            
            level_field_index['eve'] = fields_number
            fields_number = fields_number + 1
        if parameters['L_den'] == True:            
            level_field_index['den'] = fields_number
            fields_number = fields_number + 1

        # creates emission point layer and get roads powers info
        roads_powers = self.emission_points_creation(roads_layer,roads_layer_details,research_ray,receiver_points_feat_all_dict,receiver_points_spIndex,emission_points_layer_path,emission_points_writer,emission_points_memory_layer)

        # gets features amd creates SpatialIndex for emission points
        emission_points_feat_all_dict = {}
        if emission_points_layer_path <> "":
            del emission_points_writer
            emission_points_layer_name = unicode(ftools_utils.getShapefileName( emission_points_layer_path ))
            emission_points_layer = QgsVectorLayer(emission_points_layer_path, emission_points_layer_name, "ogr")
            
            emission_points_feat_all = emission_points_layer.dataProvider().getFeatures()
            emission_points_road_id_field_index = emission_points_layer.fieldNameIndex("id_road")
            
            QgsMapLayerRegistry.instance().addMapLayers([emission_points_layer])
        
        else:
            emission_points_feat_all = emission_points_memory_layer.getFeatures()    
            emission_points_road_id_field_index = emission_points_memory_layer.fieldNameIndex("id_road")
        
        emission_points_spIndex = QgsSpatialIndex()            
        for emission_points_feat in emission_points_feat_all:
            emission_points_spIndex.insertFeature(emission_points_feat)
            emission_points_feat_all_dict[emission_points_feat.id()] = emission_points_feat

        # creates rays and calculates noise levels
        receiver_point_field_level = self.noise_levels_computation(receiver_points_layer,roads_layer_details,emission_points_road_id_field_index,parameters,obstacles_layer,obstacles_spIndex,obstacles_feat_all_dict,emission_points_spIndex,emission_points_feat_all_dict,rays_layer_path,rays_writer,research_ray,roads_powers,level_field_index)
        
        # puts the sound level in the receivers points attribute table
        level_fields = []
        if roads_layer_details['diu'] == True:
            level_fields.append(QgsField('diu', QVariant.Double,len=5,prec=1))
        if roads_layer_details['nig'] == True:
            level_fields.append(QgsField('nig', QVariant.Double,len=5,prec=1))
        if roads_layer_details['day'] == True:
            level_fields.append(QgsField('day', QVariant.Double,len=5,prec=1))
        if roads_layer_details['eve'] == True:
            level_fields.append(QgsField('eve', QVariant.Double,len=5,prec=1))
        if parameters['L_den'] == True:  
            level_fields.append(QgsField('den', QVariant.Double,len=5,prec=1))
                
        receiver_points_layer.dataProvider().addAttributes( level_fields )
        receiver_points_layer.updateFields()
        receiver_points_layer.dataProvider().changeAttributeValues(receiver_point_field_level)  
        
        # render with noise colours          
        level_fields_new = ftools_utils.getFieldList(receiver_points_layer)
        if len(level_fields_new) > 0:
            render(receiver_points_layer,level_fields_new[len(level_fields_new)-1].name())
        