# -*- coding: utf-8 -*-
"""
/***************************************************************************
 OGFlushPlugin
                                 A QGIS plugin
 This plugin creates a sewer network from geodata
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-17
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Arbeitsgruppe Siedlungshydrologie, Institut für Siedlungs- und Industriewasserwirtschaft, Technische Universität Dresden
        email                : diego.novoa_vazquez@tu-dresden.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from PyQt5.QtWidgets import QMessageBox, QLabel
from qgis.core import QgsProject, QgsMapLayer, QgsWkbTypes, Qgis, QgsVectorLayer, QgsRasterLayer
from PyQt5.QtCore import Qt, QUrl

# import processing
import locale, sys
import os, shutil

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .og_flush_dialog import OGFlushPluginDialog


class OGFlushPlugin:
                     
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'OGFlushPlugin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&OG.FLUSH')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('OGFlushPlugin', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/og_flush/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Design a Sewer Network'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&OG.FLUSH'),
                action)
            self.iface.removeToolBarIcon(action)

#%%

    """Main functions to populate dropboxes in GUI"""
    def select_output_file(self):
        directory_url = QFileDialog.getExistingDirectoryUrl(
            self.dlg, "Select output directory", QUrl())

        # Extracting the full path from the QUrl object
        directory_path = directory_url.toLocalFile() if directory_url.isLocalFile() else ""

        self.dlg.output_name.setText(directory_path)
            
    
    def select_input_raster(self,combobox_name):
        filters = "Raster Files (*.tif *.asc *.bin);;All Files (*)"
        filename, _filter = QFileDialog.getOpenFileName(
                     self.dlg, "Select input raster ","", filters)
        # Add the selected filename to combobox
        if filename:
            # Add the selected filename to the given combobox
            combobox_name.addItem(filename)
            # Obtain index of newly-added item
            index = combobox_name.findText(filename)
            # Set the combobox to select the new item
            combobox_name.setCurrentIndex(index)
            
    def select_input_shape(self,combobox_name):
        
        filename, _filter = QFileDialog.getOpenFileName(
          self.dlg, "Select input raster ","", '*.shp')
        # Add the selected filename to combobox
        if filename:
            
            # Add the selected filename to the given combobox
            combobox_name.addItem(filename)
            # Obtain index of newly-added item
            index = combobox_name.findText(filename)
            # Set the combobox to select the new item
            combobox_name.setCurrentIndex(index)        
    
    def add_selected_raster_layer(self, combobox_name):
        
        filename = combobox_name.currentText()

      
        raster_layer = QgsRasterLayer(filename, os.path.basename(filename), "gdal")
        if raster_layer.isValid():
            QgsProject.instance().addMapLayer(raster_layer)

            return raster_layer

    def add_selected_shape_layer(self, combobox_name):

        filename = combobox_name.currentText()


        vector_layer = QgsVectorLayer(filename, os.path.basename(filename), "ogr")
        if vector_layer.isValid():
            QgsProject.instance().addMapLayer(vector_layer)

            return vector_layer          

    def connect_rasterbox_and_button(self, combobox, button):
        # button.clicked.connect(lambda: self.add_selected_raster_layer(combobox))
        button.clicked.connect(lambda: self.select_input_raster(combobox))

    def connect_shapebox_and_button(self, combobox, button):
        # button.clicked.connect(lambda: self.add_selected_shape_layer(combobox))
        button.clicked.connect(lambda: self.select_input_shape(combobox))
    
    '''Controls for GUI'''
    def show_error_message(self, message):
        QMessageBox.critical(None, 'Error', message, QMessageBox.Ok)
        
    def cleanup_temp_files(self, temp_files, temp_folder):
        """Clean up temporary files and folders"""
        try:
            # Remove temporary files
            for file in temp_files:
                if os.path.exists(file):
                    os.remove(file)
    
            # Remove temporary folder
            if os.path.exists(temp_folder):
                shutil.rmtree(temp_folder)
    
            print("Temporary files and folder removed.")
        except Exception as e:
            print(f"Error while cleaning up temporary files: {str(e)}")
    
    def reset_progress(self):
        """Reset progress bar and clear all dialog fields and references"""
        # Reset progress bar
        self.dlg.progressBar.setValue(0)
        self.dlg.progressBar.setVisible(False)
        self.processing_cancelled = False
        
        # Force UI update for the end
        QCoreApplication.processEvents()
        self.run_completed = True
    
    def cancel_processing(self):
        """Set the cancel flag when user clicks cancel"""
        self.processing_cancelled = True
        QMessageBox.information(None, "Processing Cancelled", 
                               "Processing has been cancelled by user.")
    
    """Check if required dependencies are available"""
    def check_dependencies(self):
        missing_deps = []
        try:
            import numpy as np
        except ImportError:
            missing_deps.append("numpy")
        try:
            import pandas as pd
        except ImportError:
            missing_deps.append("pandas")
        try:
            import networkx as nx
        except ImportError:
            missing_deps.append("networkx")
        
        return missing_deps

                                                                     
#%%
    def run_button_clicked(self, rr):
        '''Main function'''
        # Reset progress bar
        self.reset_progress()
        
        # Disable the run button
        self.dlg.Run_pb.setEnabled(False)
        
        try:
            # Initialize progress bar
            self.dlg.progressBar.setVisible(True)
            
            # Set to determinate mode
            self.dlg.progressBar.setMinimum(0)
            self.dlg.progressBar.setMaximum(100)
            self.dlg.progressBar.setValue(0) 
            
            # Apply QGIS-like style for the progress bar
            self.dlg.progressBar.setStyleSheet("""
                QProgressBar {
                    border: 1px solid #999999;
                    border-radius: 3px;
                    background-color: #f0f0f0;
                    text-align: center;
                    color: #333333;
                    font-weight: bold;
                }
                QProgressBar::chunk {
                    background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                        stop:0 #4CAF50, stop:0.5 #66BB6A, stop:1 #4CAF50);
                    border-radius: 2px;
                }
            """)
            
            QCoreApplication.processEvents()  # Initial UI update
            
            # Check dependencies before processing
            missing_deps = self.check_dependencies()
            if missing_deps:
                lib_list = ", ".join(missing_deps)
                error_message = f"""Missing Required Dependencies: {lib_list}
                
                This plugin requires the following Python libraries to function:
                - numpy
                - pandas  
                - networkx
                
                Please install these dependencies following the installation guide in the README and try again.
                The instructions can be found in the plugin documentation by following the link showed on the plugin UI."""
                QMessageBox.critical(None, 'Missing Dependencies', error_message, QMessageBox.Ok)
                self.reset_progress()
                
                # Hopefully graceful exit
                return
            
            # Set a flag for cancellation
            self.processing_cancelled = False
            
            # Existing parameter collection
            depth                   = self.dlg.pipe_depth.value()
            min_slope               = self.dlg.min_slope_pipe.value()
            min_slope_buildings     = self.dlg.build_min_sl.value()
            min_slope_streets       = self.dlg.st_min_slope.value()
            thr                     = self.dlg.dist_thresh.value()
            max_slope_buildings     = self.dlg.build_max_slope.value()
            max_slope_streets       = self.dlg.st_max_sl.value()
            z                       = self.dlg.return_period.value()
            r15                     = self.dlg.r15_rain.value()
            Tfmin                   = self.dlg.tf_min.value()
            fgw                     = self.dlg.consumption_perc.value()
            annual_pop_growth       = self.dlg.pop_grow.value()
            consumption             = self.dlg.consumption.value()
            years_pop               = self.dlg.years_growth.value()
            meshness_perc           = self.dlg.meshness_perc.value()
            
            min_slope = min_slope *0.01
            qs=(consumption/86400)
            
            combined    = self.dlg.mixed_cb.isChecked()
            gw_inf      = self.dlg.gw_cb.isChecked()
            
            if (gw_inf==False):
                fgw = 0
            
            '''Define main folders'''
            sys.path.append(os.path.dirname(os.path.abspath(__file__)))
            sys.path.append(os.path.dirname(os.path.realpath(__file__)))
            os.chdir(os.path.dirname(os.path.realpath(__file__)))   
            sys.path.append(os.path.dirname(os.path.abspath(__file__)))
            
            '''Import NumPy and custom function scripts'''
            import numpy as np
            from . import functions_SG0 as sg0
            from . import functions_SG1 as sg1
            from . import functions_SG2 as sg2
            from . import functions_SG3 as sg3
            from . import functions_SG4 as sg4
            from . import functions_SG5 as sg5
            
            '''Identify loaded layers to populate the fields in GUI'''
            layers = QgsProject.instance().mapLayers().values()
            raster_layers = [layer for layer in layers 
                             if layer.type() == QgsMapLayer.RasterLayer]
            raster_layers = [''] + raster_layers
       
                    
            vector_layers = [layer for layer in layers 
                             if layer.type() == QgsMapLayer.VectorLayer]
            
            point_layers = [layer for layer in vector_layers 
                            if layer.geometryType() == QgsWkbTypes.PointGeometry]
            point_layers = ['']+point_layers
                    
            line_layers = [layer for layer in vector_layers 
                            if layer.geometryType() == QgsWkbTypes.LineGeometry]
            line_layers = ['']+line_layers
            poly_layers = [layer for layer in vector_layers 
                            if layer.geometryType() == QgsWkbTypes.PolygonGeometry]
            poly_layers = ['']+poly_layers
            
            
            def sel_layer_r (box, layerlist):
                index = box.currentIndex()
    
                if (index < len(layerlist)):
                    sel = list(layerlist)[index]
                    
                else:
                    sel = self.add_selected_raster_layer(box)
    
                return sel
            
            def sel_layer_s (box, layerlist):
                index = box.currentIndex()
    
                if (index < len(layerlist)):
                    sel = list(layerlist)[index]
                else:
                    sel = self.add_selected_shape_layer(box)
                return sel
    
            
            dem             = sel_layer_r(self.dlg.dembox, raster_layers)
            streets_osm     = sel_layer_s(self.dlg.streetbox, line_layers)
            buildings       = sel_layer_s(self.dlg.buildbox, poly_layers)
            study_area      = sel_layer_s(self.dlg.catchbox, poly_layers)
            outlet          = sel_layer_s(self.dlg.outletbox, point_layers)
            output          = self.dlg.output_name.text()   
            
            if not dem:
                
                self.show_error_message("Please select a DEM layer.")
                return
    
            if not streets_osm:
                self.show_error_message("Please select a Street layer.")
                return
            if not buildings:
                self.show_error_message("Please select a Building layer.")
                return
            if not study_area:
                self.show_error_message("Please select a Study Area/Catchment layer.")
                return
            if not outlet:
                self.show_error_message("Please select a Outlet layer.")
                return
    
     
            if combined == True:
                pop_density = sel_layer_s(self.dlg.popbox, point_layers)
                if not pop_density:
                    self.show_error_message("Please select a population density layer.")
                    return
            if not output:
                self.show_error_message("Please select a location for your outputs.")
                return
        
            
            ''' IDentify all CRSs used and use the most common one for all opetrations'''
            path_shapefiles = output
    
            crs = buildings.crs()
            epsg = crs.postgisSrid()
            
            crs = study_area.crs()
            epsg = [epsg,crs.postgisSrid()]
             
            crs = streets_osm.crs()
            epsg = epsg+[crs.postgisSrid()]
            
            crs = outlet.crs()
            epsg = epsg+[crs.postgisSrid()]
            
            crs = dem.crs()
            epsg = epsg+[crs.postgisSrid()]
            
            epsg = [x for x in epsg if x > 0]
            
            epsg = max(set(epsg), key=epsg.count)
            
            print(f"EPSG: {epsg}")
            
            '''Create group for outputs'''
            group_name = "Sewer_Network_Data"
            root = QgsProject.instance().layerTreeRoot()
            group = root.findGroup(group_name)
            if not group:
                group = root.addGroup(group_name)
    
            
    #%%
            
            '''Section 0:
                Check the validity of the outlet locations with respect to the study area boundary'''
            
            # Set or reset validation check flag now before the main section runs every time
            buffer_threshold = 30 # in metres
            
            # sg0.validate_outlet_distance(outlet, study_area, buffer_threshold)
            
            # Validate the distance only if it's a new run, and because this is the simplest method that could work
            distance = sg0.calculate_outlet_distance(outlet, study_area)
            if distance > buffer_threshold:
                raise Exception(f"Outlet too far: {distance:.2f}m from boundary (max: {buffer_threshold}m)")
            
            '''Section 1:
                Identify outlet and snap it to most suitable location
                Identify the relevant streets and extract intersections
                Create a Minimum Spanning Tree'''
            #---------------------------------------------------------------------------------------------------------------
            
            snap_out, temp_connected_streets = sg1.clip_streets(streets_osm, study_area, outlet)
            
            connected_streets_nodes, connected_streets, out_final = sg1.vertex_extraction(temp_connected_streets, dem,
                                                                     depth, snap_out)
            
            MST_streetsX = sg1.create_mst(out_final, connected_streets, connected_streets_nodes, epsg)
            
            MST_streets = sg1.MST_streets2(MST_streetsX, connected_streets_nodes)
            
            
            self.dlg.progressBar.setValue(10)
            QCoreApplication.processEvents()
            
            
            '''Section 2:
                Correct the direction of the pipes to ensure gravitational flow to outlet
                Correction of invert elevations and slopes'''
            #---------------------------------------------------------------------------------------------------------------
            
            # Extract df from shapefiles
            df_links = sg1.shapefile_to_dataframe(MST_streets)
            df_nodes = sg1.shapefile_to_dataframe(connected_streets_nodes)
            df_outlet = sg1.shapefile_to_dataframe(out_final)
            
            del snap_out, temp_connected_streets
            
            # Correct direction and elevation
            corr2 = sg1.correct_dir2(df_links, df_outlet)
            
            df_links, df_nodes = sg1.corr_elev_new(df_links, df_nodes, corr2, min_slope, df_outlet)
            
            # Update attributes after correction to nodes and pipes
            selected_columns = ['LINK_ID', 'FROM_node','TO_node']
            corr_dir = df_links[selected_columns].values
            
            base_sewer = sg1.get_base_sewer(MST_streets, corr_dir)
            
            selected_columns = ['NODE_ID', 'x','y','invert_elev']
            inv_elev = df_nodes[selected_columns].values
            
            base_sewer_nodes = sg1.get_sewer_nodes(inv_elev, connected_streets_nodes)
            
            outfall = sg1.update_outlet(out_final, base_sewer_nodes)
            
            
            self.dlg.progressBar.setValue(25)
            QCoreApplication.processEvents()
            
            
            '''Section 3:
                Determine the contributing areas
                Connect the areas to the approporaite outlet
                Estimate dry weather flow'''
            #---------------------------------------------------------------------------------------------------------------
            
            # Set maximum and minimum slopes within a dictionary 
            slope_limits = {
                'min_slope_buildings': min_slope_buildings,
                'max_slope_buildings': max_slope_buildings,
                'min_slope_streets': min_slope_streets,
                'max_slope_streets': max_slope_streets
            }
            
            # Process buildings and connect to the network
            connected_buildings = sg2.process_buildings(
                buildings, 
                study_area, 
                base_sewer, 
                base_sewer_nodes, 
                dem,
                slope_limits, 
                thr,
                'BU'
                )
            
            sg1.save_to_disk(connected_buildings, os.path.join(path_shapefiles,'Connected_Buildings.shp'))
            
            connected_buildings = QgsVectorLayer(os.path.join(path_shapefiles, 'Connected_Buildings.shp'), "Connected_Buildings", "ogr")
            
            QgsProject.instance().addMapLayer(connected_buildings, False)
            group.addLayer(connected_buildings)
            
            
            # Process streets and connect to the network
            connected_street_areas = sg2.process_streets(
                connected_streets,
                base_sewer_nodes, 
                dem,
                slope_limits, 
                'ST'
            )
            
            sg1.save_to_disk(connected_street_areas, os.path.join(path_shapefiles,'Connected_Street_Areas.shp'))
            
            connected_street_areas = QgsVectorLayer(os.path.join(path_shapefiles, 'Connected_Street_Areas.shp'), "Connected_Street_Areas", "ogr")
                    
            QgsProject.instance().addMapLayer(connected_street_areas, False)
            group.addLayer(connected_street_areas)
            
            
            # Calculate dry weather flow if the network is a combined system, using population density
            if combined==1:
                population_density = sg2.process_population(
                    pop_density,
                    connected_buildings,
                    annual_pop_growth,
                    years_pop
                )
            
            
            self.dlg.progressBar.setValue(50)
            QCoreApplication.processEvents()
            
            
            '''Section 4:
                Extract the necessary information for designing pipes (links) and subcatchments in Section 5'''
            #---------------------------------------------------------------------------------------------------------------
            
            # Create arrays from input shapefiles to design the system
            pipes_array = sg2.shapefile_to_array(base_sewer)
            nodes_array = sg2.shapefile_to_array(base_sewer_nodes)
            connected_buildings_array = sg2.shapefile_to_array(connected_buildings)
            connected_street_areas_array = sg2.shapefile_to_array(connected_street_areas)
            
            max_depth = nodes_array[:, 3] - nodes_array[:, 4]
            new_nodes_array = np.column_stack((nodes_array, max_depth))
            
            # Assign the old array with a new name for nodes to keep life simple
            nodes_array = node_design_data = new_nodes_array
            
            
            # Get the pipe information
            pipe_design_data = sg2.process_pipe_data(
                pipes_array,
                nodes_array,
                os.path.join(path_shapefiles, 'pipe_data_design.csv')
            )
            
            # Load population data if combined system
            pop_data = None
            if combined:
                pop_data = sg2.shapefile_to_array(population_density)
            
            # Get the subcatchments information
            subcatchment_design_data = sg2.process_subcatchments(
                connected_buildings_array,
                connected_street_areas_array,
                slope_limits,
                combined,
                pop_data
            )
            
            # Get the data (for deisgn) as dataframes and optionally save them to drive
            nodedf, pipedf, subdf = sg2.design_data(
                node_design_data,
                pipe_design_data,
                subcatchment_design_data,
                path_shapefiles,
                save_csv=True
            )
            
            
            self.dlg.progressBar.setValue(70)
            QCoreApplication.processEvents()
            
            
            '''Section 5:
                Design the pipes (links)'''
            #---------------------------------------------------------------------------------------------------------------
            
            number_it = 1
            head, pipe_dim, subdf2 =  sg3.pipe_dim_prep(pipedf,subdf,sg3.psit,r15,qs,fgw,Tfmin,z,number_it)
            number_it = number_it+1
    
            # Last iteration now was first iteration
            lastiter=head
    
            pipe_dim = sg3.pipe_dimension(pipe_dim,lastiter,subdf2,Tfmin,z,r15,number_it)
            pipe_dim = pipe_dim.rename(columns={'D':'Diameter'})
            
            pipe_dim.to_csv(os.path.join(path_shapefiles, 'Total_results_gw.csv'),
                          index=True)  
            
            
            pipe_dim2 = pipe_dim.drop(columns=['cumQ','Qtot','it','sqdw','cuml','qdw','qgw','sqgw','Qmax','sumtf','okqdw','cumd'])
            
            head = head.rename(columns={'D':'Diameter'})
            
            base_sewer_joined = sg1.join_designed_data(base_sewer, pipe_dim2)
            
            self.dlg.progressBar.setValue(90)
            QCoreApplication.processEvents()
            
            
            '''Section 6:
                Add extra pipes into the primary design network (optional)
                Finalise the main designed network, with nodes and outfall'''
            #---------------------------------------------------------------------------------------------------------------
            
            # Create a shapefile with skipped street features sorted to add meshness, or with None
            if meshness_perc > 0:
                # Extract the skipped streets with the necessary fields
                skipped_streets = sg4.extract_skipped_streets(connected_streets, MST_streets)
                skipped_streets_with_fields = sg1.MST_streets2(skipped_streets, connected_streets_nodes)
                
                # Get the skipped links
                skipped_links = sg4.correct_node_elevations(skipped_streets_with_fields, base_sewer_nodes)
                
                # Add slope, diameter and Q
                skipped_links_with_more_info = sg4.add_further_data_to_links(skipped_links, pipe_dim, head)
                
                # Sort and assign a rank
                skipped_links_sorted = sg4.sort_features(skipped_links_with_more_info)
            else:
                skipped_links_sorted = None
            
            # Get the final designed netwrok with the additional links as meshness option
            designed_sewer_network = sg4.add_skipped_links_to_designed(base_sewer_joined, skipped_links_sorted, meshness_perc)
            sg1.save_to_disk(designed_sewer_network, os.path.join(path_shapefiles,'Designed_Sewer_Network.shp'))
            
            # Save the nodes and the outfall
            sg1.save_to_disk(base_sewer_nodes, os.path.join(path_shapefiles,'Base_Sewer_Nodes.shp'))
            sg1.save_to_disk(outfall, os.path.join(path_shapefiles,'Outfall.shp'))
            
            # Load the nodes, outfall and the designed network on QGIS
            base_sewer_nodes= QgsVectorLayer(os.path.join(path_shapefiles, 'base_sewer_Nodes.shp'), 
                                             "Base_Sewer_Nodes", "ogr")
            outfall = QgsVectorLayer(os.path.join(path_shapefiles, 'Outfall.shp'), "Outfall", "ogr")
            designed_sewer_network= QgsVectorLayer(os.path.join(path_shapefiles, 'Designed_Sewer_Network.shp'), 
                                             "Designed_Sewer_Network", "ogr")
            
            QgsProject.instance().addMapLayer(base_sewer_nodes, False)
            QgsProject.instance().addMapLayer(outfall, False)
            QgsProject.instance().addMapLayer(designed_sewer_network, False)
            
            group.addLayer(base_sewer_nodes)
            group.addLayer(outfall)
            group.addLayer(designed_sewer_network)
            
            # Save the results to CSV
            pipe_result = sg2.shapefile_to_dataframe(designed_sewer_network)
            pipe_result.to_csv(os.path.join(path_shapefiles, 'pipe_result.csv'), index=False)
            
            
            self.dlg.progressBar.setValue(95)
            QCoreApplication.processEvents()
            
            
            '''Section 7:
                Create a SWMM model (.inp) file based on the dsigned network'''
            #---------------------------------------------------------------------------------------------------------------
            
            links_array = pipe_result.to_numpy()
            outfall_array = sg2.shapefile_to_array(outfall)
            
            inp_config = {
                'nodes_array': nodes_array,
                'links_array': links_array,
                'outfall_array': outfall_array,
                'building_layer': connected_buildings,
                'street_layer': connected_street_areas,
                'model_path': path_shapefiles,
                'min_slope_buildings': min_slope_buildings,
                'max_slope_buildings': max_slope_buildings,
                'min_slope_streets': min_slope_streets,
                'max_slope_streets': max_slope_streets
            }
            
            output_file_path = sg5.create_swmm_model(inp_config)
            
            if output_file_path:
                print(f"SWMM model file successfully created at {output_file_path}")
            else:
                print("Falied to write SWMM model file. Check logs for details.")
            
            
            self.dlg.progressBar.setValue(100)
            QCoreApplication.processEvents()
            
            # Write success message
            self.iface.messageBar().pushMessage(
                    "Success", "Output files succesfully created",
                    level=Qgis.Success, duration=3)
            print("Processing has been completed.")
        
        except Exception as e:
            # Handle any errors that occur during processing
            error_msg = f"An error occurred during processing: {str(e)}"
            self.show_error_message(error_msg)
            print(f"Error in run_button_clicked: {str(e)}")
        
        finally:
            # Re-enable the run button
            self.dlg.Run_pb.setEnabled(True)
            
        return
        

#%%

    def run(self):
        """Run method that performs all the real work"""
        locale.setlocale(locale.LC_ALL, 'en_US.UTF-8')

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = OGFlushPluginDialog()
            # Connect another comboboxes with corresponding pushbutons
            
            label = self.dlg.findChild(QLabel, "label_5")
            if label:
                label.setOpenExternalLinks(True)
                label.setTextInteractionFlags(Qt.TextBrowserInteraction)
            
            self.dlg.pb_output.clicked.connect(self.select_output_file)
   
            self.connect_rasterbox_and_button(self.dlg.dembox, self.dlg.pb_DEM)
            self.connect_shapebox_and_button(self.dlg.popbox, self.dlg.pb_pop)
            
            self.connect_shapebox_and_button(self.dlg.buildbox, self.dlg.pb_build)
            self.connect_shapebox_and_button(self.dlg.streetbox, self.dlg.pb_st)
            self.connect_shapebox_and_button(self.dlg.catchbox, self.dlg.pb_catch)
            self.connect_shapebox_and_button(self.dlg.outletbox, self.dlg.pb_out)
            
            # Add the run button connection here, to ensure only one connection during the first time plugin opens
            self.dlg.Run_pb.clicked.connect(self.run_button_clicked)
            
               
        # Fetch the currently loaded layers
        layers = QgsProject.instance().mapLayers().values()
        raster_layers = [layer for layer in layers 
                         if layer.type() == QgsMapLayer.RasterLayer]
                
        vector_layers = [layer for layer in layers 
                         if layer.type() == QgsMapLayer.VectorLayer]
        
        point_layers = [layer for layer in vector_layers 
                        if layer.geometryType() == QgsWkbTypes.PointGeometry]
        
        line_layers = [layer for layer in vector_layers 
                        if layer.geometryType() == QgsWkbTypes.LineGeometry]
        
        poly_layers = [layer for layer in vector_layers 
                        if layer.geometryType() == QgsWkbTypes.PolygonGeometry]
        
        
        raster_name = [layer.name() for layer in layers 
                         if layer.type() == QgsMapLayer.RasterLayer]
        raster_name = ['']+raster_name
        
        point_name = [layer.name() for layer in vector_layers 
                        if layer.geometryType() == QgsWkbTypes.PointGeometry]
        point_name = ['']+point_name
        
        line_name = [layer.name() for layer in vector_layers 
                        if layer.geometryType() == QgsWkbTypes.LineGeometry]
        line_name = ['']+line_name
        
        poly_name = [layer.name() for layer in vector_layers 
                        if layer.geometryType() == QgsWkbTypes.PolygonGeometry]
        poly_name = ['']+poly_name

        # Clear the contents of the comboBox from previous runs
        self.dlg.dembox.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.dembox.addItems(raster_name)
        
        self.dlg.buildbox.clear()
        self.dlg.buildbox.addItems(poly_name)
        
        self.dlg.streetbox.clear()
        self.dlg.streetbox.addItems(line_name)
        
        self.dlg.catchbox.clear()
        self.dlg.catchbox.addItems(poly_name)
        
        self.dlg.outletbox.clear()
        self.dlg.outletbox.addItems(point_name)
        
        self.dlg.popbox.clear()
        self.dlg.popbox.addItems(point_name)
        
        # # Disconnect any existing connections first, then connect
        # try:
        #     self.dlg.Run_pb.clicked.disconnect(self.run_button_clicked)
        # except TypeError:
        #     pass  # No existing connections to disconnect
        # self.dlg.Run_pb.clicked.connect(self.run_button_clicked)
        
        # show the dialog
        self.dlg.show()
        
        # Run the dialog event loop
        result = self.dlg.exec_()
        
        # See if OK was pressed
        if result:
            pass
