# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Offsetor
                                 A QGIS plugin
 Calculates points from the offset positions off a base line
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-06
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Alexander von Lunen
        email                : avl1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QMetaType, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem, QFileDialog
from qgis.core import *
from qgis.gui import QgsMapLayerComboBox, QgsFeaturePickerWidget

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .offsetor_dialog import OffsetorDialog
import os.path
import csv
import logging
import math


class Offsetor:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'Offsetor_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Offsetor')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # for debugging
        #formatter = '%(message)s'
        #logfilename=r'/home/avl/tmp/pyqgs.log'
        #logging.basicConfig(filename=logfilename, level=logging.DEBUG, format=formatter)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Offsetor', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/offsetor/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Offset points'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

        self.map_layers = QgsProject.instance().mapLayers().values()
        self.allow_list = [
                                lyr.id() for lyr in self.map_layers if lyr.crs().isGeographic() == False
                                and lyr.geometryType()== QgsWkbTypes.PointGeometry
                            ]
        self.except_list = [l for l in self.map_layers if l.id() not in self.allow_list]

        self.pointA = None
        self.pointB = None


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Offsetor'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = OffsetorDialog()

        # helper functions
        def magnitude(x1,y1):
            return math.sqrt(x1*x1+y1*y1)

        def normalize_vector(v_x,v_y):
            u_x = v_x / magnitude(v_x,v_y)
            u_y = v_y / magnitude(v_x,v_y)
            return(u_x,u_y)

        def find_point(v_x,v_y,d,u):
            p_x = v_x - (u[0]*d)
            p_y = v_y - (u[1]*d)
            return (p_x,p_y)

        def calc_ortho(a_x,a_y,v_x,v_y):
            v_x -= a_x
            v_y -= a_y
            u_x = 1
            u_y = -v_x*(1/v_y)
            return (u_x,u_y)

        # define event handlers
        def onButtonA():
            feat = self.dlg.m_feature.feature()
            if self.pointB != feat: # only allow selection if not selected for the other point
                self.pointA = feat
                self.dlg.coordA.setText(str(feat.id()))

        def onButtonB():
            feat = self.dlg.m_feature.feature()
            if self.pointA != feat: # only allow selection if not selected for the other point
                self.pointB = feat
                self.dlg.coordB.setText(str(feat.id()))

        def onLayerChanged(lyr):
            self.dlg.m_feature.setLayer(lyr)
            self.dlg.m_feature.setDisplayExpression('$id')
            self.pointA = None
            self.pointB = None
            self.dlg.coordA.clear()
            self.dlg.coordB.clear()

        def onImport():
            filename, _filter = QFileDialog.getOpenFileName(self.dlg, "Select   import file ","", '*.csv')
            if not filename:
                return
            
            self.dlg.m_distances.clearContents()

            y = 0
            with open(filename, "r") as fileInput:
                for row in csv.reader(fileInput):
                #logging.debug(','.join(row))
                    if y > 0:
                        self.dlg.m_distances.insertRow(y)
                    item_id = QTableWidgetItem(row[0])
                    item_a = QTableWidgetItem(row[1])
                    item_b = QTableWidgetItem(row[2])
                    self.dlg.m_distances.setItem(y, 0, item_id)
                    self.dlg.m_distances.setItem(y, 1, item_a)
                    self.dlg.m_distances.setItem(y, 2, item_b)
                    y+=1
            
        def onAddRow():
            n = self.dlg.m_distances.rowCount()
            self.dlg.m_distances.insertRow(n)

        def onRemoveRow():
            l = self.dlg.m_distances.currentRow()
            self.dlg.m_distances.removeRow(l)

        # set up the UI elements
        self.dlg.m_layer.setExceptedLayerList(self.except_list)
        layer = self.dlg.m_layer.currentLayer()
        picker = self.dlg.m_feature
        picker.setLayer(layer)
        picker.setDisplayExpression('$id') #show ids in combobox
        self.dlg.m_layer.layerChanged.connect(onLayerChanged)
        self.dlg.buttonA.clicked.connect(onButtonA)
        self.dlg.buttonB.clicked.connect(onButtonB)
        self.dlg.m_distances.setRowCount(1)
        self.dlg.m_distances.setColumnCount(3)
        self.dlg.m_distances.setHorizontalHeaderLabels(["Id", "Base line", "Offset"])
        self.dlg.m_import.clicked.connect(onImport)
        self.dlg.buttonAdd.clicked.connect(onAddRow)
        self.dlg.buttonRemove.clicked.connect(onRemoveRow)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()

        # See if OK was pressed
        if result:
            if self.pointA == None or self.pointB == None:
               self.iface.messageBar().pushMessage("Error", "Points A/B not set!", level=Qgis.Critical, duration=3)

               return

            startpoint = self.pointA.geometry()
            endpoint = self.pointB.geometry()
            x1_orig = startpoint.asPoint().x()
            y1_orig = startpoint.asPoint().y()
            x2_orig = endpoint.asPoint().x()
            y2_orig = endpoint.asPoint().y()
            i = 0

            left = self.dlg.leftradio.isChecked()

            # create memory layer, to add the trilaterated points to
            memory_layer = QgsVectorLayer("Point", "Offset", "memory")
            memory_layer.setCrs(layer.crs()) # set the CRS to the same as the source layer
            mr = memory_layer.dataProvider()
            mr.addAttributes([QgsField(name="id", type=QVariant.Int)])
            memory_layer.updateFields()

            while i < self.dlg.m_distances.rowCount():
                if self.dlg.m_distances.item(i, 0) == None:
                    self.iface.messageBar().pushMessage("Error", "ID not set!", level=Qgis.Critical, duration=3)
                    return
                else:          
                    iid = int(self.dlg.m_distances.item(i, 0).text())

                if self.dlg.m_distances.item(i, 1) == None:
                    self.iface.messageBar().pushMessage("Error", "Distance on baseline not set!", level=Qgis.Critical, duration=3)
                    return
                else:          
                    idist = float(self.dlg.m_distances.item(i, 1).text())

                if self.dlg.m_distances.item(i, 2) == None:
                    self.iface.messageBar().pushMessage("Error", "Offset not set!", level=Qgis.Critical, duration=3)
                    return
                else:          
                    ioff = float(self.dlg.m_distances.item(i, 2).text())

                if not left:
                    ioff *= -1

                u = normalize_vector(x1_orig-x2_orig,y1_orig-y2_orig)
                p = find_point(x1_orig,y1_orig,idist,u)
                o = calc_ortho(x1_orig,y1_orig,x2_orig,y2_orig)
                ou = normalize_vector(o[0],o[1])
                c = find_point(p[0],p[1],ioff,ou)

                # add offset point to memory layer, using the id from the row in the table
                ft = QgsFeature()
                if self.dlg.basePoints.isChecked() == True:
                    ft.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(p[0],p[1])))
                    ft.setAttributes([iid])
                    mr.addFeatures([ft])
                ft.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(c[0],c[1])))
                ft.setAttributes([iid])
                mr.addFeatures([ft])
                i+=1


            # add memory layer to project
            QgsProject.instance().addMapLayer(memory_layer)
            memory_layer.updateExtents()

