# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ODKTrace2WKT
                                 A QGIS plugin
 This plugin converts ODK geotraces to Well-Known Text
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-09-07
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Samweli Mwakisambwe
        email                : smwltwesa6@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

from qgis.core import *
from qgis.utils import iface
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .odktrace2wkt_dialog import ODKTrace2WKTDialog
from collections import defaultdict

from .qgis_utilities import (
    display_warning_message_box,
    display_information_message_box,
    temp_dir,
    unique_filename,
    enable_busy_cursor,
    disable_busy_cursor
)

import os.path

import csv
import re
import tempfile


class ODKTrace2WKT:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ODKTrace2WKT_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&ODKTrace2WKT')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ODKTrace2WKT', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/odktrace2wkt/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Convert geotrace to Well-Known Text'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&ODKTrace2WKT'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation)
        # and keep reference
        # Only create GUI ONCE in callback, so that it will only
        # load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg = ODKTrace2WKTDialog()

        # Initialize delimiters values

        self.delimiters = [";", r"\t", ",", ":"]

        # If it is the second time gui load,
        # detect again the delimiter
        if self.dlg.inputQgsFileWidget.filePath() != '':
            delimiter = self.detect_csv_delimiter(
                self.dlg.inputQgsFileWidget.filePath()
            )
            delimiter_index = self.delimiters.index(delimiter)
        else:
            delimiter_index = 0

        # Set the delimiter in the combo box
        self.dlg.delimiter_combo_box.clear()
        self.dlg.delimiter_combo_box.addItems(
            self.delimiters)
        self.dlg.delimiter_combo_box.setCurrentIndex(
            delimiter_index
        )

        self.dlg.outputQgsFileWidget.lineEdit().\
            setPlaceholderText(
            "[Save to Temporary File]"
            )

        # Track changes in file path edits
        self.dlg.inputQgsFileWidget.fileChanged.connect(
            self.set_inputs
        )
        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            if self.dlg.outputQgsFileWidget.filePath() is '':
                tmpdir = temp_dir('output')
                tmpfile = unique_filename(dir=tmpdir, suffix='.csv')
                output = tmpfile
            else:
                output = self.dlg.outputQgsFileWidget.filePath()
            try:
                enable_busy_cursor()

                self.main(
                    self.dlg.inputQgsFileWidget.filePath(),
                    None,
                    self.dlg.column_combo_box.currentText(),
                    output
                )

                self.loadFile(
                    output,
                    self.dlg.delimiter_combo_box.currentText(),
                    self.dlg.column_combo_box.currentText()
                )

                disable_busy_cursor()
                display_information_message_box(
                    self.dlg,
                    self.dlg.tr('Success'),
                    self.dlg.tr('Output created at "{}"'.format(output)))

            except Exception as exception:

                disable_busy_cursor()
                display_warning_message_box(
                    self.dlg,
                    self.dlg.tr('Error'),
                    self.dlg.tr('{}'.format(str(exception))))


    def set_inputs(self):
        """ Set the delimiter and location column detected in the file
        """
        try:

            path = self.dlg.inputQgsFileWidget.filePath()

            delimiter = self.detect_csv_delimiter(path)

            self.dlg.delimiter_combo_box.setCurrentIndex(
               self.delimiters.index(delimiter))

            fields, column, index = \
                self.detect_location_column(
                    path,
                    delimiter
                )

            self.dlg.column_combo_box.clear()
            self.dlg.column_combo_box.addItems(fields)
            self.dlg.column_combo_box.setCurrentIndex(index)

        except Exception as exception:
            pass

    def detect_csv_delimiter(self, csv_file):
        """Takes a csv file and returns its delimiter, uses Sniffer library"""

        # Initializing delimiter with a default value
        delimiter = ','

        with open(csv_file, 'r') as csvfile:
            dialect = csv.Sniffer().sniff(csvfile.readline())
            delimiter = dialect.delimiter

        return delimiter

    def detect_location_column(self, csv_file, delimiter):
        """Takes a csv file from odk  and returns column
        which contains geotraces values
        """

        pattern_reg = r'\b([-+]?)([\d]{1,2})(((\.)(\d+)))(\s*)'\
                      r'([-+]?)([\d]{1,3})(((\.)(\d+)))(\s*)'\
                      r'([-+]?)([\d]{1,5})(((\.)(\d+)))(\s*)'\
                      r'([-+]?)([\d]{1,2})(((\.)(\d+)))\b'

        pattern = re.compile(pattern_reg)

        column = ''

        with open(csv_file) as csvfile:
            reader = csv.reader(csvfile, delimiter=delimiter)

            header = next(reader)
            # Using default dict to initialize keys with
            # zero default value
            header_count = defaultdict(int)

            # Search in the first five rows only.
            rows = 5

            for row in reader:
                index = 0
                if rows <= 0:
                    break
                for value in row:
                    match = pattern.search(value)
                    if match:
                        header_count[header[index]] += 1
                    index += 1
                rows -= 1

            column = max(
                header_count.keys(),
                key=(lambda k: header_count[k])) if header_count else ''

            # Default to first field if no column is found
            index = 0 if column == '' else header.index(column)

        return header, column, index

    def main(self, infile, column,
             column_name, output=None):
        """Iterates through a CSV and writes a CSV with
        converted linestrings.
        """

        # Avoid choking the CSV library with a long linestring
        csv.field_size_limit(100000000)

        # Get the chosen delimiter
        delimiter = self.dlg.delimiter_combo_box.currentText()

        with open(infile) as line_data:
            reader = csv.reader(line_data, delimiter=delimiter)
            of = output if output else \
                '{}_{}.csv'.format(infile, '_results')
            with open(of, 'w') as outfile:
                writer = csv.writer(outfile, delimiter=delimiter)
                header = next(reader)
                colindex = int(column) - 1 if column \
                    else header.index(column_name)
                writer.writerow(header)

                for row in reader:
                    node_string = ''.join(row[colindex])
                    outrow = row
                    outrow[colindex] = \
                        self.wkt_linestring_from_nodes(node_string)
                    writer.writerow(outrow)

    def wkt_linestring_from_nodes(self, node_string):
        """Takes a string of arbitrarily long strings separated by semicolons
        where the first two items in the string are expected to be lat and long.
        Returns a string containing those coordinates as a Well-Known Text
        linestring (with long first and lat second, therefore x,y).
        """
        nodes = node_string.split(';')
        if nodes:
            WKT_type = "LINESTRING" if len(nodes) > 1 else "POINT"
            coord_pair_list = []
            for node in nodes:
                coords = node.strip().split()
                if len(coords) >= 2:
                    # can be >2 incl elev & precision values
                    # Reverse coords; Lon first, then Lat (as per WKT spec)
                    coord_pair = '{} {}'.format(coords[1], coords[0])
                    coord_pair_list.append(coord_pair)
            line_coord_string = ', '.join(coord_pair_list)
            linestring = '{}({})'.format(WKT_type, line_coord_string)
            return linestring
        else:
            return None

    def loadFile(self, path, delimiter, column):
        """ Load csv file into qgis
        """
        if not path.startswith('/'):
            path = '/' + path
        output_file_uri = 'file://{}?delimiter={}&crs=EPSG:4326&wktField={}'.\
            format(
                path,
                delimiter,
                column
            )

        vlayer = QgsVectorLayer(output_file_uri, 'Output file', 'delimitedtext')

        iface.addVectorLayer(output_file_uri, 'Output Layer', 'delimitedtext')
        iface.mapCanvas().refresh()
