# -*- coding: utf-8 -*-
import sys
import os
import unittest

# Add thirdparty and plugin root to path
# Assumes this test file is in <plugin_root>/test/
test_dir = os.path.dirname(os.path.abspath(__file__))
plugin_dir = os.path.dirname(test_dir)
thirdparty_dir = os.path.join(plugin_dir, 'thirdparty')

if thirdparty_dir not in sys.path:
    sys.path.insert(0, thirdparty_dir)
if plugin_dir not in sys.path:
    sys.path.insert(0, plugin_dir)

try:
    from sql_where_graphql_converter import SqlWhereGraphQLConverter
except ImportError:
    # Try creating the file if it can't be imported, but we expect it to exist
    raise

class TestSqlWhereGraphQLConverter(unittest.TestCase):
    def setUp(self):
        self.converter = SqlWhereGraphQLConverter()

    def convert(self, sql):
        return self.converter.sql_where_to_graphql(sql)

    def test_basic_equality(self):
        # "field" = 'value'
        res = self.convert('"name" = \'John\'')
        self.assertEqual(res, {'name': {'equals': 'John'}})

    def test_basic_comparison(self):
        # > < >= <= !=
        self.assertEqual(self.convert('"age" > 18'), {'age': {'greater_than': 18}})
        self.assertEqual(self.convert('"age" < 18'), {'age': {'less_than': 18}})
        self.assertEqual(self.convert('"age" >= 18'), {'age': {'greater_than_or_equals': 18}})
        self.assertEqual(self.convert('"age" <= 18'), {'age': {'less_than_or_equals': 18}})
        self.assertEqual(self.convert('"age" != 18'), {'age': {'not_equals': 18}})

    def test_likes(self):
        # LIKE 'val' -> equals if no %
        self.assertEqual(self.convert('"name" LIKE \'John\''), {'name': {'equals': 'John'}})
        # ILIKE 'val' -> iequals if no %
        self.assertEqual(self.convert('"name" ILIKE \'John\''), {'name': {'iequals': 'John'}})
        
        # LIKE 'val%' -> starts_with
        self.assertEqual(self.convert('"name" LIKE \'Jo%\''), {'name': {'starts_with': 'Jo'}})
        # LIKE '%val' -> ends_with
        self.assertEqual(self.convert('"name" LIKE \'%hn\''), {'name': {'ends_with': 'hn'}})
        # LIKE '%val%' -> like (implied contains) - Wait, code implementation check:
        # If starts_pct and ends_pct: 'LIKE' mapped to 'like'
        self.assertEqual(self.convert('"name" LIKE \'%oh%\''), {'name': {'like': 'oh'}})

    def test_not_likes(self):
        self.assertEqual(self.convert('"name" NOT LIKE \'John\''), {'name': {'not_equals': 'John'}})
        self.assertEqual(self.convert('"name" NOT LIKE \'Jo%\''), {'name': {'not_starts_with': 'Jo'}})

    def test_is_null(self):
        self.assertEqual(self.convert('"x" IS NULL'), {'x': {'is_null': True}})
        self.assertEqual(self.convert('"x" IS NOT NULL'), {'x': {'is_not_null': True}})
        self.assertEqual(self.convert('"x" IS EMPTY'), {'x': {'is_empty': True}})
        self.assertEqual(self.convert('"x" IS NOT EMPTY'), {'x': {'is_not_empty': True}})

    def test_in(self):
        # IN (1, 2)
        res = self.convert('"id" IN (1, 2)')
        self.assertEqual(res, {'id': {'in': [1, 2]}})
        
        # NOT IN
        res = self.convert('"id" NOT IN (1, 2)')
        self.assertEqual(res, {'id': {'not_in': [1, 2]}})

    def test_logical_and(self):
        res = self.convert('"a" = 1 AND "b" = 2')
        # Structure depends on implementation, likely {'AND': [...]}
        self.assertTrue('AND' in res)
        self.assertEqual(len(res['AND']), 2)

    def test_logical_or(self):
        res = self.convert('"a" = 1 OR "b" = 2')
        self.assertTrue('OR' in res)
        self.assertEqual(len(res['OR']), 2)

    def test_literal_first(self):
        # 'value' = "field" -> "field" = 'value'
        self.assertEqual(self.convert("'John' = \"name\""), {'name': {'equals': 'John'}})
        # 18 < "age" -> "age" > 18
        self.assertEqual(self.convert("18 < \"age\""), {'age': {'greater_than': 18}})

    def test_composite_wildcard(self):
        # 'a%b' -> composite
        res = self.convert('"x" LIKE \'a%b\'')
        self.assertTrue('AND' in res)
        self.assertEqual(len(res['AND']), 2)
        # Verify content
        # Expecting [{'x': {'starts_with': 'a'}}, {'x': {'ends_with': 'b'}}]
        items = res['AND']
        self.assertEqual(items[0]['x']['starts_with'], 'a')
        self.assertEqual(items[1]['x']['ends_with'], 'b')

if __name__ == '__main__':
    unittest.main()
