import sys
from unittest.mock import MagicMock, patch

# Helper to mock modules if they don't exist
def mock_module_if_missing(mod_name, **api):
    if mod_name not in sys.modules:
        try:
            __import__(mod_name)
        except ImportError:
            m = MagicMock()
            for k, v in api.items():
                setattr(m, k, v)
            sys.modules[mod_name] = m
            # Also mock parts if dot notation
            parts = mod_name.split('.')
            if len(parts) > 1:
                parent = sys.modules.get('.'.join(parts[:-1]))
                if parent:
                    setattr(parent, parts[-1], m)

# Mock QGIS environment
mock_module_if_missing('qgis')
mock_module_if_missing('qgis.core')
mock_module_if_missing('qgis.PyQt')
mock_module_if_missing('qgis.PyQt.QtCore', QSettings=MagicMock())
mock_module_if_missing('qgis.PyQt.QtWidgets')

# Mock requests
mock_module_if_missing('requests')

# Mock obm_connect.config
# We need to ensure we can import from it
if 'obm_connect' not in sys.modules and 'obm_connect.config' not in sys.modules:
    try:
        import obm_connect.config
    except ImportError:
         # Mocking obm_connect package structure
         m_pkg = MagicMock()
         sys.modules['obm_connect'] = m_pkg
         
         m_conf = MagicMock()
         m_conf.build_oauth_url = lambda x: x + "/oauth"
         m_conf.DEFAULT_OBM_SERVER = "http://default"
         sys.modules['obm_connect.config'] = m_conf
         m_pkg.config = m_conf

import unittest
from datetime import datetime, timedelta

# Import auth_manager
try:
    from auth_manager import ObmAuthManager
except ImportError:
    # If strictly running as script in test/ dir
    import os
    sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))
    from auth_manager import ObmAuthManager

class TestObmAuthManager(unittest.TestCase):
    def setUp(self):
        self.parent_mock = MagicMock()
        # Mock tr (translate) to return the string itself
        self.parent_mock.tr.side_effect = lambda x: x
        self.parent_mock.DEBUG = False
        self.auth_manager = ObmAuthManager(self.parent_mock)

    def test_initial_state(self):
        self.assertIsNone(self.auth_manager.get_access_token())
        self.assertFalse(self.auth_manager.is_token_valid())

    def test_set_tokens_manually(self):
        self.auth_manager._access_token = "fake_token"
        # expires in the past
        self.auth_manager._token_expires_time = datetime.now() - timedelta(seconds=10)
        self.assertFalse(self.auth_manager.is_token_valid())

        # expires in the future
        self.auth_manager._token_expires_time = datetime.now() + timedelta(seconds=100)
        self.assertTrue(self.auth_manager.is_token_valid())

    @patch('auth_manager.requests.post')
    def test_password_grant_success(self, mock_post):
        # Mock successful response
        mock_response = MagicMock()
        mock_response.status_code = 200
        mock_response.json.return_value = {
            "access_token": "new_access_token",
            "refresh_token": "new_refresh_token",
            "expires_in": 3600
        }
        mock_post.return_value = mock_response

        # Mock QSettings to avoid writing to real registry/ini
        with patch('auth_manager.QSettings') as mock_settings_class:
            mock_settings = mock_settings_class.return_value
            mock_settings.value.return_value = None
            
            # Ensure instance state is clean
            self.auth_manager._access_token = None
            self.auth_manager._refresh_token = None

            success = self.auth_manager.ensure_valid_token("http://server", "user@example.com", "password")
            
            self.assertTrue(success)
            self.assertEqual(self.auth_manager.get_access_token(), "new_access_token")
            self.assertEqual(self.auth_manager._refresh_token, "new_refresh_token")
            self.assertTrue(self.auth_manager.is_token_valid())
            
            # Verify request
            mock_post.assert_called_once()
            args, kwargs = mock_post.call_args
            self.assertIn("data", kwargs)
            self.assertEqual(kwargs['data']['grant_type'], 'password')
            self.assertEqual(kwargs['data']['username'], 'user@example.com')

    @patch('auth_manager.requests.post')
    def test_password_grant_failure(self, mock_post):
        mock_response = MagicMock()
        mock_response.status_code = 401
        mock_response.text = "Unauthorized"
        mock_post.return_value = mock_response

        with patch('auth_manager.QSettings') as mock_settings_class:
            mock_settings = mock_settings_class.return_value
            mock_settings.value.return_value = None
            
            # Ensure instance state is clean
            self.auth_manager._access_token = None
            self.auth_manager._refresh_token = None
            
            success = self.auth_manager.ensure_valid_token("http://server", "user@example.com", "wrong_pass")
            self.assertFalse(success)
            self.assertIsNone(self.auth_manager.get_access_token())

    def test_ensure_valid_token_uses_existing_valid_token(self):
        self.auth_manager._access_token = "valid_existing"
        self.auth_manager._token_expires_time = datetime.now() + timedelta(seconds=3600)
        
        with patch('auth_manager.requests.post') as mock_post:
            success = self.auth_manager.ensure_valid_token("http://server", "u", "p")
            self.assertTrue(success)
            mock_post.assert_not_called()

    @patch('auth_manager.requests.post')
    def test_refresh_token_flow(self, mock_post):
        refresh_token = "existing_refresh_token"
        self.auth_manager.set_refresh_token(refresh_token)
        self.auth_manager._access_token = None # Clear this to force a flow
        
        mock_response = MagicMock()
        mock_response.status_code = 200
        mock_response.json.return_value = {
            "access_token": "refreshed_access_token",
            "refresh_token": "refreshed_refresh_token",
            "expires_in": 3600
        }
        mock_post.return_value = mock_response

        with patch('auth_manager.QSettings') as mock_settings_class:
            mock_settings = mock_settings_class.return_value
            # Define side effect to return our refresh token when requested
            def settings_side_effect(key, default=None, type=None):
                if key == "obm_connect/refresh_token":
                    return refresh_token
                return None
            mock_settings.value.side_effect = settings_side_effect

            success = self.auth_manager.ensure_valid_token("http://server", "u", "p")
            self.assertTrue(success)
            self.assertEqual(self.auth_manager.get_access_token(), "refreshed_access_token")
            self.assertEqual(self.auth_manager._refresh_token, "refreshed_refresh_token")
            
            args, kwargs = mock_post.call_args
            self.assertEqual(kwargs['data']['grant_type'], 'refresh_token')
            self.assertEqual(kwargs['data']['refresh_token'], 'existing_refresh_token')

    @patch('auth_manager.requests.post')
    def test_refresh_token_fails_then_password_grant(self, mock_post):
        self.auth_manager._access_token = None
        refresh_token = "bad_refresh_token"
        self.auth_manager.set_refresh_token(refresh_token)

        # First call (refresh) fails
        resp_fail = MagicMock()
        resp_fail.status_code = 400
        
        # Second call (password) succeeds
        resp_success = MagicMock()
        resp_success.status_code = 200
        resp_success.json.return_value = {
            "access_token": "password_access_token",
            "expires_in": 3600
        }

        mock_post.side_effect = [resp_fail, resp_success]

        with patch('auth_manager.QSettings') as mock_settings_class:
            mock_settings = mock_settings_class.return_value
            # Define side effect to return our refresh token when requested
            def settings_side_effect(key, default=None, type=None):
                if key == "obm_connect/refresh_token":
                    return refresh_token
                return None
            mock_settings.value.side_effect = settings_side_effect
            
            success = self.auth_manager.ensure_valid_token("http://server", "u", "p")
            
            self.assertTrue(success)
            self.assertEqual(self.auth_manager.get_access_token(), "password_access_token")
            self.assertEqual(mock_post.call_count, 2)

    def test_clear_tokens(self):
        self.auth_manager._access_token = "t"
        self.auth_manager._refresh_token = "r"
        with patch('auth_manager.QSettings') as mock_settings:
            self.auth_manager.clear_tokens()
            self.assertIsNone(self.auth_manager.get_access_token())
            self.assertIsNone(self.auth_manager._refresh_token)

if __name__ == '__main__':
    unittest.main()
