# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
import requests
from qgis.core import Qgis
from qgis.PyQt.QtCore import QSettings
from obm_connect.config import build_oauth_url, DEFAULT_OBM_SERVER, SCOPE_GET_DATA, SCOPE_WEBPROFILE

class ObmAuthManager:
    """
    Handles OBM authentication logic (password grant and refresh token).
    """
    def __init__(self, parent_widget):
        self.parent = parent_widget
        self._access_token = None
        self._refresh_token = None
        self._token_expires_time = datetime.now()

    def get_access_token(self):
        return self._access_token

    def is_token_valid(self):
        return self._access_token and datetime.now() < self._token_expires_time

    def ensure_valid_token(self, server_url, email, password):
        """
        Ensure there is a valid access token.
        1. Attempt to use existing access_token (verify via network call).
        2. Attempt refresh with refresh_token.
        3. Perform password grant if others fail.
        Returns True on success, False otherwise.
        """
        # 1. If we have a token in memory and it's not expired, use it.
        if self.is_token_valid():
            return True

        # 2. Try to load from settings if memory is empty
        if not self._access_token:
            settings = QSettings()
            self._access_token = settings.value("obm_connect/access_token")
            self._refresh_token = settings.value("obm_connect/refresh_token")
            exp = settings.value("obm_connect/expires_at")
            if exp:
                try:
                    self._token_expires_time = datetime.fromisoformat(exp)
                except Exception:
                    # If we can't parse it or it's missing, keep it as 'now' which will trigger refresh
                    pass
            
        # 3. If token is still invalid (missing or expired), try refresh
        if not self.is_token_valid():
            token_url = build_oauth_url(server_url)
            client_id = "mobile"
            client_secret = "123"

            if self._refresh_token:
                self.parent.iface.messageBar().pushMessage(self.parent.tr("OBM Connect"), self.parent.tr("Token expired, attempting refresh."))
                if self._refresh_token_flow(token_url, client_id, client_secret):
                    return True

            # 4. Password grant fallback
            self.parent.iface.messageBar().pushMessage(self.parent.tr("OBM Connect"), self.parent.tr("Session expired, please log in again."))
            return self._password_grant_flow(token_url, client_id, client_secret, email, password, server_url)

        return True

    def _refresh_token_flow(self, token_url, client_id, client_secret):
        try:
            payload = {"grant_type": "refresh_token", "refresh_token": self._refresh_token}
            resp = requests.post(token_url, data=payload, auth=(client_id, client_secret), headers={"Accept": "application/json"}, timeout=15)
            
            if resp.status_code == 200:
                self._handle_token_response(resp.json())
                self.parent.iface.messageBar().pushMessage(self.parent.tr("OBM Connect"), self.parent.tr("Access token refreshed."))
                if self.parent.DEBUG:
                    self.parent.iface.messageBar().pushMessage("DEBUG", "Refresh token: {token}".format(token=self._refresh_token))
                return True
            else:
                self.parent.iface.messageBar().pushMessage(self.parent.tr("OAuth"), self.parent.tr("Refresh token is invalid or expired."), duration=5)
                return False
        except Exception as exc:
            self.parent.iface.messageBar().pushCritical(self.parent.tr("OAuth"), self.parent.tr("Refresh token error: {err}").format(err=str(exc)))
            return False

    def _password_grant_flow(self, token_url, client_id, client_secret, email, password, server_url):
        if not password:
            self._show_password_error()
            return False

        scopes = SCOPE_GET_DATA + " " + SCOPE_WEBPROFILE
        payload = {"grant_type": "password", "username": email, "password": password, "scope": scopes}
        
        try:
            resp = requests.post(token_url, data=payload, auth=(client_id, client_secret), headers={"Accept": "application/json"}, timeout=15)
            if resp.status_code == 200:
                self._handle_token_response(resp.json())
                self.persist_tokens(server_url, email)
                self.parent.iface.messageBar().pushMessage(self.parent.tr("OBM Connect"), self.parent.tr("Access token obtained."))
                if self.parent.DEBUG:
                    self.parent.iface.messageBar().pushMessage("DEBUG", "Scopes: {scopes} ----- Access token received: {token}".format(scopes=scopes, token=self._access_token))
                return True
            else:
                self.parent.iface.messageBar().pushCritical(self.parent.tr("OAuth"), self.parent.tr("HTTP error: {code} - {text}").format(code=resp.status_code, text=resp.text))
                return False
        except Exception as exc:
            self.parent.iface.messageBar().pushCritical(self.parent.tr("OAuth"), self.parent.tr("Exception: {err}").format(err=str(exc)))
            return False

    def _handle_token_response(self, data):
        access = data.get("access_token")
        refresh = data.get("refresh_token", self._refresh_token)
        expires_in = int(data.get("expires_in", 3600))
        
        if access:
            self._access_token = access
            self._refresh_token = refresh
            self._token_expires_time = datetime.now() + timedelta(seconds=expires_in - 1)

    def persist_tokens(self, server, email):
        try:
            remember = False
            if hasattr(self.parent, "rememberMeCheckBox"):
                remember = bool(self.parent.rememberMeCheckBox.isChecked())
            
            settings = QSettings()
            settings.setValue("obm_connect/remember", remember)
            
            if remember:
                settings.setValue("obm_connect/access_token", self._access_token)
                settings.setValue("obm_connect/refresh_token", self._refresh_token)
                settings.setValue("obm_connect/server", server)
                settings.setValue("obm_connect/email", email)
                if self._token_expires_time:
                    settings.setValue("obm_connect/expires_at", self._token_expires_time.isoformat())
        except Exception:
            pass

    def _show_password_error(self):
        try:
            self.parent.iface.messageBar().pushMessage("Error", self.parent.tr("Please provide the password."), level=Qgis.Critical, duration=4)
        except Exception:
            from qgis.PyQt.QtWidgets import QMessageBox
            QMessageBox.warning(self.parent, "Error", self.parent.tr("Please provide the password."))
        
        if hasattr(self.parent, "passwordLineEdit"):
            try:
                self.parent.passwordLineEdit.setFocus()
                self.parent.passwordLineEdit.setStyleSheet("border: 1px solid #d9534f;")
            except Exception:
                pass

    def clear_tokens(self):
        self._access_token = None
        self._refresh_token = None
        self._token_expires_time = datetime.now()
        
        try:
            settings = QSettings()
            settings.remove("obm_connect/access_token")
            settings.remove("obm_connect/refresh_token")
            settings.remove("obm_connect/email")
            settings.remove("obm_connect/server")
            settings.remove("obm_connect/expires_at")
            settings.setValue("obm_connect/remember", False)
        except Exception:
            pass

    def set_refresh_token(self, token):
        self._refresh_token = token
