# python
# test_obm_connect_filter.py

import os
os.environ["QT_QPA_PLATFORM"] = "offscreen"

import unittest
from obm_connect_filter import SqlWhereGraphQLConverter

from qgis.core import QgsApplication
from PyQt5.QtWidgets import QApplication
import sys

# Qt alkalmazás példányosítása (ha nincs még)
app = QApplication.instance() or QApplication(sys.argv)

# QGIS alkalmazás példányosítása
qgs = QgsApplication([], False)
qgs.initQgis()

class ObmConnectFilterFactory:
    @staticmethod
    def make():
        return SqlWhereGraphQLConverter()

class TestBuildGraphQLFilter(unittest.TestCase):
    def setUp(self):
        self.dlg = ObmConnectFilterFactory.make()

    def test_operator_mappings_basic(self):
        d = self.dlg.build_graphql_filter("name", "=", "John")
        self.assertEqual(d, {"name": {"equals": "John"}})

        d = self.dlg.build_graphql_filter("age", ">=", 18)
        self.assertEqual(d, {"age": {"greater_than_or_equals": 18}})

        d = self.dlg.build_graphql_filter("score", "<", 3.14)
        self.assertEqual(d, {"score": {"less_than": 3.14}})

        d = self.dlg.build_graphql_filter("flag", "=", True)
        self.assertEqual(d, {"flag": {"equals": True}})

    def test_like_ilike_not_ilike(self):
        self.assertEqual(self.dlg.build_graphql_filter("notes", "LIKE", "%foo%"), {"notes": {"like": "%foo%"}})
        self.assertEqual(self.dlg.build_graphql_filter("desc", "ILIKE", "Bar%"), {"desc": {"ilike": "Bar%"}})
        self.assertEqual(self.dlg.build_graphql_filter("title", "NOT ILIKE", "baz%"), {"title": {'not_istarts_with': 'baz'}})

    def test_in_and_not_in_variants(self):
        # list input
        self.assertEqual(self.dlg.build_graphql_filter("status", "IN", ["a", "b"]), {"status": {"in": ["a", "b"]}})
        # string input should be split
        self.assertEqual(self.dlg.build_graphql_filter("nums", "IN", "(1,2)"), {"nums": {"in": [1,2]}})
        # not in
        self.assertEqual(self.dlg.build_graphql_filter("x", "NOT IN", [1,2]), {"x": {"not_in": [1,2]}})

    def test_starts_ends_and_null_empty(self):
        self.assertEqual(self.dlg.build_graphql_filter("s", "STARTS_WITH", "pre"), {"s": {"starts_with": "pre"}})
        self.assertEqual(self.dlg.build_graphql_filter("s", "ENDS_WITH", "suf"), {"s": {"ends_with": "suf"}})
        self.assertEqual(self.dlg.build_graphql_filter("g", "IS NULL", None), {"g": {"is_null": True}})
        self.assertEqual(self.dlg.build_graphql_filter("g", "IS NOT NULL", None), {"g": {"is_not_null": True}})
        self.assertEqual(self.dlg.build_graphql_filter("e", "IS EMPTY", None), {"e": {"is_empty": True}})
        self.assertEqual(self.dlg.build_graphql_filter("e", "IS NOT EMPTY", None), {"e": {"is_not_empty": True}})

class TestSqlWhereToGraphqlParsing(unittest.TestCase):
    def setUp(self):
        self.dlg = ObmConnectFilterFactory.make()

    def assertSql(self, where_expr, expected):
        got = self.dlg.sql_where_to_graphql(where_expr)
        self.assertEqual(got, expected, msg=f"WHERE: {where_expr}\nGOT: {got}\nEXPECTED: {expected}")

    def test_simple_types(self):
        # string (quoted)
        self.assertSql('"name" = \'John\'', {"name": {"equals": "John"}})
        # int
        self.assertSql('age >= 18', {"age": {"greater_than_or_equals": 18}})
        # float
        self.assertSql('score < 3.14', {"score": {"less_than": 3.14}})
        # boolean
        self.assertSql('active = true', {"active": {"equals": True}})
        self.assertSql('active = FALSE', {"active": {"equals": False}})
        # date/time as quoted strings
        self.assertSql("\"the_date\" = '2020-01-01'", {"the_date": {"equals": "2020-01-01"}})
        self.assertSql("\"the_time\" = '12:34:56'", {"the_time": {"equals": "12:34:56"}})

    def test_like_variants(self):
        self.assertSql("\"notes\" LIKE '%foo%'", {"notes": {"like": "foo"}})
        self.assertSql("\"desc\" ILIKE 'Bar%'", {"desc": {"istarts_with": "Bar"}})
        self.assertSql("\"title\" NOT ILIKE 'baz%'", {"title": {"not_istarts_with": "baz"}})
        self.assertSql("\"title\" NOT LIKE '%baz%'", {"title": {"not_like": "baz"}})
        self.assertSql("\"notes\" LIKE 'foo%'", {"notes": {"starts_with": "foo"}})
        self.assertSql("\"desc\" LIKE '%bar'", {"desc": {"ends_with": "bar"}})
        self.assertSql("\"species\" LIKE 'a%b%c'", {"AND": [{"species": {"starts_with": "a%b"}},{"species": {"ends_with": "c"}}]})
        self.assertSql("\"species\" nOt  LIKE 'a%b%c'", {"OR": [{"species": {"not_starts_with": "a%b"}},{"species": {"not_ends_with": "c"}}]})
        self.assertSql("\"species\" ILIKE 'a%b%c'", {"AND": [{"species": {"istarts_with": "a%b"}},{"species": {"iends_with": "c"}}]})
        self.assertSql("\"species\" not ILIKE 'a%b%c'", {"OR": [{"species": {"not_istarts_with": "a%b"}},{"species": {"not_iends_with": "c"}}]})
        self.assertSql("\"species\" IS  NOT  NULL AND \"species\" LIKE 'a%b%c'", {"AND": [{"species": {"is_not_null": True}}, {"AND": [{"species": {"starts_with": "a%b"}},{"species": {"ends_with": "c"}}]}]})
        self.assertSql("\"notes\" LIKE 'foo'", {"notes": {"equals": "foo"}})
        self.assertSql("\"notes\" NOT LIKE 'foo'", {"notes": {"not_equals": "foo"}})
        self.assertSql("\"desc\" ILIKE 'Bar'", {"desc": {"iequals": "Bar"}})
        self.assertSql("\"title\" NOT ILIKE 'baz'", {"title": {'not_iequals': 'baz'}})

    def test_in_and_not_iin(self):
        self.assertSql("\"status\" IN ('a','b')", {"status": {"in": ["a", "b"]}})
        self.assertSql("\"x\" NOT  IN (1,2)", {"x": {"not_in": [1, 2]}})
        self.assertSql("\"status\" IIN ('a','b')", {"status": {"iin": ["a", "b"]}})
        self.assertSql("\"x\" NOT  IIN (1,2)", {"x": {"not_iin": [1, 2]}})

    def test_is_null_and_empty(self):
        self.assertSql("\"col\" IS NULL", {"col": {"is_null": True}})
        self.assertSql("\"col\" IS NOT NULL", {"col": {"is_not_null": True}})
        self.assertSql("\"empty_field\" IS EMPTY", {"empty_field": {"is_empty": True}})
        self.assertSql("\"empty_field\" IS NOT EMPTY", {"empty_field": {"is_not_empty": True}})
        self.assertSql("\"species\" IS  not  NULL  AND \"corpse\" IS  NULL ", {"AND": [{"species": {"is_not_null": True}}, {"corpse": {"is_null": True}}]})

    def test_logical_combinations(self):
        # AND
        self.assertSql('"a" = 1 AND "b" = 2', {"AND": [ {"a": {"equals": 1}}, {"b": {"equals": 2}} ]})
        # OR
        self.assertSql('"a" = 1 OR "b" = 2', {"OR": [ {"a": {"equals": 1}}, {"b": {"equals": 2}} ]})
        # NOT unary
        self.assertSql('NOT "a" = 1', {"NOT": {"a": {"equals": 1}}})
        # parenthesis grouping -> (a=1 AND b=2) OR c=3
        self.assertSql('("a" = 1 AND "b" = 2) OR "c" = 3', {"OR": [ {"AND": [ {"a": {"equals":1}}, {"b": {"equals":2}} ]}, {"c": {"equals":3}} ]})

    def test_mixed_complex_expression(self):
        expr = " (\"name\" = 'Alice' AND \"score\" >= 4.5) OR (\"active\" = false AND \"tags\" IN ('x','y')) AND NOT \"notes\" ILIKE '%skip%' "
        result = self.dlg.sql_where_to_graphql(expr)
        self.assertTrue(isinstance(result, dict))
        dumped = str(result)
        self.assertIn("Alice", dumped)
        self.assertIn("4.5", dumped)
        self.assertIn("False", dumped)
        self.assertIn("x", dumped)
        self.assertIn("y", dumped)
        print("\nRESULTING GraphQL FILTER:", result)
        print("ORIGINAL:(name = 'Alice' AND score >= 4.5) OR (active = false AND tags IN ('x','y')) AND NOT notes ILIKE '%skip%'")

    # ---------------------------
    # NEW: literal-first scenarios
    # ---------------------------

    def test_literal_first_equals_and_not_equals(self):
        self.assertSql("'John' = \"name\"", {"name": {"equals": "John"}})
        self.assertSql("5 = \"age\"", {"age": {"equals": 5}})
        self.assertSql("'X' != \"code\"", {"code": {"not_equals": "X"}})

    def test_literal_first_directional_ops(self):
        # flip operator when swapping sides
        self.assertSql("5 > \"age\"", {"age": {"less_than": 5}})
        self.assertSql("5 >= \"age\"", {"age": {"less_than_or_equals": 5}})
        self.assertSql("5 < \"age\"", {"age": {"greater_than": 5}})
        self.assertSql("5 <= \"age\"", {"age": {"greater_than_or_equals": 5}})

    def test_literal_first_like_positive(self):
        # %foo% -> like "foo" (contains)
        self.assertSql("'%foo%' LIKE \"notes\"", {"notes": {"like": "foo"}})
        # Bar% -> starts_with
        self.assertSql("'Bar%' LIKE \"desc\"", {"desc": {"starts_with": "Bar"}})
        # %bar -> ends_with
        self.assertSql("'%bar' LIKE \"desc\"", {"desc": {"ends_with": "bar"}})
        # internal % composite -> AND of starts/ends
        self.assertSql("'a%b%c' LIKE \"species\"", {"AND": [{"species": {"starts_with": "a%b"}}, {"species": {"ends_with": "c"}}]})

    def test_literal_first_ilike_and_not_like(self):
        # ILIKE variants
        self.assertSql("'Pre%' ILIKE \"title\"", {"title": {"istarts_with": "Pre"}})
        self.assertSql("'%post' ILIKE \"title\"", {"title": {"iends_with": "post"}})
        self.assertSql("'%mid%' ILIKE \"title\"", {"title": {"ilike": "mid"}})
        # NOT LIKE/ILIKE
        self.assertSql("'baz%' NOT LIKE \"title\"", {"title": {"not_starts_with": "baz"}})
        self.assertSql("'baz%' NOT  ILIKE  \"title\"", {"title": {"not_istarts_with": "baz"}})
        # composite negated -> OR of not_starts_with / not_ends_with
        self.assertSql("'a%b%c' not like \"species\"", {"OR": [{"species": {"not_starts_with": "a%b"}}, {"species": {"not_ends_with": "c"}}]})
        self.assertSql("'a%b%c' NOT ILIKE \"species\"", {"OR": [{"species": {"not_istarts_with": "a%b"}}, {"species": {"not_iends_with": "c"}}]})

    def test_literal_first_with_not_and_parentheses(self):
        # NOT wrapping a literal-first comparison
        self.assertSql("NOT  'John' = \"name\"", {"NOT": {"name": {"equals": "John"}}})
        # Grouped OR with mixed sides
        self.assertSql("('X' = \"code\" OR 5 > \"age\")", {"OR": [{"code": {"equals": "X"}}, {"age": {"less_than": 5}}]})

    def test_literal_first_boolean_and_case_insensitive(self):
        # boolean literal first
        # self.assertSql("TRUE = \"active\"", {"active": {"equals": True}})
        # self.assertSql("false = \"active\"", {"active": {"equals": False}})
        # mixed case operators and extra spaces
        self.assertSql("'Abc%'   iLiKe   \"name\"", {"name": {"istarts_with": "Abc"}})

    def test_literal_first_underscore_wildcard(self):
        # underscore matches exactly one char; remains part of pattern normalization
        self.assertSql("'A_c%' LIKE \"val\"", {"val": {"starts_with": "A_c"}})
        self.assertSql("'%x_y' ILIKE \"val\"", {"val": {"iends_with": "x_y"}})

class TestOperatorSpacingNormalization(unittest.TestCase):
    def setUp(self):
        self.dlg = ObmConnectFilterFactory.make()

    def test_sql_where_parsing_handles_multiple_spaces_in_operator(self):
        # NOT  ILIKE with trailing % -> parser should normalize to not_istarts_with
        expr = '"title" NOT  ILIKE \'baz%\''
        got = self.dlg.sql_where_to_graphql(expr)
        self.assertEqual(got, {"title": {"not_istarts_with": "baz"}})

        # NOT   IN with extra spaces should parse as not_in with numeric list
        expr = '"x" NOT   IN (1,2)'
        got = self.dlg.sql_where_to_graphql(expr)
        self.assertEqual(got, {"x": {"not_in": [1, 2]}})

        # Multiple spaces and mixed case in IS NOT NULL
        expr = '"col" IS   Not   NULL'
        got = self.dlg.sql_where_to_graphql(expr)
        self.assertEqual(got, {"col": {"is_not_null": True}})

        # NOT with parentheses grouping and extra spaces
        expr = 'NOT   ("a" = 1  OR   "b" = 2)'
        got = self.dlg.sql_where_to_graphql(expr)
        self.assertTrue(isinstance(got, dict))
        self.assertIn("NOT", got)
        self.assertEqual(got, {"NOT": {"OR": [{"a": {"equals": 1}}, {"b": {"equals": 2}}]}})

if __name__ == "__main__":
    unittest.main()
