# coding=utf-8
"""DockWidget test.

.. note:: This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

"""

__author__ = 'gezsaj@gmail.com'
__date__ = '2025-07-21'
__copyright__ = 'Copyright 2025, Attila Gáspár'

import os
os.environ["QT_QPA_PLATFORM"] = "offscreen"

import unittest
from types import SimpleNamespace
from unittest import mock
import requests

from qgis.PyQt import QtWidgets
from qgis.PyQt.QtGui import QStandardItemModel, QStandardItem

from qgis.core import QgsApplication
from PyQt5.QtWidgets import QApplication
import sys

# Initialize Qt and QGIS applications to avoid crash during widget instantiation
app = QApplication.instance() or QApplication(sys.argv)
qgs = QgsApplication([], False)
qgs.initQgis()

# Import the module under test
import obm_connect_dockwidget as dockmod
from obm_connect_dockwidget import ObmConnectDockWidget


class FakeMessageBar:
    def __init__(self):
        self.last = None

    def pushMessage(self, *args, **kwargs):
        self.last = ("message", args, kwargs)

    def pushWarning(self, *args, **kwargs):
        self.last = ("warning", args, kwargs)

    def pushCritical(self, *args, **kwargs):
        self.last = ("critical", args, kwargs)


class FakeIface:
    def __init__(self):
        self._mb = FakeMessageBar()

    def messageBar(self):
        return self._mb


class ObmConnectDockWidgetTest(unittest.TestCase):
    """Test dockwidget works."""

    def setUp(self):
        self.iface = FakeIface()
        # Mock QSettings to prevent auto-login during initialization
        with mock.patch('obm_connect_dockwidget.QSettings') as mock_settings_class:
            mock_settings = mock_settings_class.return_value
            mock_settings.value.return_value = None
            self.dock = ObmConnectDockWidget(self.iface)

    def tearDown(self):
        self.dock = None

    def test_initial_state(self):
        # Project combo disabled and default tab index 0
        self.assertFalse(self.dock.projectComboBox.isEnabled())
        self.assertEqual(self.dock.tabWidget.currentIndex(), 0)

    @mock.patch('obm_connect_dockwidget.requests.get')
    def test_load_projects_with_list_response(self, mock_get):
        # Mock a requests.Response-like object returning a list of projects
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.json.return_value = [
            {"name": "Project A", "project_table": "proj_a"},
            {"name": "Project B", "project_table": "proj_b"}
        ]
        mock_get.return_value = resp

        self.dock.load_projects("example.com")

        # Combo should be enabled and contain the projects + placeholder
        self.assertTrue(self.dock.projectComboBox.isEnabled())
        # first item is placeholder
        self.assertEqual(self.dock.projectComboBox.count(), 1 + 2)
        # internal project list stored
        self.assertEqual(len(self.dock._project_list), 2)
        self.assertEqual(self.dock.projectComboBox.itemText(1), "Project A")

    @mock.patch('obm_connect_dockwidget.requests.get')
    def test_load_projects_with_data_key_response(self, mock_get):
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.json.return_value = {"data": [{"name": "OnlyProject", "project_table": "onlyproj"}]}
        mock_get.return_value = resp

        self.dock.load_projects("example.com")
        self.assertTrue(self.dock.projectComboBox.isEnabled())
        self.assertEqual(self.dock.projectComboBox.count(), 2)
        self.assertEqual(self.dock.projectComboBox.itemText(1), "OnlyProject")

    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'validate_auth', return_value=True)
    def test_on_project_selected_populates_table_view(self, mock_ensure_token, mock_get):
        # Prepare project in internal list
        project = {"project_table": "projx", "name": "Project X"}
        self.dock._project_list = [project]

        # Mock response for data-tables endpoint returning list of tables
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.headers = {"Content-Type": "application/json"}
        resp.json.return_value = [
            {"name": "projx_table1", "schema": "s1"},
            {"name": "projx_table2", "schema": "s2"}
        ]
        mock_get.return_value = resp

        # Call with idx=1 (first real project)
        self.dock.on_project_selected(1)

        model = self.dock.tablesTableView.model()
        self.assertIsNotNone(model)
        # Two rows expected
        self.assertEqual(model.rowCount(), 2)
        # Since raw names start with project_table + "_", display name should start with underscore
        first_display = model.item(0, 0).text()
        self.assertTrue(first_display.startswith("_"))
        self.assertEqual(first_display, "_table1")


    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'validate_auth', return_value=True)
    def test_on_project_selected_404_shows_unavailable_and_resets(self, mock_ensure_token, mock_get):
        project = {"project_table": "projx", "name": "Project X"}
        self.dock._project_list = [project]

        resp = mock.Mock()
        resp.raise_for_status.side_effect = requests.exceptions.HTTPError("404 Client Error")
        mock_get.return_value = resp

        with mock.patch.object(self.dock, "_reset_state_and_views") as mock_reset:
            self.dock.on_project_selected(1)
            mock_reset.assert_called_once()


    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'validate_auth', return_value=True)
    def test_on_project_selected_non_json_content_type_shows_unavailable_and_resets(self, mock_ensure_token, mock_get):
        project = {"project_table": "projx", "name": "Project X"}
        self.dock._project_list = [project]

        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.headers = {"Content-Type": "text/html; charset=utf-8"}  # not JSON
        resp.text = "<html>Not JSON</html>"
        mock_get.return_value = resp

        with mock.patch.object(self.dock, "_reset_state_and_views") as mock_reset:
            self.dock.on_project_selected(1)
            mock_reset.assert_called_once()

if __name__ == "__main__":
    unittest.main()

