from qgis.utils import iface
from qgis.core import QgsProject, QgsFeatureRequest

from qgis.PyQt import QtWidgets, QtGui, QtCore
from urllib.request import urlopen
import os
import json

def tr(message):
    return QtCore.QCoreApplication.translate('ObmConnect', message)

class PannableLabel(QtWidgets.QLabel):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setMouseTracking(True)
        self._drag_pos = None
        self.offset = QtCore.QPoint(0, 0)

    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        if event.button() == QtCore.Qt.LeftButton:
            self._drag_pos = event.pos()

    def mouseMoveEvent(self, event):
        super().mouseMoveEvent(event)
        if self._drag_pos is not None:
            delta = event.pos() - self._drag_pos
            self._drag_pos = event.pos()
            parent = self.parent()
            if parent:
                parent.pan_image(delta)

    def mouseReleaseEvent(self, event):
        super().mouseReleaseEvent(event)
        if event.button() == QtCore.Qt.LeftButton:
            self._drag_pos = None

class SelectionDialog(QtWidgets.QDialog):
    def __init__(self, features, field_name, parent=None):
        super().__init__(parent)
        self.setWindowTitle(tr("Select Feature"))
        self.resize(400, 300)
        self.layout = QtWidgets.QVBoxLayout(self)
        self.list_widget = QtWidgets.QListWidget()
        self.features = []
        self.field_name = field_name
        
        for feat in features:
            val = feat[field_name]
            # Filter out null/empty obm_files_id
            if val == QtCore.QVariant() or val is None:
                continue
            s_val = str(val).strip()
            if not s_val or s_val == 'NULL':
                continue

            # Get obm_id
            obm_id_val = feat.id()
            try:
                # Try to find obm_id field (case insensitive check usually handled by QGIS but fieldNameIndex is case sensitive often, let's try standard names)
                idx = feat.fieldNameIndex('obm_id')
                if idx == -1:
                    idx = feat.fieldNameIndex('OBM_ID')
                
                if idx != -1:
                    obm_id_val = feat[idx]
            except:
                pass
            
            self.list_widget.addItem(f"OBM ID: {obm_id_val} - {s_val}")
            self.features.append(feat)
            
        self.layout.addWidget(self.list_widget)
        
        self.buttons = QtWidgets.QDialogButtonBox(
            QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel
        )
        self.buttons.accepted.connect(self.accept)
        self.buttons.rejected.connect(self.reject)
        self.layout.addWidget(self.buttons)
        
    def get_selected_feature(self):
        row = self.list_widget.currentRow()
        if row >= 0 and row < len(self.features):
            return self.features[row]
        return None

class ImageViewer(QtWidgets.QDialog):
    def __init__(self, urls, parent=None):
        super().__init__(parent)
        self.setWindowTitle(tr("Thumbnail picture"))
        self.index = 0
        self.pixmap = None
        self.scale = 1.0

        self.urls = urls
        for i, url in enumerate(self.urls):
            if not url.startswith('http'):
                self.urls[i] = f"%GETPHOTO%{url.lstrip('/')}"

        self.offset = QtCore.QPoint(0, 0)
        self.label = PannableLabel(self)
        self.label.setAlignment(QtCore.Qt.AlignCenter)
        self.label.setMinimumSize(300, 300)
        self.label.installEventFilter(self)

        self.prev_btn = QtWidgets.QPushButton("<")
        self.next_btn = QtWidgets.QPushButton(">")
        self.download_btn = QtWidgets.QPushButton(tr("Download original"))
        self.info_lbl = QtWidgets.QLabel(alignment=QtCore.Qt.AlignCenter)

        btn_layout = QtWidgets.QHBoxLayout()
        btn_layout.addWidget(self.prev_btn)
        btn_layout.addWidget(self.info_lbl, stretch=1)
        btn_layout.addWidget(self.next_btn)
        btn_layout.addWidget(self.download_btn)

        layout = QtWidgets.QVBoxLayout(self)
        layout.addWidget(self.label)
        layout.addLayout(btn_layout)

        self.prev_btn.clicked.connect(self.prev_image)
        self.next_btn.clicked.connect(self.next_image)
        self.download_btn.clicked.connect(self.download_image)

        self.update_buttons()
        self.load_image()

    def load_image(self):
        self.offset = QtCore.QPoint(0, 0)
        self.scale = 1.0
        self.download_btn.setEnabled(False)
        if not self.urls:
            iface.messageBar().pushMessage("OBM Connect", tr("No image available."), duration=3)
            QtCore.QTimer.singleShot(0, self.close)
            return
        url = self.urls[self.index]
        
        # Update index info
        self.info_lbl.setText(f"{self.index+1}/{len(self.urls)}")

        try:
            if "ref=" in url:
                filename = url.split("ref=")[-1]
            else:
                filename = os.path.basename(url)
            if url.endswith(".pdf"):
                self.label.setText(f'<a href="{url}">{filename}</a>')
                self.label.setTextFormat(QtCore.Qt.RichText)
                self.label.setOpenExternalLinks(True)
                self.label.setWordWrap(True)
                return  
            data = urlopen(url).read()
            pm = QtGui.QPixmap()
            if pm.loadFromData(data):
                self.pixmap = pm
                self.update_pixmap()
                self.download_btn.setEnabled(True)
            else:
                self.pixmap = None
                # Not an image, check if it is JSON or just a link
                try:
                    text_content = data.decode('utf-8')
                    try:
                        json_obj = json.loads(text_content)
                        # It is JSON
                        server_msg = json.dumps(json_obj.get("message", "No specific error message provided"))
                        self.label.setText(
                            tr("File not found or unavailable: '{}'. <br>(Server response: {})").format(filename, server_msg)
                        )                        
                        self.label.setWordWrap(True)
                    except ValueError:
                        # Not JSON, show as clickable link
                        self.label.setText(f'<a href="{url}">{filename}</a>')
                        self.label.setTextFormat(Qt.RichText)
                        self.label.setOpenExternalLinks(True)
                        self.label.setWordWrap(True)
                except Exception:
                    # Fallback to link
                    self.label.setText(f'<a href="{url}">{filename}</a>')
                    self.label.setOpenExternalLinks(True)

        except Exception:
            # Image not found or error
            self.label.setText(f"{url}")
            self.pixmap = None


    def pan_image(self, delta):
        if self.scale > 1.0 and self.pixmap:
            self.offset += delta
            self.update_pixmap()

    def eventFilter(self, obj, event):
        if obj == self.label and event.type() == QtCore.QEvent.Wheel:
            delta = event.angleDelta().y()
            step = 0.15
            if delta > 0:
                self.scale *= (1 + step)
            elif delta < 0:
                self.scale /= (1 + step)
            self.scale = max(0.1, min(self.scale, 8.0))
            
            self.update_pixmap()
            return True
        return super().eventFilter(obj, event)

    def clamp_offset(self, scaled_pixmap):
        label_size = self.label.size()
        pixmap_size = scaled_pixmap.size()

        max_x = max(0, (pixmap_size.width() - label_size.width()) // 2)
        max_y = max(0, (pixmap_size.height() - label_size.height()) // 2)

        self.offset.setX(max(-max_x, min(self.offset.x(), max_x)))
        self.offset.setY(max(-max_y, min(self.offset.y(), max_y)))

    def update_pixmap(self):
        if not self.pixmap:
            return
        label_size = self.label.size()
        scaled = self.pixmap.scaled(
            int(label_size.width()*self.scale),
            int(label_size.height()*self.scale),
            QtCore.Qt.KeepAspectRatio,
            QtCore.Qt.SmoothTransformation,
        )
        self.clamp_offset(scaled)
        pm = QtGui.QPixmap(label_size)
        pm.fill(QtCore.Qt.transparent)
        painter = QtGui.QPainter(pm)
        x = int((label_size.width() - scaled.width())/2 + self.offset.x())
        y = int((label_size.height() - scaled.height())/2 + self.offset.y())
        painter.drawPixmap(x, y, scaled)
        painter.end()
        self.label.setPixmap(pm)

    def resizeEvent(self, ev):
        super().resizeEvent(ev)
        self.update_pixmap()

    def update_buttons(self):
        self.prev_btn.setEnabled(self.index > 0)
        self.next_btn.setEnabled(self.index < len(self.urls)-1)

    def next_image(self):
        if self.index < len(self.urls)-1:
            self.index += 1
            self.update_buttons()
            self.load_image()
        self.offset = QtCore.QPoint(0, 0)

    def prev_image(self):
        if self.index > 0:
            self.index -= 1
            self.update_buttons()
            self.load_image()
        self.offset = QtCore.QPoint(0, 0)

    def download_image(self):
        base_url = self.urls[self.index]
        if "&getfullimage" not in base_url:
            download_url = base_url + "&getfullimage"
        else:
            download_url = base_url
        try:
            filename = os.path.basename(download_url.split("ref=")[-1].split("&")[0])
            save_path, _ = QtWidgets.QFileDialog.getSaveFileName(self, tr("Save as..."), filename)
            if save_path:
                img_data = urlopen(download_url).read()
                with open(save_path, "wb") as f:
                    f.write(img_data)
        except Exception as e:
            QtWidgets.QMessageBox.warning(self, tr("Error"), tr("Error in downloading: {}").format(e))


def main_obm_action():
    # Name of the attribute with the filenames or urls
    field_name = "obm_files_id"
    urls_text = r'[% "obm_files_id" %]'
    layer_id = r'[% @layer_id %]'
    found_fids_str = r'[% @obm_found_fids %]'

    iface.messageBar().pushMessage("OBM Connect", f"Layer ID: {layer_id}", duration=5)

    layer = None
    if layer_id and not layer_id.startswith('[%'):
        layer = QgsProject.instance().mapLayer(layer_id)

    if not layer:
        layer = iface.activeLayer()

    if not layer:
        iface.messageBar().pushMessage("OBM Connect", tr("No active layer found."), duration=5)
        return

    # Check if multiple features were found by map tool
    if found_fids_str and not found_fids_str.startswith('[%') and found_fids_str != 'NULL' and ',' in found_fids_str:
        try:
            fids = [int(fid) for fid in found_fids_str.split(',') if fid.strip()]
            if len(fids) > 1:
                features = list(layer.getFeatures(QgsFeatureRequest(fids)))
                if features:
                    dlg = SelectionDialog(features, field_name, iface.mainWindow())
                    if dlg.features:
                        if dlg.exec_() == QtWidgets.QDialog.Accepted:
                            feat = dlg.get_selected_feature()
                            if feat:
                                val = feat[field_name]
                                urls_text = str(val) if val is not None else 'NULL'
                        else:
                            return # Cancelled
        except Exception as e:
            iface.messageBar().pushMessage("OBM Connect", tr("Error selecting feature: {}").format(e), duration=5)

    if layer.fields().indexOf(field_name) == -1:
        iface.messageBar().pushMessage(
            "OBM Connect", 
            tr("The '{}' column does not exist in this layer.").format(field_name), 
            duration=5
        )
        return

    if not urls_text or urls_text == '' or urls_text == 'NULL' or 'NULL' in urls_text:
        iface.messageBar().pushMessage("OBM Connect", tr("The '{}' column is empty.").format(field_name), duration=5)
        return

    urls = [u.strip() for u in urls_text.split(';') if u.strip()]

    if not urls:
        iface.messageBar().pushMessage("OBM Connect", tr("No valid file names or URLs found."), duration=5)
        return

    if not hasattr(iface, "_obm_img_viewers"):
        iface._obm_img_viewers = []

    valid_viewers = []
    for v in iface._obm_img_viewers:
        try:
            if v and not sip.isdeleted(v):
                valid_viewers.append(v)
        except:
            pass
    iface._obm_img_viewers = valid_viewers

    try:
        viewer = ImageViewer(urls, iface.mainWindow())
        iface._obm_img_viewers.append(viewer)
        
        def on_viewer_closed(result):
            if hasattr(iface, "_obm_img_viewers"):
                if viewer in iface._obm_img_viewers:
                    iface._obm_img_viewers.remove(viewer)
        
        viewer.finished.connect(on_viewer_closed)
        viewer.show()
        
    except NameError:
         iface.messageBar().pushMessage("OBM Connect", tr("Error: ImageViewer class is not defined."), duration=5)


main_obm_action()
