from PyQt5 import QtWidgets, uic, QtGui, QtCore
from qgis.core import (
    Qgis,
    QgsProject,
    QgsVectorLayer,
    QgsFields,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsWkbTypes
)
from PyQt5.QtCore import QVariant, Qt
import json
import os
import requests
import csv
try:
    from .sql_where_graphql_converter import SqlWhereGraphQLConverter
    from .sql_where_validator import SqlWhereValidator, ValidatorResult, Token, SqlTokenizer
except ImportError:
    from sql_where_graphql_converter import SqlWhereGraphQLConverter
    from sql_where_validator import SqlWhereValidator, ValidatorResult, Token, SqlTokenizer
import sys
import html
import tempfile
import random
import string
import re
from typing import Union

from qgis.core import QgsAction, QgsActionManager, QgsProject, QgsAttributeTableConfig
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QFileDialog

sys.path.append(os.path.dirname(os.path.abspath(__file__)))
from config import build_project_api_url, build_projects_url, build_unique_values_url, build_getphoto_url, DEFAULT_OBM_SERVER




class ObmConnectFilterDialog(QtWidgets.QDialog):
    def tr(self, msg: str) -> str:
        return QtCore.QCoreApplication.translate("ObmConnectFilter", msg)

    def __init__(self, project_domain, parent=None, debug=False):
        super().__init__(parent)
        self.project_domain = project_domain
        uic.loadUi(os.path.join(os.path.dirname(__file__), "obm_connect_filter.ui"), self)
        self.setFixedSize(self.size())

        self.DEBUG = debug if 'debug' in locals() else False
        self.main_widget = parent
        # ensure iface is available when dialog is created from the dockwidget
        self.iface = None
        if parent is not None:
            # parent may be the dockwidget instance that has .iface
            if hasattr(parent, "iface"):
                self.iface = parent.iface
            # or main_widget wrapper
            elif hasattr(parent, "main_widget") and getattr(parent, "main_widget") is not None and hasattr(parent.main_widget, "iface"):
                self.iface = parent.main_widget.iface

        # Wire OK/Cancel
        try:
            self.buttonBox.accepted.disconnect(self.accept)
        except TypeError:
            pass
        self.buttonBox.accepted.connect(self.on_ok_clicked)
        self.buttonBox.rejected.connect(self.on_cancel_clicked)

        # Wire field list double-click to insert quoted field name
        self.fieldListView.doubleClicked.connect(self._on_field_double_clicked)
        self.fieldListView.clicked.connect(self._set_unique_count_on_all_button)
        self.valueListView.doubleClicked.connect(self._on_value_double_clicked)
        self._highlighted_row = None


        # Map operator buttons to tokens and connect
        self._wire_operator_buttons()

        # Wire Test button to test() method
        if hasattr(self, "pushButton_test"):
            self.pushButton_test.clicked.connect(self.test)

        # Wire GraphQL button
        if hasattr(self, "pushButton_graphql"):
            self.pushButton_graphql.clicked.connect(self.on_graphql_button_clicked)

        # Initialize value list and wire sample/all buttons and search
        self.value_list_full = []
        self.sampleButton.clicked.connect(self.on_sample_button_clicked)
        self.allButton.clicked.connect(self.on_all_button_clicked)
        self.lineEdit_search.textChanged.connect(self.on_search_text_changed)


    # ---------- UI helpers ----------
    def _wire_operator_buttons(self):
        """Connect operator buttons to insert their token at the cursor with spaces."""
        btn_map = {
            "pushButton_equal": " = ",
            "pushButton_less": " < ",
            "pushButton_greater": " > ",
            "pushButton_lessequal": " <= ",
            "pushButton_greaterequal": " >= ",
            "pushButton_notequal": " != ",
            "pushButton_like": " LIKE ",
            "pushButton_ilike": " ILIKE ",
            "pushButton_wildcard": "%",
            "pushButton_in": " IN (",
            "pushButton_notin": " NOT IN (",
            "pushButton_and": " AND ",
            "pushButton_or": " OR ",
            "pushButton_not": " NOT ",
            "pushButton_iin": " IIN (",
            "pushButton_notiin": " NOT IIN (",
            "pushButton_isnull": " IS NULL ",
            "pushButton_isnotnull": " IS NOT NULL ",
            }
        for obj_name, token in btn_map.items():
            btn = getattr(self, obj_name, None)
            if btn is not None:
                btn.clicked.connect(lambda checked=False, t=token: self._insert_token(t))

    def _on_field_double_clicked(self, index: QtCore.QModelIndex):
        """Insert the selected field name wrapped in double quotes."""
        field_name = str(index.data())
        self._insert_text(f"\"{field_name}\"")

    def _on_value_double_clicked(self, index: QtCore.QModelIndex):
        """Insert value into the filter editor using API field types."""
        try:
            val = index.data()
            if val is None:
                self._insert_text("NULL")
                return

            text = str(val).strip()
            if text == "":
                self._insert_text("''")
                return

            # Get selected field name
            field_name = ""
            model = self.fieldListView.model()
            if model is not None:
                sel = self.fieldListView.selectionModel().currentIndex()
                if sel.isValid():
                    field_name = str(model.itemFromIndex(sel).text()).strip()

            # Get field type from parent._last_fields_json
            field_type = None
            parent = self.main_widget or self.parent()
            last_fields = getattr(parent, "_last_fields_json", None)
            if isinstance(last_fields, dict):
                for f in last_fields.get("fields", []):
                    if f.get("name") == field_name:
                        field_type = f.get("type")
                        break
            
            # Determine if quotes are needed
            # For Number and Boolean, no need to wrap in '
            should_quote = True
            is_boolean = False
            
            if field_type:
                ft = field_type.lower()
                if ft == "boolean":
                    should_quote = False
                    is_boolean = True
                elif ft in ["integer", "numeric", "smallint", "bigint", "decimal", "real", "double precision", "float"]:
                    should_quote = False
            
            if not should_quote:
                if is_boolean:
                    lower_text = text.lower()
                    if lower_text in ("true", "t", "yes", "y", "1"):
                        self._insert_text("true")
                    else:
                        self._insert_text("false")
                else:
                    self._insert_text(text)
            else:
                text_escaped = text.replace("'", "''")
                self._insert_text(f"'{text_escaped}'")

        except Exception:
            pass

    def _insert_token(self, token: str):
        """Insert operator token with a leading and trailing space at the cursor."""
        self._insert_text(f"{token}")

    def _insert_text(self, text: str):
        """Insert arbitrary text at the current cursor position in the filter editor."""
        editor: QtWidgets.QPlainTextEdit = self.filterEditor
        cursor: QtGui.QTextCursor = editor.textCursor()
        cursor.insertText(text)  # inserts at cursor, replacing any selection
        editor.setTextCursor(cursor)
        editor.setFocus()

    def highlight_row(self, row_index, color='#7CBEE3'):
        """Highlight the given row index and restore the previous row's style."""
        model = self.fieldListView.model()

        # Reset previously highlighted row to default background
        if self._highlighted_row is not None:
            previous = model.item(self._highlighted_row)
            if previous is not None:
                previous.setBackground(QtGui.QBrush())

        # Set background color for the newly selected row
        current = model.item(row_index)
        if current is not None:
            current.setBackground(QtGui.QColor(color))
        self._highlighted_row = row_index
        self.valueListView.repaint()

    def _set_unique_count_on_all_button(self, cur=None, prev=None):
        """
        Find the selected field in parent._last_fields_json and update the "All" button
        to include the unique-value count (if available).
        """
        model = getattr(self, "fieldListView").model()
        if model is None or model.rowCount() == 0:
            raise RuntimeError(self.tr("No field list available"))
        sel = self.fieldListView.selectionModel().currentIndex()

        if sel is None or not sel.isValid():
            sel = model.index(0, 0)
        field_name = str(model.itemFromIndex(sel).text()).strip()
        if not field_name:
            raise RuntimeError(self.tr("No field selected"))
        
        parent = self.main_widget or self.parent()
        unique_values = None
        last_fields_json = getattr(parent, "_last_fields_json", None)
        for field in last_fields_json.get("fields", []):
            if field.get("name") == field_name:
                unique_values = field.get("unique_values")
                break
        # Fallback value if unknown
        if unique_values is None:
            unique_values = "?"
        self.unique_values = unique_values
        # Update button label
        self.allButton.setText(self.tr("All ({0})").format(self.unique_values))

    def on_sample_button_clicked(self):
        """Download a sample of unique values (limit 25) from the API and populate the list."""
        if self.unique_values == "?" or self.unique_values == 0 or self.unique_values == "":
            return
        parent = self.main_widget or self.parent()
        try:
            obm_server = (parent.obmServerLineEdit.text().strip() if parent and hasattr(parent, "obmServerLineEdit") else DEFAULT_OBM_SERVER) or DEFAULT_OBM_SERVER

            # field from fieldListView selection
            model = getattr(self, "fieldListView").model()
            if model is None or model.rowCount() == 0:
                raise RuntimeError(self.tr("No field list available"))
            sel = self.fieldListView.selectionModel().currentIndex()
            if sel is None or not sel.isValid():
                # fallback to first row
                sel = model.index(0, 0)
            field_name = str(model.itemFromIndex(sel).text()).strip()
            if not field_name:
                raise RuntimeError(self.tr("No field selected"))
            self.highlight_row(sel.row())

            project_name, schema, table_name, fields = self._get_last_fields_info(parent)

            url = build_unique_values_url(self.project_domain, schema, table_name, field_name)

            headers = {}
            if parent and hasattr(parent, "auth_manager"):
                token = parent.auth_manager.get_access_token()
                if token:
                    headers["Authorization"] = f"{token}"
            params = "?limit=25"
            resp = requests.get(url + params, headers=headers, timeout=15)
            resp.raise_for_status()
            self.value_list_full = resp.json()

            self.apply_filter_to_valuelist()
        except Exception as e:
            if self.DEBUG:
                raise
            QtWidgets.QMessageBox.critical(self, self.tr("API error"), str(e))

    def on_all_button_clicked(self):
        """Download all unique values (large limit) from the API and populate the list."""
        if self.unique_values == "?" or self.unique_values == 0 or self.unique_values == "":
            return

        parent = self.main_widget or self.parent()
        try:
            # build using same logic as sample
            obm_server = (parent.obmServerLineEdit.text().strip() if parent and hasattr(parent, "obmServerLineEdit") else DEFAULT_OBM_SERVER) or DEFAULT_OBM_SERVER

            model = getattr(self, "fieldListView").model()
            if model is None or model.rowCount() == 0:
                raise RuntimeError(self.tr("No field list available"))
            sel = self.fieldListView.selectionModel().currentIndex()
            if sel is None or not sel.isValid():
                sel = model.index(0, 0)
            field_name = str(model.itemFromIndex(sel).text()).strip()
            if not field_name:
                raise RuntimeError(self.tr("No field selected"))
            self.highlight_row(sel.row())

            project_name, schema, table_name, fields = self._get_last_fields_info(parent)

            url = build_unique_values_url(self.project_domain, schema, table_name, field_name)

            params = "?limit=1000000"
            # ensure Authorization header if available on parent
            headers = {}
            if parent and hasattr(parent, "auth_manager"):
                token = parent.auth_manager.get_access_token()
                if token:
                    headers["Authorization"] = f"{token}"

            resp = requests.get(url + params, headers=headers, timeout=60)
            resp.raise_for_status()
            self.value_list_full = resp.json()
            self.apply_filter_to_valuelist()
        except Exception as e:
            if self.DEBUG:
                raise
            QtWidgets.QMessageBox.critical(self, self.tr("API error"), str(e))

    def on_search_text_changed(self):
        """Filter the currently loaded value list (client-side, no API call)."""
        self.apply_filter_to_valuelist()

    def apply_filter_to_valuelist(self):
        """Filter the full downloaded value list by the search box (case-insensitive substring)."""
        search = self.lineEdit_search.text().strip().lower()
        if search:
            filtered = [
                v for v in self.value_list_full
                if v is not None and search in str(v).lower()
            ]
        else:
            filtered = [v for v in self.value_list_full if v is not None]
        model = QtGui.QStandardItemModel()
        for v in filtered:
            item = QtGui.QStandardItem(str(v))
            model.appendRow(item)
        self.valueListView.setModel(model)

    def _get_last_fields_info(self, parent):
        """
        Return (project, schema, table_name, fields) using parent._last_fields_json.
        Safe defaults provided when information is missing.
        """
        last = getattr(parent, "_last_fields_json", None)
        project = None
        schema = "public"
        table_name = None
        fields = []

        if isinstance(last, dict):
            fields_json = last.get("fields", []) or []
            project = last.get("project", None)
            schema = last.get("schema", "public")
            table_name = last.get("name", None)
            fields = [item["name"] for item in fields_json if isinstance(item, dict) and "name" in item]
        return project, schema, table_name, fields

    def _get_geometry_fields(self, parent):
        """
        Extract geometry fields from parent._last_fields_json.
        """
        last = getattr(parent, "_last_fields_json", None)
        geom_fields = []
        if isinstance(last, dict):
            fields_json = last.get("fields", []) or []
            for f in fields_json:
                if isinstance(f, dict):
                    ftype = (f.get("type") or "").lower()
                    if ftype == "geometry" or "geometry_column_details" in f:
                        name = f.get("name")
                        if name:
                            geom_fields.append(name)
        return geom_fields

    def _build_graphql_spatial_payload(self, schema_name, table_name, geom_field, filters_dict, limit, offset=0, include_srid=True):
        fields_string = (
            "{"
            " total_count"
            " feature_collection {"
            " type"
            " features {"
            " type"
            " geometry { type coordinates srid }"
            " properties"
            " }"
            " }"
            " }"
        )
        query = (
            "query getFilteredObmData($filters: ObmDataFilterInput, $limit: Int, $offset: Int) { "
            f"spatialObmDataList(primaryGeometry: \"{geom_field}\" filters: $filters, limit: $limit, offset: $offset) "
            f"{fields_string} "
            "}"
        )
        return {
            "schema": schema_name or "public",
            "table_name": table_name,
            "query": query,
            "variables": {
                "filters": filters_dict,
                "limit": max(0, int(limit) if isinstance(limit, int) else 1),
                "offset": int(offset) if isinstance(offset, int) else 0
            }
        }

    def _build_graphql_data_payload(self, parent, schema_name, table_name, filters_dict, limit, offset=0, geom_fields=None):
        # Build fields string from last_fields_json

        fields = ""
        last_fields_json = getattr(parent, "_last_fields_json", None)
        for field in last_fields_json.get("fields", []):
            fields += f"{field.get('name')}, "

        fields_string = (
            "{"
            " total_count"
            f" items {{ {fields} }}"
            " }"
        )
        
        # Enforce that all geometry fields are NULL for non-spatial queries
        if geom_fields:
            if not isinstance(filters_dict, dict):
                filters_dict = {}
            
            null_checks = [{g: {"is_null": True}} for g in geom_fields]
            
            if "AND" not in filters_dict:
                filters_dict["AND"] = []
            
            if isinstance(filters_dict["AND"], list):
                filters_dict["AND"].extend(null_checks)

        query = (
            "query getFilteredObmData($filters: ObmDataFilterInput, $limit: Int, $offset: Int) { "
            "obmDataList(filters: $filters, limit: $limit, offset: $offset) "
            f"{fields_string} "
            "}"
        )
        return {
            "schema": schema_name or "public",
            "table_name": table_name,
            "query": query,
            "variables": {
                "filters": filters_dict,
                "limit": max(0, int(limit) if isinstance(limit, int) else 1),
                "offset": int(offset) if isinstance(offset, int) else 0
            }
        }

    def call_graphql_request(self, parent, graphql: dict, timeout: int = 30) -> Union[dict, None]:
        """
        Generic GraphQL POST call for an OBM project.

        Args:
            parent: main window or component holding obmServerLineEdit and optional _access_token
            graphql: GraphQL payload (dict)
            timeout: HTTP timeout in seconds

        Returns:
            dict: full JSON response or None on error. Errors or missing expected keys result in None.
        """
        obm_server = parent.obmServerLineEdit.text().strip() or DEFAULT_OBM_SERVER
        url = build_project_api_url(self.project_domain)
        if not url.endswith("/"):
            url += "/"
        url += "get-data"

        headers = {}
        if parent and hasattr(parent, "auth_manager"):
            token = parent.auth_manager.get_access_token()
            if token:
                headers["Authorization"] = f"{token}"
        try:
            resp = requests.post(url, json=graphql, headers=headers, timeout=timeout)
            resp.raise_for_status()
            resp_json = resp.json() if resp.content else {}

            if self.DEBUG:
                import copy
                display_json = copy.deepcopy(resp_json)

                try:
                    features_list = display_json.get('data', {}).get('spatialObmDataList', {}).get('feature_collection', {}).get('features')
                    if isinstance(features_list, list):
                        display_json['data']['spatialObmDataList']['feature_collection']['features'] = features_list[:2]
                    else:
                        items_list = display_json.get('data', {}).get('obmDataList', {}).get('items')
                        if isinstance(items_list, list):
                            display_json['data']['obmDataList']['items'] = items_list[:2]
                        
                except Exception:
                    pass

                debug_message = "Response JSON [max 2 records]:\n{json}".format(
                    json=json.dumps(display_json, indent=2, ensure_ascii=False)
                )
                
                QtWidgets.QMessageBox.information(
                    self,
                    "DEBUG",
                    debug_message
                )


            # GraphQL response may contain 'errors' according to spec
            errors = resp_json.get("errors")
            if isinstance(errors, list) and errors:
                msgs = []
                for e in errors:
                    if isinstance(e, dict):
                        msg = e.get("message") or "GraphQL error"
                        locs = e.get("locations")
                        if isinstance(locs, list) and locs:
                            loc_strs = []
                            for loc in locs:
                                line = loc.get("line")
                                col = loc.get("column")
                                if line is not None and col is not None:
                                    loc_strs.append(f"(line {line}, col {col})")
                            if loc_strs:
                                msg += " " + " ".join(loc_strs)
                    else:
                        msg = str(e) if e is not None else "GraphQL error"
                    msgs.append(msg)

                # Show a detailed API error message to the user
                if self.DEBUG:
                    # Append request/response debug info
                    strgraphql = "\n\nGraphQL request:\n{graphql}".format(
                        graphql=json.dumps(graphql, indent=2, ensure_ascii=False)
                    )
                    msgs.append(strgraphql)
                    strresponse = "\n\nFull response:\n{response}".format(
                        response=json.dumps(resp_json, indent=2, ensure_ascii=False)
                    )
                    msgs.append(strresponse)
                QtWidgets.QMessageBox.critical(self, self.tr("API error"), "\n".join(msgs))
                return None

            # Ensure we have the expected data structure
            data = resp_json.get("data", {})
            if not isinstance(data, dict) or ("spatialObmDataList" not in data and "obmDataList" not in data):
                QtWidgets.QMessageBox.warning(
                    self,
                    self.tr("No data"),
                    self.tr("The response did not include 'data.spatialObmDataList' or 'data.obmDataList'.")
                )
                return None

            return resp_json

        except Exception as e:
            if self.DEBUG:
                raise
            QtWidgets.QMessageBox.critical(
                self,
                self.tr("Error"),
                self.tr("Failed to send GraphQL request: {err}").format(err=str(e))
            )
            return None

    def run_filter_and_fetch_data(self, limit: int = 0, debug: bool = False):
        """
        Common logic for validating filter, building GraphQL payload, sending request, and returning results.
        Used by both test() and on_ok_clicked().
        Returns:
            dict: {
                'validator_result': ValidatorResult,
                'graphql': dict,
                'response_json': dict or None,
                'geojson_result': dict or None,
                'data_result': list or None,
                'fields': list,
                'project_name': str,
                'schema_name': str,
                'table_name': str,
                'geom_field': str or None,
                'filter_expr': str,
                'error': str or None
            }
        """
        parent = self.main_widget or self.parent()
        result = {
            'validator_result': None,
            'graphql': None,
            'response_json': None,
            'geojson_result': None,
            'data_result': None,
            'fields': [],
            'project_name': None,
            'schema_name': None,
            'table_name': None,
            'geom_field': None,
            'filter_expr': "",
            'error': None
        }

        if not parent or not hasattr(parent, "validate_auth"):
            result['error'] = "Main window or token validation function not found."
            return result

        if not parent.validate_auth():
            result['error'] = "No valid token."
            return result

        expr = self.filterEditor.toPlainText().strip()
        result['filter_expr'] = expr

        project_name, schema_name, table_name, fields = self._get_last_fields_info(parent)
        result['fields'] = fields
        result['project_name'] = project_name
        result['schema_name'] = schema_name
        result['table_name'] = table_name

        if not fields:
            result['error'] = "Table fields are unknown."
            return result

        # Geometry field from UI
        geom_field = None
        try:
            tv = getattr(parent, "tablesTableView", None)
            if tv is not None and tv.model() is not None:
                sel = tv.selectionModel().currentIndex()
                row = sel.row() if sel.isValid() else -1
                if row >= 0:
                    item_geom = tv.model().item(row, 2)
                    if item_geom is not None:
                        geom_field = item_geom.text().strip()
        except Exception:
            geom_field = None
        result['geom_field'] = geom_field

        converter = SqlWhereGraphQLConverter()
        filters_dict = converter.sql_where_to_graphql(expr)

        if geom_field:
            graphql = self._build_graphql_spatial_payload(
                schema_name=schema_name,
                table_name=table_name,
                geom_field=geom_field,
                filters_dict=filters_dict,
                limit=limit,
                offset=0
            )
        else:
            # Fetch all geometry fields to enforce NULL check
            geom_fields = self._get_geometry_fields(parent)
            graphql = self._build_graphql_data_payload(
                parent,
                schema_name=schema_name,
                table_name=table_name,
                filters_dict=filters_dict,
                limit=limit,
                offset=0,
                geom_fields=geom_fields
            )
        result['graphql'] = graphql

        validator = SqlWhereValidator()
        validator_result = validator.validate(
            expr,
            field_names=fields,
            case_sensitive_fields=False,
            deep_sql_check=True
        )
        result['validator_result'] = validator_result

        if not validator_result.ok:
            return result

        response_json = self.call_graphql_request(parent=parent, graphql=graphql)
        result['response_json'] = response_json

        if not response_json:
            return result

        data = response_json.get("data", {})
        
        if geom_field:
            geojson_result = data.get("spatialObmDataList") or {}

            if isinstance(geojson_result, dict) and geom_field:
                if "feature_collection" in geojson_result:
                    geojson_result[table_name + "-" + geom_field] = geojson_result.pop("feature_collection")

            geojson_result["metadata"] = {
                "source": "OBM Connect",
                "project": project_name,
                "schema": schema_name,
                "table": table_name,
                "geometry_field": geom_field,
                "filter_expression": expr,
                "record_limit": limit,
                "last_updated": QtCore.QDateTime.currentDateTime().toString(Qt.ISODate),
                "api_url": build_project_api_url(self.project_domain),
                "graphql_query": graphql
            }
            result['geojson_result'] = geojson_result
        else:
            obm_data_list = data.get("obmDataList") or {}
            data_result = obm_data_list.get("items") or []
            result['data_result'] = data_result

        return result

    def test(self):
        """Run a lightweight test using current filter expression."""
        res = self.run_filter_and_fetch_data(limit=0, debug=self.DEBUG)
        box = QtWidgets.QMessageBox(self)
        box.setWindowTitle(self.tr("Test"))
        box.setTextFormat(Qt.RichText)
        css_mono = "font-family: 'DejaVu Sans Mono','Consolas','Courier New',monospace; white-space: pre; margin:0"

        if res['error']:
            box.setText(self.tr(res['error']))
            box.exec_()
            return

        validator_result = res['validator_result']
        expr = res['filter_expr']
        if validator_result.ok:
            data = res['response_json'].get("data", {}) if res['response_json'] else {}
            slist = data.get("spatialObmDataList", {}) if isinstance(data, dict) else {}
            total_count = None
            if isinstance(slist, dict) and "total_count" in slist:
                total_count = slist.get("total_count")
            else:
                dlist = data.get("obmDataList", {}) if isinstance(data, dict) else {}
                if isinstance(dlist, dict):
                    total_count = dlist.get("total_count")

            if total_count is None:
                total_count = 0
            message = "<br>" + self.tr("Number of rows in the current filter: {count}").format(count=total_count)
        else:
            err_lines = ["- " + str(e) for e in validator_result.errors]
            error_slice_html = ""
            if validator_result.error_index is not None:
                start = max(0, validator_result.error_index - 20)
                end = min(len(expr), (validator_result.error_index + (validator_result.error_length or 1) + 20))
                snippet = expr[start:end]
                caret = " " * int(validator_result.error_index - start) + "^" * max(1, int(validator_result.error_length or 1))
                error_slice_html = f"<pre style=\"{css_mono}\">{html.escape(snippet)}\n{html.escape(caret)}</pre>"
            message = "<br>" + self.tr("The filter expression (WHERE) is NOT valid SQL.") + "<br>"
            if error_slice_html:
                message += ("<br>" + self.tr("Errors near:") + "<br>" +
                    error_slice_html + "<br>"
                    )
            if err_lines:
                safe_errors = [html.escape(e) for e in err_lines] 
                message += "<br>".join(safe_errors)
        box.setText(message)
        box.exec_()

        if self.DEBUG:
            debug_message = "Would run test with:\nFilter: {expr}".format(expr=expr)
            debug_message += "\nGraphQL equivalent:\n{graphql}".format(
                graphql=json.dumps(res['graphql'], indent=2, ensure_ascii=False)
            )
            debug_message += "\n\nResponse with metadata:\n{response_json}".format(
                response_json=json.dumps(res['response_json'], indent=2, ensure_ascii=False)
            )
            debug_message += "\n\nSQL Validator WHERE expression analysis:\n"
            tokenizer = SqlTokenizer(expr)
            tokens = tokenizer.tokenize()
            debug_message += "- Number of tokens: {count}".format(count=len(tokens))
            debug_message += "\n- tokens: {tokens}".format(tokens=tokens)
            debug_message += "\nOriginal error index: {error_index}".format(error_index=validator_result.error_index)
            debug_message += "\nOriginal error length: {error_length}".format(error_length=validator_result.error_length)
            QtWidgets.QMessageBox.information(
                self,
                "DEBUG",
                debug_message
            )

    def on_graphql_button_clicked(self):
        """Handle GraphQL button click: show dialog for user to paste GraphQL JSON."""
        parent = self.main_widget or self.parent()
        
        # Create input dialog
        dialog = QtWidgets.QDialog(self)
        dialog.setWindowTitle(self.tr("GraphQL Query"))
        dialog.setMinimumSize(600, 400)
        
        layout = QtWidgets.QVBoxLayout(dialog)
        
        label = QtWidgets.QLabel(self.tr("Paste your GraphQL JSON here:"))
        layout.addWidget(label)
        
        text_edit = QtWidgets.QPlainTextEdit()
        text_edit.setPlaceholderText(self.tr("Paste GraphQL JSON from clipboard..."))
        layout.addWidget(text_edit)
        
        button_box = QtWidgets.QDialogButtonBox(
            QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel
        )
        button_box.accepted.connect(dialog.accept)
        button_box.rejected.connect(dialog.reject)
        layout.addWidget(button_box)
        
        if dialog.exec_() != QtWidgets.QDialog.Accepted:
            return
        
        graphql_text = text_edit.toPlainText().strip()
        if not graphql_text:
            QtWidgets.QMessageBox.warning(self, self.tr("Error"), self.tr("GraphQL JSON is empty."))
            return
        
        # Parse JSON
        try:
            graphql_obj = json.loads(graphql_text)
        except Exception as e:
            QtWidgets.QMessageBox.warning(
                self, 
                self.tr("Error"), 
                self.tr("Invalid JSON: {err}").format(err=str(e))
            )
            return
        
        # Get project info
        if not parent or not hasattr(parent, "validate_auth"):
            QtWidgets.QMessageBox.warning(self, self.tr("Error"), self.tr("Main window not found."))
            return
        
        if not parent.validate_auth():
            QtWidgets.QMessageBox.warning(self, self.tr("Error"), self.tr("No valid token."))
            return
        
        project_name, schema_name, table_name, fields = self._get_last_fields_info(parent)
        
        # Send GraphQL request
        response_json = self.call_graphql_request(parent=parent, graphql=graphql_obj)
        
        if not response_json:
            return
        
        # Extract geojson_result
        data = response_json.get("data", {})
        geojson_result = data.get("spatialObmDataList") or {}
        
        # Get geom_field from UI
        geom_field = None
        try:
            tv = getattr(parent, "tablesTableView", None)
            if tv is not None and tv.model() is not None:
                sel = tv.selectionModel().currentIndex()
                row = sel.row() if sel.isValid() else -1
                if row >= 0:
                    item_geom = tv.model().item(row, 2)
                    if item_geom is not None:
                        geom_field = item_geom.text().strip()
        except Exception:
            geom_field = None
        
        if isinstance(geojson_result, dict) and geom_field:
            if "feature_collection" in geojson_result:
                geojson_result[table_name + "-" + geom_field] = geojson_result.pop("feature_collection")
        
        geojson_result["metadata"] = {
            "source": "OBM Connect (GraphQL)",
            "project": project_name,
            "schema": schema_name,
            "table": table_name,
            "geometry_field": geom_field,
            "filter_expression": "GraphQL direct query",
            "last_updated": QtCore.QDateTime.currentDateTime().toString(Qt.ISODate),
            "api_url": build_project_api_url(self.project_domain),
            "graphql_query": graphql_obj
        }
        
        if not geojson_result:
            QtWidgets.QMessageBox.information(
                self,
                self.tr("Empty result"),
                self.tr("No features returned.")
            )
            return
        
        layer_name = table_name + "-" + geom_field if geom_field else table_name
        self.createLayerFromGeojson(geojson_result, layer_name)
        self.accept()

    def on_ok_clicked(self):
        limit_txt = self.lineEdit_recNum.text().strip() if hasattr(self, "lineEdit_recNum") else "100"
        try:
            limit = max(1, int(limit_txt))
        except Exception:
            limit = 100

        res = self.run_filter_and_fetch_data(limit=limit, debug=self.DEBUG)
        if res['error']:
            QtWidgets.QMessageBox.warning(self, self.tr("Error"), self.tr(res['error']))
            return

        validator_result = res['validator_result']
        if not validator_result.ok:
            QtWidgets.QMessageBox.warning(self, self.tr("Error"), "\n".join(validator_result.errors))
            return

        if self.DEBUG:
            temp_dir = tempfile.gettempdir()
            with open(os.path.join(temp_dir, "response.geojson"), 'w', encoding='utf-8') as f:
                json.dump(res['response_json'], f, ensure_ascii=False, indent=2)
            with open(os.path.join(temp_dir, "request.geojson"), 'w', encoding='utf-8') as f:
                json.dump(res['graphql'], f, ensure_ascii=False)

        geojson_result = res['geojson_result']
        data_result = res.get('data_result')
        table_name = res['table_name']
        geom_field = res['geom_field']
        expr = res['filter_expr']

        if not geojson_result and not data_result:
            QtWidgets.QMessageBox.information(
                self,
                self.tr("Empty result"),
                self.tr("No features returned for the given filters.")
            )
            return

        if self.DEBUG:
            temp_dir = tempfile.gettempdir()
            if geojson_result:
                with open(os.path.join(temp_dir, "geojson_result.geojson"), 'w', encoding='utf-8') as f:
                    json.dump(geojson_result, f, ensure_ascii=False, indent=2)
            if data_result:
                with open(os.path.join(temp_dir, "data_result.json"), 'w', encoding='utf-8') as f:
                    json.dump(data_result, f, ensure_ascii=False, indent=2)

        if geojson_result:
            layer_name = table_name + "-" + geom_field if geom_field else table_name
            self.createLayerFromGeojson(geojson_result, layer_name)
        elif data_result:
            self.createTableFromData(data_result, table_name)

        layer = self.iface.activeLayer()
        if expr and layer is not None:
            layer.setSubsetString(expr)
        self.accept()

    def on_cancel_clicked(self):
        self.close()

    def add_picture_action_to_layer(self, layer):
        """
        Add the Picture action to a vector layer.
        Args:
            layer: QgsVectorLayer to add the action to
        """
        if layer is None or not layer.isValid():
            return

        mgr = layer.actions()
        script_dir = os.path.dirname(os.path.abspath(__file__))
        action_code_path = os.path.join(script_dir, "picture_action.py")

        try:
            with open(action_code_path, "r", encoding="utf-8") as f:
                action_code = f.read()
        except Exception as e:
            if self.DEBUG:
                self.iface.messageBar().pushWarning(
                    self.tr("DEBUG"),
                    self.tr("Failed to load picture_action.py: {err}").format(err=str(e))
                )
            return

        action_code = action_code.replace("%GETPHOTO%", build_getphoto_url(self.project_domain))

        action = QgsAction(
            QgsAction.GenericPython,
            'Picture',
            action_code,
            "",
            False
        )

        action.setActionScopes({
            'Feature',
            'Canvas',
            'Form',
            'Layer',
            'Field'
        })

        mgr.addAction(action)

        # set the action to be visible in the attribute table (Show in attribute table)
        if layer:
            config = layer.attributeTableConfig()
            config.setActionWidgetVisible(True)
            config.setActionWidgetStyle(QgsAttributeTableConfig.ButtonList)
            layer.setAttributeTableConfig(config)

    def createTableFromData(self, data_list, layer_name):
        """
        Save non-spatial data to CSV and load it into QGIS.
        """
        if not data_list:
            return

        # Determine target path
        use_temp = True
        target_dir = None
        
        if self.main_widget:
            if hasattr(self.main_widget, "tempFileCheckBox"):
                use_temp = self.main_widget.tempFileCheckBox.isChecked()
            if not use_temp and hasattr(self.main_widget, "dirLineEdit"):
                target_dir = self.main_widget.dirLineEdit.text().strip()

        if self.DEBUG:
            self.iface.messageBar().pushMessage(
                "DEBUG",
                "use_temp: {use_temp}, target_dir: {target_dir}".format(
                    use_temp=use_temp,
                    target_dir=target_dir
                )
            )


        final_path = None

        if use_temp:
            # Generate random folder name OBMCon-XXXXXX
            random_chars = ''.join(random.choice(string.ascii_letters + string.digits) for _ in range(6))
            custom_dir = f"OBMCon-{random_chars}"
            temp_base = tempfile.gettempdir()
            final_dir = os.path.join(temp_base, custom_dir)
            os.makedirs(final_dir, exist_ok=True)
            final_path = os.path.join(final_dir, f"{layer_name}.csv")
        else:
            # Prompt for filename
            default_path = os.path.join(target_dir, f"{layer_name}.csv") if target_dir else f"{layer_name}.csv"
            filename, _ = QFileDialog.getSaveFileName(
                self,
                self.tr("Save CSV"),
                default_path,
                self.tr("CSV files (*.csv)")
            )
            if filename:
                final_path = filename
            else:
                return # User canceled

        fieldnames = data_list[0].keys()

        try:
            with open(final_path, 'w', newline='', encoding='utf-8') as csvfile:
                writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                writer.writeheader()
                for record in data_list:
                    if isinstance(record, dict):
                        writer.writerow(record)
            
            if self.DEBUG:
                self.iface.messageBar().pushMessage(
                    self.tr("DEBUG"), 
                    self.tr("CSV written to: {path}").format(path=final_path), 
                    duration=5
                )

            # Load into QGIS
            uri = f"file:///{final_path}?delimiter=,&detectTypes=yes&geomType=none"
            
            layer = self.iface.addVectorLayer(uri, layer_name, "delimitedtext")
            if not layer or not layer.isValid():
                layer = self.iface.addVectorLayer(final_path, layer_name, "ogr")

            if layer and layer.isValid():
                if self.DEBUG:
                    self.iface.messageBar().pushMessage(
                        self.tr("DEBUG"),
                        self.tr("Added table layer: {name}").format(name=layer.name()),
                        duration=3
                    )
                # Add Picture action to this data layers
                self.add_picture_action_to_layer(layer)

        except Exception as e:
            QtWidgets.QMessageBox.critical(self, self.tr("Error"), self.tr("Failed to save or load table: {err}").format(err=str(e)))


    def createLayerFromGeojson(self, json_obj, layer_name):
        # Determine target directory
        use_temp = True
        target_dir = None
        
        if self.main_widget:
            if hasattr(self.main_widget, "tempFileCheckBox"):
                use_temp = self.main_widget.tempFileCheckBox.isChecked()
            if not use_temp and hasattr(self.main_widget, "dirLineEdit"):
                target_dir = self.main_widget.dirLineEdit.text().strip()

        final_path = None

        if use_temp:
            # Generate random folder name OBMCon-XXXXXX
            random_chars = ''.join(random.choice(string.ascii_letters + string.digits) for _ in range(6))
            custom_dir = f"OBMCon-{random_chars}"
            temp_base = tempfile.gettempdir()
            final_dir = os.path.join(temp_base, custom_dir)
            os.makedirs(final_dir, exist_ok=True)
            final_path = os.path.join(final_dir, f"{layer_name}.geojson")
        else:
             # Prompt for filename
            default_path = os.path.join(target_dir, f"{layer_name}.geojson") if target_dir else f"{layer_name}.geojson"
            filename, _ = QFileDialog.getSaveFileName(
                self,
                self.tr("Save GeoJSON"),
                default_path,
                self.tr("GeoJSON files (*.geojson)")
            )
            if filename:
                final_path = filename
            else:
                return # User canceled
        
        if self.DEBUG:
            self.iface.messageBar().pushMessage(
                self.tr("DEBUG"), 
                self.tr("Creating geojson at: {path}").format(path=final_path), 
                duration=5
            )

        try:
            with open(final_path, 'w', encoding='utf-8') as f:
                json.dump(json_obj, f, ensure_ascii=False)
            if self.DEBUG:
                self.iface.messageBar().pushMessage(
                    self.tr("DEBUG"), 
                    self.tr("Geojson written."), 
                    duration=5
                )
            
            # Get existing layers before adding the new one
            layers_before = set(QgsProject.instance().mapLayers().keys())
            
            added_layer = None
            try:
                if hasattr(self, "iface") and self.iface:
                    added_layer = self.iface.addVectorLayer(final_path, layer_name, "ogr")
                else:
                    raise RuntimeError(self.tr("No iface available"))
            except Exception:
                if self.DEBUG:
                    raise
                added_layer = None

            # User canceled or failed to add: return silently
            if added_layer is None:
                return None

            # Get all layers after adding - QGIS may create multiple layers from GeoJSON
            layers_after = set(QgsProject.instance().mapLayers().keys())
            new_layer_ids = layers_after - layers_before
            
            # Collect all newly added layers
            new_layers = []
            for layer_id in new_layer_ids:
                layer = QgsProject.instance().mapLayer(layer_id)
                if layer is not None and layer.isValid():
                    new_layers.append(layer)
            
            # If no new layers detected, fall back to the returned layer
            if not new_layers and added_layer is not None and added_layer.isValid():
                new_layers = [added_layer]
            
            # Add Picture action to all new layers
            for vl in new_layers:
                self.add_picture_action_to_layer(vl)
                if self.DEBUG:
                    self.iface.messageBar().pushMessage(
                        self.tr("DEBUG"),
                        self.tr("Added Picture action to layer: {name}").format(name=vl.name()),
                        duration=3
                    )

            # Return the first layer (usually the one addVectorLayer returned)
            return added_layer if added_layer and added_layer.isValid() else (new_layers[0] if new_layers else None)

        except Exception as e:
            # try:
            #     os.unlink(final_path)
            # except Exception:
            #     pass
            if self.DEBUG:
                raise
            QtWidgets.QMessageBox.critical(self, self.tr("Error"), self.tr("Failed to create layer: {err}").format(err=str(e)))



#-- PDS API to GeoJSON ------------------------------------------
    from qgis.core import QgsGeometry, QgsJsonExporter, QgsFeature

    def obm_json_to_geojson_qgis(self, obm_json_data):
        """
        Convert OBM JSON format into a GeoJSON FeatureCollection using QGIS core classes.

        Args:
            obm_json_data (dict or str): OBM API JSON response

        Returns:
            dict: GeoJSON FeatureCollection
        """
        # If a JSON string was provided, parse it
        if isinstance(obm_json_data, str):
            obm_json_data = json.loads(obm_json_data)
        
        # Basic structure validation
        if not isinstance(obm_json_data, dict) or 'data' not in obm_json_data:
            raise ValueError("Invalid OBM JSON structure. Expected dict with 'data' key.")
        
        records = obm_json_data['data']
        if not isinstance(records, list):
            raise ValueError("Expected 'data' to be a list of records.")
        
        # Build GeoJSON FeatureCollection
        geojson = {
            "type": "FeatureCollection",
            "features": []
        }
        
        for record in records:
            if not isinstance(record, dict):
                continue
                
            # Geometry processing via QgsGeometry (WKT expected in 'obm_geometry')
            wkt_geometry = record.get('obm_geometry')
            if not wkt_geometry:
                continue  # Skip records without geometry
            
            try:
                # Convert WKT -> QgsGeometry -> GeoJSON geometry
                qgs_geometry = QgsGeometry.fromWkt(wkt_geometry)
                if qgs_geometry.isEmpty():
                    print(f"Warning: Empty geometry for WKT: {wkt_geometry}")
                    continue
                    
                # Use QgsGeometry.asJson for GeoJSON geometry text, then parse
                geojson_geometry_str = qgs_geometry.asJson(precision=8)
                geojson_geometry = json.loads(geojson_geometry_str)
                
            except Exception as e:
                print(f"Warning: Failed to parse geometry '{wkt_geometry}': {e}")
                continue
            
            # Build properties excluding the geometry field
            properties = {}
            for key, value in record.items():
                if key != 'obm_geometry':
                    properties[key] = value
            
            # Create feature entry
            feature = {
                "type": "Feature",
                "geometry": geojson_geometry,
                "properties": properties
            }
            
            geojson["features"].append(feature)
        
        return geojson



