# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ObmConnect
                             A QGIS plugin
 Plugin for connecting to OpenBioMaps and loading spatial layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
 -------------------
     begin                : 2025-07-21
     git sha              : $Format:%H$
     copyright            : (C) 2025 by Attila Gáspár
     email                : gezsaj@gmail.com
***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                    *
 ***************************************************************************/
"""

from __future__ import annotations

from pathlib import Path
import importlib
import os
import shutil
import subprocess
import sys
import tempfile
import glob

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QObject
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QDockWidget, QToolButton

from .obm_connect_dockwidget import ObmConnectDockWidget

from qgis.gui import QgsMapTool
from qgis.core import QgsFeature, QgsRectangle, QgsAction, QgsExpressionContext, QgsExpressionContextScope

# ---------- Plugin implementation ----------

class ObmConnect:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """
        :param iface: QGIS interface for manipulating the application at runtime.
        :type iface: QgisInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # i18n
        self._install_translator()

        # UI state
        self.actions: list[QAction | list[QAction] | tuple[QAction, ...]] = []
        self.menu = self.tr("&OBM Connect")
        self.toolbar = self.iface.addToolBar("ObmConnect")
        self.toolbar.setObjectName("ObmConnect")

        self.plugin_is_active = False
        self.dockwidget: ObmConnectDockWidget | None = None

        # Best-effort temp cleanup
        self.cleanup_all_geojson_temp()


    # ------------- Utilities -------------

    def _install_translator(self) -> None:
        """Install Qt translator if a locale file exists."""
        locale = (QSettings().value("locale/userLocale") or "")[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", f"ObmConnect_{locale}.qm")
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    # noinspection PyMethodMayBeStatic
    def tr(self, message: str):
        """Translate text via Qt translation API."""
        return QCoreApplication.translate("ObmConnect", message)

    def add_action(
        self,
        icon_path: str,
        text: str,
        callback,
        enabled_flag: bool = True,
        add_to_menu: bool = True,
        add_to_toolbar: bool = True,
        status_tip: str | None = None,
        whats_this: str | None = None,
        parent=None,
    ) -> QAction:
        """Create and register a QAction with optional menu/toolbar placement."""
        action = QAction(QIcon(icon_path), text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip:
            action.setStatusTip(status_tip)
        if whats_this:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)
        if add_to_menu:
            self.iface.pluginMenu().addAction(action)

        self.actions.append(action)
        return action

    def cleanup_all_geojson_temp(self) -> None:
        """Delete all OBMCon-* directories under system temp path."""
        temp_dir = tempfile.gettempdir()
        pattern = os.path.join(temp_dir, "OBMCon-*")
        for dirpath in glob.glob(pattern):
            if os.path.isdir(dirpath):
                try:
                    shutil.rmtree(dirpath)
                except OSError:
                    # Best-effort cleanup; ignore locked or transient issues
                    pass

    # ------------- QGIS entrypoints -------------

    def initGui(self) -> None:
        """Create the menu entry and toolbar icon in the QGIS GUI."""
        # icon_path = ":/plugins/obm_connect/icon.png"
        icon_path = os.path.join(self.plugin_dir, "icon.png")
        self.add_action(
            icon_path=icon_path,
            text=self.tr("OBM Connect"),
            callback=self.run,
            parent=self.iface.mainWindow(),
            status_tip=self.tr("Connect to OpenBioMaps."),
        )

        self.add_picture_maptool()

    def on_close_plugin (self) -> None:
        """Cleanup when the dockwidget is closed."""
        if self.dockwidget:
            try:
                self.dockwidget.closing_plugin.disconnect(self.on_close_plugin )
            except Exception:
                pass
        self.plugin_is_active = False

    def unload(self) -> None:
        """Remove menu items and toolbar icons from QGIS GUI."""
        plugins_menu = self.iface.pluginMenu()

        for action in self.actions:
            if isinstance(action, (list, tuple)):
                for a in action:
                    if isinstance(a, QAction):
                        try:
                            plugins_menu.removeAction(a)
                        except Exception:
                            pass
                        try:
                            self.iface.removeToolBarIcon(a)
                        except Exception:
                            pass
            else:
                if isinstance(action, QAction):
                    try:
                        plugins_menu.removeAction(action)
                    except Exception:
                        pass
                    try:
                        self.iface.removeToolBarIcon(action)
                    except Exception:
                        pass

        # Delete toolbar reference if present
        if hasattr(self, "toolbar"):
            try:
                del self.toolbar
            except Exception:
                pass

        # Remove the dockwidget
        if self.dockwidget:
            try:
                self.dockwidget.closing_plugin.disconnect(self.on_close_plugin)
            except Exception:
                pass
            
            self.iface.removeDockWidget(self.dockwidget)
            self.dockwidget = None
            self.plugin_is_active = False

    def run(self) -> None:
        """Run method that loads and starts the plugin."""
        if self.plugin_is_active:
            # Already active: just raise the dock
            if self.dockwidget:
                self._raise_dock()
            return

        self.plugin_is_active = True

        if self.dockwidget is None:
            self.dockwidget = ObmConnectDockWidget(self.iface)
            self.dockwidget.setObjectName("ObmConnectDockWidget")
            self.dockwidget.setWindowTitle(self.tr("OBM Connect"))
            self.dockwidget.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
            self.dockwidget.connectPushButton.setDefault(True)

        self.dockwidget.closing_plugin.connect(self.on_close_plugin )

        # Prefer modern API (QGIS 3.14+), fallback to legacy if unavailable
        try:
            # tabifyWith param is optional; omitting allows QGIS to choose reasonable peers
            # raiseTab=True ensures visibility after adding
            self.iface.addTabifiedDockWidget(
                Qt.LeftDockWidgetArea,
                self.dockwidget,
                [],  # let QGIS pick; avoids fragile objectName dependencies
                True,
            )
        except Exception:
            self._add_dock_legacy()

        self._raise_dock()

    def _add_dock_legacy(self) -> None:
        """Fallback for older QGIS versions without addTabifiedDockWidget."""
        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dockwidget)

        # Try to tabify with Layers panel if available
        main_window = self.iface.mainWindow()
        layers_dock = main_window.findChild(QDockWidget, "Layers")
        if not layers_dock:
            # Heuristic search by object name hints
            for dock in main_window.findChildren(QDockWidget):
                name = (dock.objectName() or "").lower()
                if "layer" in name or "legend" in name:
                    layers_dock = dock
                    break

        if layers_dock:
            main_window.tabifyDockWidget(layers_dock, self.dockwidget)

        self.dockwidget.show()

    def _raise_dock(self) -> None:
        """Ensure the dockwidget is visible and focused."""
        if not self.dockwidget:
            return
        try:
            # Preferred: ensure visible tab in tabified area
            self.dockwidget.raise_()
        except Exception:
            pass
        self.dockwidget.show()

    def add_picture_maptool(self):
        icon_path = os.path.join(self.plugin_dir, "picture48.png")
        self.picture_action = QAction(QIcon(icon_path), "Picture", self.iface.mainWindow())
        self.picture_action.setCheckable(True)

        group = self.iface.mapToolActionGroup()
        if group is not None:
            group.addAction(self.picture_action)

        self.picture_tool = PictureActionTool(self.iface)
        self.picture_tool.setAction(self.picture_action)

        def on_picture_action_triggered():
            # This is where you actually enable it!
            self.iface.mapCanvas().setMapTool(self.picture_tool)
            self.picture_action.setChecked(True)

        self.picture_action.triggered.connect(on_picture_action_triggered)

        self.toolbar.addAction(self.picture_action)
        self.actions.append(self.picture_action)


class PictureActionTool(QgsMapTool):

    def __init__(self, iface):
        super().__init__(iface.mapCanvas())
        self.iface = iface

    def canvasReleaseEvent(self, event):
        layer = self.iface.activeLayer()
        if not layer:
             self.iface.messageBar().pushWarning("OBM Connect", "No active layer.", duration=5)
             return

        # Check for Picture action
        action_manager = layer.actions()
        action_id = None
        for act in action_manager.actions():
            if act.type() == getattr(act, "GenericPython", 7) and act.name() == "Picture":
                action_id = act.id()
                break
        
        if not action_id:
            self.iface.messageBar().pushWarning("OBM Connect", "The active vector layer has no Picture actions")
            return

        # Click coordinates on the map (in layer coordinates)
        point = self.toLayerCoordinates(layer, event.mapPoint())
        tolerance = self.calc_tolerance()         
        # Search for a circle around the click
        search_rect = QgsRectangle(
            point.x() - tolerance, point.y() - tolerance,
            point.x() + tolerance, point.y() + tolerance
        )
        feats = [f for f in layer.getFeatures(search_rect) if f.geometry().intersects(search_rect)]
        if not feats:
            self.iface.mainWindow().statusBar().showMessage("No features found at this position.", 2000)
            return
        feature = feats[0]

        # Run action
        context = QgsExpressionContext()
        scope = QgsExpressionContextScope("layer_action_scope")
        # Pass all found feature IDs to the action scope
        found_fids = [str(f.id()) for f in feats]
        scope.setVariable("obm_found_fids", ",".join(found_fids))
        context.appendScope(scope)

        layer.actions().doAction(
            action_id,    # QUuid returned by addAction() or actions().actions()[-1].id() etc.
            feature,
            context
        )

    def calc_tolerance(self, px=5):
        """Returns a tolerance (in layer units), e.g., within a 5-pixel radius."""
        canvas = self.iface.mapCanvas()
        pt1 = canvas.getCoordinateTransform().toMapCoordinates(0, 0)
        pt2 = canvas.getCoordinateTransform().toMapCoordinates(px, px)
        return abs(pt2.x() - pt1.x())  # Usually one is enough in X and Y directions
    
# class AttribTableButtonInjector(QObject):
#     def __init__(self, iface):
#         super().__init__()
#         self.iface = iface
#         # watch for the creation of every new QWidget
#         iface.mainWindow().installEventFilter(self)
#         self.active_buttons = {}  # {dock:status}

#     def eventFilter(self, source, event):
#         # only run if a new attribute table is created!
#         from qgis.gui import QgsAttributeTableDialog
#         # 17 == QEvent.ChildAdded birth of a new widget
#         if event.type() == 17:
#             child = event.child()
#             if isinstance(child, QgsAttributeTableDialog):
#                 self.inject_button(child)
#         return False 

#     def inject_button(self, dock):
#         button = QToolButton(dock)
#         button.setText("Picture")
#         button.setCheckable(True)
#         button.setToolTip("Picture action: clicking on a record starts a picture")
#         dock.layout().addWidget(button)
#         self.active_buttons[dock] = False

#         def toggle_action_mode(checked):
#             self.active_buttons[dock] = checked

#         button.toggled.connect(toggle_action_mode)

#         def on_row_clicked(index):
#             if self.active_buttons[dock]:
#                 layer = dock.layer()
#                 selected_row = index.row()
#                 model = dock.tableView().model()
#                 feature = model.data(model.index(selected_row, 0), role=model.FeatureRole)
#                 # Extracting the QUuid of the "Picture" action
#                 action_id = None
#                 for act in layer.actions().actions():
#                     if act.type() == QgsAction.GenericPython and act.name() == "Picture":
#                         action_id = act.id()
#                         break
#                 if action_id:
#                     layer.actions().doAction(action_id, feature, QgsExpressionContext())

#         dock.tableView().clicked.connect(on_row_clicked)