# coding=utf-8
"""DockWidget test.

.. note:: This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

"""

__author__ = 'gezsaj@gmail.com'
__date__ = '2025-07-21'
__copyright__ = 'Copyright 2025, Attila Gáspár'

import unittest
from types import SimpleNamespace
from unittest import mock
import requests

from qgis.PyQt import QtWidgets
from qgis.PyQt.QtGui import QStandardItemModel, QStandardItem

# from test.utilities import get_qgis_app
# QGIS_APP = get_qgis_app()

# Import the module under test
import obm_connect_dockwidget as dockmod
from obm_connect_dockwidget import ObmConnectDockWidget


class FakeMessageBar:
    def __init__(self):
        self.last = None

    def pushMessage(self, *args, **kwargs):
        self.last = ("message", args, kwargs)

    def pushWarning(self, *args, **kwargs):
        self.last = ("warning", args, kwargs)

    def pushCritical(self, *args, **kwargs):
        self.last = ("critical", args, kwargs)


class FakeIface:
    def __init__(self):
        self._mb = FakeMessageBar()

    def messageBar(self):
        return self._mb


class ObmConnectDockWidgetTest(unittest.TestCase):
    """Test dockwidget works."""

    def setUp(self):
        self.iface = FakeIface()
        # Create widget with fake iface so messageBar calls do not raise
        self.dock = ObmConnectDockWidget(self.iface)

    def tearDown(self):
        self.dock = None

    def test_initial_state(self):
        # Project combo disabled and default tab index 0
        self.assertFalse(self.dock.projectComboBox.isEnabled())
        self.assertEqual(self.dock.tabWidget.currentIndex(), 0)

    @mock.patch('obm_connect_dockwidget.requests.get')
    def test_load_projects_with_list_response(self, mock_get):
        # Mock a requests.Response-like object returning a list of projects
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.json.return_value = [
            {"name": "Project A", "project_table": "proj_a"},
            {"name": "Project B", "project_table": "proj_b"}
        ]
        mock_get.return_value = resp

        self.dock.load_projects("example.com")

        # Combo should be enabled and contain the projects + placeholder
        self.assertTrue(self.dock.projectComboBox.isEnabled())
        # first item is placeholder
        self.assertEqual(self.dock.projectComboBox.count(), 1 + 2)
        # internal project list stored
        self.assertEqual(len(self.dock._project_list), 2)
        self.assertEqual(self.dock.projectComboBox.itemText(1), "Project A")

    @mock.patch('obm_connect_dockwidget.requests.get')
    def test_load_projects_with_data_key_response(self, mock_get):
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.json.return_value = {"data": [{"name": "OnlyProject", "project_table": "onlyproj"}]}
        mock_get.return_value = resp

        self.dock.load_projects("example.com")
        self.assertTrue(self.dock.projectComboBox.isEnabled())
        self.assertEqual(self.dock.projectComboBox.count(), 2)
        self.assertEqual(self.dock.projectComboBox.itemText(1), "OnlyProject")

    @mock.patch('obm_connect_dockwidget.requests.post')
    def test_ensure_valid_token_refresh_success(self, mock_post):
        # Set a refresh token so refresh flow is attempted
        self.dock._refresh_token = "existing_refresh"
        # Prepare fake token endpoint response for refresh grant
        resp = mock.Mock()
        resp.status_code = 200
        resp.json.return_value = {
            "access_token": "new_access",
            "refresh_token": "new_refresh",
            "expires_in": 3600
        }
        mock_post.return_value = resp

        # Ensure obmServerLineEdit has some value (used to build URL)
        self.dock.obmServerLineEdit.setText("example.com")

        ok = self.dock.ensure_valid_token()
        self.assertTrue(ok)
        self.assertEqual(self.dock._access_token, "new_access")
        self.assertEqual(self.dock._refresh_token, "new_refresh")

    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'ensure_valid_token', return_value=True)
    def test_on_project_selected_populates_table_view(self, mock_ensure_token, mock_get):
        # Prepare project in internal list
        project = {"project_table": "projx", "name": "Project X"}
        self.dock._project_list = [project]

        # Mock response for data-tables endpoint returning list of tables
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.headers = {"Content-Type": "application/json"}
        resp.json.return_value = [
            {"name": "projx_table1", "schema": "s1"},
            {"name": "projx_table2", "schema": "s2"}
        ]
        mock_get.return_value = resp

        # Call with idx=1 (first real project)
        self.dock.on_project_selected(1)

        model = self.dock.tablesTableView.model()
        self.assertIsNotNone(model)
        # Two rows expected
        self.assertEqual(model.rowCount(), 2)
        # Since raw names start with project_table + "_", display name should start with underscore
        first_display = model.item(0, 0).text()
        self.assertTrue(first_display.startswith("_"))
        self.assertEqual(first_display, "_table1")


    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'ensure_valid_token', return_value=True)
    def test_on_project_selected_404_shows_unavailable_and_resets(self, mock_ensure_token, mock_get):
        project = {"project_table": "projx", "name": "Project X"}
        self.dock._project_list = [project]

        resp = mock.Mock()
        resp.raise_for_status.side_effect = requests.exceptions.HTTPError("404 Client Error")
        mock_get.return_value = resp

        with mock.patch.object(self.dock, "_reset_state_and_views") as mock_reset:
            self.dock.on_project_selected(1)
            mock_reset.assert_called_once()


    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'ensure_valid_token', return_value=True)
    def test_on_project_selected_non_json_content_type_shows_unavailable_and_resets(self, mock_ensure_token, mock_get):
        project = {"project_table": "projx", "name": "Project X"}
        self.dock._project_list = [project]

        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.headers = {"Content-Type": "text/html; charset=utf-8"}  # not JSON
        resp.text = "<html>Not JSON</html>"
        mock_get.return_value = resp

        with mock.patch.object(self.dock, "_reset_state_and_views") as mock_reset:
            self.dock.on_project_selected(1)
            mock_reset.assert_called_once()

    # @mock.patch('obm_connect_dockwidget.requests.get')
    # @mock.patch.object(dockmod.ObmConnectDockWidget, 'ensure_valid_token', return_value=True)
    # def test_on_table_double_clicked_opens_filter_dialog(self, mock_ensure_token, mock_get):
    #     print("Starting test_on_table_double_clicked_opens_filter_dialog...")
    #     # Prepare a fake project and table view model with one table
    #     project = {"project_table": "projz", "name": "Project Z"}
    #     self.dock._project_list = [project]

    #     model = QStandardItemModel()
    #     model.setHorizontalHeaderLabels(["Table name", "Schema"])
    #     item = QStandardItem("_tableA")
    #     model.appendRow([item, QStandardItem("s")])
    #     self.dock.tablesTableView.setModel(model)

    #     print("DBG model row count:", model.rowCount())

    #     # Place the dock into a state with a valid access token and server
    #     # self.dock._access_token = "tok"
    #     # self.dock._token_expires_time = datetime.now().timestamp() + 3600
    #     # self.dock.obmServerLineEdit.setText("example.com")

    #     print("DBG access_token:", self.dock._access_token)

    #     # Mock fields response
    #     resp = mock.Mock()
    #     resp.raise_for_status = mock.Mock()
    #     resp.json.return_value = {"fields": [{"name": "col1"}, {"name": "col2"}]}
    #     mock_get.return_value = resp


    #     print("DBG rows:", model.rowCount(), "token_exp:", getattr(self.dock, "_token_expires_time", None))

    #     # Replace the dialog class used to present fields to capture the model set
    #     class FakeFieldView:
    #         def __init__(self):
    #             self.model = None
    #         def setModel(self, m):
    #             self.model = m

    #     class FakeDialog:
    #         last_instance = None
    #         def __init__(self, parent=None):
    #             self.fieldListView = FakeFieldView()
    #             FakeDialog.last_instance = self
    #         def exec_(self):
    #             return None

    #     with mock.patch('obm_connect_dockwidget.ObmConnectFilterDialog', new=FakeDialog):
    #         # Create a QModelIndex for row 0, column 0
    #         idx = self.dock.tablesTableView.model().index(0, 0)
    #         # call method under test
    #         self.dock.on_table_double_clicked(idx)

    #         # # Ensure dialog was created and model has two rows
    #         # self.assertIsNotNone(FakeDialog.last_instance)
    #         # fm = FakeDialog.last_instance.fieldListView.model
    #         # self.assertIsNotNone(fm)
    #         # self.assertEqual(fm.rowCount(), 2)
    #         # # verify first field name matches expected
    #         # first_field_item = fm.item(0, 0)
    #         # self.assertIsNotNone(first_field_item)
    #         # self.assertEqual(first_field_item.text(), "col1")


    #         # Invoke the handler
    #         print("Invoking on_table_double_clicked...")

    #         self.dock.on_table_double_clicked(idx)

    #         print("Invoked on_table_double_clicked...")
    #         # try:
    #         #     self.dock.on_table_double_clicked(1)
    #         # except Exception as e:
    #         #     self.fail(f"on_table_double_clicked raised: {e!r}")


    #         # Ensure the dialog was created
    #         print("Ensuring dialog creation...")
    #         self.assertIsNotNone(
    #             FakeDialog.last_instance,
    #             msg="Dialog was not created"
    #         )

    #         # Extract model from the dialog's field list view
    #         fm = FakeDialog.last_instance.fieldListView.model
    #         self.assertIsNotNone(
    #             fm,
    #             msg="Dialog.fieldListView.model is None"
    #         )

    #         # Check row count
    #         rows = fm.rowCount() if fm else None
    #         self.assertEqual(
    #             rows, 2,
    #             msg=f"Expected 2 rows, got: {rows}"
    #         )

    #         # Check the first item existence
    #         first_field_item = fm.item(0, 0) if fm else None
    #         self.assertIsNotNone(
    #             first_field_item,
    #             msg="First table cell is None"
    #         )

    #         # Check the first item's text
    #         val = first_field_item.text() if first_field_item else None
    #         self.assertEqual(
    #             val, "col1",
    #             msg=f"First field name mismatch, got: {val!r}"
    #         )



    # from unittest import mock
    # from qgis.PyQt.QtGui import QStandardItemModel, QStandardItem

    @mock.patch('obm_connect_dockwidget.requests.get')
    @mock.patch.object(dockmod.ObmConnectDockWidget, 'ensure_valid_token', return_value=True)
    def test_on_table_double_clicked_opens_filter_dialog(self, mock_ensure_token, mock_get):
        # Prepare a fake project and set the combo to select it
        project = {"project_table": "projz", "name": "Project Z"}
        self.dock._project_list = [project]
        self.dock.projectComboBox.clear()
        self.dock.projectComboBox.addItem("Select project")
        self.dock.projectComboBox.addItem(project["name"])
        self.dock.projectComboBox.setCurrentIndex(1)

        # Table model with one displayed row "_tableA"
        model = QStandardItemModel()
        model.setHorizontalHeaderLabels(["Table name", "Schema"])
        item = QStandardItem("_tableA")
        model.appendRow([item, QStandardItem("s")])
        self.dock.tablesTableView.setModel(model)

        # Mock fields response for the GET request
        resp = mock.Mock()
        resp.raise_for_status = mock.Mock()
        resp.headers = {"Content-Type": "application/json"}
        resp.json.return_value = {"fields": [{"name": "col1"}, {"name": "col2"}]}
        mock_get.return_value = resp

        # Fake dialog and field view that match runtime expectations
        class _DummySignal:
            def connect(self, *_args, **_kwargs):
                return None

        class _DummySelModel:
            def __init__(self):
                self.currentChanged = _DummySignal()

        class FakeFieldView:
            def __init__(self):
                self.model = None
            def setModel(self, m):
                self.model = m
            def selectionModel(self):
                return _DummySelModel()

        class FakeDialog:
            last_instance = None
            def __init__(self, parent=None, debug=False):
                self.fieldListView = FakeFieldView()
                FakeDialog.last_instance = self
            def _set_unique_count_on_all_button(self):
                # no-op in test
                return None
            def exec_(self):
                return None

        # Patch exactly where the symbol is looked up at runtime
        with mock.patch.object(dockmod, 'ObmConnectFilterDialog', new=FakeDialog):
            # Use a valid QModelIndex for row 0, col 0
            idx = self.dock.tablesTableView.model().index(0, 0)

            # Invoke the slot under test
            self.dock.on_table_double_clicked(idx)

            # Assertions
            self.assertIsNotNone(FakeDialog.last_instance, msg="Dialog was not created")
            fm = FakeDialog.last_instance.fieldListView.model
            self.assertIsNotNone(fm, msg="Dialog.fieldListView.model is None")

            rows = fm.rowCount() if fm else None
            self.assertEqual(rows, 2, msg=f"Expected 2 rows, got: {rows}")

            first_field_item = fm.item(0, 0) if fm else None
            self.assertIsNotNone(first_field_item, msg="First table cell is None")

            val = first_field_item.text() if first_field_item else None
            self.assertEqual(val, "col1", msg=f"First field name mismatch, got: {val!r}")




if __name__ == "__main__":
    unittest.main()

