# python
from obm_connect_filter import SqlWhereValidator
from PyQt5 import QtWidgets
import unittest

class TestSqlWhereValidator(unittest.TestCase):
    def setUp(self):
        # Instantiate validator
        self.validator = SqlWhereValidator()
        # Patch QMessageBox methods to avoid GUI popups during tests
        self._orig_info = QtWidgets.QMessageBox.information
        self._orig_warn = QtWidgets.QMessageBox.warning
        self._orig_crit = QtWidgets.QMessageBox.critical
        QtWidgets.QMessageBox.information = lambda *a, **k: None
        QtWidgets.QMessageBox.warning = lambda *a, **k: None
        QtWidgets.QMessageBox.critical = lambda *a, **k: None

        # Common field list that matches test expressions (lowercase accepted)
        self.fields = [
            "name", "age", "score", "active", "the_date", "the_time",
            "notes", "desc", "title", "species", "corpse", "status",
            "x", "col", "empty_field", "a", "b", "c", "tags", "code",
            "val", "s", "g", "e"
        ]

    def tearDown(self):
        # Restore QMessageBox
        QtWidgets.QMessageBox.information = self._orig_info
        QtWidgets.QMessageBox.warning = self._orig_warn
        QtWidgets.QMessageBox.critical = self._orig_crit

    def validate_ok(self, expr):
        res = self.validator.validate(expr, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=True)
        self.assertTrue(res.ok, msg=f"Expected valid but got errors: {expr} -> {res.errors}")

    def validate_not_ok(self, expr):
        res = self.validator.validate(expr, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=True)
        self.assertFalse(res.ok, msg=f"Expected invalid but reported ok: {expr}")
        self.assertTrue(res.errors, msg="Expected error messages for invalid expression")

    def test_valid_basic_comparisons(self):
        # basic comparisons (ensure fields are double-quoted for validator)
        self.validate_ok('"name" = \'John\'')
        self.validate_ok('"age" >= 18')
        self.validate_ok('"score" < 3.14')
        self.validate_ok('"active" = true')
        self.validate_ok('"the_date" = \'2020-01-01\'')
        self.validate_ok('"the_time" = \'12:34:56\'')
    def test_valid_like_ilike_not_like(self):
        # LIKE / ILIKE / NOT ILIKE
        self.validate_ok('"notes" LIKE \'%foo%\'')
        self.validate_ok('"desc" ILIKE \'Bar%\'')
        self.validate_ok('"title" NOT ILIKE \'baz%\'')
        self.validate_ok('"title" NOT LIKE \'%baz%\'')
        self.validate_ok('"notes" LIKE \'foo%\'')
        self.validate_ok('"desc" LIKE \'%bar\'')
    def test_valid_composite_wildcard(self):
        # Composite wildcard
        self.validate_ok('"species" LIKE \'a%b%c\'')
    def test_valid_in(self):
        # IN / NOT IN
        self.validate_ok('"status" IN (\'a\',\'b\')')
        self.validate_ok('"x" NOT IN (1,2)')
    def test_valid_is_null_is_empty(self):
        # IS NULL / IS NOT NULL / EMPTY
        self.validate_ok('"col" IS NULL')
        self.validate_ok('"col" IS NOT NULL')
        self.validate_ok('"empty_field" IS EMPTY')
        self.validate_ok('"empty_field" IS NOT EMPTY')
    def test_valid_logical_combinations(self):
        # Logical combinations
        self.validate_ok('"a" = 1 AND "b" = 2')
        self.validate_ok('"a" = 1 OR "b" = 2')
        self.validate_ok('NOT "a" = 1')
        self.validate_ok('("a" = 1 AND "b" = 2) OR "c" = 3')
    def test_valid_literal_first(self):
        # Literal-first cases are accepted (literal left, quoted identifier right)
        self.validate_ok("'John' = \"name\"")
        self.validate_ok("5 = \"age\"")
        self.validate_ok("'Bar%' LIKE \"desc\"")
        self.validate_ok("'a%b%c' LIKE \"species\"")
        self.validate_ok("'Abc%'   iLiKe   \"name\"")
        # underscore wildcard
        self.validate_ok("'A_c%' LIKE \"val\"")
        self.validate_ok("'%x_y' ILIKE \"val\"")

    def test_invalid_unterminated_quote_and_paren(self):
        # Unterminated single quote
        expr = '"name" = \'John'
        res = self.validator.validate(expr, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res.ok)
        # Expect single quote related error kind
        self.assertIsNotNone(res.error_kind)
        self.assertIn('single_quote', str(res.error_kind))
        # Unbalanced parentheses (missing closing)
        expr2 = '("a" = 1'
        res2 = self.validator.validate(expr2, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res2.ok)
        self.assertIsNotNone(res2.error_kind)
        self.assertIn('paren_open', str(res2.error_kind) or '')

    def test_invalid_unquoted_identifier_and_adjacent_values(self):
        # Unquoted identifier should be flagged
        expr = 'age = 1'
        res = self.validator.validate(expr, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res.ok)
        self.assertTrue(any('Unquoted identifier' in e or 'Unquoted' in e for e in res.errors))
        # Adjacent values without operator
        expr2 = '"a" = 1 2'
        res2 = self.validator.validate(expr2, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res2.ok)
        self.assertTrue(any('Missing operator' in e or 'adjacent' in e.lower() for e in res2.errors))

    def test_invalid_trailing_logic_and_missing_in_parenthesis(self):
        expr = '"a" = 1 AND'
        res = self.validator.validate(expr, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res.ok)
        self.assertTrue(any('Trailing logical operator' in e or 'dangling' in e.lower() for e in res.errors))
        # IN without parentheses
        expr2 = '"x" IN 1,2'
        res2 = self.validator.validate(expr2, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res2.ok)
        self.assertTrue(any('must be followed by a parenthesized list' in e or 'IN' in e for e in res2.errors))

    def test_invalid_both_identifiers_or_both_literals(self):
        # Identifiers on both sides
        expr = '"a" = "b"'
        res = self.validator.validate(expr, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res.ok)
        self.assertTrue(any('identifiers' in e.lower() or 'both_identifiers' in (res.error_kind or '') for e in res.errors))
        # Literals on both sides
        expr2 = "'a' = 'b'"
        res2 = self.validator.validate(expr2, field_names=self.fields, case_sensitive_fields=False, deep_sql_check=False)
        self.assertFalse(res2.ok)
        self.assertTrue(any('literals' in e.lower() or 'both_literals' in (res2.error_kind or '') for e in res2.errors))