# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ObmConnect
                             A QGIS plugin
 Plugin for connecting to OpenBioMaps and loading spatial layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
 -------------------
     begin                : 2025-07-21
     git sha              : $Format:%H$
     copyright            : (C) 2025 by Attila Gáspár
     email                : gezsaj@gmail.com
***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                    *
 ***************************************************************************/
"""

from __future__ import annotations

from pathlib import Path
import importlib
import os
import shutil
import subprocess
import sys
import tempfile
import glob

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QDockWidget

from .obm_connect_dockwidget import ObmConnectDockWidget


# ---------- Plugin implementation ----------

class ObmConnect:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """
        :param iface: QGIS interface for manipulating the application at runtime.
        :type iface: QgisInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # i18n
        self._install_translator()

        # UI state
        self.actions: list[QAction | list[QAction] | tuple[QAction, ...]] = []
        self.menu = self.tr("&OBM Connect")
        self.toolbar = self.iface.addToolBar("ObmConnect")
        self.toolbar.setObjectName("ObmConnect")

        self.pluginIsActive = False
        self.dockwidget: ObmConnectDockWidget | None = None

        # Best-effort temp cleanup
        self.cleanup_all_geojson_temp()

    # ------------- Utilities -------------

    def _install_translator(self) -> None:
        """Install Qt translator if a locale file exists."""
        locale = (QSettings().value("locale/userLocale") or "")[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", f"ObmConnect_{locale}.qm")
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    # noinspection PyMethodMayBeStatic
    def tr(self, message: str):
        """Translate text via Qt translation API."""
        return QCoreApplication.translate("ObmConnect", message)

    def add_action(
        self,
        icon_path: str,
        text: str,
        callback,
        enabled_flag: bool = True,
        add_to_menu: bool = True,
        add_to_toolbar: bool = True,
        status_tip: str | None = None,
        whats_this: str | None = None,
        parent=None,
    ) -> QAction:
        """Create and register a QAction with optional menu/toolbar placement."""
        action = QAction(QIcon(icon_path), text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip:
            action.setStatusTip(status_tip)
        if whats_this:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)
        if add_to_menu:
            self.iface.pluginMenu().addAction(action)

        self.actions.append(action)
        return action

    def cleanup_all_geojson_temp(self) -> None:
        """Delete all OBMCon-* directories under system temp path."""
        temp_dir = tempfile.gettempdir()
        pattern = os.path.join(temp_dir, "OBMCon-*")
        for dirpath in glob.glob(pattern):
            if os.path.isdir(dirpath):
                try:
                    shutil.rmtree(dirpath)
                except OSError:
                    # Best-effort cleanup; ignore locked or transient issues
                    pass

    # ------------- QGIS entrypoints -------------

    def initGui(self) -> None:
        """Create the menu entry and toolbar icon in the QGIS GUI."""
        # icon_path = ":/plugins/obm_connect/icon.png"
        icon_path = os.path.join(self.plugin_dir, "icon.png")
        self.add_action(
            icon_path=icon_path,
            text=self.tr("OBM Connect"),
            callback=self.run,
            parent=self.iface.mainWindow(),
            status_tip=self.tr("Connect to OpenBioMaps."),
        )

    def onClosePlugin(self) -> None:
        """Cleanup when the dockwidget is closed."""
        if self.dockwidget:
            try:
                self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)
            except Exception:
                pass
        self.pluginIsActive = False

    def unload(self) -> None:
        """Remove menu items and toolbar icons from QGIS GUI."""
        plugins_menu = self.iface.pluginMenu()

        for action in self.actions:
            if isinstance(action, (list, tuple)):
                for a in action:
                    if isinstance(a, QAction):
                        try:
                            plugins_menu.removeAction(a)
                        except Exception:
                            pass
                        try:
                            self.iface.removeToolBarIcon(a)
                        except Exception:
                            pass
            else:
                if isinstance(action, QAction):
                    try:
                        plugins_menu.removeAction(action)
                    except Exception:
                        pass
                    try:
                        self.iface.removeToolBarIcon(action)
                    except Exception:
                        pass

        # Delete toolbar reference if present
        if hasattr(self, "toolbar"):
            try:
                del self.toolbar
            except Exception:
                pass

    def run(self) -> None:
        """Run method that loads and starts the plugin."""
        if self.pluginIsActive:
            # Already active: just raise the dock
            if self.dockwidget:
                self._raise_dock()
            return

        self.pluginIsActive = True

        if self.dockwidget is None:
            self.dockwidget = ObmConnectDockWidget(self.iface)
            self.dockwidget.setObjectName("ObmConnectDockWidget")
            self.dockwidget.setWindowTitle(self.tr("OBM Connect"))
            self.dockwidget.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)

        self.dockwidget.closingPlugin.connect(self.onClosePlugin)

        # Prefer modern API (QGIS 3.14+), fallback to legacy if unavailable
        try:
            # tabifyWith param is optional; omitting allows QGIS to choose reasonable peers
            # raiseTab=True ensures visibility after adding
            self.iface.addTabifiedDockWidget(
                Qt.LeftDockWidgetArea,
                self.dockwidget,
                [],  # let QGIS pick; avoids fragile objectName dependencies
                True,
            )
        except Exception:
            self._add_dock_legacy()

        self._raise_dock()

    def _add_dock_legacy(self) -> None:
        """Fallback for older QGIS versions without addTabifiedDockWidget."""
        self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dockwidget)

        # Try to tabify with Layers panel if available
        main_window = self.iface.mainWindow()
        layers_dock = main_window.findChild(QDockWidget, "Layers")
        if not layers_dock:
            # Heuristic search by object name hints
            for dock in main_window.findChildren(QDockWidget):
                name = (dock.objectName() or "").lower()
                if "layer" in name or "legend" in name:
                    layers_dock = dock
                    break

        if layers_dock:
            main_window.tabifyDockWidget(layers_dock, self.dockwidget)

        self.dockwidget.show()

    def _raise_dock(self) -> None:
        """Ensure the dockwidget is visible and focused."""
        if not self.dockwidget:
            return
        try:
            # Preferred: ensure visible tab in tabified area
            self.dockwidget.raise_()
        except Exception:
            pass
        self.dockwidget.show()
