# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ObmConnectDockWidget
                                 A QGIS plugin
 Plugin for connecting to OpenBioMaps and loading spatial layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-07-21
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Attila Gáspár
        email                : gezsaj@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import threading
from datetime import datetime, timedelta

import requests
from qgis.core import Qgis
from qgis.PyQt import uic
from qgis.PyQt.QtCore import QCoreApplication, Qt, QSettings, pyqtSignal
from qgis.PyQt.QtGui import QStandardItemModel, QStandardItem
from qgis.PyQt import QtGui, QtWidgets
from qgis.PyQt.QtWidgets import QHeaderView

from obm_connect.obm_connect_filter import ObmConnectFilterDialog
from obm_connect.config import build_oauth_url, build_project_api_url, build_projects_url, DEFAULT_OBM_SERVER

FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "obm_connect_dockwidget_base.ui")
)


class ObmConnectDockWidget(QtWidgets.QDockWidget, FORM_CLASS):
    """
    Dock widget for OBM Connect plugin.

    Responsibilities:
    - Authenticate (OAuth password or refresh flow)
    - Load projects
    - List tables for a project and fetch table details in background threads
    - Open filter dialog for a selected table
    """

    closingPlugin = pyqtSignal()
    tableDetailsFetched = pyqtSignal(object, object)  # row_key, details dict

    def translate(self, message: str) -> str:
        return QCoreApplication.translate("ObmConnect", message)

    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.setupUi(self)

        # Development flag (keep for diagnostics)
        self.DEBUG = False

        # UI wiring
        self.connectPushButton.clicked.connect(self.on_connect_clicked)
        self.projectComboBox.currentIndexChanged.connect(self.on_project_selected)
        self.tablesTableView.doubleClicked.connect(self.on_table_double_clicked)
        self.projectComboBox.setEnabled(False)
        self.tabWidget.setCurrentIndex(0)
        self.rememberMeCheckBox.setEnabled(True)

        # Auth state
        self._access_token = None
        self._refresh_token = None
        self._token_expires_time = datetime.now()
        self._last_table_name = None
        self._project_list = []

        self.tableDetailsFetched.connect(self._on_table_details_fetched)

        # Initialize settings and try refresh if a refresh token is stored
        self._load_settings_and_attempt_refresh()

    def _load_settings_and_attempt_refresh(self):
        try:
            settings = QSettings()
            server = settings.value("obm_connect/server", DEFAULT_OBM_SERVER)
            email = settings.value("obm_connect/email", "")
            refresh_token = settings.value("obm_connect/refresh_token", "")
            remember_flag = settings.value("obm_connect/remember", False, type=bool)

            if hasattr(self, "obmServerLineEdit"):
                self.obmServerLineEdit.setText(str(server or ""))
            if hasattr(self, "emailLineEdit"):
                self.emailLineEdit.setText(str(email or ""))
            if hasattr(self, "rememberMeCheckBox"):
                try:
                    self.rememberMeCheckBox.setChecked(bool(remember_flag))
                except Exception:
                    pass

            self._refresh_token = str(refresh_token) if refresh_token else None

            if self._refresh_token:
                if self.ensure_valid_token():
                    self.load_projects(server)
                    self.tabWidget.setCurrentIndex(1)
                    self._set_info_text(server, connected=True)
                    try:
                        self.connectPushButton.setText(self.translate("Disconnect"))
                        self.rememberMeCheckBox.setEnabled(False)
                    except Exception:
                        pass
            else:
                self._set_info_text(connected=False)
        except Exception:
            # Do not fail plugin load due to settings read errors
            self._set_info_text(connected=False)

    def _set_info_text(self, server_display: str = None, connected: bool = True):
        if not hasattr(self, "InfoLineEdit"):
            return

        server_display = server_display or (
            self.obmServerLineEdit.text().strip() if hasattr(self, "obmServerLineEdit") else DEFAULT_OBM_SERVER
        )

        if connected:
            text = self.translate("Connected to {srv}").format(srv=server_display)
        else:
            text = self.translate("Not connected")

        if self.DEBUG:
            text = self.translate("DEBUG MODE! ") + text

        try:
            self.InfoLineEdit.setText(text)
        except Exception:
            pass

    def load_projects(self, obm_server: str):
        """Load projects list from server and populate projectComboBox."""
        server_url = build_projects_url(obm_server)
        if not server_url:
            self.iface.messageBar().pushWarning(self.translate("OBM Connect"), self.translate("Please provide a server URL."))
            return

        try:
            resp = requests.get(server_url, timeout=15)
            resp.raise_for_status()
            json_data = resp.json()

            # Normalize different API shapes
            if isinstance(json_data, list):
                projects = json_data
            elif isinstance(json_data, dict) and isinstance(json_data.get("data"), list):
                projects = json_data.get("data")
            elif isinstance(json_data, dict):
                values = [v for v in json_data.values() if isinstance(v, dict)]
                projects = values if values else [json_data]
            else:
                projects = []

            if not projects:
                self.iface.messageBar().pushWarning(self.translate("OBM Connect"), self.translate("No projects found or invalid response."))
                return

            self.projectComboBox.setEnabled(True)
            self.projectComboBox.clear()
            self._project_list = projects
            self.projectComboBox.addItem(self.translate("<< Select a project >>"))
            for project in projects:
                label = project.get("name") or project.get("project_table") or ""
                self.projectComboBox.addItem(label)
            self.projectComboBox.setCurrentIndex(0)

            if self.DEBUG:
                self.iface.messageBar().pushMessage(self.translate("DEBUG"), self.translate("Loaded projects from {url}").format(url=server_url))
        except Exception as exc:
            self.iface.messageBar().pushCritical(self.translate("OBM Connect"), self.translate("Error loading projects: {err}").format(err=exc))

    def ensure_valid_token(self) -> bool:
        """
        Ensure there is a valid access token.
        Attempt refresh with refresh_token, otherwise perform password grant.
        Returns True on success, False otherwise.
        """
        if self._access_token and datetime.now() < self._token_expires_time:
            return True

        server = self.obmServerLineEdit.text().strip() or DEFAULT_OBM_SERVER
        token_url = build_oauth_url(server)
        client_id = "mobile"
        client_secret = "123"

        self.iface.messageBar().pushMessage(self.translate("OBM Connect"), self.translate("Attempting to obtain access token."))

        # Try refresh token flow
        if self._refresh_token:
            try:
                payload = {"grant_type": "refresh_token", "refresh_token": self._refresh_token}
                resp = requests.post(token_url, data=payload, auth=(client_id, client_secret), headers={"Accept": "application/json"}, timeout=15)
                if resp.status_code == 200:
                    data = resp.json()
                    access = data.get("access_token")
                    refresh = data.get("refresh_token", self._refresh_token)
                    expires_in = int(data.get("expires_in", 3600))
                    if access:
                        self._access_token = access
                        self._refresh_token = refresh
                        self._token_expires_time = datetime.now() + timedelta(seconds=expires_in - 1)
                        self._persist_tokens_if_requested(server)
                        self._set_info_text(server, connected=True)
                        try:
                            self.connectPushButton.setText(self.translate("Disconnect"))
                        except Exception:
                            pass
                        self.iface.messageBar().pushMessage(self.translate("OBM Connect"), self.translate("Access token refreshed."))
                        if self.DEBUG:
                            self.iface.messageBar().pushMessage(self.translate("DEBUG"), self.translate("Refresh token: {token}").format(token=refresh))
                        return True
                else:
                    self.iface.messageBar().pushWarning(self.translate("OAuth"), self.translate("Refresh token is invalid or expired."))
            except Exception as exc:
                self.iface.messageBar().pushCritical(self.translate("OAuth"), self.translate("Refresh token error: {err}").format(err=str(exc)))

        # Password grant fallback
        email = self.emailLineEdit.text().strip() if hasattr(self, "emailLineEdit") else ""
        password = self.passwordLineEdit.text() if hasattr(self, "passwordLineEdit") else ""

        if not password:
            # Use Qgis.Critical for message level if available
            try:
                self.iface.messageBar().pushMessage("Error", self.translate("Please provide the password."), level=Qgis.Critical, duration=4)
            except Exception:
                from qgis.PyQt.QtWidgets import QMessageBox
                QMessageBox.warning(self, "Error", self.translate("Please provide the password."))
            if hasattr(self, "passwordLineEdit"):
                try:
                    self.passwordLineEdit.setFocus()
                    # optional visual hint
                    self.passwordLineEdit.setStyleSheet("border: 1px solid #d9534f;")
                except Exception:
                    pass
            self._set_not_connected()
            return False

        scopes = "get_data offline_access"
        payload = {"grant_type": "password", "username": email, "password": password, "scope": scopes}
        try:
            resp = requests.post(token_url, data=payload, auth=(client_id, client_secret), headers={"Accept": "application/json"}, timeout=15)
            if resp.status_code == 200:
                data = resp.json()
                access = data.get("access_token")
                refresh = data.get("refresh_token")
                expires_in = int(data.get("expires_in", 3600))
                if access:
                    self._access_token = access
                    self._refresh_token = refresh
                    self._token_expires_time = datetime.now() + timedelta(seconds=expires_in - 1)
                    # Persist tokens and server/email
                    try:
                        settings = QSettings()
                        settings.setValue("obm_connect/access_token", self._access_token)
                        settings.setValue("obm_connect/refresh_token", self._refresh_token)
                        settings.setValue("obm_connect/server", server)
                        settings.setValue("obm_connect/email", email)
                    except Exception:
                        pass
                    self._set_info_text(server, connected=True)
                    self.iface.messageBar().pushMessage(self.translate("OBM Connect"), self.translate("Access token obtained."))
                    return True
                else:
                    self.iface.messageBar().pushWarning(self.translate("OAuth"), self.translate("Unable to obtain access token."))
            else:
                self.iface.messageBar().pushCritical(self.translate("OAuth"), self.translate("HTTP error: {code} - {text}").format(code=resp.status_code, text=resp.text))
        except Exception as exc:
            self.iface.messageBar().pushCritical(self.translate("OAuth"), self.translate("Exception: {err}").format(err=str(exc)))

        self._set_not_connected()
        return False

    def _persist_tokens_if_requested(self, server: str):
        """Persist tokens and server/email to QSettings if the user requested 'remember'."""
        try:
            remember = False
            if hasattr(self, "rememberMeCheckBox"):
                remember = bool(self.rememberMeCheckBox.isChecked())
            if remember:
                settings = QSettings()
                settings.setValue("obm_connect/access_token", self._access_token)
                settings.setValue("obm_connect/refresh_token", self._refresh_token)
                settings.setValue("obm_connect/server", server)
                if hasattr(self, "emailLineEdit"):
                    settings.setValue("obm_connect/email", self.emailLineEdit.text().strip())
                settings.setValue("obm_connect/remember", True)
        except Exception:
            pass

    def on_connect_clicked(self):
        """Handle connect/disconnect button."""
        try:
            btn_text = self.connectPushButton.text()
        except Exception:
            btn_text = ""

        if btn_text and btn_text.lower().strip() == self.translate("Disconnect").lower():
            # Logout: clear tokens and stored credentials
            self._access_token = None
            self._refresh_token = None
            self._token_expires_time = datetime.now()
            self._set_not_connected()
            try:
                self.connectPushButton.setText(self.translate("Connect"))
                self.emailLineEdit.clear()
                self.passwordLineEdit.clear()
                self.rememberMeCheckBox.setEnabled(True)
            except Exception:
                pass
            try:
                self.projectComboBox.setEnabled(False)
                self.tabWidget.setCurrentIndex(0)
            except Exception:
                pass
            try:
                settings = QSettings()
                settings.remove("obm_connect/access_token")
                settings.remove("obm_connect/refresh_token")
                settings.remove("obm_connect/email")
                settings.remove("obm_connect/server")
                settings.setValue("obm_connect/remember", False)
            except Exception:
                pass
            self.iface.messageBar().pushMessage(self.translate("OBM Connect"), self.translate("Disconnected and stored credentials removed."))
            return

        # Connect flow (password grant)
        email = self.emailLineEdit.text()
        password = self.passwordLineEdit.text()
        server = self.obmServerLineEdit.text().strip() or DEFAULT_OBM_SERVER
        token_url = build_oauth_url(server)
        client_id = "mobile"
        client_secret = "123"
        scopes = "get_data get_tables"

        payload = {"grant_type": "password", "username": email, "password": password, "scope": scopes}
        try:
            resp = requests.post(token_url, data=payload, auth=(client_id, client_secret), headers={"Accept": "application/json"}, timeout=15)
            if resp.status_code == 200:
                data = resp.json()
                access = data.get("access_token")
                refresh = data.get("refresh_token")
                expires_in = int(data.get("expires_in", 3600))
                if access:
                    self._access_token = access
                    self._refresh_token = refresh
                    self._token_expires_time = datetime.now() + timedelta(seconds=expires_in - 1)
                    try:
                        remember = False
                        if hasattr(self, "rememberMeCheckBox"):
                            remember = bool(self.rememberMeCheckBox.isChecked())
                        settings = QSettings()
                        settings.setValue("obm_connect/remember", bool(remember))
                        if remember:
                            settings.setValue("obm_connect/access_token", self._access_token)
                            settings.setValue("obm_connect/refresh_token", self._refresh_token)
                            settings.setValue("obm_connect/server", server)
                            settings.setValue("obm_connect/email", email)
                    except Exception:
                        pass
                    if self.DEBUG:
                        self.iface.messageBar().pushMessage(self.translate("DEBUG"), self.translate("Scopes: {scopes} ----- Access token received.").format(scopes=scopes))
                    self._set_info_text(server, connected=True)
                    try:
                        self.connectPushButton.setText(self.translate("Disconnect"))
                    except Exception:
                        pass
                    self.load_projects(server)
                    self.tabWidget.setCurrentIndex(1)
                else:
                    error_msg = data.get("error_description") or str(data)
                    self.iface.messageBar().pushWarning(self.translate("OAuth"), self.translate("Error: {err}").format(err=error_msg))
            else:
                self.iface.messageBar().pushCritical(self.translate("OAuth"), self.translate("HTTP error: {code} - {text}").format(code=resp.status_code, text=resp.text))
        except Exception as exc:
            self.iface.messageBar().pushCritical(self.translate("OAuth"), self.translate("Exception: {err}").format(err=str(exc)))

    def _reset_state_and_views(self):
        """Reset transient state and clear table/field views."""
        self._last_table_name = None
        self._last_project_name = None

        tv = getattr(self, "tablesTableView", None)
        if tv is not None:
            model = tv.model()
            if model is not None:
                model.clear()
                try:
                    model.setHorizontalHeaderLabels([self.translate("Table name")])
                except Exception:
                    pass

        fv = getattr(self, "tablesFieldsView", None)
        if fv is not None:
            fmodel = fv.model()
            if fmodel is not None:
                fmodel.clear()
                try:
                    fmodel.setHorizontalHeaderLabels([self.translate("Field name"), self.translate("Type")])
                except Exception:
                    pass

    def _set_not_connected(self):
        """Set the UI to the disconnected state and clear models."""
        self._set_info_text(connected=False)
        try:
            if hasattr(self, "tablesTableView") and self.tablesTableView is not None:
                mv = self.tablesTableView.model()
                if mv is not None:
                    mv.clear()
                self.tablesTableView.setModel(None)
        except Exception:
            pass

        try:
            if hasattr(self, "projectComboBox") and self.projectComboBox is not None:
                self.projectComboBox.clear()
                self.projectComboBox.setEnabled(False)
        except Exception:
            pass

        try:
            if hasattr(self, "tabWidget") and self.tabWidget is not None:
                self.tabWidget.setCurrentIndex(0)
        except Exception:
            pass

    def _fetch_table_details(self, api_url: str, schema: str, raw_name: str, display_name: str, row_key: str, headers: dict):
        """Background worker: fetch table metadata and emit result to main thread."""
        details = None
        try:
            url = api_url + f"data-tables/{schema}/{raw_name}"
            resp = requests.get(url, headers=headers, timeout=15)
            resp.raise_for_status()
            details = resp.json()
        except Exception:
            details = None
        finally:
            self.tableDetailsFetched.emit(row_key, {"raw_name": raw_name, "display_name": display_name, "schema": schema, "details": details})

    def _on_table_details_fetched(self, row_key: str, payload: dict):
        """Update table model rows when background details arrive."""
        tv = getattr(self, "tablesTableView", None)
        if tv is None or tv.model() is None:
            return
        model = tv.model()

        raw_name = payload.get("raw_name")
        schema = payload.get("schema")
        details = payload.get("details")

        # Find rows matching row_key or fallback to raw_name+schema
        matches = []
        for r in range(model.rowCount()):
            item0 = model.item(r, 0)
            if item0 is None:
                continue
            data = item0.data(Qt.UserRole)
            if data == row_key:
                matches.append(r)

        if not matches:
            for r in range(model.rowCount()):
                item0 = model.item(r, 0)
                if item0 is None:
                    continue
                data = item0.data(Qt.UserRole)
                if isinstance(data, dict) and data.get("raw_name") == raw_name and data.get("schema") == schema:
                    matches.append(r)

        if not matches:
            return

        geometry_keys = ["Point", "LineString", "Polygon", "MultiPoint", "MultiLine", "MultiPolygon", "GeometryCollection"]

        geom_rows = []
        record_number = ""
        fields = []
        if isinstance(details, dict):
            fields = details.get("fields", []) or []
            record_number = details.get("record_number", "")

        if isinstance(fields, list):
            for fdef in fields:
                if not isinstance(fdef, dict):
                    continue
                ftype = (fdef.get("type") or "").lower()
                if ftype == "geometry" or "geometry_column_details" in fdef:
                    gdetails = fdef.get("geometry_column_details") or {}
                    parts = []
                    total = 0
                    for k in geometry_keys:
                        try:
                            v = int(gdetails.get(k) or 0)
                        except Exception:
                            v = 0
                        if v > 0:
                            parts.append(f"{k}: {v}")
                            total += v
                    geom_types_str = "; ".join(parts)
                    num_records_value = record_number or (total if total > 0 else "")
                    geom_rows.append((fdef.get("name", ""), geom_types_str, str(num_records_value)))

        for base_row in matches:
            if geom_rows:
                first = geom_rows[0]
                model.setItem(base_row, 2, QStandardItem(str(first[0])))
                model.setItem(base_row, 3, QStandardItem(first[1]))
                model.setItem(base_row, 4, QStandardItem(first[2]))
                item0 = model.item(base_row, 0)
                if item0 is not None and not item0.data(Qt.UserRole):
                    item0.setData({"raw_name": raw_name, "schema": schema}, Qt.UserRole)
                insert_pos = base_row + 1
                for extra in geom_rows[1:]:
                    row_items = [
                        QStandardItem(item0.text() if item0 is not None else raw_name),
                        QStandardItem(str(schema)),
                        QStandardItem(str(extra[0])),
                        QStandardItem(extra[1]),
                        QStandardItem(extra[2]),
                    ]
                    row_items[0].setData({"raw_name": raw_name, "schema": schema}, Qt.UserRole)
                    model.insertRow(insert_pos, row_items)
                    insert_pos += 1
            else:
                model.setItem(base_row, 2, QStandardItem(""))
                model.setItem(base_row, 3, QStandardItem(""))
                model.setItem(base_row, 4, QStandardItem(str(record_number or "")))
                item0 = model.item(base_row, 0)
                if item0 is not None and not item0.data(Qt.UserRole):
                    item0.setData({"raw_name": raw_name, "schema": schema}, Qt.UserRole)

        tv.resizeColumnsToContents()

    def on_project_selected(self, idx: int):
        """Handle project selection: list tables and start background detail fetches."""
        
        if not (idx >= 1 and self._project_list):
            self._reset_state_and_views()
            return

        if not self.ensure_valid_token():
            QtWidgets.QMessageBox.warning(self, self.translate("Error"), self.translate("No valid token."))
            return

        project = self._project_list[idx - 1]
        project_name = project.get("project_table")
        server = self.obmServerLineEdit.text().strip()
        api_url = build_project_api_url(server, project_name)

        if self.DEBUG:
            self.iface.messageBar().pushMessage(self.translate("DEBUG"), self.translate("Loading tables from {url}").format(url=api_url + "data-tables"))

        headers = {"Authorization": self._access_token, "Accept": "application/json"}

        try:
            resp = requests.get(api_url + "data-tables", headers=headers, timeout=15)
            resp.raise_for_status()

            content_type = (resp.headers.get("Content-Type") or "").lower()
            if "application/json" not in content_type:
                self.iface.messageBar().pushCritical(self.translate("OBM Connect"), self.translate("The API is not available for the selected project."))
                self._reset_state_and_views()
                return

            try:
                data = resp.json()
            except Exception:
                self.iface.messageBar().pushCritical(self.translate("OBM Connect"), self.translate("The API returned invalid JSON."))
                self._reset_state_and_views()
                return

            tables = data if isinstance(data, list) else data.get("tables") or data.get("data") or []

            model = QStandardItemModel()
            model.setHorizontalHeaderLabels(
                [
                    self.translate("Table name"),
                    self.translate("Schema"),
                    self.translate("Geom column"),
                    self.translate("Geom Type"),
                    self.translate("Num of records"),
                ]
            )

            self.tablesTableView.setModel(model)
            self.tablesTableView.setColumnWidth(1, 50)
            self.tablesTableView.setAlternatingRowColors(True)
            self.tablesTableView.verticalHeader().setVisible(False)
            self.tablesTableView.verticalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
            self.tablesTableView.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)

            row_keys = []
            for table in tables:
                if isinstance(table, dict):
                    raw_name = table.get("name") or table.get("table") or ""
                    schema = table.get("schema") or ""
                    fallback_record_number = table.get("record_number") or ""
                else:
                    raw_name = str(table)
                    schema = ""
                    fallback_record_number = ""

                display_name = raw_name
                if project_name and raw_name.startswith(project_name + "_"):
                    display_name = raw_name[len(project_name):]

                if not schema:
                    schema = "public"

                item_name = QStandardItem(display_name)
                row_key = f"{raw_name}||{schema}"
                item_name.setData(row_key, Qt.UserRole)
                item_schema = QStandardItem(str(schema))
                item_geom = QStandardItem("")  # placeholder
                item_types = QStandardItem("")  # placeholder
                item_num = QStandardItem(str(fallback_record_number or ""))

                model.appendRow([item_name, item_schema, item_geom, item_types, item_num])
                row_keys.append((raw_name, schema, display_name, row_key))

            # Launch background detail fetchers
            for raw_name, schema, display_name, row_key in row_keys:
                thread = threading.Thread(
                    target=self._fetch_table_details,
                    args=(api_url, schema, raw_name, display_name, row_key, headers),
                    daemon=True,
                )
                thread.start()

        except requests.exceptions.HTTPError:
            self.iface.messageBar().pushCritical(self.translate("OBM Connect"), self.translate("The API is not available for the selected project."))
            self._reset_state_and_views()
        except Exception as exc:
            self.iface.messageBar().pushCritical(self.translate("OBM Connect"), self.translate("Error retrieving tables: {err}").format(err=exc))
            self._reset_state_and_views()

        self.tablesTableView.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)

    def on_table_double_clicked(self, index):
        """Open filter dialog for the double-clicked table."""
        if not self.ensure_valid_token():
            QtWidgets.QMessageBox.warning(self, self.translate("Error"), self.translate("No valid token. Please log in again."))
            return

        model = self.tablesTableView.model()
        if model is None:
            QtWidgets.QMessageBox.warning(self, self.translate("Error"), self.translate("No table model available."))
            return

        displayed_item = model.item(index.row(), 0)
        if displayed_item is None:
            QtWidgets.QMessageBox.warning(self, self.translate("Error"), self.translate("No table selected."))
            return
        displayed_name = displayed_item.text()

        schema_item = model.item(index.row(), 1)
        schema = ""
        if schema_item is not None:
            try:
                schema = schema_item.text().strip()
            except Exception:
                schema = ""
        if not schema:
            schema = "public"

        current_idx = self.projectComboBox.currentIndex()
        project = self._project_list[current_idx - 1]
        project_name = project.get("project_table") or project.get("name") or ""

        # Rebuild full table name if the displayed name was truncated
        if displayed_name.startswith("_") and project_name:
            table_name = project_name + displayed_name
        else:
            table_name = displayed_name

        self._last_table_name = table_name
        server = self.obmServerLineEdit.text().strip() or DEFAULT_OBM_SERVER
        url = build_project_api_url(server, project_name) + f"data-tables/{schema}/{table_name}"
        headers = {"Authorization": self._access_token, "Accept": "application/json"}

        try:
            if self.DEBUG:
                self.iface.messageBar().pushMessage("OBM Connect", f"URL: {url}")

            resp = requests.get(url, headers=headers, timeout=15)
            resp.raise_for_status()
            resp_json = resp.json()
            self._last_fields_json = resp_json

            fields = []
            if isinstance(resp_json, dict):
                fields = resp_json.get("fields", []) or []
            elif isinstance(resp_json, list) and resp_json and isinstance(resp_json[0], dict):
                fields = resp_json[0].get("fields", []) or []
            else:
                fields = []

            if not isinstance(fields, list):
                fields = []

            dialog = ObmConnectFilterDialog(parent=self, debug=self.DEBUG)
            list_model = QtGui.QStandardItemModel()
            for fdef in fields:
                fname = fdef.get("name") if isinstance(fdef, dict) else str(fdef)
                list_model.appendRow(QtGui.QStandardItem(str(fname)))
            dialog.fieldListView.setModel(list_model)
            sel_model = dialog.fieldListView.selectionModel()
            if sel_model is not None and hasattr(dialog, "_set_unique_count_on_all_button"):
                sel_model.currentChanged.connect(dialog._set_unique_count_on_all_button)
                dialog._set_unique_count_on_all_button()
            dialog.exec_()

        except Exception as exc:
            self.iface.messageBar().pushCritical(self.translate("OBM Connect"), self.translate("Field retrieval failed: {err}").format(err=exc))
            return

    def closeEvent(self, event):
        self.closingPlugin.emit()
        event.accept()
