from requests import get, post
from json import loads, dump


class VegrefStedfestning:
    def __init__(self, attrib: dict = {}):
        self.token: str = str('')
        self.vegref: str = str('')
        self.geo_parsed: dict = {}

        self.is_position_ok = False  # for checking when position flag is has changed or not

        self.is_attribs_set = False
        self.side_posisjon = None

        self.env: str = str()
        self.env_x: str = str()

        self.set_env()

        # setting attribs at the beginning
        if len(attrib) > 0:
            self.set_attributes(attrib)

    # set env, default is Test miljø
    def set_env(self, env: str = str('test')) -> None:
        if 'test' in env:
            self.env = 'https://nvdbapiles.test.atlas.vegvesen.no/vegnett/api/v4/veg'
            self.env_x = 'https://nvdbapiles-v3.test.atlas.vegvesen.no/veg'

        elif 'prod' in env:
            self.env = 'https://nvdbapiles.atlas.vegvesen.no/vegnett/api/v4/veg'
            self.env_x = 'https://nvdbapiles-v3.atlas.vegvesen.no/veg'

    # get env, default is v4 version miljø
    def get_env(self, env_x='v4') -> str:
        if env_x == 'v3':
            return self.env_x

        return self.env

    # get env as name ('test' or 'prod') isntead of URL endpoint (https:nvdbles.endpoint.no/etc/etc)
    def get_env_as_name(self) -> str:
        print('checking whats in env:', self.env)
        print('checking whats in env_x:', self.env_x)
        
        # if '.test' in self.env or self.env_x:
        #     return str('.test.')
        
        if '.test' in self.env:
            return str('.test.')
            
        if '.test' in self.env_x:
            return str('.test.')

        return str('.')

    # set attibutes before continue using
    def set_attributes(self, attrib: dict = {}):
        '''
        this method will set token and vegref if not done in instantiation
        format will be a dictionary ex.
        
        attrib{
            'token': 'aaa5544a-77745'
        }
        '''

        self.token = attrib.get('token', 'empty')
        # self.vegref = attrib.get('vegref', 'empty')

        if 'empty' in self.token:
            self.is_attribs_set = False

            raise Exception('Attributes not set yet ex: {token: token_value, vegref: vegref_value}')

        else:
            self.is_attribs_set = True

    # parse vegsystemreferanse
    def check_vegref(self, vegref: str = str()) -> bool:

        endpoint = self.get_env('v3')  # endpoint

        params = {'vegsystemreferanse': vegref}  # params for adding to endpoint in get request --> arg.. vegref
        header = {'X-Client': 'QGIS-Les'}

        response = get(endpoint, params=params, headers=header)

        if response.status_code == 200:
            return True

        return False

    # get position (get method)
    def get_position(self, vegref_args: dict = {}) -> bool:
        '''
        to get position, then parse_vegref must be done, but raw input can be used
        and all required attributes must be set,  and parse_vegref method will be used here
        
        wtk values can be store in a global scope variable ex: self.wtk str data type
        or self.geo_parsed dict data type for later substraction from stedfest_object method.
        '''

        # submethod for parsing wkt to normal str
        def parse_wkt(wkt: str = str()):

            a = wkt.split('(')

            b = a[1].split(')')

            c = b[0]

            return c

        # end of submethod

        geo_data: dict = {}
        response: None
        diff_vegnet_nvdbid: bool = False

        if not self.is_attribs_set:
            raise Exception('position fail: attributes must be set')

        if self.is_attribs_set:
            # position code from here, meaning that attribs are set

            self.geo_parsed: dict = {}  # to store geo json data, to substract from stedfest_object method

            endpoint = self.get_env('v3')  # endpoint

            header = {'X-Client': 'QGIS-Les'}

            # if position is type Punkt
            if vegref_args.get('stedfest_type') == 'punkt':

                params = {'vegsystemreferanse': vegref_args.get(
                    'fra')}  # params for adding to endpoint in get request --> vegref_args['fra']

                response = get(endpoint, params=params, headers=header)

                # print('before ok in get_position():', response.text)

                if response.status_code == 200:
                    object_data = response.json()

                    geo_data = {
                        'stedfest_type': vegref_args.get('stedfest_type'),
                        'geometry': object_data['geometri']['wkt'],  # geometri well known text
                        'veglenke_id': object_data['veglenkesekvens']['veglenkesekvensid'],
                        'relative_position': object_data['veglenkesekvens']['relativPosisjon'],
                        'kortform': object_data['veglenkesekvens']['kortform'],
                        'retning': object_data['vegsystemreferanse']['strekning']['retning'],
                        'vegref_parsed': object_data['vegsystemreferanse']['vegsystem']  # vegref already parsed :)
                    }

                    # print('geo_parsed in get_position():', geo_data)

                    self.geo_parsed = geo_data

                    diff_vegnet_nvdbid = True  # not that important here to check if is or not same veglenke_id

                    self.is_position_ok = True

                if not response.ok:
                    self.is_position_ok = False

            # if position is type Linje or Flate
            if vegref_args.get('stedfest_type') == 'linje' or vegref_args.get(
                    'stedfest_type') == 'hjelpelinje' or vegref_args.get('stedfest_type') == 'flate':

                params_fra = {'vegsystemreferanse': vegref_args.get(
                    'fra')}  # params for adding to endpoint in get request --> vegref_args['fra']

                response_fra = get(endpoint, params=params_fra, headers=header)

                params_til = {'vegsystemreferanse': vegref_args.get(
                    'til')}  # params for adding to endpoint in get request --> vegref_args['fra']

                response_til = get(endpoint, params=params_til, headers=header)

                # print('before ok in get_position() fra:', response_fra.text)

                # print('before ok in get_position() til:', response_til.text)

                if response_fra.status_code == 200 and response_til.status_code == 200:
                    object_data_fra = response_fra.json()

                    object_data_til = response_til.json()

                    wkt_fra = parse_wkt(object_data_fra['geometri']['wkt'])
                    wkt_til = parse_wkt(object_data_til['geometri']['wkt'])

                    # print('wkt fra', wkt_fra)
                    # print('wkt til', wkt_til)

                    line_geometry = str(f'LINESTRING ({wkt_fra}, {wkt_til})') if vegref_args.get(
                        'stedfest_type') == 'linje' or vegref_args.get('stedfest_type') == 'hjelpelinje' else str(
                        f'POLYGON ({wkt_fra}, {wkt_til})')

                    # print('line geometry:', line_geometry)

                    geo_data_fra = {
                        'veglenke_id': object_data_fra['veglenkesekvens']['veglenkesekvensid'],
                        'relative_position': object_data_fra['veglenkesekvens']['relativPosisjon'],
                        'kortform': object_data_fra['veglenkesekvens']['kortform'],
                        'retning': object_data_fra['vegsystemreferanse']['strekning']['retning'],
                        'vegref_parsed': object_data_fra['vegsystemreferanse']['vegsystem']  # vegref already parsed :)
                    }

                    geo_data_til = {
                        'veglenke_id': object_data_til['veglenkesekvens']['veglenkesekvensid'],
                        'relative_position': object_data_til['veglenkesekvens']['relativPosisjon']
                    }

                    diff_vegnet_nvdbid = True if geo_data_fra.get('veglenke_id') == geo_data_til.get(
                        'veglenke_id') else False

                    # print('diff_vegnet_nvdbid', diff_vegnet_nvdbid)

                    geo_data = {
                        'stedfest_type': vegref_args.get('stedfest_type'),
                        'geometry': line_geometry,
                        'veglenke_id': object_data_fra['veglenkesekvens']['veglenkesekvensid'],  # object_data_fra
                        'relative_position_fra': object_data_fra['veglenkesekvens']['relativPosisjon'],
                        'relative_position_til': object_data_til['veglenkesekvens']['relativPosisjon'],
                        'kortform': object_data_fra['veglenkesekvens']['kortform'],
                        'retning': object_data_fra['vegsystemreferanse']['strekning']['retning'],
                        'vegref_parsed': object_data_fra['vegsystemreferanse']['vegsystem']  # vegref already parsed :)
                    }

                    # print('geo fra:', geo_data_fra)
                    # print('geo to:', geo_data_til)
                    # print('geo_data in get_position():', geo_data)

                    self.geo_parsed = geo_data

                    self.is_position_ok = True

                if not response_fra.ok and not response_til.ok:
                    self.is_position_ok = False

        return diff_vegnet_nvdbid

    # stedfest road object
    def stedfest_object(self, extra_attribs: dict = {}, diff_geometry_on: bool = False,
                        diff_geometry: str = str()) -> None:
        '''
        here road object will re-locate to a new geo location on the map system
        but only after get_position has success when calling it from inside this class

        extra_attribs arg is a dictionary with specific data needed for ex:
        typeId, tempID, gyldighetsperiode sstart and slutt dato.
        '''
        if self.is_attribs_set and self.is_position_ok:

            # stedfest code from here

            header = {
                'X-Client': 'QGIS Skriv',
                'Content-Type': 'application/json',
                'Authorization': 'Bearer ' + self.token
            }

            endpoint = 'https://nvdbstedfesting' + self.get_env_as_name() + 'atlas.vegvesen.no/api/v1/stedfest'
            
            print( 'stedfest_object:', endpoint )
            print('getting from self.get_env_as_name():', self.get_env_as_name())
            
            body = {

                "parametere": {
                    "maksimalAvstandTilVeg": 10,
                    "omkrets": 100,
                    "beregnSideposisjon": False,
                    "beregnKjørefelt": False,
                    "konnekteringslenker": False,
                    "beholdTrafikantgruppe": False

                    # "trafikantgruppe": extra_attribs['trafikant_gruppe']
                },

                "vegobjekter": [
                    {
                        "typeId": extra_attribs['type'],
                        "tempId": extra_attribs['nvdbid'],

                        "gyldighetsperiode": {
                            "startdato": extra_attribs['start_date']
                        },

                        "geometri": {
                            "srid": str(5973),
                            "wkt": self.geo_parsed.get('geometry') if not diff_geometry_on else diff_geometry
                            # self.geo_parsed['geometry']
                        }
                    }
                ]

            }

            # print(body)

            # POSTING
            response = post(url=endpoint, headers=header, json=body)

            positions: dict = {}

            # print('before sending response in stedfest_object():', response.text)

            if response.ok:

                # print('ok response in stedfest_object():', response.text)

                data = response.text
                parsed = loads(data)  # check does we use it ?

                if self.geo_parsed.get('stedfest_type') == 'punkt':
                    positions = {'fra': self.geo_parsed.get('relative_position')}

                if self.geo_parsed.get('stedfest_type') == 'linje' or self.geo_parsed.get(
                        'stedfest_type') == 'hjelpelinje' or self.geo_parsed.get('stedfest_type') == 'flate':
                    positions = {
                        'fra': self.geo_parsed.get('relative_position_fra'),
                        'til': self.geo_parsed.get('relative_position_til')
                    }

                self.egenskaper_stedfest = {
                    'nvdbid': extra_attribs['nvdbid'],
                    'typeid': extra_attribs['type'],
                    'versjon': extra_attribs['versjon'],
                    'start_dato': extra_attribs['start_date'],
                    'operasjon': "oppdater",
                    'veglenkesekvensnvdbid': self.geo_parsed['veglenke_id'],
                    'posisjon': positions,  # according if its linje, punkt or flate,
                    'retning': self.geo_parsed['retning'],
                    'physic_geometry': self.geo_parsed.get('geometry') if not diff_geometry_on else diff_geometry
                }

            else:  # better with if not response.ok:
                print("bad response:", response.text)

    # to get vegref from geometry wkt format
    @staticmethod
    def get_vegref_from_wkt(attribs: dict = {}) -> dict:

        # stedfest code from here

        header = {
            'X-Client': 'QGIS Skriv',
            'Content-Type': 'application/json'
            # 'Authorization': 'Bearer ' + self.token
        }

        # endpoint = 'https://nvdbstedfesting' + self.get_env_as_name() + 'atlas.vegvesen.no/api/v1/stedfest'

        rest_endpoint = '.atlas.vegvesen.no/api/v1/stedfest'

        if attribs.get('env') == '':
            rest_endpoint = 'atlas.vegvesen.no/api/v1/stedfest'

        endpoint = 'https://nvdbstedfesting.' + attribs.get('env') + rest_endpoint

        body = {

            "parametere": {
                "maksimalAvstandTilVeg": 10,
                "omkrets": 100,
                "beregnSideposisjon": False,
                "beregnKjørefelt": False,
                "konnekteringslenker": False,
                "beholdTrafikantgruppe": False

                # "trafikantgruppe": extra_attribs['trafikant_gruppe']
            },

            "vegobjekter": [
                {
                    "typeId": attribs.get('type'),  # extra_attribs['type'],
                    "tempId": attribs.get('nvdbid'),  # extra_attribs['nvdbid'],

                    "gyldighetsperiode": {
                        "startdato": attribs.get('startdato')  # extra_attribs['start_date']
                    },

                    "geometri": {
                        "srid": str(5973),
                        "wkt": attribs.get('geometry')
                        # self.geo_parsed.get('geometry') if not diff_geometry_on else diff_geometry #self.geo_parsed['geometry']
                    }
                }
            ]
        }

        print('body from get_vegref_from_wkt():', body)
        print('endpoint in get_vegref_from_wkt():', endpoint)

        # POSTING
        response = post(url=endpoint, headers=header, json=body)

        # print(response.text)

        if response.status_code == 200:

            # continue only if there is not any feil
            if len(response.json()['vegobjekter'][0].get('feil')) == 0:

                # print( 'hole road object info:', response.json()['vegobjekter'][0])
                # print( 'only stedfestning info:', response.json()['vegobjekter'][0]['stedfesting'] )

                # making sue that if geometry type is hjelpelinje then treat it like a linje
                if 'hjelpelinje' in attribs.get('stedfest_type'):
                    attribs['stedfest_type'] = 'linje'

                vegref = \
                    response.json()['vegobjekter'][0]['stedfesting'][attribs.get('stedfest_type')][0][
                        'vegsystemreferanse'][
                        'kortform']

                parse = vegref.split('-')
                parse_2 = vegref.split(' ')

                vegref_from = parse[0]
                help_to = parse_2[len(parse_2) - 1] = str('m' + parse[len(parse) - 1])

                vegref_to: str = str()

                for st in parse_2:
                    vegref_to += st + ' '

                if attribs.get('stedfest_type') == 'linje' or attribs.get('stedfest_type') == 'flate':
                    return {'from': vegref_from, 'to': vegref_to.strip()}

                if attribs.get('stedfest_type') == 'punkt':
                    return {'from': vegref_from}

        return {'feil': response.text}