# -*- coding: utf-8 -*-
"""
Algorithm name
Radiance2Carbon (R2C)

Group
NTL and Carbon Analysis

Summary
Estimate carbon emissions from Nighttime Lights rasters (batch),
with multiple conversion methods, threshold filtering, and outputs:
CSV reports and PNG visualizations (trend, comparison grid, distributions).
Now includes CRS-safe boundary overlay (thin, high-contrast) and year labels.

Credits
Firman Afrianto & Adipandang Yudono

Notes
No rasterio required. Uses GDAL, NumPy, Matplotlib bundled with QGIS.
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterMultipleLayers,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterEnum,
    QgsProcessingParameterNumber,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterFileDestination,
    QgsProcessingParameterFolderDestination,
    QgsProcessingException,
    QgsRasterLayer,
    QgsWkbTypes,
    QgsCoordinateTransform,
    QgsGeometry,
)
from osgeo import gdal
import numpy as np
import csv
import os
import math
import re

# Headless Matplotlib
try:
    import matplotlib
    matplotlib.use("Agg")
    import matplotlib.pyplot as plt
    from matplotlib.colors import LinearSegmentedColormap
    _HAS_MPL = True
except Exception:
    _HAS_MPL = False


def tr(text):
    return QCoreApplication.translate("R2C_Toolbox", text)


def read_raster_to_array(path):
    ds = gdal.Open(path, gdal.GA_ReadOnly)
    if ds is None:
        raise QgsProcessingException(f"Failed to open raster {path}")
    band = ds.GetRasterBand(1)
    arr = band.ReadAsArray().astype(float)
    nodata = band.GetNoDataValue()
    if nodata is not None:
        arr[arr == nodata] = np.nan
    gt = ds.GetGeoTransform()
    # extent: left, right, bottom, top
    extent = (gt[0], gt[0] + ds.RasterXSize * gt[1],
              gt[3] + ds.RasterYSize * gt[5], gt[3])
    ds = None
    return arr, extent


def create_ntl_colormap():
    """Dark-to-bright palette for NTL."""
    colors = ['black', 'darkblue', 'blue', 'cyan', 'yellow', 'white']
    return LinearSegmentedColormap.from_list('ntl_colormap', colors, N=256)


def apply_threshold_filter(ntl_data, threshold_type="auto", custom_threshold=None):
    data = ntl_data.copy()
    valid = np.array(data, copy=True)
    valid[np.isnan(valid)] = 0
    if threshold_type == "auto":
        threshold = float(np.mean(valid) + 0.5 * np.std(valid))
    elif threshold_type == "percentile":
        positive = valid[valid > 0]
        threshold = float(np.percentile(positive, 75)) if positive.size > 0 else 0.0
    elif threshold_type == "manual" and custom_threshold is not None:
        threshold = float(custom_threshold)
    else:
        threshold = 0.0
    filtered = data.copy()
    filtered[np.isnan(filtered)] = 0.0
    filtered[filtered < threshold] = 0.0
    # restore NaN where original was NaN (exclude from stats)
    filtered[np.isnan(ntl_data)] = np.nan
    return filtered, threshold


def calculate_carbon_from_ntl(ntl_data, method="Elvidge_2009", custom_coef=0.001):
    valid = ntl_data[~np.isnan(ntl_data)]
    if valid.size == 0:
        return 0.0, {
            'total_ntl': 0.0,
            'mean_ntl': 0.0,
            'max_ntl': 0.0,
            'pixel_count': 0,
            'carbon_coefficient': 0.0,
            'carbon_per_pixel': 0.0
        }
    if method == "Elvidge_2009":
        coef = 0.0005
        carbon = float(np.sum(valid) * coef)
    elif method == "Shi_2016":
        coef = 0.0008
        carbon = float(np.sum(valid) * coef)
    elif method == "Custom_Linear":
        coef = float(custom_coef)
        carbon = float(np.sum(valid) * coef)
    elif method == "Power_Law":
        coef = 0.0001
        carbon = float(np.sum(np.power(valid, 1.2)) * coef)
    else:
        coef = 0.0006
        carbon = float(np.sum(valid) * coef)

    stats = {
        'total_ntl': float(np.sum(valid)),
        'mean_ntl': float(np.mean(valid)),
        'max_ntl': float(np.max(valid)),
        'pixel_count': int(valid.size),
        'carbon_coefficient': float(coef),
        'carbon_per_pixel': float(carbon / valid.size) if valid.size > 0 else 0.0
    }
    return carbon, stats


def parse_year_from_name(name):
    """
    Extract 4-digit year from filename (first match of 19xx or 20xx).
    Falls back to base name without extension if no year found.
    """
    base = os.path.basename(name)
    stem = os.path.splitext(base)[0]
    m = re.search(r'(19|20)\d{2}', stem)
    return m.group(0) if m else stem


def draw_boundary(ax, vlayer, transform=None, color="#FF00FF", linewidth=0.5, extent_limits=None):
    """
    Draw boundary (vector layer) on a Matplotlib axis.
    If 'transform' (QgsCoordinateTransform) is provided, geometries are reprojected
    into the raster CRS before plotting.
    After drawing, if 'extent_limits'=(left,right,bottom,top) is given, the axis
    limits are restored to prevent autoscale from hiding the raster.
    """
    if vlayer is None:
        return
    for feat in vlayer.getFeatures():
        geom = feat.geometry()
        if geom is None or geom.isEmpty():
            continue
        if transform is not None:
            g = QgsGeometry(geom)
            try:
                g.transform(transform)
                geom = g
            except Exception:
                continue
        gtype = geom.type()
        if gtype == QgsWkbTypes.PolygonGeometry:
            if geom.isMultipart():
                polys = geom.asMultiPolygon()
                for poly in polys or []:
                    if not poly:
                        continue
                    exterior = poly[0]
                    xs = [p.x() for p in exterior]
                    ys = [p.y() for p in exterior]
                    ax.plot(xs, ys, color=color, linewidth=linewidth)
            else:
                poly = geom.asPolygon()
                if poly:
                    exterior = poly[0]
                    xs = [p.x() for p in exterior]
                    ys = [p.y() for p in exterior]
                    ax.plot(xs, ys, color=color, linewidth=linewidth)
        elif gtype == QgsWkbTypes.LineGeometry:
            if geom.isMultipart():
                mlines = geom.asMultiPolyline()
                for line in mlines or []:
                    xs = [p.x() for p in line]
                    ys = [p.y() for p in line]
                    ax.plot(xs, ys, color=color, linewidth=linewidth)
            else:
                line = geom.asPolyline()
                xs = [p.x() for p in line]
                ys = [p.y() for p in line]
                ax.plot(xs, ys, color=color, linewidth=linewidth)
        else:
            # points
            pt = geom.asPoint()
            ax.plot([pt.x()], [pt.y()], marker='o', markersize=2, color=color)

    if extent_limits is not None:
        left, right, bottom, top = extent_limits
        ax.set_xlim(left, right)
        ax.set_ylim(bottom, top)


class NTLCarbonToolbox(QgsProcessingAlgorithm):
    INPUT_RASTERS = 'INPUT_RASTERS'
    METHOD = 'METHOD'
    THRESHOLD_TYPE = 'THRESHOLD_TYPE'
    CUSTOM_COEF = 'CUSTOM_COEF'
    MANUAL_THRESHOLD = 'MANUAL_THRESHOLD'
    MAKE_DISTRIB_PNG = 'MAKE_DISTRIB_PNG'
    OUT_FOLDER = 'OUT_FOLDER'
    OUT_REPORT_CSV = 'OUT_REPORT_CSV'
    OUT_STATS_CSV = 'OUT_STATS_CSV'
    OUT_TREND_PNG = 'OUT_TREND_PNG'
    OUT_GRID_PNG = 'OUT_GRID_PNG'
    BOUNDARY = 'BOUNDARY'
    BOUNDARY_WIDTH = 'BOUNDARY_WIDTH'

    METHOD_ITEMS = ["Elvidge_2009", "Shi_2016", "Power_Law", "Custom_Linear"]
    THRESHOLD_ITEMS = ["auto", "percentile", "manual", "none"]

    def tr(self, string):
        return tr(string)

    def createInstance(self):
        return NTLCarbonToolbox()

    def name(self):
        return 'radiance2carbon_r2c'

    def displayName(self):
        return self.tr('Radiance2Carbon (R2C)')

    def shortHelpString(self):
        return self.tr(
            "<p><b>A simple way to convert NTL insights into carbon emissions</b></p>"
            "<p>Batch-estimates <b>carbon emissions</b> from Nighttime Lights (NTL) rasters using several "
            "conversion methods and threshold filters. Produces <b>CSV reports</b> and <b>PNG visualizations</b> "
            "(trend, comparison grid, and per-raster distributions), with an optional <b>boundary overlay</b> "
            "that is <b>CRS-safe</b> and rendered in <b>high-contrast thin lines</b>. Grid and plots use the "
            "<b>year parsed from filenames</b> where available.</p>"

            "<p><b>What it does</b></p>"
            "<ul>"
            "<li>Reads multiple single-band NTL rasters and parses a <i>year</i> from each filename.</li>"
            "<li>Applies thresholding (auto / percentile / manual / none) to isolate significant NTL.</li>"
            "<li>Converts NTL to carbon using <i>Elvidge 2009</i>, <i>Shi 2016</i>, <i>Power Law</i>, or <i>Custom Linear</i>.</li>"
            "<li>Computes per-raster stats and writes a detailed <b>carbon report CSV</b> and <b>NTL stats CSV</b>.</li>"
            "<li>Draws optional <b>boundary overlay</b> with automatic CRS transformation to raster CRS and "
            "locks axes to keep rasters visible.</li>"
            "<li>Generates PNGs: <b>trend</b> (time series), <b>comparison grid</b> (by year), and <b>per-raster distributions</b> "
            "(NTL & carbon side-by-side).</li>"
            "</ul>"

            "<p><b>Inputs</b></p>"
            "<ul>"
            "<li><b>NTL rasters</b> (single-band), any consistent radiance-like units.</li>"
            "<li><b>Boundary layer (optional)</b> in any CRS (e.g., ESRI:54034). It is reprojected on the fly to the raster CRS.</li>"
            "</ul>"

            "<p><b>Key Parameters</b></p>"
            "<ul>"
            "<li><b>Conversion method</b>: Elvidge 2009, Shi 2016, Power Law (exp=1.2), or Custom Linear (user coefficient).</li>"
            "<li><b>Threshold type</b>: auto (mean+0.5σ), percentile (P75 of positives), manual, or none.</li>"
            "<li><b>Custom coefficient</b>: used when <i>Custom Linear</i> is selected.</li>"
            "<li><b>Manual threshold</b>: numeric cutoff when threshold type is manual.</li>"
            "<li><b>Boundary line width</b>: thin overlay (default 0.5 px). Magenta on NTL, Cyan on Carbon panels.</li>"
            "</ul>"

            "<p><b>Outputs</b></p>"
            "<ul>"
            "<li><b>Carbon report CSV</b>: Dataset (year/file), method, threshold, total carbon (Ton/KTon), total NTL, pixels, carbon per pixel, mean/max NTL.</li>"
            "<li><b>NTL stats CSV</b>: min, max, mean, std, and valid area (px) per raster.</li>"
            "<li><b>Trend PNG</b>: total carbon, carbon per pixel, total NTL (post-threshold), and NTL–carbon scatter.</li>"
            "<li><b>Comparison grid PNG</b>: NTL maps titled by <b>year</b> (with boundary overlay).</li>"
            "<li><b>Distribution PNGs</b>: side-by-side NTL and Carbon for each raster; filenames include the year.</li>"
            "</ul>"

            "<p><b>Notes</b></p>"
            "<ul>"
            "<li><b>CRS handling</b>: boundary is reprojected to the first raster's CRS; axes are reset to the raster extent to avoid autoscale issues.</li>"
            "<li><b>Year labels</b>: the first 4-digit <i>19xx/20xx</i> pattern in filenames is used; falls back to the file stem.</li>"
            "<li><b>Assumptions</b>: methods and coefficients are simplified proxies for exploratory analysis and should be calibrated with local ground truth.</li>"
            "<li><b>Dependencies</b>: GDAL, NumPy, Matplotlib (bundled with QGIS). No rasterio required.</li>"
            "</ul>"

            "<p><i>Created by</i> <b>FIRMAN AFRIANTO</b> &amp; <b>ADIPANDANG YUDONO</b></p>"
        )


    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.INPUT_RASTERS,
                self.tr('NTL rasters (single-band)'),
                layerType=QgsProcessing.TypeRaster
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.BOUNDARY,
                self.tr('Boundary layer to overlay (optional)'),
                types=[QgsProcessing.TypeVectorAnyGeometry],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.BOUNDARY_WIDTH,
                self.tr('Boundary line width (px)'),
                QgsProcessingParameterNumber.Double,
                defaultValue=0.5,  # thin by default
                minValue=0.05,
                maxValue=10.0
            )
        )

        self.addParameter(
            QgsProcessingParameterEnum(
                self.METHOD,
                self.tr('NTL-to-carbon conversion method'),
                options=self.METHOD_ITEMS,
                defaultValue=0
            )
        )

        self.addParameter(
            QgsProcessingParameterEnum(
                self.THRESHOLD_TYPE,
                self.tr('Threshold type for significant NTL areas'),
                options=self.THRESHOLD_ITEMS,
                defaultValue=0
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.CUSTOM_COEF,
                self.tr('Custom coefficient (for Custom_Linear)'),
                QgsProcessingParameterNumber.Double,
                defaultValue=0.001,
                minValue=0.0000001,
                maxValue=0.01
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MANUAL_THRESHOLD,
                self.tr('Manual threshold value (if threshold type is manual)'),
                QgsProcessingParameterNumber.Double,
                defaultValue=5.0,
                minValue=0.0
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.MAKE_DISTRIB_PNG,
                self.tr('Create per-raster distribution PNGs'),
                defaultValue=True
            )
        )

        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUT_FOLDER,
                self.tr('Output folder for PNGs')
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUT_REPORT_CSV,
                self.tr('Carbon report CSV'),
                'CSV files (*.csv)'
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUT_STATS_CSV,
                self.tr('NTL statistics per raster CSV'),
                'CSV files (*.csv)'
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUT_TREND_PNG,
                self.tr('Carbon trend PNG'),
                'PNG files (*.png)'
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUT_GRID_PNG,
                self.tr('Comparison grid PNG'),
                'PNG files (*.png)'
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        if not _HAS_MPL:
            raise QgsProcessingException('Matplotlib is not available in this QGIS environment')

        layers = self.parameterAsLayerList(parameters, self.INPUT_RASTERS, context)
        if not layers:
            raise QgsProcessingException('At least one NTL raster is required')

        boundary_layer = self.parameterAsVectorLayer(parameters, self.BOUNDARY, context)
        boundary_width = self.parameterAsDouble(parameters, self.BOUNDARY_WIDTH, context)

        method_idx = self.parameterAsEnum(parameters, self.METHOD, context)
        method = self.METHOD_ITEMS[method_idx]

        thr_idx = self.parameterAsEnum(parameters, self.THRESHOLD_TYPE, context)
        thr_type = self.THRESHOLD_ITEMS[thr_idx]

        custom_coef = self.parameterAsDouble(parameters, self.CUSTOM_COEF, context)
        manual_thr = self.parameterAsDouble(parameters, self.MANUAL_THRESHOLD, context)

        make_distrib = self.parameterAsBool(parameters, self.MAKE_DISTRIB_PNG, context)
        out_folder = self.parameterAsString(parameters, self.OUT_FOLDER, context)
        out_report_csv = self.parameterAsFileOutput(parameters, self.OUT_REPORT_CSV, context)
        out_stats_csv = self.parameterAsFileOutput(parameters, self.OUT_STATS_CSV, context)
        out_trend_png = self.parameterAsFileOutput(parameters, self.OUT_TREND_PNG, context)
        out_grid_png = self.parameterAsFileOutput(parameters, self.OUT_GRID_PNG, context)

        if not os.path.isdir(out_folder):
            os.makedirs(out_folder, exist_ok=True)

        # Prepare CRS transform for boundary -> raster CRS
        raster_crs = layers[0].crs()
        ct = None
        if boundary_layer:
            try:
                ct = QgsCoordinateTransform(boundary_layer.crs(), raster_crs, context.transformContext())
            except Exception:
                ct = None  # fallback: skip transform if it fails

        names, years, arrays, extents = [], [], [], []

        # Read all rasters
        for i, lyr in enumerate(layers):
            if not isinstance(lyr, QgsRasterLayer):
                raise QgsProcessingException('All inputs must be raster layers')
            path = lyr.source()
            feedback.pushInfo(f'Reading raster {i+1}: {os.path.basename(path)}')
            arr, extent = read_raster_to_array(path)
            base = os.path.basename(path)
            names.append(base)
            years.append(parse_year_from_name(base))
            arrays.append(arr)
            extents.append(extent)

        # Per-raster processing
        report_rows = []
        stats_rows = []

        for i, arr in enumerate(arrays):
            if feedback.isCanceled():
                break

            if thr_type == "manual":
                filtered, used_thr = apply_threshold_filter(arr, "manual", manual_thr)
            elif thr_type in ["auto", "percentile"]:
                filtered, used_thr = apply_threshold_filter(arr, thr_type, None)
            else:
                filtered, used_thr = apply_threshold_filter(arr, "none", None)

            carbon, cstats = calculate_carbon_from_ntl(filtered, method, custom_coef)

            row = {
                "Dataset": years[i],
                "FileName": names[i],
                "Method": method,
                "ThresholdType": thr_type,
                "ThresholdValue": float(used_thr),
                "Carbon_Estimate_Ton": float(carbon),
                "Carbon_Estimate_KTon": float(carbon) / 1000.0,
                "Total_NTL": float(cstats['total_ntl']),
                "Pixel_Count": int(cstats['pixel_count']),
                "Active_Pixels": int(np.nansum(filtered > 0)),
                "Carbon_Per_Pixel": float(cstats['carbon_per_pixel']),
                "Mean_NTL": float(cstats['mean_ntl']),
                "Max_NTL": float(cstats['max_ntl'])
            }
            report_rows.append(row)

            srow = {
                "Dataset": years[i],
                "FileName": names[i],
                "Min": float(np.nanmin(arr)) if np.isfinite(np.nanmin(arr)) else 0.0,
                "Max": float(np.nanmax(arr)) if np.isfinite(np.nanmax(arr)) else 0.0,
                "Mean": float(np.nanmean(arr)) if np.isfinite(np.nanmean(arr)) else 0.0,
                "Std": float(np.nanstd(arr)) if np.isfinite(np.nanstd(arr)) else 0.0,
                "Area_px": int(np.sum(~np.isnan(arr)))
            }
            stats_rows.append(srow)

            # Per-raster distribution PNG
            if make_distrib:
                try:
                    fig, axes = plt.subplots(1, 2, figsize=(12, 5))
                    ntl_cmap = create_ntl_colormap()
                    extent = extents[i]  # (left, right, bottom, top)
                    extent_list = [extent[0], extent[1], extent[2], extent[3]]

                    # Left: original NTL + boundary (magenta for contrast)
                    im1 = axes[0].imshow(arr, cmap=ntl_cmap, extent=extent_list)
                    axes[0].set_title(f'NTL (Original) — {years[i]}')
                    axes[0].set_xlabel('Longitude'); axes[0].set_ylabel('Latitude')
                    plt.colorbar(im1, ax=axes[0], shrink=0.8)
                    if boundary_layer:
                        draw_boundary(axes[0], boundary_layer, transform=ct,
                                      color="#FF00FF", linewidth=boundary_width,
                                      extent_limits=extent)

                    # Right: carbon per pixel + boundary (cyan for contrast with YlOrRd)
                    if method == "Custom_Linear":
                        coef = custom_coef
                        carbon_px = np.array(filtered, copy=True)
                        carbon_px[np.isnan(carbon_px)] = 0.0
                        carbon_px = carbon_px * float(coef)
                        carbon_px[np.isnan(filtered)] = np.nan
                    elif method == "Elvidge_2009":
                        coef = 0.0005
                        carbon_px = np.array(filtered, copy=True)
                        carbon_px[np.isnan(carbon_px)] = 0.0
                        carbon_px = carbon_px * float(coef)
                        carbon_px[np.isnan(filtered)] = np.nan
                    elif method == "Shi_2016":
                        coef = 0.0008
                        carbon_px = np.array(filtered, copy=True)
                        carbon_px[np.isnan(carbon_px)] = 0.0
                        carbon_px = carbon_px * float(coef)
                        carbon_px[np.isnan(filtered)] = np.nan
                    elif method == "Power_Law":
                        coef = 0.0001
                        valid = np.array(filtered, copy=True)
                        valid[np.isnan(valid)] = 0.0
                        carbon_px = np.power(valid, 1.2) * coef
                        carbon_px[np.isnan(filtered)] = np.nan
                    else:
                        coef = 0.0006
                        carbon_px = np.array(filtered, copy=True)
                        carbon_px[np.isnan(carbon_px)] = 0.0
                        carbon_px = carbon_px * float(coef)
                        carbon_px[np.isnan(filtered)] = np.nan

                    im2 = axes[1].imshow(carbon_px, cmap=plt.cm.YlOrRd, extent=extent_list)
                    axes[1].set_title(f'Carbon Estimate — {years[i]}')
                    axes[1].set_xlabel('Longitude'); axes[1].set_ylabel('Latitude')
                    plt.colorbar(im2, ax=axes[1], shrink=0.8)
                    if boundary_layer:
                        draw_boundary(axes[1], boundary_layer, transform=ct,
                                      color="#00FFFF", linewidth=boundary_width,
                                      extent_limits=extent)

                    fig.suptitle(f'NTL and Carbon Distribution — {names[i]}', fontsize=12)
                    out_png = os.path.join(out_folder, f"distribution_{years[i]}.png")
                    plt.tight_layout()
                    fig.savefig(out_png, dpi=200)
                    plt.close(fig)
                except Exception as e:
                    feedback.pushInfo(f"Failed to create distribution for {names[i]} due to {e}")

        # CSV outputs
        with open(out_report_csv, "w", newline="", encoding="utf-8") as f:
            writer = csv.DictWriter(f, fieldnames=list(report_rows[0].keys()))
            writer.writeheader()
            for r in report_rows:
                writer.writerow(r)

        with open(out_stats_csv, "w", newline="", encoding="utf-8") as f:
            writer = csv.DictWriter(f, fieldnames=list(stats_rows[0].keys()))
            writer.writeheader()
            for r in stats_rows:
                writer.writerow(r)

        # Carbon trend PNG
        try:
            x = list(range(1, len(report_rows) + 1))
            y_total = [r["Carbon_Estimate_Ton"] for r in report_rows]
            y_cpp = [r["Carbon_Per_Pixel"] for r in report_rows]
            y_ntl = [r["Total_NTL"] for r in report_rows]

            fig, axes = plt.subplots(2, 2, figsize=(12, 9))

            axes[0, 0].plot(x, y_total, marker='o', linewidth=2)
            axes[0, 0].set_title('Total Carbon Estimate (Ton)')
            axes[0, 0].set_xlabel('Dataset (by order)')
            axes[0, 0].set_ylabel('Ton')
            axes[0, 0].grid(True, alpha=0.3)
            axes[0, 0].set_xticks(x); axes[0, 0].set_xticklabels([parse_year_from_name(n) for n in names], rotation=45)

            axes[0, 1].plot(x, y_cpp, marker='s', linewidth=2)
            axes[0, 1].set_title('Carbon Density per Pixel')
            axes[0, 1].set_xlabel('Dataset (by order)')
            axes[0, 1].set_ylabel('Ton / pixel')
            axes[0, 1].grid(True, alpha=0.3)
            axes[0, 1].set_xticks(x); axes[0, 1].set_xticklabels([parse_year_from_name(n) for n in names], rotation=45)

            axes[1, 0].bar(x, y_ntl)
            axes[1, 0].set_title('Total NTL (post-threshold)')
            axes[1, 0].set_xlabel('Dataset (by order)')
            axes[1, 0].set_ylabel('NTL sum')
            axes[1, 0].set_xticks(x); axes[1, 0].set_xticklabels([parse_year_from_name(n) for n in names], rotation=45)

            axes[1, 1].scatter(y_ntl, y_total, s=60, alpha=0.7)
            axes[1, 1].set_title('NTL vs Carbon')
            axes[1, 1].set_xlabel('Total NTL')
            axes[1, 1].set_ylabel('Carbon (Ton)')
            axes[1, 1].grid(True, alpha=0.3)

            plt.tight_layout()
            fig.savefig(out_trend_png, dpi=200)
            plt.close(fig)
        except Exception as e:
            feedback.pushInfo(f"Failed to create trend plot due to {e}")

        # Comparison grid PNG (titles use year)
        try:
            n = len(arrays)
            cols = 2 if n > 1 else 1
            rows = int(math.ceil(n / cols))
            fig, axes = plt.subplots(rows, cols, figsize=(5 * cols, 4 * rows))
            if rows == 1 and cols == 1:
                axes = np.array([[axes]])
            elif rows == 1:
                axes = np.array([axes])
            axes_flat = axes.flatten()

            ntl_cmap = create_ntl_colormap()

            for i, ax in enumerate(axes_flat):
                if i >= n:
                    ax.axis('off')
                    continue
                arr = arrays[i]
                extent = extents[i]
                extent_list = [extent[0], extent[1], extent[2], extent[3]]
                im = ax.imshow(arr, cmap=ntl_cmap, extent=extent_list)
                ax.set_title(f"{years[i]}")
                ax.set_xlabel("Longitude")
                ax.set_ylabel("Latitude")
                ax.grid(True, alpha=0.2)
                cbar = plt.colorbar(im, ax=ax, shrink=0.8)
                cbar.set_label('Radiance')
                if boundary_layer:
                    # Magenta for contrast on NTL grid
                    draw_boundary(ax, boundary_layer, transform=ct,
                                  color="#FF00FF", linewidth=boundary_width,
                                  extent_limits=extent)

            plt.tight_layout()
            fig.savefig(out_grid_png, dpi=200)
            plt.close(fig)
        except Exception as e:
            feedback.pushInfo(f"Failed to create comparison grid due to {e}")

        feedback.pushInfo("Finished generating reports and visualizations")
        return {
            self.OUT_REPORT_CSV: out_report_csv,
            self.OUT_STATS_CSV: out_stats_csv,
            self.OUT_TREND_PNG: out_trend_png,
            self.OUT_GRID_PNG: out_grid_png,
            self.OUT_FOLDER: out_folder
        }


# Registration for QGIS Script Runner
def classFactory(iface):
    return NTLCarbonToolbox()
