# -----------------------------------------------------------
# Copyright (C) 2023 Oslandia - Jacky Volpes
# -----------------------------------------------------------
# Licensed under the terms of GNU GPL 2
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# ---------------------------------------------------------------------

"""
Classes for external storage API

Upload documentation http://namespace.lantmateriet.se/distribution/geodatakatalog/uppladdning/v1/uppladdning-api-1.2.html
Download documentation http://namespace.lantmateriet.se/distribution/geodatakatalog/nedladdning/v1/nedladdning-api-1.1.html
"""

import json
import mimetypes
from pathlib import Path

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsBlockingNetworkRequest,
    QgsExternalStorage,
    QgsExternalStorageFetchedContent,
    QgsExternalStorageStoredContent,
    QgsFeedback,
    QgsFetchedContent,
    QgsTask,
)
from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtNetwork import QNetworkRequest

from qgis.utils import iface

from .toolbelt.preferences import PlgOptionsManager


class NgpExternalStorage(QgsExternalStorage):
    """Main class for registering this external storage into QGIS"""

    def displayName(self):
        """The name of the storage"""
        return "National Geodata Platform"

    def doStore(self, filePath, url, authCfgId=""):  # pylint: disable=unused-argument
        """Do the upload, method called by store() which is already implemented in QgsExternalStorage"""

        settings = PlgOptionsManager.get_plg_settings()
        return NgpExternalStorageStoredContent(
            filePath, settings.upload_auth_cfg_id, settings.endpoint_url, settings.information_type
        )

    def doFetch(self, resourceId, authCfgId=""):
        """Do the download, method called by fetch() which is already implemented in QgsExternalStorage"""

        settings = PlgOptionsManager.get_plg_settings()
        if settings.download_auth_cfg_id == "":
            iface.messageBar().pushWarning(
                "Credentials missing", "No credentials for download in plugin settings"
            )
        if settings.endpoint_url == "":
            iface.messageBar().pushWarning(
                "URL missing", "No endpoint URL in plugin settings"
            )

        fetchedContent = QgsApplication.networkContentFetcherRegistry().fetch(
            f"{settings.endpoint_url}/nedladdning/v1/asset/{resourceId}",
            Qgis.ActionStart.Deferred,
            settings.download_auth_cfg_id,
        )
        return NgpExternalStorageFetchedContent(fetchedContent)

    def type(self):
        """The identification code of the storage"""
        return "NgpConnect"


class NgpExternalStorageStoredContent(QgsExternalStorageStoredContent):
    """
    Class for handling file upload in a background task
    """

    def __init__(self, filePath, authCfgId, url, information_type):
        super().__init__()
        self.uploadTask = NgpExternalStorageStoreTask(
            Path(filePath), authCfgId, url, information_type
        )
        self.uploadTask.taskCompleted.connect(self.uploadFinished)
        self.uploadTask.taskTerminated.connect(
            lambda: self.reportError(self.uploadTask.errorString())
        )
        self.uploadTask.progressChanged.connect(self.progressChanged.emit)

    def store(self):
        """Launch the background task"""

        self.setStatus(Qgis.ContentStatus.Running)
        QgsApplication.taskManager().addTask(self.uploadTask)

    def uploadFinished(self):
        """Broadcast that the upload is finished"""

        self.setStatus(Qgis.ContentStatus.Finished)
        self.stored.emit()

    def url(self):
        """This is the property that will be stored in the feature attribute"""
        return self.uploadTask.resourceId

    def cancel(self):
        """Connect the task to the slots which handles cancellation"""

        self.uploadTask.taskTerminated.disconnect()
        self.uploadTask.taskTerminated.connect(self.setCanceled)
        self.uploadTask.cancel()

    def setCanceled(self):
        """Broadcast that the upload is canceled"""

        self.setStatus(Qgis.ContentStatus.Canceled)
        self.canceled.emit()


class NgpExternalStorageFetchedContent(QgsExternalStorageFetchedContent):
    """
    Class for handling file download
    """

    def __init__(self, fetchedContent):
        """
        fetchedContent is a QgsFetchedContent object, that can be
        fetched by calling its download method, and that emit signals
        """

        super().__init__()
        self.fetchedContent = fetchedContent
        self.fetchedContent.fetched.connect(self.onFetched)
        self.fetchedContent.errorOccurred.connect(self.reportError)

    def reportError(self, code, errorMsg):  # pylint: disable=unused-argument
        """Call parent method without code"""

        super().reportError(errorMsg)

    def fetch(self):
        """Ask for downloading of resource"""

        if self.fetchedContent is None:
            return

        self.setStatus(Qgis.ContentStatus.Running)
        self.fetchedContent.download()  # Download resource

        # Could be already fetched/cached
        if self.fetchedContent.status() == QgsFetchedContent.Finished:
            self.setStatus(Qgis.ContentStatus.Finished)
            self.fetched.emit()

    def filePath(self):
        """Return the file path of the temporary file of fetched content"""

        if self.fetchedContent is None:
            return ""
        return self.fetchedContent.filePath()

    def onFetched(self):
        """Broadcast that the resource has been downloaded"""

        if self.fetchedContent is None:
            return

        if self.fetchedContent.status() == QgsFetchedContent.Finished:
            self.setStatus(Qgis.ContentStatus.Finished)
            self.fetched.emit()


class NgpExternalStorageStoreTask(QgsTask):
    """
    The QgsTask to upload file

    It is here in the run() method that we have to follow the
    API requirements to communicate with the server.
    """

    def __init__(self, filePath, authCfgId, url, information_type):
        super().__init__(f"Uploading {filePath}")
        self.filePath = filePath
        self.authCfgId = authCfgId
        self.feedback = QgsFeedback(self)
        self.mErrorString = ""
        self.resourceId = ""
        self.url = url
        self.information_type = information_type

    def run(self):
        """Everything here is made in a background thread"""

        if self.authCfgId == "":
            self.mErrorString = self.tr(
                "No upload authentification configured in plugin settings"
            )
            return False

        # --------------------------
        # Create resource

        req = QNetworkRequest(QUrl(self.url + "/uppladdning/v1/resurs"))
        req.setRawHeader(b"Content-Type", b"application/json")
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(self.authCfgId)
        err = request.post(
            req, bytes('{"informationstyp": "' + self.information_type + '"}', 'utf8'), feedback=self.feedback
        )
        if err != QgsBlockingNetworkRequest.NoError or self.isCanceled():
            self.mErrorString = request.errorMessage()
            return False

        self.resourceId = json.loads(bytes(request.reply().content()).decode())[
            "referensidentitet"
        ]

        # --------------------------
        # Upload file into resource

        # Prepare the request
        request.uploadProgress.connect(self.updateProgress)
        req = QNetworkRequest(
            QUrl(f"{self.url}/uppladdning/v1/resurs/{self.resourceId}/leverans")
        )
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(self.authCfgId)

        # Get file type
        mimeType = mimetypes.guess_type(str(self.filePath))[0]
        if mimeType is None:
            mimeType = "application/octet-stream"

        # Set headers and post file
        req.setRawHeader(b"Content-Type", mimeType.encode())
        req.setRawHeader(b"Leverans-Filnamn", self.filePath.name.encode())
        with self.filePath.open("rb") as fio:
            err = request.post(req, fio.read(), feedback=self.feedback)

        if err != QgsBlockingNetworkRequest.NoError or self.isCanceled():
            self.mErrorString = request.errorMessage()
            return False

        return True

    def errorString(self):
        """The error string"""

        return self.mErrorString

    def updateProgress(self, bytesReceived, bytesTotal):
        """Update the upload progress in percentage of bytes"""

        if not self.isCanceled() and bytesTotal > 0:
            self.setProgress(int(bytesReceived * 100 / bytesTotal))

    def cancel(self):
        """Cancel the background task"""

        self.feedback.cancel()
        super().cancel()
