# -----------------------------------------------------------
# Copyright (C) 2023 Oslandia - Jacky Volpes
# -----------------------------------------------------------
# Licensed under the terms of GNU GPL 2
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# ---------------------------------------------------------------------

"""
Processings for NGP API
"""

import json
import time
import tempfile
from datetime import datetime, timedelta

from processing import QgsProcessingException, QgsVectorLayer
from qgis.core import (
    QgsBlockingNetworkRequest,
    QgsEditorWidgetSetup,
    QgsFeature,
    QgsFeatureSink,
    QgsField,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingLayerPostProcessorInterface,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterField,
    QgsProcessingParameterString,
    QgsProcessingProvider,
    QgsProcessingParameterFile,
    QgsProcessingParameterBoolean,
)
import processing
from qgis.PyQt.QtCore import QUrl, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtNetwork import QNetworkRequest

from ngpconnect.__about__ import DIR_PLUGIN_ROOT, __version__
from ngpconnect.toolbelt.preferences import PlgOptionsManager
from ngpconnect.gui.dlg_settings import InformationType


class NgpProvider(QgsProcessingProvider):
    def __init__(self):
        super().__init__()
        self._tr = None

    def loadAlgorithms(self):
        """Loads all algorithms belonging to this provider."""
        updateAlg = NgpUpdateAlgorithm()
        updateAlg.setTranslationCallback(self._tr)
        self.addAlgorithm(updateAlg)
        deleteAlg = NgpDeleteAlgorithm()
        deleteAlg.setTranslationCallback(self._tr)
        self.addAlgorithm(deleteAlg)
        deleteResourceAlg = NgpDeleteResourceAlgorithm()
        deleteResourceAlg.setTranslationCallback(self._tr)
        self.addAlgorithm(deleteResourceAlg)
        deleteResourceUuidAlg = NgpDeleteResourceUuidAlgorithm()
        deleteResourceUuidAlg.setTranslationCallback(self._tr)
        self.addAlgorithm(deleteResourceUuidAlg)
        downloadAlg = NgpDownloadAlgorithm()
        downloadAlg.setTranslationCallback(self._tr)
        self.addAlgorithm(downloadAlg)
        downloadUuidAlg = NgpDownloadUuidAlgorithm()
        downloadUuidAlg.setTranslationCallback(self._tr)
        self.addAlgorithm(downloadUuidAlg)

    def id(self) -> str:
        """Unique provider id, used for identifying it. This string should be a unique, short, character only.
        This string should not be localised.

        :return: provider ID
        :rtype: str
        """
        return "ngpconnect"

    def name(self) -> str:
        """Returns the provider name, which is used to describe the provider
        within the GUI. This string should be short (e.g. "Lastools") and localised.

        :return: provider name
        :rtype: str
        """
        return self.tr("NGP Connect")

    def longName(self) -> str:
        """Longer version of the provider name, which can include
        extra details such as version numbers. E.g. "Lastools LIDAR tools". This string should
        be localised. The default implementation returns the same string as name().

        :return: provider long name
        :rtype: str
        """
        return self.tr("NGP Connect - Tools")

    def icon(self) -> QIcon:
        """QIcon used for your provider inside the Processing toolbox menu.

        :return: provider icon
        :rtype: QIcon
        """
        return QIcon(
            str(DIR_PLUGIN_ROOT / "resources" / "images" / "NGPconnect_trans.png")
        )

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def versionInfo(self) -> str:
        """Version information for the provider, or an empty string if this is not \
        applicable (e.g. for inbuilt Processing providers). For plugin based providers, \
        this should return the plugin’s version identifier.

        :return: version
        :rtype: str
        """
        return __version__


class NgpUpdateAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    FEATURES = "FEATURES"  # pylint: disable=invalid-name
    ID_FIELD = "ID_FIELD"  # pylint: disable=invalid-name
    JSON_FIELD = "JSON_FIELD"  # pylint: disable=invalid-name
    OUTPUT = "OUTPUT"  # pylint: disable=invalid-name
    VALIDATION_REPORT_FIELD_NAME = (
        "VALIDATION_REPORT_FIELD_NAME"  # pylint: disable=invalid-name
    )
    VALIDATION_STATUS_FIELD_NAME = (
        "VALIDATION_STATUS_FIELD_NAME"  # pylint: disable=invalid-name
    )

    def __init__(self):
        super().__init__()
        self._tr = None
        self.styler = NgpUpdateOutputLayerStylerInterface()

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def createInstance(self):
        updateAlg = NgpUpdateAlgorithm()
        updateAlg.setTranslationCallback(self._tr)
        return updateAlg

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "ngp_update"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("NGP Update")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("NGP Connect")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "ngp_connect"

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it.
        """
        return self.tr(
            "This algorithm sends features to the update API of Lantmäteriets' National Geodata Platform.\n"
            "The JSON data must already have been created and stored in an attribute. "
            "The tool just sends the data to the API and receives the validation report and status.\n"
            "The type of information to send have to be selected. Plans or buildings.\n"
            "The layer holding the features must be selected as input layer.\n"
            "ID field is the object id (uuid) used for the feature.\n"
            "The JSON field is a text field that holds the information about the feature structured according to the specifications by Lantmäteriet. "
            'Read more about it at <a href="https://www.lantmateriet.se/sv/nationella-geodataplattformen/">https://www.lantmateriet.se/sv/nationella-geodataplattformen/</a>\n'
            "The validation report and validation status is added as new attributes to the output layer."
        )

    def initAlgorithm(self, config=None):  # pylint: disable=unused-argument
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Features
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.FEATURES,
                self.tr("Features"),
                [QgsProcessing.TypeVectorAnyGeometry],
            )
        )

        # ID field
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_FIELD,
                self.tr("ID field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName=self.FEATURES,
            )
        )

        # JSON field
        self.addParameter(
            QgsProcessingParameterField(
                self.JSON_FIELD,
                self.tr("JSON field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName=self.FEATURES,
            )
        )

        # Output features
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("Output layer"),
                QgsProcessing.TypeVectorAnyGeometry,
            )
        )

        # Validation report field name
        self.addParameter(
            QgsProcessingParameterString(
                self.VALIDATION_REPORT_FIELD_NAME,
                self.tr("Validation report field name on output layer"),
                "validation_report",
            )
        )

        # Validation status field name
        self.addParameter(
            QgsProcessingParameterString(
                self.VALIDATION_STATUS_FIELD_NAME,
                self.tr("Validation status field name on output layer"),
                "validation_status",
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Stop if no feature
        source = self.parameterAsSource(parameters, self.FEATURES, context)
        if source.featureCount() == 0:
            feedback.pushWarning(self.tr("No feature to process"))
            return {}
        features = list(source.getFeatures())

        # Stop if an attribute is missing
        id_field = self.parameterAsFields(parameters, self.ID_FIELD, context)[0]
        json_field = self.parameterAsFields(parameters, self.JSON_FIELD, context)[0]
        available_fields = [f.name() for f in source.fields()]
        for attr_name in [id_field, json_field]:
            if attr_name not in available_fields:
                raise QgsProcessingException(
                    self.tr("The attribute '{0}' is not present in the layer").format(
                        attr_name
                    )
                )

        # Validate JSON
        for feature in features:
            try:
                json.loads(feature[json_field])
            except json.JSONDecodeError as e:
                raise QgsProcessingException(
                    self.tr("Object {0}: JSON is not conform! See error below.").format(
                        feature[id_field]
                    )
                    + f"\n{e}"
                )

        # Read plugin settings
        feedback.pushInfo(self.tr("Reading plugin parameters"))
        settings = PlgOptionsManager.get_plg_settings()
        if settings.endpoint_url == "":
            raise QgsProcessingException(
                self.tr("No endpoint URL in plugin parameters")
            )
        if settings.city_code == "":
            raise QgsProcessingException(self.tr("No city code in plugin parameters"))
        if settings.upload_auth_cfg_id == "":
            raise QgsProcessingException(self.tr("No authentication set for upload"))
        if settings.information_type == InformationType.DETAILED_PLAN.value:
            content_type_upload = "application/vnd.lm.detaljplan.v4+json"
        elif settings.information_type == InformationType.BUILDING.value:
            content_type_upload = "application/vnd.lm.byggnad.v1+json"
        elif settings.information_type == InformationType.GENERAL_PLAN.value:
            content_type_upload = "application/vnd.lm.oversiktsplan.v1+json"
        elif settings.information_type == InformationType.HEIGHTMODEL_GRID.value:
            content_type_upload = "application/vnd.lm.hojdmodellgrid.v1+json"
        elif settings.information_type == InformationType.MEASURE_POINT.value:
            content_type_upload = "application/vnd.lm.stompunkt.v1+json"
        else:
            raise QgsProcessingException(
                self.tr("Unknown information type: ") + settings.information_type
            )
        update_url = f"{settings.endpoint_url}/uppdatering/v1"

        # Prepare request object that will be used in the whole process
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(settings.upload_auth_cfg_id)

        # Get a reception ID for this transaction
        feedback.pushInfo(self.tr("Getting a reception ID"))
        req = QNetworkRequest(QUrl(f"{update_url}/mottagning"))
        req.setRawHeader(b"Content-Type", b"application/json")
        payload = {
            "leveransinfo": {
                "provider": settings.city_code,
                "informationstyp": settings.information_type,
            }
        }
        err_msg = getErrorMsg(request.post(req, json.dumps(payload).encode()))
        reply = bytes(request.reply().content()).decode()
        if err_msg:
            feedback.pushWarning(err_msg)
            feedback.pushWarning(request.errorMessage())
            raise QgsProcessingException(reply)
        res = json.loads(reply)
        reception_id = res["id"]
        if (status := res["status"]["typ"]) != "registrerad":
            err_msg = self.tr("Status should be 'registrerad' but is: '{0}'").format(
                status
            )
            if (err_msg_status := "felmeddelande") in res["status"]:
                err_msg += "\n"
                err_msg += self.tr("Error message is '{0}'").format(
                    res["status"][err_msg_status]
                )
            raise QgsProcessingException(err_msg)
        feedback.pushInfo(self.tr("Reception ID: {0}").format(reception_id))

        # Register changes
        feedback.pushInfo(self.tr("Registering changes"))
        req = QNetworkRequest(
            QUrl(f"{update_url}/mottagning/{reception_id}/forandringar")
        )
        req.setRawHeader(b"Content-Type", b"application/json")
        payload = {
            "forandringar": [
                {"objektidentitet": feature[id_field], "typ": "leverans"}
                for feature in features
            ]
        }
        err_msg = getErrorMsg(request.post(req, json.dumps(payload).encode()))
        reply = bytes(request.reply().content()).decode()
        if err_msg:
            feedback.pushWarning(err_msg)
            feedback.pushWarning(request.errorMessage())
            raise QgsProcessingException(reply)
        res = json.loads(reply)
        for object_dict in res["forandringar"]:
            if (status := object_dict["status"]["typ"]) != "registrerad":
                err_msg = self.tr(
                    "Status should be 'registrerad' but is: '{0}' for object {1}"
                ).format(status, object_dict["objektidentitet"])
                if (err_msg_status := "felmeddelande") in object_dict["status"]:
                    err_msg += "\n"
                    err_msg += self.tr("Error message is '{0}'").format(
                        object_dict["status"][err_msg_status]
                    )
                raise QgsProcessingException(err_msg)

        # Upload json objects for each feature
        feedback.pushInfo(self.tr("Uploading json objects"))
        for feature in features:
            feedback.pushInfo(self.tr("Object {0}").format(feature[id_field]))
            req = QNetworkRequest(
                QUrl(
                    f"{update_url}/mottagning/{reception_id}/forandringar/{feature[id_field]}/leverans"
                )
            )
            req.setRawHeader(b"Content-Type", content_type_upload.encode())
            err_msg = getErrorMsg(request.post(req, feature[json_field].encode()))
            reply = bytes(request.reply().content()).decode()
            if err_msg:
                feedback.pushWarning(err_msg)
                feedback.pushWarning(request.errorMessage())
                raise QgsProcessingException(reply)

        # Create output layer
        output_fields = source.fields()
        validation_report_field_name = self.parameterAsString(
            parameters, self.VALIDATION_REPORT_FIELD_NAME, context
        ).strip()
        validation_status_field_name = self.parameterAsString(
            parameters, self.VALIDATION_STATUS_FIELD_NAME, context
        ).strip()
        for field_to_add in [
            validation_report_field_name,
            validation_status_field_name,
        ]:
            if not output_fields.append(QgsField(field_to_add, QVariant.String)):
                raise QgsProcessingException(
                    self.tr(
                        "Error adding field '{0}' to output layer, maybe the field name already exists"
                    ).format(field_to_add)
                )
        sink, output_layer_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            output_fields,
            source.wkbType(),
            source.sourceCrs(),
        )
        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        # Get validation reports
        for feature in features:
            start_time = datetime.now()
            feedback.pushInfo(
                self.tr("Waiting validation report for object {0}").format(
                    feature[id_field]
                )
            )
            timeout = False
            while not timeout:
                if feedback.isCanceled():
                    return {}
                time.sleep(1)
                req = QNetworkRequest(
                    QUrl(
                        f"{update_url}/mottagning/{reception_id}/forandringar/{feature[id_field]}/leverans"
                    )
                )
                err_msg = getErrorMsg(request.get(req))
                code = request.reply().attribute(
                    QNetworkRequest.HttpStatusCodeAttribute
                )
                if code == 204:
                    feedback.pushInfo(".")
                else:
                    break
                timeout = datetime.now() - start_time > timedelta(seconds=120)
            if timeout:
                raise QgsProcessingException(
                    self.tr("Timeout occurred waiting for the report to be ready")
                )
            reply = bytes(request.reply().content()).decode()
            if err_msg:
                feedback.pushWarning(err_msg)
                feedback.pushWarning(request.errorMessage())
                raise QgsProcessingException(reply)
            report = json.loads(reply)

            # get validation reports status
            feedback.pushInfo(self.tr("Getting validation status"))
            req = QNetworkRequest(
                QUrl(
                    f"{update_url}/mottagning/{reception_id}/forandringar/{feature[id_field]}"
                )
            )
            err_msg = getErrorMsg(request.get(req))
            reply = bytes(request.reply().content()).decode()
            if err_msg:
                feedback.pushWarning(err_msg)
                feedback.pushWarning(request.errorMessage())
                raise QgsProcessingException(reply)
            res = json.loads(reply)
            validation_status = res["status"]["typ"]
            if (err_msg_status := "felmeddelande") in res["status"]:
                validation_status += f" ({res['status'][err_msg_status]})"

            # Save validation report in new feature
            output_feature = QgsFeature(output_fields)
            output_feature.setAttributes(
                feature.attributes()
                + [
                    json.dumps(
                        report,
                        sort_keys=True,
                        indent=4,
                        ensure_ascii=False,
                    ),
                    validation_status,
                ]
            )
            output_feature.setGeometry(feature.geometry())
            if not sink.addFeature(output_feature, QgsFeatureSink.FastInsert):
                raise QgsProcessingException(
                    self.tr("Unable to save the output feature")
                )

        # If output layer is loaded, choose json widget for the report field
        if context.willLoadLayerOnCompletion(output_layer_id):
            self.styler.json_field_name = validation_report_field_name
            context.layerToLoadOnCompletionDetails(output_layer_id).setPostProcessor(
                self.styler
            )

        return {self.OUTPUT: output_layer_id}


class NgpDeleteAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm that takes a vector layer performs a post network
    request against NGP using the id of the provided features for deletion.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    FEATURES = "FEATURES"  # pylint: disable=invalid-name
    ALLOW_MULTIPLE_FEATURES = "ALLOW_MULTIPLE_FEATURES"  # pylint: disable=invalid-name
    ID_FIELD = "ID_FIELD"  # pylint: disable=invalid-name
    OUTPUT = "OUTPUT"  # pylint: disable=invalid-name

    def __init__(self):
        super().__init__()
        self._tr = None

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def createInstance(self):
        deleteAlg = NgpDeleteAlgorithm()
        deleteAlg.setTranslationCallback(self._tr)
        return deleteAlg

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "ngp_delete"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("NGP Delete")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("NGP Connect")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "ngp_connect"

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it.
        """
        return self.tr(
            "This algorithm sends id of features to the update API of Lantmäteriets' National Geodata Platform for deletion.\n"
            "The layer holding the features must be selected as input layer.\n"
            "By default it is only allowed to to process single features. \n"
            "ID field is the object id (uuid) used for the feature.\n"
            'Read more about it at <a href="https://www.lantmateriet.se/sv/nationella-geodataplattformen/">https://www.lantmateriet.se/sv/nationella-geodataplattformen/</a>\n'
        )

    def initAlgorithm(self, config=None):  # pylint: disable=unused-argument
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Features
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.FEATURES,
                self.tr("Features"),
                [QgsProcessing.TypeVectorAnyGeometry],
            )
        )

        # Allow multiple features
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.ALLOW_MULTIPLE_FEATURES,
                self.tr("Allow multiple features"),
                defaultValue=False,
            )
        )

        # ID field
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_FIELD,
                self.tr("ID field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName=self.FEATURES,
            )
        )

        # Output features
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("Output layer"),
                QgsProcessing.TypeVectorAnyGeometry,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Stop if no feature
        source = self.parameterAsSource(parameters, self.FEATURES, context)
        if source.featureCount() == 0:
            feedback.pushWarning(self.tr("No feature to process"))
            return {}
        # Stop if more than one feature and its not allowed
        if source.featureCount() > 1 and not self.parameterAsBool(
            parameters, self.ALLOW_MULTIPLE_FEATURES, context
        ):
            feedback.pushWarning(
                self.tr(
                    "Multiple features to process not allowed. Change parameter value or select just one feature."
                )
            )
            return {}
        features = list(source.getFeatures())

        # Stop if an attribute is missing
        id_field = self.parameterAsFields(parameters, self.ID_FIELD, context)[0]
        available_fields = [f.name() for f in source.fields()]
        for attr_name in [id_field]:
            if attr_name not in available_fields:
                raise QgsProcessingException(
                    self.tr("The attribute '{0}' is not present in the layer").format(
                        attr_name
                    )
                )

        # Read plugin settings
        feedback.pushInfo(self.tr("Reading plugin parameters"))
        settings = PlgOptionsManager.get_plg_settings()
        if settings.endpoint_url == "":
            raise QgsProcessingException(
                self.tr("No endpoint URL in plugin parameters")
            )
        if settings.city_code == "":
            raise QgsProcessingException(self.tr("No city code in plugin parameters"))
        if settings.upload_auth_cfg_id == "":
            raise QgsProcessingException(self.tr("No authentication set for upload"))
        update_url = f"{settings.endpoint_url}/uppdatering/v1"

        # Prepare request object that will be used in the whole process
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(settings.upload_auth_cfg_id)

        # Get a reception ID for this transaction
        feedback.pushInfo(self.tr("Getting a reception ID"))
        req = QNetworkRequest(QUrl(f"{update_url}/mottagning"))
        req.setRawHeader(b"Content-Type", b"application/json")
        payload = {
            "leveransinfo": {
                "provider": settings.city_code,
                "informationstyp": settings.information_type,
            }
        }
        err_msg = getErrorMsg(request.post(req, json.dumps(payload).encode()))
        reply = bytes(request.reply().content()).decode()
        if err_msg:
            feedback.pushWarning(err_msg)
            feedback.pushWarning(request.errorMessage())
            raise QgsProcessingException(reply)
        res = json.loads(reply)
        reception_id = res["id"]
        if (status := res["status"]["typ"]) != "registrerad":
            err_msg = self.tr("Status should be 'registrerad' but is: '{0}'").format(
                status
            )
            if (err_msg_status := "felmeddelande") in res["status"]:
                err_msg += "\n"
                err_msg += self.tr("Error message is '{0}'").format(
                    res["status"][err_msg_status]
                )
            raise QgsProcessingException(err_msg)
        feedback.pushInfo(self.tr("Reception ID: {0}").format(reception_id))

        # Register changes
        feedback.pushInfo(self.tr("Registering changes"))
        req = QNetworkRequest(
            QUrl(f"{update_url}/mottagning/{reception_id}/forandringar")
        )
        req.setRawHeader(b"Content-Type", b"application/json")
        payload = {
            "forandringar": [
                {"objektidentitet": feature[id_field], "typ": "radering"}
                for feature in features
            ]
        }
        err_msg = getErrorMsg(request.post(req, json.dumps(payload).encode()))
        reply = bytes(request.reply().content()).decode()
        if err_msg:
            feedback.pushWarning(err_msg)
            feedback.pushWarning(request.errorMessage())
            raise QgsProcessingException(reply)
        res = json.loads(reply)
        for object_dict in res["forandringar"]:
            if (status := object_dict["status"]["typ"]) != "mottagen":
                err_msg = self.tr(
                    "Status should be 'mottagen' but is: '{0}' for object {1}"
                ).format(status, object_dict["objektidentitet"])
                if (err_msg_status := "felmeddelande") in object_dict["status"]:
                    err_msg += "\n"
                    err_msg += self.tr("Error message is '{0}'").format(
                        object_dict["status"][err_msg_status]
                    )
                raise QgsProcessingException(err_msg)

        # Create output layer
        output_fields = source.fields()
        sink, output_layer_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            output_fields,
            source.wkbType(),
            source.sourceCrs(),
        )
        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        # Get save features to output
        for feature in features:
            output_feature = QgsFeature(output_fields)
            output_feature.setAttributes(feature.attributes())
            output_feature.setGeometry(feature.geometry())
            if not sink.addFeature(output_feature, QgsFeatureSink.FastInsert):
                raise QgsProcessingException(
                    self.tr("Unable to save the output feature")
                )

        return {self.OUTPUT: output_layer_id}


class NgpDeleteResourceAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm that takes a vector layer performs a delete network
    request against NGP using the id of the provided features.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    FEATURES = "FEATURES"  # pylint: disable=invalid-name
    ALLOW_MULTIPLE_FEATURES = "ALLOW_MULTIPLE_FEATURES"  # pylint: disable=invalid-name
    ID_FIELD = "ID_FIELD"  # pylint: disable=invalid-name

    def __init__(self):
        super().__init__()
        self._tr = None

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def createInstance(self):
        deleteResourceAlg = NgpDeleteResourceAlgorithm()
        deleteResourceAlg.setTranslationCallback(self._tr)
        return deleteResourceAlg

    def name(self):
        return "ngp_delete_resource"

    def displayName(self):
        return self.tr("NGP Delete Resource")

    def group(self):
        return self.tr("NGP Connect")

    def groupId(self):
        return "ngp_connect"

    def shortHelpString(self):
        return self.tr(
            "This algorithm sends resource id for deletion to the of upload API of Lantmäteriets' National Geodata Platform.\n"
            "The layer holding the resources must be selected as input layer.\n"
            "By default it is only allowed to to process single features. \n"
            "ID field is the resource id (uuid) used for the resource.\n"
            'Read more about it at <a href="https://www.lantmateriet.se/sv/nationella-geodataplattformen/">https://www.lantmateriet.se/sv/nationella-geodataplattformen/</a>\n'
        )

    def initAlgorithm(self, config=None):  # pylint: disable=unused-argument
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Features
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.FEATURES,
                self.tr("Features"),
                [QgsProcessing.TypeVectorAnyGeometry],
            )
        )

        # Allow multiple features
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.ALLOW_MULTIPLE_FEATURES,
                self.tr("Allow multiple features"),
                defaultValue=False,
            )
        )

        # ID field
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_FIELD,
                self.tr("ID field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName=self.FEATURES,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Stop if no feature
        source = self.parameterAsSource(parameters, self.FEATURES, context)
        if source.featureCount() == 0:
            feedback.pushWarning(self.tr("No feature to process"))
            return {}
        # Stop if more than one feature and its not allowed
        if source.featureCount() > 1 and not self.parameterAsBool(
            parameters, self.ALLOW_MULTIPLE_FEATURES, context
        ):
            feedback.pushWarning(
                self.tr(
                    "Multiple features to process not allowed. Change parameter value or select just one feature."
                )
            )
            return {}
        features = list(source.getFeatures())

        # Stop if an attribute is missing
        id_field = self.parameterAsFields(parameters, self.ID_FIELD, context)[0]
        available_fields = [f.name() for f in source.fields()]
        for attr_name in [id_field]:
            if attr_name not in available_fields:
                raise QgsProcessingException(
                    self.tr("The attribute '{0}' is not present in the layer").format(
                        attr_name
                    )
                )

        # Read plugin settings
        feedback.pushInfo(self.tr("Reading plugin parameters"))
        settings = PlgOptionsManager.get_plg_settings()
        if settings.endpoint_url == "":
            raise QgsProcessingException(
                self.tr("No endpoint URL in plugin parameters")
            )
        if settings.upload_auth_cfg_id == "":
            raise QgsProcessingException(self.tr("No authentication set for upload"))
        upload_url = f"{settings.endpoint_url}/uppladdning/v1"

        # Prepare request object that will be used in the whole process
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(settings.upload_auth_cfg_id)

        # Send delete request for the features
        feedback.pushInfo(self.tr("Deletes resources"))
        for feature in features:
            feedback.pushInfo(self.tr("Object {0}").format(feature[id_field]))
            req = QNetworkRequest(QUrl(f"{upload_url}/resurs/{feature[id_field]}"))
            err_msg = getErrorMsg(request.deleteResource(req))
            reply = bytes(request.reply().content()).decode()
            if err_msg:
                feedback.pushWarning(err_msg)
                feedback.pushWarning(request.errorMessage())
                raise QgsProcessingException(reply)

        return {}


class NgpDeleteResourceUuidAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm that an uuid value performs a delete network
    request against NGP using the uuid.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    UUID = "UUID"  # pylint: disable=invalid-name

    def __init__(self):
        super().__init__()
        self._tr = None

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def createInstance(self):
        deleteResourceUuidAlg = NgpDeleteResourceUuidAlgorithm()
        deleteResourceUuidAlg.setTranslationCallback(self._tr)
        return deleteResourceUuidAlg

    def name(self):
        return "ngp_delete_resource_uuid"

    def displayName(self):
        return self.tr("NGP Delete Resource UUID")

    def group(self):
        return self.tr("NGP Connect")

    def groupId(self):
        return "ngp_connect"

    def shortHelpString(self):
        return self.tr(
            "This algorithm sends resource uuid for deletion to the of upload API of Lantmäteriets' National Geodata Platform.\n"
            'Read more about it at <a href="https://www.lantmateriet.se/sv/nationella-geodataplattformen/">https://www.lantmateriet.se/sv/nationella-geodataplattformen/</a>\n'
        )

    def initAlgorithm(self, config=None):  # pylint: disable=unused-argument
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # UUID value
        self.addParameter(
            QgsProcessingParameterString(
                self.UUID, self.tr("UUID value for the resource")
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Stop if no uuid
        uuid_value = self.parameterAsString(parameters, self.UUID, context)
        if uuid_value == "":
            feedback.pushWarning(self.tr("No UUID to process"))
            return {}

        # Read plugin settings
        feedback.pushInfo(self.tr("Reading plugin parameters"))
        settings = PlgOptionsManager.get_plg_settings()
        if settings.endpoint_url == "":
            raise QgsProcessingException(
                self.tr("No endpoint URL in plugin parameters")
            )
        if settings.upload_auth_cfg_id == "":
            raise QgsProcessingException(self.tr("No authentication set for upload"))
        upload_url = f"{settings.endpoint_url}/uppladdning/v1"

        # Prepare request object that will be used in the whole process
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(settings.upload_auth_cfg_id)

        # Send delete request for the features
        feedback.pushInfo(self.tr("Deletes resource"))
        feedback.pushInfo(self.tr("Object {0}").format(uuid_value))
        req = QNetworkRequest(QUrl(f"{upload_url}/resurs/{uuid_value}"))
        err_msg = getErrorMsg(request.deleteResource(req))
        reply = bytes(request.reply().content()).decode()
        if err_msg:
            feedback.pushWarning(err_msg)
            feedback.pushWarning(request.errorMessage())
            raise QgsProcessingException(reply)

        return {}


class NgpDownloadAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm that takes a vector layer performs a download network
    request against NGP using the id of the provided features.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    FEATURES = "FEATURES"  # pylint: disable=invalid-name
    ID_FIELD = "ID_FIELD"  # pylint: disable=invalid-name
    OUTPUT = "OUTPUT"  # pylint: disable=invalid-name

    def __init__(self):
        super().__init__()
        self._tr = None

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def createInstance(self):
        downloadAlg = NgpDownloadAlgorithm()
        downloadAlg.setTranslationCallback(self._tr)
        return downloadAlg

    def name(self):
        return "ngp_download"

    def displayName(self):
        return self.tr("NGP Download")

    def group(self):
        return self.tr("NGP Connect")

    def groupId(self):
        return "ngp_connect"

    def shortHelpString(self):
        return self.tr(
            "This algorithm sends id (uuid) to the of download API of Lantmäteriets' National Geodata Platform.\n"
            "The layer holding the resources must be selected as input layer.\n"
            "ID field is the resource id (uuid) used for the resource.\n"
            "An output directory must be selected \n"
            'Read more about it at <a href="https://www.lantmateriet.se/sv/nationella-geodataplattformen/">https://www.lantmateriet.se/sv/nationella-geodataplattformen/</a>\n'
        )

    def initAlgorithm(self, config=None):  # pylint: disable=unused-argument
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Features
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.FEATURES,
                self.tr("Features"),
                [QgsProcessing.TypeVectorAnyGeometry],
            )
        )

        # ID field
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_FIELD,
                self.tr("ID field"),
                type=QgsProcessingParameterField.String,
                parentLayerParameterName=self.FEATURES,
            )
        )

        # Output directory
        self.addParameter(
            QgsProcessingParameterFile(
                self.OUTPUT,
                self.tr("Output directory"),
                behavior=QgsProcessingParameterFile.Folder,
                optional=True,
                defaultValue=None,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # Stop if no feature
        source = self.parameterAsSource(parameters, self.FEATURES, context)
        if source.featureCount() == 0:
            feedback.pushWarning(self.tr("No feature to process"))
            return {}
        features = list(source.getFeatures())

        # Stop if an attribute is missing
        id_field = self.parameterAsFields(parameters, self.ID_FIELD, context)[0]
        available_fields = [f.name() for f in source.fields()]
        for attr_name in [id_field]:
            if attr_name not in available_fields:
                raise QgsProcessingException(
                    self.tr("The attribute '{0}' is not present in the layer").format(
                        attr_name
                    )
                )
        
        #Get file directory, if empry, get tempdir
        folder = self.parameterAsFile(parameters, self.OUTPUT, context)
        if folder == "":
            folder = tempfile.gettempdir()
            
        # Read plugin settings
        feedback.pushInfo(self.tr("Reading plugin parameters"))
        settings = PlgOptionsManager.get_plg_settings()
        if settings.endpoint_url == "":
            raise QgsProcessingException(
                self.tr("No endpoint URL in plugin parameters")
            )
        if settings.download_auth_cfg_id == "":
            raise QgsProcessingException(self.tr("No authentication set for download"))
        download_url = f"{settings.endpoint_url}/nedladdning/v1"

        # Prepare request object that will be used in the whole process
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(settings.download_auth_cfg_id)

        # Send get request for the features
        feedback.pushInfo(self.tr("Downloading data"))
        for feature in features:
            req = QNetworkRequest(QUrl(f"{download_url}/asset/{feature[id_field]}"))
            err_msg = getErrorMsg(request.get(req))
            reply = request.reply()
            contentdisposition = bytes(reply.rawHeader(b"content-disposition")).decode()
            content = reply.content()
            filename = reply.extractFileNameFromContentDispositionHeader(
                contentdisposition
            )
            filepath = f"{folder}\\{filename}"
            feedback.pushInfo(self.tr("File downloaded to {0}").format(filepath))

            # Open a new file for writing
            try:
                with open(filepath, mode="wb") as resp_file:
                    resp_file.write(content)  # Write the reply content to the file
            except OSError:
                feedback.pushInfo(
                    self.tr("File already open")
                )  # File already exists, is open and can't be written

            if feedback.isCanceled():
                return {}
        return {}


class NgpDownloadUuidAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm that takes a uuid sting an dperforma a downlaod network
    request against NGP.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    UUID = "UUID"  # pylint: disable=invalid-name

    def __init__(self):
        super().__init__()
        self._tr = None

    def tr(self, message: str) -> str:
        """Translate if translation callback is set"""
        if self._tr is not None:
            return self._tr(message)
        return message

    def setTranslationCallback(self, trCallback):
        """Set the translation callback to use the translation manager"""
        self._tr = trCallback

    def createInstance(self):
        downloadUuidAlg = NgpDownloadUuidAlgorithm()
        downloadUuidAlg.setTranslationCallback(self._tr)
        return downloadUuidAlg

    def name(self):
        return "ngp_download_uuid"

    def displayName(self):
        return self.tr("NGP Download UUID")

    def group(self):
        return self.tr("NGP Connect")

    def groupId(self):
        return "ngp_connect"

    def shortHelpString(self):
        return self.tr(
            "This algorithm sends resource uuid to the download API of Lantmäteriets' National Geodata Platform.\n"
            "After download the file is opened. This requires QGIS 3.40 or later. \n"
            'Read more about it at <a href="https://www.lantmateriet.se/sv/nationella-geodataplattformen/">https://www.lantmateriet.se/sv/nationella-geodataplattformen/</a>\n'
        )

    def initAlgorithm(self, config=None):  # pylint: disable=unused-argument
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # UUID value
        self.addParameter(
            QgsProcessingParameterString(
                self.UUID, self.tr("UUID value for the resource")
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Stop if no uuid
        uuid_value = self.parameterAsString(parameters, self.UUID, context)
        if uuid_value == "":
            feedback.pushWarning(self.tr("No UUID to process"))
            return {}

        # Read plugin settings
        settings = PlgOptionsManager.get_plg_settings()
        if settings.endpoint_url == "":
            raise QgsProcessingException(
                self.tr("No endpoint URL in plugin parameters")
            )
        if settings.download_auth_cfg_id == "":
            raise QgsProcessingException(self.tr("No authentication set for download"))
        download_url = f"{settings.endpoint_url}/nedladdning/v1"

        # Prepare request object that will be used in the whole process
        feedback.pushInfo(self.tr("Reading plugin parameters"))
        request = QgsBlockingNetworkRequest()
        request.setAuthCfg(settings.download_auth_cfg_id)

        # Send request for the download
        feedback.pushInfo(self.tr("Downloading resource"))
        req = QNetworkRequest(QUrl(f"{download_url}/asset/{uuid_value}"))
        err_msg = getErrorMsg(request.get(req))
        reply = request.reply()
        contentdisposition = bytes(reply.rawHeader(b"content-disposition")).decode()
        content = reply.content()
        filename = reply.extractFileNameFromContentDispositionHeader(contentdisposition)
        filepath = f"{tempfile.gettempdir()}\\{filename}"

        # Open a new file for writing
        try:
            with open(filepath, mode="wb") as resp_file:
                resp_file.write(content)  # Write the reply content to the file
        except OSError:
            feedback.pushInfo(
                self.tr("File already open")
            )  # File already exists, is open and can't be written

        # Open file with native algorithm QGIS >= 3.40
        alg_params = {"URL": filepath}
        processing.run(
            "native:openurl",
            alg_params,
            context=context,
            feedback=feedback,
            is_child_algorithm=True,
        )

        contentbytes = bytes(content)
        if err_msg:
            feedback.pushWarning(err_msg)
            feedback.pushWarning(request.errorMessage())
            raise QgsProcessingException(contentbytes)

        return {}


class NgpUpdateOutputLayerStylerInterface(QgsProcessingLayerPostProcessorInterface):
    def __init__(self):
        super().__init__()
        self.json_field_name = None

    def postProcessLayer(self, layer, context, feedback):
        if (
            not layer
            or not layer.isValid()
            or not isinstance(layer, QgsVectorLayer)
            or self.json_field_name is None
        ):
            return

        layer.setEditorWidgetSetup(
            layer.fields().indexFromName(self.json_field_name),
            QgsEditorWidgetSetup("JsonEdit", {"DefaultView": 1}),
        )


def getErrorMsg(error_code):
    """
    Translates QgsBlockingNetworkRequest error code into a string
    """

    if error_code == QgsBlockingNetworkRequest.ErrorCode.NoError:
        return ""
    if error_code == QgsBlockingNetworkRequest.ErrorCode.TimeoutError:
        return "Timeout Error"
    if error_code == QgsBlockingNetworkRequest.ErrorCode.ServerExceptionError:
        return "Server Error"
    if error_code == QgsBlockingNetworkRequest.ErrorCode.NetworkError:
        return "Network Error"
    return "Unknown Error"
