# -*- coding: utf-8 -*-
"""
Local persistence for NetworkStore connection presets.
Non-sensitive data are stored in JSON; credentials are stored in QGIS auth DB.
"""

from __future__ import annotations

import json
import os
from dataclasses import dataclass, asdict
from typing import Optional, List

from qgis.core import QgsApplication, QgsAuthMethodConfig


@dataclass
class NetworkStoreConfig:
    name: str
    base_url: str
    authcfg: Optional[str] = None

    @staticmethod
    def from_dict(data: dict) -> "NetworkStoreConfig":
        return NetworkStoreConfig(
            name=str(data.get("name") or "").strip(),
            base_url=str(data.get("base_url") or "").strip(),
            authcfg=str(data.get("authcfg")).strip() or None,
        )


class NetworkStoreSettingsStore:
    def __init__(self) -> None:
        self._path = os.path.join(
            QgsApplication.qgisSettingsDirPath(),
            "network-store-qgis",
            "networkstore_settings.json",
        )
        os.makedirs(os.path.dirname(self._path), exist_ok=True)

    def _upsert_authcfg(
        self, username: str, password: str, existing_authcfg: Optional[str]
    ) -> str:
        authm = QgsApplication.authManager()
        cfg = QgsAuthMethodConfig()
        cfg.setName(existing_authcfg or f"networkstore:{username}")
        cfg.setMethod("Basic")
        cfg.setConfig("username", username)
        cfg.setConfig("password", password)
        authm.storeAuthenticationConfig(cfg)
        return cfg.id()

    def _delete_authcfg(self, authcfg: Optional[str]) -> None:
        if not authcfg:
            return
        try:
            QgsApplication.authManager().removeAuthenticationConfig(authcfg)
        except Exception:
            pass

    def _read_all(self) -> List[NetworkStoreConfig]:
        try:
            with open(self._path, "r", encoding="utf-8") as f:
                data = json.load(f)
        except Exception:
            return []
        if not isinstance(data, list):
            return []
        configs: List[NetworkStoreConfig] = []
        for item in data:
            try:
                cfg = NetworkStoreConfig.from_dict(item)
                if cfg.name:
                    configs.append(cfg)
            except Exception:
                continue
        return configs

    def _write_all(self, configs: List[NetworkStoreConfig]) -> None:
        serialized = [asdict(c) for c in configs]
        with open(self._path, "w", encoding="utf-8") as f:
            json.dump(serialized, f, indent=2, ensure_ascii=False)

    def list(self) -> List[NetworkStoreConfig]:
        return self._read_all()

    def get(self, name: str) -> Optional[NetworkStoreConfig]:
        name = name.strip()
        for cfg in self._read_all():
            if cfg.name == name:
                return cfg
        return None

    def add_or_update(
        self,
        name: str,
        base_url: str,
        username: Optional[str],
        password: Optional[str],
    ) -> NetworkStoreConfig:
        name = name.strip()
        base_url = base_url.strip()
        if not name or not base_url:
            raise ValueError("Name and base_url are required.")

        configs = self._read_all()
        existing = None
        for cfg in configs:
            if cfg.name == name:
                existing = cfg
                break

        authcfg = existing.authcfg if existing else None
        if username and password:
            authcfg = self._upsert_authcfg(username, password, authcfg)

        new_cfg = NetworkStoreConfig(name=name, base_url=base_url, authcfg=authcfg)

        if existing:
            configs = [new_cfg if c.name == name else c for c in configs]
        else:
            configs.append(new_cfg)
        self._write_all(configs)
        return new_cfg

    def delete(self, name: str) -> bool:
        name = name.strip()
        configs = self._read_all()
        removed = False
        keep: List[NetworkStoreConfig] = []
        for cfg in configs:
            if cfg.name == name:
                removed = True
                self._delete_authcfg(cfg.authcfg)
            else:
                keep.append(cfg)
        if removed:
            self._write_all(keep)
        return removed
