# -*- coding: utf-8 -*-
"""
/***************************************************************************
 NetworkStorePlugin
                                 A QGIS plugin
 export layers to kisters network store
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-01-18
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Attila Bibok
        email                : Attila.bibok@kisters.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
__author__ = "Attila Bibok / KISTERS North America"
__date__ = "2022-05-28"
__copyright__ = "(C) 2022 by Attila Bibok / KISTERS North America"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

import os
import sys
import inspect
from os.path import exists
from pathlib import Path

from qgis.core import (
    QgsApplication,
    QgsProject,
    QgsLayerTreeLayer,
    QgsVectorLayer,
    QgsLayerTreeGroup
)
from .kisters_processing_provider import kisters_processingProvider
from .utils import (
    load_icons,
    clean_icons,
    apply_style_from_qml,
    saving_gpkg,
    add_gpkg_layer,
    add_feature,
)
from .inspect_model_library import get_class_schema

cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
pluginPath = os.path.dirname(__file__)

if cmd_folder not in sys.path:
    sys.path.insert(0, cmd_folder)

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .network_store_dialog import NetworkStorePluginDialog
from .network_store_dialog_export import NetworkStorePluginDialogExport
from .network_store_dialog_post import NetworkStorePluginDialogPost
from .network_store_dialog_get import NetworkStorePluginDialogGet
import os.path

from .utils import new_layer, save_with_description, add_fields_from_schema


class NetworkStorePlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # copy icons to their location

        # copy the .SVGs to profiles/$profile/svg
        load_icons()
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "NetworkStorePlugin_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Network Store Plugin")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # processing provider
        self.provider = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("NetworkStorePlugin", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(pluginPath, "icon.png")
        icon_store_up = os.path.join(pluginPath, "icon_store_up.png")
        icon_store_down = os.path.join(pluginPath, "icon_store_down.png")
        icon_gpkg = os.path.join(pluginPath, "icon_store_file.png")
        self.add_action(
            icon_path,
            text=self.tr("Create blank network_store layers"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        self.add_action(
            icon_gpkg,
            text=self.tr("Export layers to JSON"),
            callback=self.run_export,
            # add_to_toolbar = False,
            parent=self.iface.mainWindow(),
        )
        self.add_action(
            icon_store_up,
            text=self.tr("Post layers to network store"),
            callback=self.run_post,
            # add_to_toolbar = False,
            parent=self.iface.mainWindow(),
        )
        self.add_action(
            icon_store_down,
            text=self.tr("Get layers from network store"),
            callback=self.run_get,
            # add_to_toolbar = False,
            parent=self.iface.mainWindow(),
        )


        # will be set False in run()
        self.first_start = True

        # Processing Provider
        self.initProcessing()

    def initProcessing(self):
        """Init Processing provider for QGIS >= 3.8."""
        self.provider = kisters_processingProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Network Store Plugin"), action)
            self.iface.removeToolBarIcon(action)

        # removed copied svgs
        clean_icons()

    def get_layer_in_group(
        self, project_name, group_name, layer_name
    ) -> QgsVectorLayer:
        root = QgsProject.instance().layerTreeRoot()
        print(root.children())
        for child in root.children():
            if isinstance(child, QgsLayerTreeGroup):
                if child.name() == project_name:
                    print(child.children())
                    for gchild in child.children():
                        if isinstance(gchild, QgsLayerTreeGroup):
                            if gchild.name() == group_name:
                                print(gchild.children())
                                for ggchild in gchild.children():
                                    if ggchild.name() == layer_name:
                                        return ggchild.layer()
        raise ValueError(
            f"Layer ref not found in {project_name}/{group_name}/{layer_name}"
        )

    def run(self):
        """Generate a new blank model from the base dialog"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = NetworkStorePluginDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # validate checkboxes
            node_class_list = []
            link_class_list = []

            self._checked_items: dict = self.dlg.get_checked_items()

            if "Nodes" in self._checked_items:
                node_class_list = self._checked_items["Nodes"]

            if "Links" in self._checked_items:
                link_class_list = self._checked_items["Links"]

            _apply_default_style = self.dlg.checkBox_ApplyDefaultStyle.isChecked()

            gpkg_path = self.dlg.mQgsFileWidget_geopackage.filePath()
            base_path = Path(__file__).parent

            # create project group
            root = QgsProject.instance().layerTreeRoot()
            group_project = root.addGroup(Path(gpkg_path).stem)
            subgroup_links = group_project.addGroup("Links")
            subgroup_nodes = group_project.addGroup("Nodes")

            for nodeClass in node_class_list:
                # create new layer in memory, do not add to layer tree
                _newlayer = QgsProject.instance().addMapLayer(
                    QgsVectorLayer("Point", nodeClass, "memory"), False
                )
                # add fields from schema
                schema = get_class_schema("nodes", nodeClass)
                add_fields_from_schema(_newlayer, schema)

                if _apply_default_style:
                    _style_path = (base_path / f"styles/{nodeClass}.qml").resolve()
                    if exists(_style_path):
                        _ = _newlayer.loadNamedStyle(str(_style_path))
                    else:
                        print(f"Style .QML: {str(_style_path)} not found")
                saving_gpkg(_newlayer, gpkg_path)
                savedlayer = add_gpkg_layer(self.iface, gpkg_path, nodeClass)
                _newlayer = subgroup_nodes.insertChildNode(
                    1, QgsLayerTreeLayer(savedlayer)
                )  # move it to the subgroup

            for linkClass in link_class_list:
                _newlayer = QgsProject.instance().addMapLayer(
                    QgsVectorLayer("Linestring", linkClass, "memory"), False
                )
                schema = get_class_schema("links", linkClass)
                add_fields_from_schema(_newlayer, schema)
                if _apply_default_style:
                    _style_path = (base_path / f"styles/{linkClass}.qml").resolve()
                    if exists(_style_path):
                        _ = _newlayer.loadNamedStyle(str(_style_path))
                    else:
                        print(f"Style .QML: {str(_style_path)} not found")
                saving_gpkg(_newlayer, gpkg_path)
                savedlayer = add_gpkg_layer(self.iface, gpkg_path, linkClass)
                _newlayer = subgroup_links.insertChildNode(
                    1, QgsLayerTreeLayer(savedlayer)
                )  # move it to the subgroup

    def run_export(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = NetworkStorePluginDialogExport()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def run_post(self):
        """Run method that performs all the real work"""

        def _walk_children(group):
            layers = []
            for child in group.children():
                if isinstance(child, QgsLayerTreeLayer):
                    layers.append(child)
                if isinstance(child, QgsLayerTreeGroup):
                    layers.extend(_walk_children(child))
            return layers

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = NetworkStorePluginDialogPost()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # get the selected groups layers
            selected = self.dlg.group_comboBox.currentText()

            root = QgsProject.instance().layerTreeRoot()

            group = root.findGroup(selected)
            layers = _walk_children(group=group)

            links, nodes = self.dlg.parse_layers(layers=layers)

            self.dlg.post_network(nodes=nodes, links=links)

    def run_get(self):
        """Run method to get a network from the network store"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            # self.first_start = False
            self.dlg = NetworkStorePluginDialogGet()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            nodes, links = self.dlg.get_network()

            # create new memory layers and groups
            # TODO: create a method to get all the element classes from _BaseLink and _BaseNode
            node_class_list = [
                "Junction",
                "FlowBoundary",
                "LevelBoundary",
                "Storage",
            ]
            link_class_list = [
                "Channel",
                "Delay",
                "FlowControlledStructure",
                "Pipe",
                "Weir",
                "Drain",
                "Pump",
                "Turbine",
                "TopDownRectangularOrifice",
                "TopDownSphericalOrifice",
            ]

            node_layer_ref = {}
            link_layer_ref = {}

            _apply_default_style = True

            base_path = Path(__file__).parent

            # create project group
            root = QgsProject.instance().layerTreeRoot()
            group_project = root.addGroup(self.dlg.mLineEdit_networkID.value())
            subgroup_nodes = group_project.addGroup("Nodes")
            subgroup_links = group_project.addGroup("Links")

            for linkClass in link_class_list:
                _newlayer = QgsVectorLayer("Linestring", linkClass, "memory")
                _newMapLayer = QgsProject.instance().addMapLayer(_newlayer, False)
                schema = get_class_schema("links", linkClass)
                add_fields_from_schema(_newMapLayer, schema)
                if _apply_default_style:
                    _style_path = (base_path / f"styles/{linkClass}.qml").resolve()
                    if exists(_style_path):
                        _ = _newMapLayer.loadNamedStyle(str(_style_path))
                    else:
                        print(f"Style .QML: {str(_style_path)} not found")
                subgroup_links.insertChildNode(-1,QgsLayerTreeLayer(_newlayer))
                link_layer_ref[linkClass] = _newlayer

            for nodeClass in node_class_list:
                # create new layer in memory, do not add to layer tree
                _newlayer = QgsVectorLayer("Point", nodeClass, "memory")
                _newMapLayer = QgsProject.instance().addMapLayer(_newlayer, False)

                # add fields from schema
                schema = get_class_schema("nodes", nodeClass)
                add_fields_from_schema(_newMapLayer, schema)

                if _apply_default_style:
                    _style_path = (base_path / f"styles/{nodeClass}.qml").resolve()
                    if exists(_style_path):
                        # _ = _newlayer.loadNamedStyle(str(_style_path))
                        _ = _newMapLayer.loadNamedStyle(str(_style_path))
                    else:
                        print(f"Style .QML: {str(_style_path)} not found")
                subgroup_nodes.insertChildNode(-1,QgsLayerTreeLayer(_newlayer))
                node_layer_ref[nodeClass] = _newlayer

            # populate the layers with features
            for element in nodes:
                # find corresponding layer
                try:
                    add_feature(node_layer_ref[element.element_class], element)
                except KeyError:
                    print(
                        f"Warning: element class {element.element_class} is not in the supported layers!"
                        f" Supported layers are: {node_layer_ref.keys()}"
                    )

            for element in links:
                # find corresponding layer
                try:
                    add_feature(link_layer_ref[element.element_class], element)
                except KeyError:
                    print(
                        f"Warning: element class {element.element_class} is not in the supported layers!"
                        f" Supported layers are: {link_layer_ref.keys()}"
                    )
            # refresh tree and canvas after adding features
            QgsProject.instance().reloadAllLayers()
            self.iface.mapCanvas().refresh()
