from datetime import datetime
from qgis.core import (
    QgsVectorFileWriter,
    QgsVectorLayer,
    QgsVectorLayerUtils,
    QgsWkbTypes,
    QgsProject,
    QgsField,
    QgsFieldConstraints,
    QgsReadWriteContext,
    QgsProject,
    QgsVectorFileWriter,
    QgsCoordinateTransform,
    QgsPointXY,
    QgsField,
    QgsVectorLayer,
    QgsCoordinateReferenceSystem,
    QgsFeature,
    QgsPoint,
    QgsLineString,
    QgsGeometry,
)
from qgis.PyQt.QtCore import QVariant
from .type_mapping import type_pyd_to_Qvar, field_exceptions
from typing import Dict, List
from copy import copy
import shutil
import os
import json
import ast

import errno
from pathlib import Path
from osgeo import ogr
from qgis.PyQt import uic, QtWidgets
from qgis.PyQt.QtXml import QDomDocument

from kisters.network_store.model_library.util import elements_mapping

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer

def load_icons():
    src_dir = os.path.join(os.path.dirname(__file__), "icons/")
    target_dir = os.path.join(os.path.dirname(__file__), "../../../svg/icons/")
    try:
        if os.path.exists(target_dir):
            shutil.rmtree(target_dir)
        shutil.copytree(src_dir, target_dir)
    except OSError as e:
        if e.errno == errno.ENOTDIR:
            shutil.copy(src_dir, target_dir)
        else:
            raise OSError(e)


def clean_icons():
    target_dir = os.path.join(os.path.dirname(__file__), "../../../svg/icons/")
    try:
        if os.path.exists(target_dir):
            shutil.rmtree(target_dir)
    except OSError as e:
        raise OSError(e)


def new_layer(
    geom_type="Point", layer_name: str = "unnamed_layer", provider: str = "ogr"
):
    layer = QgsVectorLayer(geom_type, layer_name, provider)
    print(layer.isValid())
    return layer


def add_gpkg_layer(
    iface, gpkg, layer
):  # adds EXISTING layer to the project from a gpkg
    layers = [l.GetName() for l in ogr.Open(gpkg)]
    if layer in layers:
        return QgsProject.instance().addMapLayer(
            QgsVectorLayer(gpkg + "|layername=" + layer, layer, "ogr"), False
        )
        # return iface.addVectorLayer(gpkg + "|layername=" + layer, layer, 'ogr')
    else:
        print('Error: there is no layer named "{}" in {}!'.format(layer, gpkg))


def add_fields_from_schema(layer: QgsVectorLayer, schema):
    _schema = copy(schema)

    fields = list(_schema["properties"].keys())
    not_null_fields = _schema["required"]
    optional_fields = list(set(fields) - set(not_null_fields))

    for f in not_null_fields:
        p = _schema["properties"][f]
        if "type" in p:
            if p["type"] in type_pyd_to_Qvar:
                layer.dataProvider().addAttributes(
                    [
                        QgsField(
                            name=f,
                            type=type_pyd_to_Qvar[p["type"]],
                            comment=p["description"] if "description" in p else "",
                        )
                    ]
                )
            # elif p['type'] == 'object':
            #     #TODO import the object too with a p['title']. prefix
            #     continue
            else:
                raise TypeError(
                    f"Unsupported type: {p['type']} see type_mapping.py for reference"
                )
        if (
            "allOf" in p
        ):  # Union of values. Types are static here, use it as a string and parse it later
            layer.dataProvider().addAttributes(
                [
                    QgsField(
                        name=f,
                        type=QVariant.String,
                        comment=p["description"] if "description" in p else ""
                    )
                ]
            )
        layer.updateFields()

    for f in optional_fields:
        p = _schema["properties"][f]
        if "type" in p:
            if p["type"] in type_pyd_to_Qvar:
                layer.dataProvider().addAttributes(
                    [
                        QgsField(
                            name=f,
                            type=type_pyd_to_Qvar[p["type"]],
                            comment=p["description"] if "description" in p else "",
                        )
                    ]
                )
            elif p["type"] == "object":
                layer.dataProvider().addAttributes(
                    [
                        QgsField(
                            name=f,
                            type=type_pyd_to_Qvar[p["type"]],
                            comment=p["description"] if "description" in p else "",
                        )
                    ]
                )
                continue
            else:
                raise TypeError(
                    f"Unsupported type: {p['type']} see type_mapping.py for reference"
                )
        if (
            "allOf" in p
        ):  # Union of values. Types are static here, use it as a string and parse it later
            layer.dataProvider().addAttributes(
                [
                    QgsField(
                        name=f,
                        type=QVariant.String,
                        comment=p["description"] if "description" in p else "",
                    )
                ]
            )
        layer.updateFields()


def generate_attribute_form(layer: QgsVectorLayer, schema):
    pass


def apply_style_from_qml(layer: QgsVectorLayer, style_path: str):
    if Path(style_path).suffix == ".qml":
        layer.loadNamedStyle(style_path)
    else:
        raise OSError(f"Invalid extension: {Path(style_path).suffix} it must be .QML")


def saving_gpkg(styled_layer, out_path):
    context = QgsProject.instance().transformContext()
    name = styled_layer.name()
    options = QgsVectorFileWriter.SaveVectorOptions()
    options.layerName = name
    if os.path.exists(out_path):
        options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer
    else:
        options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
    options.fileEncoding = styled_layer.dataProvider().encoding()
    options.driverName = "GPKG"
    QgsVectorFileWriter.writeAsVectorFormatV2(styled_layer, out_path, context, options)
    doc = QDomDocument()
    readWriteContext = context = QgsReadWriteContext()
    styled_layer.exportNamedStyle(doc)
    gpkg_layer = QgsVectorLayer(f"{out_path}|layername={name}", name, "ogr")
    gpkg_layer.importNamedStyle(doc)
    gpkg_layer.saveStyleToDatabase(name, "", True, "")


def save_with_description(layer, outputfile, type=QgsWkbTypes.PointZ):
    options = QgsVectorFileWriter.SaveVectorOptions()
    # options.overrideGeometryType = type

    # If geopackage exists, append layer to it, else create it
    if os.path.exists(outputfile):
        options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer
    else:
        options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
    # Use input layer abstract and name in the geopackage
    options.layerOptions = [f"DESCRIPTION={layer.abstract()}"]
    options.layerName = layer.name()

    error, error_string = QgsVectorFileWriter.writeAsVectorFormat(
        layer, outputfile, options
    )
    if error != QgsVectorFileWriter.NoError:
        print("Oh an error has happened: {details}".format(details=error_string))

    return (error, error_string)


from kisters.network_store.model_library.base import Location, BaseLink, BaseNode
from typing import List


def location_to_point(location: Location) -> QgsPoint:
    return QgsPoint(location.x, location.y, location.z)


def location_to_pointXY(location: Location) -> QgsPointXY:
    return QgsPointXY(location.x, location.y)


def polyline_from_vertices(vertices: List[Location]) -> List[QgsPoint]:
    points = []
    for loc in vertices:
        points.append(location_to_point(loc))
    return points


def add_point_feature(
    layer: QgsVectorLayer, element, crs_tr: QgsCoordinateTransform = None
):
    if issubclass(type(element), BaseNode):
        fields = layer.fields()
        feature = QgsFeature(fields)
        for field in fields:
            ftype = field.type()
            var = getattr(element, field.name())

            if field.name() not in field_exceptions:
                type_name = type(var).__name__
                if type_name == "NoneType":
                    continue
                try:
                    if type_name in type_pyd_to_Qvar:
                        if type_pyd_to_Qvar[type_name] == ftype:
                            feature.setAttribute(field.name(), var)
                        else:
                            print(
                                f"Mismatched types: element - {type(var).__name__}, "
                                f"{ftype}"
                            )
                    else:
                        if type_name == "list":
                            feature.setAttribute(
                                field.name(), json.dumps([dict(v) for v in var])
                            )
                        elif type_name == "datetime":
                            feature.setAttribute(
                                field.name(), var.strftime("%Y-%m-%dT%H:%M:%SZ")
                            )
                        else:
                            try:
                                feature.setAttribute(field.name(), json.dumps(var.dict(exclude_none=True)))
                            except Exception:
                                feature.setAttribute(field.name(), var.value) # it's a workaround, going to be solved after 3.11 using StrEnum
                except Exception as e:
                    print(f"Error: {e}")
        # inform the feature of its fields
        # feature.setFields(fields)
        layerPoint = location_to_pointXY(element.location)
        geom = QgsGeometry.fromPointXY(layerPoint)
        if crs_tr:
            geom.transform(crs_tr)
        feature.setGeometry(geom)
        return feature
    else:
        raise TypeError(
            f"Wrong element added to a node layer. " "Only BaseNode is accepted!"
        )


def add_link_feature(
    layer: QgsVectorLayer, element, crs_tr: QgsCoordinateTransform = None
):
    if issubclass(type(element), BaseLink):
        fields = layer.fields()
        feature = QgsFeature(fields)
        for field in fields:
            ftype = field.type()
            var = getattr(element, field.name())

            _field_name = field.name()
            if _field_name not in field_exceptions:
                type_name = type(var).__name__
                if type_name == "NoneType":
                    continue
                try:
                    if type_name in type_pyd_to_Qvar:
                        if type_pyd_to_Qvar[type_name] == ftype:
                            feature.setAttribute(field.name(), var)
                        else:
                            print(
                                f"Mismatched types: element - {type(var).__name__}, "
                                f"{ftype}"
                            )
                    else:
                        if type_name == "list":
                            feature.setAttribute(
                                field.name(), json.dumps([dict(v) for v in var])
                            )
                        elif type_name == "datetime":
                            feature.setAttribute(
                                field.name(), var.strftime("%Y-%m-%dT%H:%M:%SZ")
                            )
                        else:
                            try:
                                feature.setAttribute(field.name(), json.dumps(var.dict(exclude_none=True)))
                            except Exception:
                                feature.setAttribute(field.name(), var.value) # it's a workaround, going to be solved after 3.11 using StrEnum
                except Exception as e:
                    print(f"Error: {e}")

        # inform the feature of its fields
        # feature.setFields(fields)
        layerPoint = polyline_from_vertices(element.vertices)
        geom = QgsGeometry.fromPolyline(layerPoint)
        if crs_tr:
            geom.transform(crs_tr)
        feature.setGeometry(geom)
        return feature
    else:
        raise TypeError(
            f"Wrong element added to a link layer. " "Only BaseLink is accepted!"
        )


def add_feature(layer: QgsVectorLayer, element):
    sourceCrs = QgsCoordinateReferenceSystem(3857)
    destCrs = QgsCoordinateReferenceSystem(4326)
    tr = QgsCoordinateTransform(sourceCrs, destCrs, QgsProject.instance())
    geomtype = layer.geometryType()
    if geomtype == QgsWkbTypes.PointGeometry:
        feat = add_point_feature(layer, element, tr)
        (res, outFeats) = layer.dataProvider().addFeatures([feat])
    elif layer.geometryType() == QgsWkbTypes.LineGeometry:
        feat = add_link_feature(layer, element, tr)
        (res, outFeats) = layer.dataProvider().addFeatures([feat])


def append_geometry(
    element: Dict,
    feature: QgsFeature,
    geomtype: QgsWkbTypes,
    tr: QgsCoordinateTransform,
) -> Dict:
    # add geometry
    geom = feature.geometry()
    if geomtype == QgsWkbTypes.PointGeometry:
        print("")
        p = tr.transform(geom.asPoint())
        if isinstance(p, QgsPointXY):
            element["location"] = {"x": p.x(), "y": p.y()}
        elif isinstance(p, QgsPoint):
            element["location"] = {"x": p.x(), "y": p.y(), "z": p.z()}
        else:
            raise TypeError(f"geometry class {type(p).__name__} not supported, only QgsPoint and QgsPointXY")
        # Point feature -> location:Location
    elif geomtype == QgsWkbTypes.LineGeometry:
        # Line feature -> vertices:[Location]\
        vertices = []
        polyline = geom.asPolyline()
        for p in polyline:
            p = tr.transform(p)
            if isinstance(p, QgsPointXY):
                vertices.append({"x": p.x(), "y": p.y()})
            elif isinstance(p, QgsPoint):
                vertices.append({"x": p.x(), "y": p.y(), "z": p.z()})
            else:
                raise TypeError(f"geometry class {type(p).__name__} not supported, only QgsPoint and QgsPointXY")
        element["vertices"] = vertices
    else:
        raise TypeError(
            f"Type: {geomtype} is not supported, only Point and Line geometries"
        )


def extract_features(layer: QgsVectorLayer) -> List:
    sourceCrs = layer.crs()
    destCrs = QgsCoordinateReferenceSystem(3857)
    # reproject to webmercator all the time
    tr = QgsCoordinateTransform(sourceCrs, destCrs, QgsProject.instance())

    parsed_element_list = []
    geomtype = layer.geometryType()
    for feature in layer.getFeatures():
        element = {}
        for field in feature.fields():
            if not feature[field.name()] is None:
                if isinstance(feature[field.name()], str):
                    try:
                        element[field.name()] = ast.literal_eval(feature[field.name()])
                    except Exception:
                        element[field.name()] = feature[field.name()]
                else:
                    element[field.name()] = feature[field.name()]

        append_geometry(element, feature, geomtype, tr)

        element_class = elements_mapping[element["domain"]][element["collection"]][element["element_class"]]
        obj = element_class.parse_obj(element)
        parsed_element_list.append(obj)
    return parsed_element_list
