# -*- coding: utf-8 -*-
"""
/***************************************************************************
 NetClus
                                 A QGIS plugin
 This plugin make cluster nalysis base on network layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-04-14
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Surya Hafizh
        email                : suryahafizh979@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtWidgets import*
from qgis.core import*



# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .Net_clus_dialog import NetClusDialog
import os.path

import networkx as nx
import numpy as np
import pandas as pd
from shapely.geometry import Point
from sklearn.cluster import KMeans, AgglomerativeClustering
import math
from scipy.spatial import cKDTree
from tqdm import tqdm



class NetClus:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'NetClus_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        #Cancel button
        self.is_canceled = False
  

        # Declare instance attributes
        self.actions = []
        self.layer_names = []
        self.all_layers = []
        self.menu = self.tr(u'&NetClus')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('NetClus', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(self.plugin_dir, "icon.png")
        self.add_action(
            icon_path,
            text=self.tr(u'NetClus'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&NetClus'),
                action)
            self.iface.removeToolBarIcon(action)

    
    def run(self):
        """Run method that performs all the real work"""                                                                                        
        

        # Buat dialog hanya sekali
        if self.first_start == True:
            self.first_start = False
            self.dlg = NetClusDialog()
            self.dlg.pushButton.clicked.connect(self.run_clustering)  # Connect sekali saja
            self.dlg.button_box.rejected.connect(self.cancel_process)


        self.dlg.comboBox.clear()
        self.dlg.comboBox_2.clear()    
        self.layer_names.clear()

        for layer in QgsProject.instance().mapLayers().values():
            self.layer_names.append(layer.name())
            self.all_layers.append(layer)

        self.dlg.comboBox.addItems(self.layer_names)
        self.dlg.comboBox_2.addItems(self.layer_names)
        self.dlg.progressBar.setValue(0)
        
        self.dlg.show()

    def cancel_process(self):
        """Fungsi untuk membatalkan proses clustering"""
        self.is_canceled = True
        self.dlg.progressBar.setValue(0)  # 🔥 Langsung reset progress bar ke 0
        self.iface.messageBar().pushMessage("NetClus", "❌ Proses clustering dibatalkan oleh pengguna.", level=Qgis.Warning)


    def run_clustering(self):
        
        self.is_canceled = False  # Reset dulu
        
        point_layer_name = self.dlg.comboBox.currentText()
        line_layer_name = self.dlg.comboBox_2.currentText()
        show_result = self.dlg.checkBox.isChecked()
        n_clusters = self.dlg.mQgsSpinBox.value()
        output_path = self.dlg.mQgsFileWidget.filePath()
        use_max_distance = self.dlg.checkMaxDistance.isChecked()
        max_distance = self.dlg.spinMaxDistance.value()


        point_layer = QgsProject.instance().mapLayersByName(point_layer_name)[0]
        line_layer = QgsProject.instance().mapLayersByName(line_layer_name)[0]

        if not point_layer.isValid() or not line_layer.isValid():
            self.iface.messageBar().pushWarning("NetClus", "Layer tidak valid.")
            return

   
        self.dlg.progressBar.setValue(10)

        # === Bangun Graph Jalan ===
        G = nx.Graph()
        for feat in line_layer.getFeatures():
            if self.is_canceled:
                return
            geom = feat.geometry()
            lines = geom.asMultiPolyline() if geom.isMultipart() else [geom.asPolyline()]
            for line in lines:
                for i in range(len(line) - 1):
                    u, v = tuple(line[i]), tuple(line[i + 1])
                    G.add_edge(u, v, weight=Point(u).distance(Point(v)))

        nodes = list(G.nodes)
        node_coords = np.array([list(n) for n in nodes])
        tree = cKDTree(node_coords)

        self.dlg.progressBar.setValue(20)

        # === Ambil Titik Dummy ===
        features = list(point_layer.getFeatures())
        dummy_coords = []
        dummy_geometries = []

        for f in features:
            if self.is_canceled:
                return
            geom = f.geometry()
            if geom.type() == QgsWkbTypes.PointGeometry:
                pt = geom.asPoint()
            elif geom.type() == QgsWkbTypes.PolygonGeometry:
                pt = geom.centroid().asPoint()
            else:
                continue
            dummy_coords.append([pt.x(), pt.y()])
            dummy_geometries.append(geom)

        dummy_coords = np.array(dummy_coords)

        n_features = len(dummy_coords)

        self.dlg.progressBar.setValue(30)

        # === Logika Hybrid vs Langsung Agglomerative ===
        if n_features <= 1000:
            # === Langsung Agglomerative ===
            print("ℹ️ Data kecil: langsung Agglomerative clustering.")

            _, idxs = tree.query(dummy_coords, k=1)
            nearest_nodes = [nodes[i] for i in idxs]

            nodes_unique = list(set(nearest_nodes))
            G_dm = nx.DiGraph(G)
            node_dm = pd.DataFrame(index=nodes_unique, columns=nodes_unique, dtype=float)

            for i, u in enumerate(nodes_unique):
                if u not in G_dm:
                    continue
                lengths = nx.single_source_dijkstra_path_length(G_dm, u, weight='weight')
                for v in nodes_unique:
                    if v in lengths:
                        node_dm.at[u, v] = lengths[v]

                progress = 40 + int(20 * (i + 1) / len(nodes_unique))
                self.dlg.progressBar.setValue(progress)

            ndm = node_dm.reindex(index=nearest_nodes, columns=nearest_nodes).fillna(9999)

            self.dlg.progressBar.setValue(70)

            if use_max_distance:
                model = AgglomerativeClustering(distance_threshold=max_distance, n_clusters=None, metric='precomputed', linkage='complete')
            else:
                model = AgglomerativeClustering(n_clusters=n_clusters, metric='precomputed', linkage='complete')

            labels = model.fit_predict(ndm)

        else:
            # === Hybrid KMeans → Agglomerative ===
            print("ℹ️ Data besar: gunakan Hybrid Mode (KMeans + Agglomerative).")

            jumlah_kmeans = min(1500, max(50, int(math.log(n_features) * 100)))
            print(f"ℹ️ Pre-clustering pakai {jumlah_kmeans} cluster.")

            kmeans = KMeans(n_clusters=jumlah_kmeans, n_init='auto', random_state=42)
            kmeans_labels = kmeans.fit_predict(dummy_coords)

            mean_coords = []
            for k in range(jumlah_kmeans):
                if self.is_canceled:
                    return
                pts = dummy_coords[kmeans_labels == k]
                mean = pts.mean(axis=0)
                mean_coords.append(mean)
            mean_coords = np.array(mean_coords)

            _, idxs = tree.query(mean_coords, k=1)
            mean_nodes = [nodes[i] for i in idxs]

            nodes_unique = list(set(mean_nodes))
            G_dm = nx.DiGraph(G)
            node_dm = pd.DataFrame(index=nodes_unique, columns=nodes_unique, dtype=float)

            for i, u in enumerate(nodes_unique):
                if self.is_canceled:
                    print("❌ Proses clustering dibatalkan saat perhitungan jarak.")
                    return  # 🔥 Langsung stop loop kalau cancel
                if u not in G_dm:
                    continue
                lengths = nx.single_source_dijkstra_path_length(G_dm, u, weight='weight')
                for v in nodes_unique:
                    if v in lengths:
                      node_dm.at[u, v] = lengths[v]

                progress = 40 + int(20 * (i + 1) / len(nodes_unique))
                self.dlg.progressBar.setValue(progress)

            ndm = node_dm.reindex(index=mean_nodes, columns=mean_nodes).fillna(9999)

            self.dlg.progressBar.setValue(70)

            if use_max_distance:
                model = AgglomerativeClustering(distance_threshold=max_distance, n_clusters=None, metric='precomputed', linkage='complete')
            else:
                model = AgglomerativeClustering(n_clusters=n_clusters, metric='precomputed', linkage='complete')

            agglo_labels = model.fit_predict(ndm)

            # === Mapping dari KMeans ke Final Cluster ===
            kmeans_to_final = {k: int(agglo_labels[i]) for i, k in enumerate(range(jumlah_kmeans))}
            labels = np.array([kmeans_to_final[kmeans_labels[i]] for i in range(n_features)])

        self.dlg.progressBar.setValue(80)

        # === Buat Layer Output ===
        if self.is_canceled:
           self.iface.messageBar().pushMessage("NetClus", "⚠️ Proses dibatalkan, tidak membuat layer output.", level=Qgis.Warning)
           self.dlg.progressBar.setValue(0)
           return
        
        if show_result:
            geom_type = "Polygon" if point_layer.geometryType() == QgsWkbTypes.PolygonGeometry else "Point"
            out_layer = QgsVectorLayer(f"{geom_type}?crs=EPSG:4326", "Clustering_Result", "memory")
            prov = out_layer.dataProvider()
            prov.addAttributes([QgsField("cluster", QVariant.Int)])
            out_layer.updateFields()

            for i, geom in enumerate(dummy_geometries):
                feat = QgsFeature()
                feat.setGeometry(geom)
                feat.setAttributes([int(labels[i])])
                prov.addFeature(feat)

            out_layer.updateExtents()

            # Remove existing layers
            existing_layers = [lyr for lyr in QgsProject.instance().mapLayers().values() if lyr.name() == "Clustering_Result"]
            for lyr in existing_layers:
                QgsProject.instance().removeMapLayer(lyr)

            QgsProject.instance().addMapLayer(out_layer)

            if output_path and output_path.endswith(".shp"):
                QgsVectorFileWriter.writeAsVectorFormat(out_layer, output_path, "UTF-8", out_layer.crs(), "ESRI Shapefile")

            self.iface.messageBar().pushMessage("NetClus", "✅ Klasterisasi selesai.", level=Qgis.Info)

        self.dlg.progressBar.setValue(100)
        self.dlg.progressBar.setValue(0)