# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Nadi
                                 A QGIS plugin
 Nadi (River) connections tool
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-12-21
        copyright            : (C) 2023 by Gaurav Atreya
        email                : allmanpride@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gaurav Atreya'
__date__ = '2023-12-21'
__copyright__ = '(C) 2023 by Gaurav Atreya'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingAlgorithm,
    QgsProcessingParameterString,
    QgsProcessingParameterEnum,
    QgsProcessingParameterFolderDestination,
    QgsProcessingLayerPostProcessorInterface,
    QgsVectorLayer,
    QgsLineSymbol,
    QgsMarkerSymbol,
    QgsFillSymbol,
)
import pathlib
from .nadi_exe import qgis_nadi_proc


class NadiDownload(QgsProcessingAlgorithm):
    """
    Downloads the data from USGS NWIS service.
    """

    USGS_SITE_ID = 'USGS_SITE_ID'
    DATA_TYPE = 'DATA_TYPE'
    OUTPUT = 'OUTPUT'

    ALL_DATA_TYPES = ["upstream", "downstream", "tributaries", "basin", "nwis-site"]

    def initAlgorithm(self, config):
        """
        Takes usgs site number, data type and the output directory
        """
        self.addParameter(
            QgsProcessingParameterString(
                self.USGS_SITE_ID,
                self.tr('USGS Site ID'),
            )
        )
        self.addParameter(
            QgsProcessingParameterEnum(
                self.DATA_TYPE,
                self.tr('Data Type'),
                self.ALL_DATA_TYPES,
                False,
                0
            )
        )

        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT,
                self.tr('Output Directory for GeoJSON')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Calls nadi-gis executable and loads the resutls back to QGIS
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.

        usgs_id = self.parameterAsString(parameters, self.USGS_SITE_ID, context)
        data_type = self.ALL_DATA_TYPES[self.parameterAsInt(parameters, self.DATA_TYPE, context)]
        output_dir = pathlib.Path(self.parameterAsFile(
            parameters, self.OUTPUT, context
        ))
        output = output_dir.joinpath(f'{usgs_id}_{data_type}.json').as_posix()

        # main command, ignore spatial reference and verbose for progress
        cmd = ["usgs", "-v", "-s", usgs_id, "-d", data_type, "-o", output_dir.as_posix()]

        proc = qgis_nadi_proc(feedback, cmd)
        res = proc.run(feedback)

        if feedback.isCanceled():
            feedback.pushInfo("Cancelled")
        elif res != 0:
            feedback.reportError("Error")
        else:
            feedback.pushInfo("Completed")

        context.layerToLoadOnCompletionDetails(output).setPostProcessor(LayerPostProcessor.create(data_type))
        return {self.OUTPUT: output}

    def name(self):
        return 'download'

    def displayName(self):
        return self.tr("Download USGS Data")

    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), "download.png"))

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'Vector'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return NadiDownload()


class LayerPostProcessor(QgsProcessingLayerPostProcessorInterface):

    instance = None

    def __init__(self, data_type):
        self.data_type = data_type
        super().__init__()

    def postProcessLayer(self, layer, context, feedback):
        if not isinstance(layer, QgsVectorLayer):
            return
        renderer = layer.renderer().clone()
        if self.data_type == "upstream":
            symbol = QgsLineSymbol.createSimple({'line_color': 'blue', 'line_width': '1.0'})
            renderer.setSymbol(symbol)
        elif self.data_type == "downstream":
            symbol = QgsLineSymbol.createSimple({'line_color': 'blue', 'line_width': '1.0'})
            renderer.setSymbol(symbol)
        elif self.data_type == "tributaries":
            symbol = QgsLineSymbol.createSimple({'line_color': 'blue', 'line_width': '0.5'})
            renderer.setSymbol(symbol)
        elif self.data_type == "basin":
            symbol = QgsFillSymbol.createSimple({'color': 'green', "opacity": "0.4"})
            renderer.setSymbol(symbol)
        elif self.data_type == "nwis-site":
            symbol = QgsMarkerSymbol.createSimple({'color': 'orange', 'size':'0.5'})
            renderer.setSymbol(symbol)
        layer.setRenderer(renderer)

    @staticmethod
    def create(data_type) -> 'LayerPostProcessor':
        LayerPostProcessor.instance = LayerPostProcessor(data_type)
        return LayerPostProcessor.instance
