from qgis.PyQt.QtWidgets import QAction, QDockWidget, QLabel, QVBoxLayout, QPushButton, QWidget, QSizePolicy, QHBoxLayout, QLineEdit
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import Qt
from qgis.utils import iface
from qgis.core import QgsProject, Qgis, QgsVectorLayer
import os
import json

class FeatureNavigator:
    def __init__(self, iface):
        self.iface = iface
        self.layer = None
        self.index = 0
        self.feature_list = []
        self.state_file = os.path.join(os.path.dirname(__file__), "state.json")
        self.load_state()

        self.dock_widget = None
        self.icon_path = os.path.join(os.path.dirname(__file__), 'icons', 'navigator_icon.png')
        self.toolbar = self.iface.addToolBar("Feature Navigator")

    def initGui(self):
        """Initializes the toolbar and navigation controls."""
        icon = QIcon(self.icon_path) if os.path.exists(self.icon_path) else QIcon()
        self.toggle_action = QAction(icon, "Open Feature Navigator", self.iface.mainWindow())
        self.toggle_action.triggered.connect(self.toggle_plugin)
        self.toolbar.addAction(self.toggle_action)

        self.iface.layerTreeView().currentLayerChanged.connect(self.on_layer_changed)

    def toggle_plugin(self):
        """Toggles the dock widget on or off."""
        if not self.dock_widget:
            self.dock_widget = FeatureDockWidget(self)
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dock_widget)
        else:
            self.iface.removeDockWidget(self.dock_widget)
            self.dock_widget = None

    def load_state(self):
        """Loads the saved state of the plugin."""
        if os.path.exists(self.state_file):
            with open(self.state_file, "r") as f:
                state = json.load(f)
                self.index = state.get("index", 0)

    def save_state(self):
        """Saves the current state of the plugin."""
        state = {"index": self.index}
        with open(self.state_file, "w") as f:
            json.dump(state, f)

    def unload(self):
        """Cleans up resources when the plugin is deactivated."""
        self.save_state()
        if self.dock_widget:
            self.iface.removeDockWidget(self.dock_widget)
            self.dock_widget = None
        if self.toolbar:
            self.toolbar.clear()
            self.toolbar = None

    def load_layer(self):
        """Loads the active layer and features if it is a vector layer."""
        self.layer = self.iface.activeLayer()
        if not self.layer:
            self.feature_list = []
            if self.dock_widget:
                self.dock_widget.update_info(None, 0, 0)
            return

        if isinstance(self.layer, QgsVectorLayer):
            # Only process vector layers
            self.feature_list = list(self.layer.getFeatures())
            self.index = 0
            if self.feature_list and self.dock_widget:
                self.dock_widget.update_info(self.feature_list[self.index], self.index, len(self.feature_list))
        else:
            # If the layer is not vector (raster or other type), clear the feature list
            self.feature_list = []
            if self.dock_widget:
                self.dock_widget.update_info(None, 0, 0)
            if self.dock_widget and self.dock_widget.isVisible():
                self.iface.messageBar().pushMessage(
                    "Info", "This tool only works with vector layers.", level=Qgis.Info
                )

    def navigate(self, step):
        """Navigates between features."""
        if not self.layer or not self.feature_list:
            self.load_layer()
        if self.feature_list:
            self.index = (self.index + step) % len(self.feature_list)
            self.show_feature(self.feature_list[self.index])
            if self.dock_widget:
                self.dock_widget.update_info(self.feature_list[self.index], self.index, len(self.feature_list))

    def show_feature(self, feature):
        """Selects and shows the feature on the map."""
        if self.layer:
            self.layer.removeSelection()
            self.layer.selectByIds([feature.id()])
            if self.layer.selectedFeatureCount() > 0:
                self.iface.mapCanvas().zoomToSelected(self.layer)

    def zoom_to_id(self, feature_id):
        """Zooms to a specific feature by its ID."""
        if self.layer and self.feature_list:
            # Ajustar el ID ingresado por el usuario para que sea compatible con índices basados en 0
            adjusted_id = feature_id - 1

            # Verificar si el ID ajustado está dentro de los límites
            if 0 <= adjusted_id < len(self.feature_list):
                feature = self.feature_list[adjusted_id]
                self.show_feature(feature)
            else:
                self.iface.messageBar().pushMessage("Error", "No feature found with that ID.", level=Qgis.Warning)

    def on_layer_changed(self, layer):
        """Updates the active layer and features if it is a vector layer."""
        if isinstance(layer, QgsVectorLayer):
            self.layer = layer
            self.load_layer()
        else:
            # If it is not a vector layer, clear the feature list and show a message
            self.layer = None
            self.feature_list = []
            if self.dock_widget:
                self.dock_widget.update_info(None, 0, 0)
            # Ensure the message is shown only if the dock_widget is visible
            if self.dock_widget and self.dock_widget.isVisible():
                self.iface.messageBar().pushMessage(
                    "Info", "This tool only works with vector layers.", level=Qgis.Info
                )

class FeatureDockWidget(QDockWidget):
    def __init__(self, navigator):
        super().__init__("Feature Navigator")
        self.navigator = navigator

        # UI Elements
        self.label = QLabel("No features selected.")
        self.id_input = QLineEdit()
        self.id_input.setPlaceholderText("Feature ID")
        self.id_input.returnPressed.connect(self.zoom_to_id)  # Connects the Enter event

        # Create a layout with navigation buttons
        button_layout = self.create_navigation_buttons()

        # Main layout
        layout = QVBoxLayout()
        layout.addWidget(self.label)
        layout.addWidget(self.id_input)
        layout.addLayout(button_layout)

        widget = QWidget()
        widget.setLayout(layout)
        self.setWidget(widget)

    def update_info(self, feature, index, total):
        """Updates the information displayed in the dock."""
        if feature:
            self.label.setText(f"Feature {index + 1} of {total}")
        else:
            self.label.setText("No features selected.")

    def create_navigation_buttons(self):
        """Creates the navigation buttons and their actions."""
        actions = [
            ("<<", lambda: self.navigator.navigate(-self.navigator.index)),
            ("<", lambda: self.navigator.navigate(-1)),
            (">", lambda: self.navigator.navigate(1)),
            (">>", lambda: self.navigator.navigate(len(self.navigator.feature_list) - self.navigator.index - 1))
        ]

        button_layout = QHBoxLayout()
        for text, action in actions:
            button = self.create_navigation_button(text, action)
            button_layout.addWidget(button)

        return button_layout

    @staticmethod
    def create_navigation_button(text, action):
        """Creates a navigation button with its connected action."""
        button = QPushButton(text)
        button.clicked.connect(action)
        return button

    def zoom_to_id(self):
        """Processes the entered ID and zooms to it."""
        try:
            feature_id = int(self.id_input.text())
            self.navigator.zoom_to_id(feature_id)
        except ValueError:
            self.navigator.iface.messageBar().pushMessage("Error", "Please enter a valid ID.", level=Qgis.Warning)