"""
/***************************************************************************
 MS Planetary Computer Manager task processing
                             -------------------
        begin                : 2025-10-15
        copyright            : (C) 2025 by Luiz Motta
        email                : motta.luiz@gmail.com

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 """

import os
import json
import xml.etree.ElementTree as ET

from qgis.core import (
    Qgis,
    QgsMapLayer
)

from .stacprocessor import (
    QgisInterface,
    StacProcessor,
    TaskProcessor,
    StacClient,
    tr
)

class MSPCStacProcessor(StacProcessor):
    def __init__(self,
            iface:QgisInterface,
            task_processor:TaskProcessor,
            stac_client:StacClient
        ):
        super().__init__( iface, task_processor, stac_client )

        self.ltv = iface.layerTreeView()

    def _setToken(self, collection_id:str)->bool:
        r = self._client.setToken( collection_id )
        if not r['is_ok']:
            msg = tr('Error requesting a token - {}.').format( r['message'] )
            self.requestProcessData.emit({
                'type': 'message_bar',
                'data': { 'text': msg, 'level': Qgis.Critical }
            })
            return False
        
        return True

    def setCollection(self, collection:dict)->None:
        super().setCollection( collection )
        self._setToken( collection['id'] )

    def updateTokenUrls(self):
        def getSourceFilenamesVRT(source:str)->set:
            tree = ET.parse( source )
            root = tree.getroot()

            return set( [elem.text for elem in root.iter('SourceFilename')] )
        
        def updateToken(filepath:str, tokens:dict)->None:
            tree = ET.parse( filepath )
            root = tree.getroot()

            if not collection_id in tokens:
                if not self._setToken( collection_id ):
                    return
            tokens[ collection_id ] = self._client.token
            
            for elem in root.iter('SourceFilename'):
                url = elem.text.split('?')[0]
                elem.text = f"{url}?{tokens[ collection_id ]}"

            tree.write( filepath, encoding='UTF-8', xml_declaration=True)

        def getSourceTypeMSPC(layer:QgsMapLayer)->tuple:
            if  not layer.type() == QgsMapLayer.RasterLayer or not layer.providerType() == 'gdal':
                return None, None, None
            
            filepath = layer.source()
            if not filepath.lower().endswith('.vrt'):
                return None, None, None
            
            filepath_collection = f"{filepath}.{self._client.TAG_ATT}.json"
            if not os.path.exists( filepath_collection):
                return None, None, None
            
            with open( filepath_collection, "r", encoding="utf-8") as f:
                data = json.load(f)

            return filepath, data['source_type'], data['collection_id']

        def reloadData(layer:QgsMapLayer):
            def removeIndicators()->None:
                node = lyr_tree_root.findLayer(layer.id())
                for ind in self.ltv.indicators(node):
                    self.ltv.removeIndicator(node, ind)
                    node.setItemVisibilityChecked(False)
                    node.setItemVisibilityChecked(True)

            layer.dataProvider().reloadData()
            layer.triggerRepaint()
            removeIndicators()

        lyr_tree_root = self.project.layerTreeRoot()
        tokens = {}
        total_update = 0
        for layer in self.project.mapLayers().values():
            ( filepath, source_type, collection_id ) = getSourceTypeMSPC( layer )
            if source_type is None:
                continue

            if source_type == 'URL':
                updateToken( filepath, tokens )
                reloadData( layer )
                total_update += 1
                continue

            dirname = os.path.dirname( filepath )
            for source_vrt in getSourceFilenamesVRT( filepath ):
                filepath_vrt = os.path.join( dirname, source_vrt )
                updateToken( filepath_vrt, tokens )
                reloadData( layer )
                total_update += 1
            
        msg = tr('Updated {} layers').format( total_update )
        self.requestProcessData.emit({
            'type': 'message_bar',
            'data': {'text': msg, 'level': Qgis.Info }
        })
        
