# -*- coding: utf-8 -*-
"""
/***************************************************************************
 MS Planetary Compute Catalog
                                 A QGIS plugin
 Plugin to access Microsoft Planetary Compute for show COG scenes
                             -------------------
        begin                : 2025-10-15
        copyright            : (C) 2025 by Luiz Motta
        email                : motta.luiz@gmail.com

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 This script initializes the plugin, making it known to QGIS.
"""

__author__ = "Luiz Motta"
__date__ = "2025-09-02"
__copyright__ = "(C) 2025, Luiz Motta"
__revision__ = "$Format:%H$"


import os

from typing import Callable

from qgis.PyQt.QtCore import (
    QObject,
    QDir,
    pyqtSlot
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import  QAction, QToolButton, QMenu

from qgis.gui import QgisInterface
from qgis.core import QgsApplication

from .mspc.taskmanager import TaskProcessor
from .mspc.mspc_stacclient import MSPCStacClient
from .mspc.mspc_stacprocessor import MSPCStacProcessor
from .mspc.config import configCollection

from .mspc.catalogwidget import CatalogWidget
from .mspc.catalog import Catalog

from .mspc.translate import setTranslation, tr


def classFactory(iface:QgisInterface):
    return MSPCCatalogPlugin(iface)


class MSPCCatalogPlugin(QObject):
    def __init__(self, iface:QgisInterface):
        super().__init__()
        self.iface = iface

        setTranslation( type(self).__name__, os.path.dirname(__file__) )

        self.plugin_name = 'MSPCCatalog'
        self.actions = {}
        self.toolButton = QToolButton()
        self.toolButton.setMenu( QMenu() )
        self.toolButton.setPopupMode( QToolButton.MenuButtonPopup )
        self.titleTool = tr('MS Planetary Compute Catalog')

        # Catalog Widget
        task_processor = TaskProcessor( self.iface, 'MSPC Catalog' )
        client = MSPCStacClient()
        self._processor = MSPCStacProcessor( iface, task_processor, client )
        self._config_collection = configCollection()

        self.catalog = None # initGui

    def initGui(self)->None:
        def createAction(
                icon:QIcon,
                title:str,
                slot:Callable[[None], None],
                menu:QMenu,
                isCheckable:bool=False
            ):
            action = QAction( icon, title, self.iface.mainWindow() )
            action.triggered.connect( slot )
            action.setToolTip( f"MS Planetary Compute {title}" )
            action.setCheckable( isCheckable )
            self.iface.addPluginToWebMenu( f"&{self.titleTool}" , action )
            menu.addAction( action )
            return action

        # Actions
        self.action = self.iface.addWebToolBarWidget( self.toolButton )
        menu = self.toolButton.menu()
        path = QDir( os.path.dirname(__file__) )
        icon = QIcon( path.filePath('resources/mspc_catalog.svg'))
        self.actions['tool'] = createAction( icon, tr('Catalog'), self.on_MSPCClicked, menu, True )
        icon = QgsApplication.getThemeIcon('mActionRefresh.svg')
        self.actions['token'] = createAction( icon, tr('Update token'), self.on_UpdateToken, menu )
        self.toolButton.setDefaultAction( self.actions['tool'] )

        # Catalog Widget
        widget = CatalogWidget( self.iface, self._config_collection, 'mspccatalogwidget' )
        self.catalog = Catalog( widget, self._processor )
        self.catalog.addWidget()

    def unload(self)->None:
        for k in self.actions:
            action = self.actions[ k ]
            self.iface.removePluginWebMenu( f"&{self.titleTool}", action )
            self.iface.webToolBar().removeAction( action )
            self.iface.unregisterMainWindowAction( action )
            try:
                action.triggered.disconnect( self.on_MSPCClicked )
            except Exception:
                pass
            action.deleteLater()
        self.iface.webToolBar().removeAction( self.action )

        del self.catalog

    @pyqtSlot(bool)
    def on_MSPCClicked(self, enabled:bool)->None:
        self.catalog.enabled( enabled )

    @pyqtSlot(bool)
    def on_UpdateToken(self, enabled:bool)->None:
        self.catalog.processor.updateTokenUrls()
