# -*- coding: utf-8 -*-
"""
/***************************************************************************
 MS Planetary Compute Catalog
                             -------------------
        begin                : 2025-10-15
        copyright            : (C) 2025 by Luiz Motta
        email                : motta.luiz@gmail.com

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 """

from typing import Callable

from qgis.core import Qgis

from qgis.PyQt.QtCore import pyqtSignal

from .translate import tr

from .stacclient import StacClient

class MSPCStacClient(StacClient):
    def __init__(self):
        super().__init__()

        self.TAG_ATT = 'MSPC'
        self.STAC_URL = 'https://planetarycomputer.microsoft.com/api/stac/v1'
        self.TOKEN_URL = 'https://planetarycomputer.microsoft.com/api/sas/v1/token/{}'

        self._verify_ssl = False
        
        self.token = None # str

    def search(
            self,
            bbox:list,
            dates:list,
            requestProcessData:pyqtSignal,
            isCanceled:Callable[[str], None]
        )->dict:
        def getNextUrlFromResult(result:dict)->str:
            url_next = [ item['href'] for item in result['links'] if item['rel'] == 'next']
            url_next = url_next[0] if len( url_next) else None

            return url_next

        getNameFromFeature = lambda feature: feature['properties'][ self.collection['name'] ]
        getCRSFromFeature = lambda feature: str( feature['properties']['proj:epsg'] )
        
        r = self._setCollectionsCOGBandsMeta()
        if not r['is_ok']:
            requestProcessData.emit( {
                'type': 'message_bar',
                'data':{ 'text': r['message'], 'level': Qgis.Critical }
            })
            return False

        self._request_count = 1

        self._features.clear()
        r = self._searchStacItems(
            bbox, dates,
            requestProcessData,
            isCanceled,
            getNameFromFeature, getCRSFromFeature, getNextUrlFromResult
        )

        if not r['is_ok']:
            requestProcessData.emit( {
                'type': 'message_bar',
                'data':{ 'text': r['message'], 'level': Qgis.Critical }
            })
            return False

        returned = r['returned']
        if not returned:
            msg = tr("No scenes found in '{}' collection").format( self.collection['id'] )
            requestProcessData.emit( {
                'type': 'message_bar',
                'data':{ 'text': msg, 'level': Qgis.Info }
            })
            return True

        self._features = r['features']

        if r['url_next'] is None:
            self._messageTotalFeatures( returned, requestProcessData )
            return True

        total_returned = r['returned']
       
        while True:
            self._request_count += 1
            r = self._fetchNextPage(
                r['url_next'],
                requestProcessData,
                isCanceled,
                getNameFromFeature, getCRSFromFeature, getNextUrlFromResult
            )
            if not r['is_ok']:
                requestProcessData.emit( {
                    'type': 'message_bar',
                    'data':{ 'text': r['message'], 'level': Qgis.Critical }
                })
                return False

            total_returned += r['returned']

            if r['url_next'] is None:
                self._messageTotalFeatures( total_returned, requestProcessData )
                return True

    def setToken(self, collection_id:str)->dict:
        r = self._getResponse( { 'url': self.TOKEN_URL.format( collection_id ) })
        if not r['is_ok']:
            return r
        
        result = r['response'].json()
        r['response'].close()

        self.token = result['token']

        return { 'is_ok': True }

    def getScenesByDateOrbitsCRS(self, spatial_resolution:str)->dict:
        scene_list = {}
        for asset, data in self._features.items():
            p = data['properties']
            scene_key = f"{p['datetime'].split('T')[0]}_{p[ self._feat_key_orbit_crs ]}"
            urls = { asset: { band: f"{value['href']}?{self.token}" for band, value in data['bands'].items() if value[ self._feat_key_spatial_res ] == spatial_resolution } }
            if not scene_key in scene_list:
                scene_list[ scene_key ] = [ urls ]
                continue
            scene_list[ scene_key ].append( urls )
        
        return scene_list

