# -*- coding: utf-8 -*-
"""
/***************************************************************************
 MRTPS
                                 A QGIS plugin
 Set transparency pixel for multiple raster layers
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-11-07
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Łukasz Świątek
        email                : lukasz.swiatek1996@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import (
    Qgis,
    QgsProject,
    QgsPathResolver,
    QgsMessageLog,
    QgsRasterLayer,
    QgsMapLayer,
    QgsRasterTransparency
)
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .mrtps_dockwidget import MRTPSDockWidget
import os.path


class MRTPS:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'MRTPS_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Multi Raster Transparency Pixel Setter')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'MRTPS')
        self.toolbar.setObjectName(u'MRTPS')

        #print "** INITIALIZING MRTPS"

        self.pluginIsActive = False
        self.dockwidget = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('MRTPS', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/mrtps/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Set transparency pixel for rasters'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING MRTPS"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD MRTPS"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Multi Raster Transparency Pixel Setter'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------
    def load_table(self):
        
        # ładuje wsyztskie warstwy nie tylko widoczne
        self.dockwidget.listWidget.clear()
        for l in QgsProject.instance().mapLayers().values():
        
            # określa czy to raster
            if l.type() != QgsMapLayer.VectorLayer:
                # QgsMessageLog.logMessage(f"{l.name()}", "mrtps", level=Qgis.Info)

                self.dockwidget.listWidget.addItem(l.name());
                
    def valid_float(self, value, top=255.0):
        if top < value or value < 0:
            value = top
        return value
        
    def clear_transparency(self):
    
        for raster in self.dockwidget.listWidget.selectedItems():
        
            layers = QgsProject.instance().mapLayersByName(raster.text())
            
            layer = layers[0]
                    
            transparency = QgsRasterTransparency()
            transparency.setTransparentThreeValuePixelList([])
            
            renderer = layer.renderer()
            renderer.setRasterTransparency(transparency)
            
    def set_transparency_by_xml(self):
    
        filename = QFileDialog.getOpenFileName(self.dockwidget, "Select xml file with transparency", "", "text file (*.txt *.xml)")
        

        try:
            with open(str(filename[0]), "r", encoding="cp1252") as file:
                QgsMessageLog.logMessage(f"1", "mrtps", level=Qgis.Info)
                lines = file.readlines()
                QgsMessageLog.logMessage(f"2", "mrtps", level=Qgis.Info)
        except Exception as e:
            QgsMessageLog.logMessage(f"Can't open file: {str(filename[0])}\n{e}", "mrtps", level=Qgis.Info)
            return
            
        values = []
        for line in lines:
            QgsMessageLog.logMessage(f"{line}", "mrtps", level=Qgis.Info)
            if line.startswith("#"):
                continue
            line = line.replace("\n", "")
            rgb = line.split("\t")
            if len(rgb) >= 3:
                values.append(rgb)
    
        for raster in self.dockwidget.listWidget.selectedItems():
        
            layers = QgsProject.instance().mapLayersByName(raster.text())
            
            layer = layers[0]
            transparency = QgsRasterTransparency()
           
            tp = [QgsRasterTransparency.TransparentSingleValuePixel(minimum=float(v[0]), maximum=float(v[1]),
                opacity=(1  - (float(v[2]) / 100))) for v in values]
            transparency.setTransparentSingleValuePixelList(tp)
            
            renderer = layer.renderer()
            renderer.setRasterTransparency(transparency)
        
            
    def set_transparency_by_txt(self):
    
        filename = QFileDialog.getOpenFileName(self.dockwidget, "Select xml file with transparency", "", "text file (*.txt *.xml)")
        

        try:
            with open(str(filename[0]), "r", encoding="cp1252") as file:
                QgsMessageLog.logMessage(f"1", "mrtps", level=Qgis.Info)
                lines = file.readlines()
                QgsMessageLog.logMessage(f"2", "mrtps", level=Qgis.Info)
        except Exception as e:
            QgsMessageLog.logMessage(f"Can't open file: {str(filename[0])}\n{e}", "mrtps", level=Qgis.Info)
            return
            
        values = []
        for line in lines:
            QgsMessageLog.logMessage(f"{line}", "mrtps", level=Qgis.Info)
            if line.startswith("#"):
                continue
            line = line.replace("\n", "")
            rgb = line.split("\t")
            if len(rgb) >= 4:
                values.append(rgb)
    
        for raster in self.dockwidget.listWidget.selectedItems():
        
            layers = QgsProject.instance().mapLayersByName(raster.text())
            
            layer = layers[0]
            transparency = QgsRasterTransparency()
            tp = [QgsRasterTransparency.TransparentThreeValuePixel(red=float(v[0]),
                green=float(v[1]), blue=float(v[2]),
                opacity=(1  - (float(v[3]) / 100))) for v in values]
            transparency.setTransparentThreeValuePixelList(tp)
            
            renderer = layer.renderer()
            renderer.setRasterTransparency(transparency)
    
        
    def set_transparency(self):
    
        r = self.dockwidget.leR.text()
        try:
            r = float(r)
        except:
            r = 255.0
            
        r = self.valid_float(r)
            
        g = self.dockwidget.leG.text()
        try:
            g = float(g)
        except:
            g = 255.0
        
        g = self.valid_float(g)
        
        b = self.dockwidget.leB.text()
        try:
            b = float(b)
        except:
            b = 255.0
            
        b = self.valid_float(b)
        
        QgsMessageLog.logMessage(f"Red: {r}\tGreen: {g}\tBlue: {b}", "mrtps", level=Qgis.Info)
            
        if (tr := self.dockwidget.leT.text()).isnumeric():
            tr = 1 - (float(tr) / 100)
            
        else:
            tr = 0.0
            
        tr = self.valid_float(tr, 1.0)
            
        
        for raster in self.dockwidget.listWidget.selectedItems():
        
            layers = QgsProject.instance().mapLayersByName(raster.text())
            
            layer = layers[0]
                    
            transparency = QgsRasterTransparency()
            tp = QgsRasterTransparency.TransparentThreeValuePixel(red=r, green=g, blue=b, opacity=tr)
            transparency.setTransparentThreeValuePixelList([tp])
            
            renderer = layer.renderer()
            renderer.setRasterTransparency(transparency)
        
         
    def set_transparency_single(self):
    
        # TODO add feature to work with float after comma values
        vvv = self.dockwidget.leMin.text().isnumeric()
    
        min = self.dockwidget.leMin.text()
        try:
            min = float(min)
        except:
            min = 0.0
            

        max = self.dockwidget.leMax.text()
        try:
            max = float(max)
        except:
            max = 1.0
            
        QgsMessageLog.logMessage(f"Min: {min}\tMax: {max}", "mrtps", level=Qgis.Info)
        
            
        if (tr := self.dockwidget.leT2.text()).isnumeric():
            tr = 1 - (float(tr) / 100)
            
        else:
            tr = 0.0
            
        tr = self.valid_float(tr, 1.0)
        
        imin = self.dockwidget.cbMin.isChecked() 
        imax = self.dockwidget.cbMax.isChecked() 
        
            
        
        for raster in self.dockwidget.listWidget.selectedItems():
        
            layers = QgsProject.instance().mapLayersByName(raster.text())
            
            layer = layers[0]
                    
            transparency = QgsRasterTransparency()
            tp = QgsRasterTransparency.TransparentSingleValuePixel(minimum=min, maximum=max,
                includeMinimum=imin, includeMaximum=imax,
                opacity=tr)
                
            transparency.setTransparentSingleValuePixelList([tp])
            
            renderer = layer.renderer()
            renderer.setRasterTransparency(transparency)

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING MRTPS"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = MRTPSDockWidget()
                
                self.dockwidget.pbRefresh.clicked.connect(lambda: self.load_table())
                
                self.dockwidget.pbAddT.clicked.connect(lambda: self.set_transparency())
                self.dockwidget.pbClearT.clicked.connect(lambda: self.clear_transparency())
                self.dockwidget.pbXML.clicked.connect(lambda: self.set_transparency_by_txt())
                
                self.dockwidget.pbOneP.clicked.connect(lambda: self.set_transparency_single())
                self.dockwidget.pbXML2.clicked.connect(lambda: self.set_transparency_by_xml())
                
                self.dockwidget.pbAll.clicked.connect(lambda: self.dockwidget.listWidget.selectAll())
                self.dockwidget.pbClear.clicked.connect(lambda: self.dockwidget.listWidget.clearSelection())
                
                
                
            self.load_table()

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
