import numpy as np
import cv2 as cv
from osgeo import gdal,ogr,osr
import shapely

def max_square4bi_matrix(matrix):
    """
    给出了一个 m x n 的矩阵 matrix ，里面每个格子填充了 0 或者 1 ，找到只包含 1 的最大的正方形面积并返回,
    DP (bottom-up) approach
    参考：https://www.techgeekbuzz.com/blog/maximum-size-square-sub-matrices-with-all-1s/
    :matrix: 0或1组成的二值图
    :return 最大正方形的右下角位置索引(row,col）和正方形的边长
    """
    M = len(matrix)
    N = len(matrix[0])
    dp = np.zeros((M,N)).astype("uint32")
    for n in range(N):
        dp[0][n] = int(matrix[0][n])
    for m in range(M):
        dp[m][0] = int(matrix[m][0])
    for i in range(1, M):
        for j in range(1, N):
            if matrix[i][j] == 0:
                continue
            dp[i][j] = min(dp[i - 1][j - 1], dp[i - 1][j], dp[i][j - 1]) + 1
    ind = np.unravel_index(np.argmax(dp, axis=None), dp.shape)
    return ind,dp[ind]

def create_mask_for_polygon(polygon, scale=1, keep_origin=False):
    '''
    将坐标表示的多边形表示为二值矩阵，多边形内部填充1，外部为0
    :param polygon:
    :param scale:
    :param relative：返回的mask的原点与polygon的坐标原点是否保持一致，true表示一致，则返回的origin=（0,0），否则为polygon的最小值（minx，miny）
    :return:
    '''
    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2
    if scale!=1:
        polygon=(polygon*scale).astype('int32')

    bbox = cv.boundingRect(polygon)
    origin = bbox[0:2]
    if not keep_origin:
        rows = bbox[3]
        cols = bbox[2]
        #print_log('bounding box:',bbox)
        mask = np.zeros([rows, cols], dtype=np.uint8)
        zero_centered_x = polygon[:, :, 0] - bbox[0]
        zero_centered_y = polygon[:, :, 1] - bbox[1]
        polygon = np.dstack((zero_centered_x, zero_centered_y))
        cv.fillPoly(mask, polygon, 1)
    else:
        rows = origin[1]+bbox[3]
        cols = origin[0]+bbox[2]
        #print_log('bounding box:',bbox)
        mask = np.zeros([rows, cols], dtype=np.uint8)
        cv.fillPoly(mask, polygon, 1)
        origin=[0,0]

    return origin, mask

def index2coord(index):
    '''
    将矩阵行列坐标转换为直角坐标系xy坐标
    :param index:
    :return:
    '''
    return (index[1],index[0])


def rotate(polygon, angle, center,use_redian=False):
    '''
    围绕某一中心点对多边形进行旋转，angle在use_redian=False时单位为度，use_redian=True时，为弧度
    :param polygon:
    :param angle:
    :param center:
    :param use_redian:
    :return:
    '''
    def R(angle, use_redian=False):
        '''
        生成一个旋转运算的核，angle在use_redian=False时单位为度，use_redian=True时，为弧度
        :param angle:
        :param use_redian:
        :return:
        '''
        if use_redian:
            cos_a = np.cos(angle)
            sin_a = np.sin(angle)
        else:
            cos_a = np.cos(angle * 180 / np.pi)
            sin_a = np.sin(angle * 180 / np.pi)
        return np.array([[cos_a, -sin_a], [sin_a, cos_a]])

    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2
    if angle==0:
        poly_new=polygon
    else:
        poly_new = (polygon - center).dot(R(angle,use_redian).T) + center
    return poly_new

def centroid(polygon):
    '''
    基于opencv的moments计算结果，求取多边形的质心
    :param polygon:
    :return:
    '''
    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2

    M = cv.moments(polygon[0])
    cx = int(M['m10'] / M['m00'])
    cy = int(M['m01'] / M['m00'])
    return (cx , cy)

def largest_ortho_square(polygon,scale=1):
    '''
    返回多边形内部边与坐标轴平行的面积最大的正方形
    :param polygon:
    :param scale:
    :return:
    '''
    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2
    origin,mask=create_mask_for_polygon(polygon, scale)
    index,size=max_square4bi_matrix(mask)
    tl=index2coord(index - size + 1)
    rb=index2coord(index)
    tl = (tl[0]+origin[0],tl[1]+origin[1])
    rb = (rb[0]+origin[0],rb[1]+origin[1])
    rect=np.array([tl,  rb]).reshape(1,-1,2)
    return rect,size

def get_edge_slopes(polygon,return_redian=True):
    '''
    计算多边形每条边的倾角，在此基础上返回将其旋转到与坐标轴平行所需的角度
    :param polygon:
    :param return_redian:
    :return:
    '''
    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2
    import math
    slopes=[]
    pts=polygon[0]
    for i in range(len(pts)-1):
        # print_log("pt2,pt1:",pts[i+1],pts[i])
        s=math.atan((pts[i+1][1]-pts[i][1])/(pts[i+1][0]-pts[i][0]))
        # print_log("倾角：",s,math.degrees(s))
        if s < 0: s = -s
        # print_log("反角：",s,math.degrees(s))
        if return_redian:
            slopes.append(math.pi / 2-s)
        else:
            slopes.append(90-math.degrees(s))
    return slopes

def largest_square_angle(polygon,ok_length=256,align_edges=True,scale=0.1,angle_resolution_degree=1):
    '''
    搜索多边形内部最大正方形的偏转角度。为了提升计算速度，按照scale缩小多边形。
    先采用各个边的角度(假定最大正方形的边和多边形某条边平行的可能性更大），试算包含的最大正方形的边长是否达到设定阈值
    然后再在0-90度范围内逐次进行穷举计算，寻找符合要求的正方形，如果没有达到边长要求的，最后返回对应边长最大的旋转角
    :param polygon:
    :param ok_length:
    :param align_edges:
    :param scale:
    :param angle_resolution_degree:
    :return:
    '''
    center=centroid(polygon)
    max_len = 0
    angle = 0
    angles=get_edge_slopes(polygon, return_redian=False) if align_edges else range(0,90,angle_resolution_degree)

    for ang in angles:
        poly=rotate(polygon,ang,center).astype("int32")
        rect,side_len=largest_ortho_square(poly,scale)

        # print_log("质心旋转角——正方形边长：",(ang,side_len/scale))
        if side_len>max_len:
            max_len=side_len
            angle=ang
        if max_len>ok_length*scale:
            break
    return angle

def largest_rotated_square(polygon):
    '''
    获取多边形内部最大的正方形，其边与坐标轴不一定平行
    :param polygon:
    :return:
    '''
    center=centroid(polygon)
    angle=largest_square_angle(polygon,align_edges=False,scale=0.1,angle_resolution_degree=2)
    poly = rotate(polygon, angle, center).astype("int32")
    square, side_len = largest_ortho_square(poly)
    square=square[0]
    temp = [square[0][0], square[0][1], square[1][0] , square[0][1], square[1][0],square[1][1], square[0][0],square[1][1]]
    r_square = rotate(np.array(temp).reshape(1,-1, 2), -angle, center).astype("int32")
    # #重新计算以r_square以重心为中心的偏转角度作为返回的角度
    # square = r_square.reshape((-1, 2))
    # origin = np.min(square, axis=0)
    # shifted = square - origin
    # maxxy = np.max(shifted, axis=0)
    # minxy = np.min(shifted, axis=0)
    # x = y = xmin = ymax = 0
    # for pt in shifted:
    #     if pt[0] == minxy[0]:
    #         xmin = pt[0]
    #         y = pt[1]
    #     if pt[1] == maxxy[1]:
    #         x = pt[0]
    #         ymax = pt[1]
    # angle = 0 if ymax == y else math.atan((x - xmin) / (ymax - y)) * 180 / math.pi  # 左上边与y轴的夹角
    return r_square,side_len,angle

def max_inner_circle(polygon):
    '''
    为多边形内最大内接圆（LIC）的外切正方形与多边形相交的部分生成一个mask，mask的坐标原点为多边形坐标的最小值（minx，miny）
    找出多边形内部最大的圆，返回圆心位置(x,y)和半径r
    最大内接圆的外切正方形：circumscribing square of Largest Inscribed Circle （CSRIC）
    :param polygon:
    :return:
    '''
    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2
    origin, mask = create_mask_for_polygon(polygon)
    raw_dist = np.empty(mask.shape, dtype=np.float32)
    for i in range(mask.shape[0]):
        for j in range(mask.shape[1]):
            raw_dist[i, j] = cv.pointPolygonTest(polygon - origin, (j, i), True)
    # 获取最大值即内接圆半径，中心点坐标
    _, maxVal, _, center = cv.minMaxLoc(raw_dist)
    radius = np.int32(abs(maxVal))
    return (center[0]+origin[0],center[1]+origin[1]),radius

def create_mask_for_outer_square_of_LIC(polygon):
    '''
    为多边形内最大内接圆（LIC）的外切正方形与多边形相交的部分生成一个mask，mask的坐标原点为多边形坐标的最小值（minx，miny）
    找出多边形内部最大的圆，返回圆心位置(x,y)和半径r
    最大内接圆的外切正方形：circumscribing square of Largest Inscribed Circle （CSRIC）
    :param polygon:
    :return:
    '''
    assert polygon.shape[0] == 1
    assert polygon.shape[1] > 2
    assert polygon.shape[2] == 2
    origin,mask=create_mask_for_polygon(polygon)

    raw_dist = np.empty(mask.shape, dtype=np.float32)
    for i in range(mask.shape[0]):
        for j in range(mask.shape[1]):
            raw_dist[i, j] = cv.pointPolygonTest(polygon-origin, (j, i), True)
    # 获取最大值即内接圆半径，中心点坐标
    _, maxVal, _, center = cv.minMaxLoc(raw_dist)
    r = np.int32(abs(maxVal))

    # # 画出最大内接圆
    # result = cv.cvtColor(mask*255, cv.COLOR_GRAY2BGR)
    # cv.circle(result, maxDistPt, radius, (0, 255, 0), 2, 1, 0)
    # cv.imshow('Maximum inscribed circle', result)
    # cv.waitKey(0)
    outer_square=np.array([center[0]-r,center[1]-r,center[0]-r,center[1]+r,center[0]+r,center[1]+r,center[0]+r,center[1]-r]).reshape(1,-1,2)
    _, square_mask = create_mask_for_polygon(outer_square,keep_origin=True)

    intersection=mask and square_mask
    return  intersection,origin
    #return (center_of_circle[0]+origin[0],center_of_circle[1]+origin[1]),radius
def proper_square(polygon,ok_length=256,inside=True):
    '''
    获取达到边长条件、够用的正方形。首先提取最大内部正方形，如果不满足条件，再通过内切圆判断
    :param polygon:
    :param ok_length:
    :param inside: 指定偏好： False表示返回的正方形尽可能大一些，可能会超出图斑范围，但超出部分后续处理时需用裁除；
                            True表示返回的正方形全部位于图斑内部
    :param align_edges:
    :param scale:
    :param angle_resolution_degree:
    :return:
    '''
    square, side_length = largest_ortho_square(polygon)
    if side_length > ok_length:
        return square, side_length, 0, inside
    else:
        center, r = max_inner_circle(polygon)
        if not inside and r*2 >=ok_length:
            square = np.array([center[0] - r, center[1] - r, center[0] + r, center[1] + r]).reshape(1, -1, 2).astype('int32')
            return square, r*2, 0, inside
        else:
            square, side_length, angle=largest_rotated_square(polygon)
            return square, side_length, angle,inside

def clip_image_by_mask(src_img,mask):
