# -*- coding: utf-8 -*-
"""
/***************************************************************************
 MoniQue
                                 A QGIS plugin
 Monoplotting oblique images.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-02-07
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Sebastian Mikolka-Flöry
        email                : s.floery@gmx.at
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QApplication, QMenu

from qgis.core import QgsProject, QgsVectorLayer, QgsJsonUtils, QgsFeatureRequest
from qgis.gui import QgsMapToolPan

# # Initialize Qt resources from file resources.py
# from .resources import *
from .gui.dlg_main import MainDialog
from .gui.dlg_convert import ConvertDialog

import os.path
import json
import open3d as o3d
import numpy as np
from .camera import Camera

class MoniQue:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.iface.actionMapTips().trigger()    #enable showing map tips
        
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        self.icon_dir = os.path.join(self.plugin_dir, "gfx", "icon")
        
        self.project_name = None
        
        #paths to the styling files
        self.cam_qml_path = os.path.join(self.plugin_dir, "gfx", "qml", "camera.qml")
        self.img_line_qml_path = os.path.join(self.plugin_dir, "gfx", "qml", "lines_img.qml")
        self.map_line_qml_path = os.path.join(self.plugin_dir, "gfx", "qml", "lines_map.qml")
        self.map_region_qml_path = os.path.join(self.plugin_dir, "gfx", "qml", "region_map.qml")
        self.map_gcps_qml_path = os.path.join(self.plugin_dir, "gfx", "qml", "gcps_map.qml")
        self.img_gcps_qml_path = os.path.join(self.plugin_dir, "gfx", "qml", "gcps_img.qml")
        
        #map_canvas is the canvas of the QGIS main window
        self.map_canvas = iface.mapCanvas()
        
        self.camera_collection = {}
        
        #set tools
        self.map_pan_tool = QgsMapToolPan(self.map_canvas)

        # Declare instance attributes
        self.actions = []
    
    # def add_action(
    #     self,
    #     icon_path,
    #     text,
    #     callback,
    #     enabled_flag=True,
    #     add_to_menu=True,
    #     add_to_toolbar=True,
    #     status_tip=None,
    #     whats_this=None,
    #     parent=None):
    #     """Add a toolbar icon to the toolbar.

    #     :param icon_path: Path to the icon for this action. Can be a resource
    #         path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
    #     :type icon_path: str

    #     :param text: Text that should be shown in menu items for this action.
    #     :type text: str

    #     :param callback: Function to be called when the action is triggered.
    #     :type callback: function

    #     :param enabled_flag: A flag indicating if the action should be enabled
    #         by default. Defaults to True.
    #     :type enabled_flag: bool

    #     :param add_to_menu: Flag indicating whether the action should also
    #         be added to the menu. Defaults to True.
    #     :type add_to_menu: bool

    #     :param add_to_toolbar: Flag indicating whether the action should also
    #         be added to the toolbar. Defaults to True.
    #     :type add_to_toolbar: bool

    #     :param status_tip: Optional text to show in a popup when mouse pointer
    #         hovers over the action.
    #     :type status_tip: str

    #     :param parent: Parent widget for the new action. Defaults None.
    #     :type parent: QWidget

    #     :param whats_this: Optional text to show in the status bar when the
    #         mouse pointer hovers over the action.

    #     :returns: The action that was created. Note that the action is also
    #         added to self.actions list.
    #     :rtype: QAction
    #     """

    #     icon = QIcon(icon_path)
    #     action = QAction(icon, text, parent)
    #     action.triggered.connect(callback)
    #     action.setEnabled(enabled_flag)

    #     if status_tip is not None:
    #         action.setStatusTip(status_tip)

    #     if whats_this is not None:
    #         action.setWhatsThis(whats_this)

    #     if add_to_toolbar:
    #         # Adds plugin icon to Plugins toolbar
    #         self.iface.addToolBarIcon(action)

    #     if add_to_menu:
    #         self.iface.addPluginToMenu(self.menu, action)

    #     self.actions.append(action)

    #     return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        launch_icon = QIcon(os.path.join(self.plugin_dir, "icon.png"))
        launch_action = QAction(launch_icon, 'Run moniQue...', self.iface.mainWindow())
        launch_action.triggered.connect(self.open_main_dlg)
        launch_action.setEnabled(True)
        
        
        convert_icon = QIcon(os.path.join(self.icon_dir, "mActionAddMeshLayer.png"))
        convert_action = QAction(convert_icon, "Convert DTM to mesh...", self.iface.mainWindow())
        convert_action.triggered.connect(self.open_convert_dlg)
        convert_action.setEnabled(True)
        
        self.iface.addPluginToMenu("moniQue", launch_action)
        self.iface.addPluginToMenu("moniQue", convert_action)
        
        self.iface.addToolBarIcon(launch_action)
        
        self.actions.append(launch_action)
        self.actions.append(convert_action)
        
        # icon_path = ':/plugins/monique/icon.png'
        # main_menu = self.add_action(os.path.join(self.plugin_dir, "icon.png"), text='moniQue', callback=self.run, parent=self.iface.mainWindow())
        # QAction("DTM to Mesh", )
        # self.add_action(os.path.join(self.plugin_dir, "icon.png"), text='moniQue', callback=self.run, parent=main_menu)
        
        # will be set False in run()
        # self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu('moniQue', action)
            self.iface.removeToolBarIcon(action)

    def on_load_project_signal(self, data):
        
        self.gpkg_path = data["gpkg_path"]
        self.project_name = os.path.basename(self.gpkg_path).rsplit(".")[0]
        
        QApplication.instance().setOverrideCursor(Qt.WaitCursor)
        
        self.reset_plugin()
        
        #load layers from geopackage
        gpkg_reg_lyr = self.gpkg_path + "|layername=region"
        self.reg_lyr = QgsVectorLayer(gpkg_reg_lyr, "region", "ogr")
        self.reg_lyr.loadNamedStyle(self.map_region_qml_path)
            
        gpkg_cam_lyr = self.gpkg_path + "|layername=cameras"
        self.cam_lyr = QgsVectorLayer(gpkg_cam_lyr, "cameras", "ogr")           
        self.cam_lyr.loadNamedStyle(self.cam_qml_path)
                        
        gpkg_map_lines_lyr = self.gpkg_path + "|layername=lines"
        self.map_line_lyr = QgsVectorLayer(gpkg_map_lines_lyr, "lines", "ogr")
        self.map_line_lyr.loadNamedStyle(self.map_line_qml_path)
       
        gpkg_img_lines_lyr = self.gpkg_path + "|layername=lines_img"
        self.img_line_lyr = QgsVectorLayer(gpkg_img_lines_lyr, "lines_img", "ogr")
        self.img_line_lyr.loadNamedStyle(self.img_line_qml_path)       

        gpkg_map_gcps_lyr = self.gpkg_path + "|layername=gcps"
        self.map_gcps_lyr = QgsVectorLayer(gpkg_map_gcps_lyr, "gcps", "ogr")
        self.map_gcps_lyr.loadNamedStyle(self.map_gcps_qml_path)       
        
        # # map_gcps_lyr.committedFeaturesAdded.connect(self.map_gcp_added)
        # # self.map_gcps_lyr.selectionChanged.connect(self.map_gcp_selected)
        # # self.map_gcps_lyr.featuresDeleted.connect(self.map_gcp_deleted)
        # self.map_gcps_lyr.geometryChanged.connect(self.map_gcp_changed)
        
        gpkg_img_gcps_lyr = self.gpkg_path + "|layername=gcps_img"
        self.img_gcps_lyr = QgsVectorLayer(gpkg_img_gcps_lyr, "gcps_img", "ogr")
        self.img_gcps_lyr.loadNamedStyle(self.img_gcps_qml_path)       
        
        root = QgsProject.instance().layerTreeRoot()
        monoGroup = root.insertGroup(0, self.project_name)
        monoGroup.addLayer(self.map_line_lyr) 
        monoGroup.addLayer(self.cam_lyr) 
        # monoGroup.addLayer(self.img_line_lyr) 
        monoGroup.addLayer(self.reg_lyr)
        monoGroup.addLayer(self.map_gcps_lyr)
        # monoGroup.addLayer(self.img_gcps_lyr)

        expression = "iid = 'sth_not_existing'"
        self.img_line_lyr.setSubsetString(expression) #show only those lines which correspond to the currently selected image
        self.img_gcps_lyr.setSubsetString(expression)
        self.map_gcps_lyr.setSubsetString(expression)
        
        #define layers which should be shown/considered in which canvas
        self.map_canvas.setLayers([self.map_line_lyr, self.cam_lyr, self.reg_lyr, self.map_gcps_lyr])
        self.map_canvas.setExtent(self.reg_lyr.extent())
        self.map_canvas.refresh()
        
        self.dlg_main.setWindowTitle(self.project_name)
        self.crs = self.cam_lyr.crs()
        # self.activate_buttons()
               
        # self.mono_tool.set_layers(self.img_line_lyr, self.map_line_lyr)
        # self.select_tool.set_layers(self.img_line_lyr, self.map_line_lyr)
        # self.vertex_tool.set_layers(self.img_line_lyr, self.map_line_lyr)
        # self.img_picker_tool.set_layers(self.img_gcps_lyr, self.map_gcps_lyr)
        # self.map_picker_tool.set_layers(self.img_gcps_lyr, self.map_gcps_lyr)
        
        # # self.orient_tool.set_layers(self.img_gcps_lyr, self.map_gcps_lyr)
        
        self.layer_collection = {"reg_lyr":self.reg_lyr,
                                 "cam_lyr":self.cam_lyr,
                                 "img_line_lyr":self.img_line_lyr,
                                 "img_gcps_lyr":self.img_gcps_lyr,
                                 "map_line_lyr":self.map_line_lyr,
                                 "map_gcps_lyr":self.map_gcps_lyr}
        self.dlg_main.set_layers(self.layer_collection)
        
        self.load_mesh()
        
        self.dlg_main.activate_gui_elements()
        
        
        self.load_cameras_from_gpkg()
        
        QApplication.instance().restoreOverrideCursor()
    
    def load_cameras_from_gpkg(self):
        
        cam_feats = self.cam_lyr.getFeatures() 
        for feat in cam_feats:
            
            feat_json = json.loads(QgsJsonUtils.exportAttributes(feat))
            del feat_json["fid"]
            cam = Camera(**feat_json)

            self.camera_collection[cam.iid] = cam
            self.dlg_main.add_camera_to_list(cam)
    
    def load_mesh(self):
            
        reg_feat = list(self.reg_lyr.getFeatures())[0]
        mesh_path = reg_feat["path"]
        if not os.path.exists(mesh_path):
            #!raise Error message that mesh could no be found/loaded
            pass
        
        reg_minx = reg_feat["minx"]
        reg_miny = reg_feat["miny"]
        reg_minz = reg_feat["minz"]
        
        mesh = o3d.io.read_triangle_mesh(mesh_path)
        
        verts = np.asarray(mesh.vertices)
        verts -= [reg_minx, reg_miny, reg_minz]
        tris = np.asarray(mesh.triangles)
        
        o3d_mesh = o3d.geometry.TriangleMesh(o3d.utility.Vector3dVector(verts),
                                             o3d.utility.Vector3iVector(tris))
        o3d_mesh.compute_vertex_normals()
        
        scene = o3d.t.geometry.RaycastingScene()
        scene.add_triangles(o3d.t.geometry.TriangleMesh.from_legacy(o3d_mesh))
        
        self.ray_scene = scene
        self.dlg_main.add_mesh_to_obj_canvas(o3d_mesh, [reg_minx, reg_miny, reg_minz])
        
        # self.mono_tool.set_scene(scene)
        # self.vertex_tool.set_scene(scene)
        
    def reset_plugin(self):
        
        root = QgsProject.instance().layerTreeRoot()
        
        #project_name is only set if anything was loaded; 
        #if its not available than we don't have to do anything
        if self.project_name:
            monoGroup = root.findGroup(self.project_name)
            if monoGroup:
                root.removeChildNode(monoGroup)
            
            # # self.clear_highlighted_features()    
            
            self.map_canvas.refresh()            
            self.map_canvas.setMapTool(self.map_pan_tool)
            
            self.dlg_main.setWindowTitle("moniQue")
    
    def open_convert_dlg(self):
        self.dlg_convert = ConvertDialog(icon_dir=self.icon_dir, parent=self)
        self.dlg_convert.show()
    
    def open_main_dlg(self):
                
        self.dlg_main = MainDialog(plugin_dir=self.plugin_dir, parent=self)
        
        self.dlg_main.camera_collection = self.camera_collection
        self.dlg_main.load_project_signal.connect(self.on_load_project_signal)
        self.dlg_main.close_dialog_signal.connect(self.reset_plugin)
        
        self.dlg_main.show()

