# -*- coding: utf-8 -*-

"""
/***************************************************************************
 VerticalSessions
                                 A QGIS plugin
 Used to Vertical Sessions of Lidar Point Cloud
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-07-05
        copyright            : (C) 2024 Consultoria Técnica de Cartografia da Topocart - BR
        email                : adriano@topocart.com.br; isaac.antunes@topocart.com.br
        git sha              : $Format:%H$
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 This script initializes the plugin, making it known to QGIS.
"""

import math
import os
from datetime import datetime as dt
import matplotlib
from PyQt5.QtCore import QPoint, QThread, pyqtSignal, QRegExp, QVariant
from PyQt5.QtGui import QPainter, QPen, QRegExpValidator
from PyQt5.QtWidgets import QButtonGroup, QHBoxLayout, QSpinBox
from qgis.PyQt.QtCore import QSettings, Qt, QSize, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QCursor, QPixmap, QIcon, QColor, QFont
from qgis.PyQt.QtWidgets import (QAction, QScrollArea, QGridLayout, QPushButton, QComboBox, QLabel, QCheckBox,
                                 QLineEdit, QFrame, QWidget, QSizePolicy, QSpacerItem, QDockWidget, QSplitter)
from qgis.PyQt import QtCore
from qgis.core import (QgsPointXY, QgsFields, QgsField, QgsFeature, QgsPoint,QgsGeometry, QgsProject, QgsWkbTypes,
                       QgsVectorLayer, QgsMapLayerProxyModel)
from qgis.gui import QgsMapToolIdentify, QgsRubberBand, QgsMapLayerComboBox, QgsAdvancedDigitizingDockWidget

plugin_path = os.path.dirname(os.path.dirname(__file__))
# sys.path.append(os.path.join(os.path.join(plugin_path, 'libs')))


class VerticalSessions:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.name_ = 'Vertical Sessions'
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            plugin_path,
            'i18n',
            '{}_{}.qm'.format(self.name_.replace(' ',''), locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes

        self.actions = []
        self.menu = self.tr(f'&T {self.name_}')
        self.dic_prj_conn = {}
        self.dic_icon = {}

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate(self.name_.replace(' ',''), message)

    def add_action(self, icon_path, text, callback, enabled_flag=True, add_to_menu=True, add_to_toolbar=True,
                   status_tip=None, whats_this=None, parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon()
        icon.addPixmap(QPixmap(icon_path))
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        print('initGui')
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        # self.dock = QDockWidget('T - Inventário de Via.')

        self.dock = QgsAdvancedDigitizingDockWidget(self.iface.mapCanvas())
        self.dock.setWindowTitle(f'T - {self.name_}.')

        self.inv_wd = WdVS(self.iface, parent=self.dock, main=self)
        self.inv_wd.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.dock.setWidget(self.inv_wd)
        self.dock.setObjectName(f"{self.name_} Panel")
        self.dock.setMinimumHeight(60)

        self.iface.addDockWidget(Qt.BottomDockWidgetArea, self.dock)
        icon_path = os.path.join(plugin_path, 'icons/icon_vs.png')

        self.add_action(
            icon_path,
            text=self.tr(''),
            callback=self.call_vs,
            parent=self.iface.mainWindow())

        self.first_start = True


    def unload(self):
        print('unload')
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(f'T - {self.name_}.'),
                action)
            self.iface.removeToolBarIcon(action)
        for x in self.iface.mainWindow().findChildren(QDockWidget):
            if x.objectName() == f"{self.name_} Panel":
                del x

    def call_vs(self):
        if not self.dock.isVisible():
            self.dock.setVisible(True)
        # self.inv_wd.get_inv()

    def plugin_version(self):
        meta_file = plugin_path + "/metadata.txt"
        # print(meta_file)
        with open(meta_file) as meta:
            mt = meta.readlines()
            for l_ in mt:
                if l_[:8] == "version=":
                    return l_[8:].replace('\n', '')
        return '0.0'


class WdVS(QWidget):
    def __init__(self, iface, parent=None, main=None):
        super(WdVS, self).__init__(parent)

        self.filter_dlg = None
        self.setObjectName('WdVS')
        # Save reference to the QGIS interface
        self.iface = iface
        self.parent = parent
        self.main = main
        if os.getlogin() == 'adriano.caliman' or os.getlogin() == 'issac.antunes':
            self.iface.actionShowPythonDialog().trigger()
        self.dic_layers_tools = {
            'fp': {},
            'fp_stats': {},
            }
        self.config_rubber_bands()
        self.canvas = self.iface.mapCanvas()

        self.dic_current_cell = {'row': 0, 'color': None}

        lg = self.create_layout()
        self.setLayout(lg)

        self.cloud_thread = None
        self.cloud_node_group = None
        self.w_ = 2
        self.z_ = None
        self.cloud_layer_catalog = None
        self.dic_cur_ax = {}

    def create_layout(self):
        gl_prof = QGridLayout()
        gl_prof.setContentsMargins(0, 0, 0, 0)
        gl_prof.setSpacing(1)
        spt_left = QSplitter(Qt.Horizontal)
        # gl_.addWidget(spt_left, 1, 0)
        gl_prof.addWidget(spt_left, 0, 0)

        wd_tool = QWidget()
        sp_ = QSizePolicy()
        sp_.setHorizontalPolicy(QSizePolicy.Minimum)
        sp_.setHorizontalStretch(0)
        sp_.setVerticalPolicy(QSizePolicy.Expanding)
        wd_tool.setSizePolicy(sp_)
        gl_tool = QGridLayout()
        gl_tool.setContentsMargins(0, 0, 0, 0)
        wd_tool.setLayout(gl_tool)
        spt_left.addWidget(wd_tool)

        self.lb_session_logo = QLabel()
        self.lb_session_logo.setFixedSize(QSize(20, 20))
        pixmap_ = QPixmap(os.path.join(plugin_path, 'icons/icon_vs.png'))
        scaled_ = pixmap_.scaled(self.lb_session_logo.size(), QtCore.Qt.KeepAspectRatio)
        self.lb_session_logo.setPixmap(scaled_)
        r_ = 0
        gl_tool.addWidget(self.lb_session_logo, r_, 0)

        self.lb_topo_logo = QLabel('Footprint')
        self.lb_topo_logo.setFixedSize(QSize(35, 15))
        pixmap_ = QPixmap(os.path.join(plugin_path, 'icons/topo_logo.png'))
        scaled_ = pixmap_.scaled(self.lb_topo_logo.size(), QtCore.Qt.KeepAspectRatio)
        self.lb_topo_logo.setPixmap(scaled_)
        gl_tool.addWidget(self.lb_topo_logo, r_, 1)

        gl_tool.addItem(QSpacerItem(0, 0, QSizePolicy.Expanding, QSizePolicy.Minimum), r_, 2)

        self.lb_version = QLabel(f'v{self.main.plugin_version()}')
        self.lb_version.setAlignment(Qt.AlignRight)
        gl_tool.addWidget(self.lb_version, r_, 3)

        self.pb_def_fp = QPushButton('Set Session Footprint')
        self.pb_def_fp.setIcon(QIcon(os.path.join(plugin_path, f'icons/icon_session_fp.png')))
        r_ += 1
        gl_tool.addWidget(self.pb_def_fp, r_, 0, 1, 4)

        self.chk_hand_width = QCheckBox('Fixed Width:')
        self.chk_hand_width.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        r_ += 1
        gl_tool.addWidget(self.chk_hand_width, r_, 1)

        self.le_hand_width = QLineEdit()
        self.le_hand_width.setEnabled(False)
        self.le_hand_width.setText('10.0')
        self.le_hand_width.setMaximumWidth(50)
        self.le_hand_width.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        validator = QRegExpValidator(QRegExp(r'[0-9].+'))
        self.le_hand_width.setValidator(validator)
        gl_tool.addWidget(self.le_hand_width, r_, 2, 1, 2)

        self.chk_hand_length = QCheckBox('Fixed Length:')
        self.chk_hand_length.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        r_ += 1
        gl_tool.addWidget(self.chk_hand_length, r_, 1)

        self.le_hand_length = QLineEdit()
        self.le_hand_length.setEnabled(False)
        self.le_hand_length.setText('100.0')
        self.le_hand_length.setMaximumWidth(50)
        self.le_hand_length.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        validator = QRegExpValidator(QRegExp(r'[0-9].+'))
        self.le_hand_length.setValidator(validator)
        gl_tool.addWidget(self.le_hand_length, r_, 2, 1, 2)

        f_ = QFrame()
        f_.setFrameShape(QFrame.HLine)
        r_ += 1
        gl_tool.addWidget(f_, r_, 1, 1, 3)

        self.btn_group = QButtonGroup()
        self.pb_previous = QPushButton()
        self.btn_group.addButton(self.pb_previous,id=-10)
        self.pb_previous.setIcon(QIcon(os.path.join(plugin_path, f'icons/icon_previous.png')))
        self.pb_previous.setToolTip('Previous/ Left Session')
        self.pb_previous.setMaximumWidth(50)
        r_ += 1
        gl_tool.addWidget(self.pb_previous, r_, 1, 1, 1)

        self.pb_next = QPushButton()
        self.btn_group.addButton(self.pb_next, id=10)
        self.pb_next.setIcon(QIcon(os.path.join(plugin_path, f'icons/icon_next.png')))
        self.pb_next.setToolTip('Next/ Right Session')
        self.pb_next.setMaximumWidth(50)
        # r_ += 1
        gl_tool.addWidget(self.pb_next, r_, 2, 1, 2)

        self.lb_interval = QLabel('Interval:')
        self.lb_interval.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        r_ += 1
        gl_tool.addWidget(self.lb_interval, r_, 1)

        self.le_interval = QLineEdit()
        self.le_interval.setEnabled(True)
        self.le_interval.setText('10.0')
        self.le_interval.setMaximumWidth(50)
        self.le_interval.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        validator = QRegExpValidator(QRegExp(r'[0-9].+'))
        self.le_interval.setValidator(validator)
        gl_tool.addWidget(self.le_interval, r_, 2, 1, 2)

        self.chk_follow_line = QCheckBox('Follow Alignment:')
        self.chk_follow_line.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        r_ += 1
        gl_tool.addWidget(self.chk_follow_line, r_, 1)

        r_ += 1
        lb_ = QLabel('Layer:')
        gl_tool.addWidget(lb_, r_, 1, 1, 1)
        self.mlcb_layer = QgsMapLayerComboBox(self)
        self.mlcb_layer.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.mlcb_layer.setEnabled(False)
        self.mlcb_layer.setToolTip('It need be a line Layer')
        gl_tool.addWidget(self.mlcb_layer, r_, 1, 1, 3)

        self.pb_get_feature = QPushButton('Select Feature')
        self.pb_get_feature.setIcon(QIcon(os.path.join(plugin_path, f'icons/icon_axis.png')))
        self.pb_get_feature.setEnabled(False)
        r_ += 1
        gl_tool.addWidget(self.pb_get_feature, r_, 1, 1, 2)

        self.le_cur_step = QLineEdit()
        self.le_cur_step.setEnabled(False)
        self.le_cur_step.setText('-')
        self.le_cur_step.setMaximumWidth(50)
        self.le_cur_step.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        validator = QRegExpValidator(QRegExp(r'[0-9].+'))
        self.le_cur_step.setValidator(validator)
        gl_tool.addWidget(self.le_cur_step, r_, 3)

        r_ += 1
        gl_tool.addItem(QSpacerItem(10, 10, QSizePolicy.Minimum, QSizePolicy.Expanding), r_, 0)

        wd_layer= QWidget()
        gl_layer= QGridLayout()
        gl_layer.setContentsMargins(0, 0, 0, 0)
        wd_layer.setLayout(gl_layer)
        spt_left.addWidget(wd_layer)

        rr_ = 0
        gl_layer.addWidget(QLabel('Symbology Settings:'), rr_, 1, 1, 2)

        lh_ = QHBoxLayout()
        lh_.setContentsMargins(0, 0, 0, 0)
        lh_.setSpacing(1)
        lh_.addWidget(QLabel('Point Size:'))
        self.spb_point_size = QSpinBox()
        self.spb_point_size.setValue(3)
        self.spb_point_size.setMaximumWidth(45)
        lh_.addWidget(self.spb_point_size)
        rr_ += 1
        gl_layer.addLayout(lh_, rr_, 1, 1, 2)

        rr_ += 1
        gl_layer.addWidget(QLabel('Symbology:'), rr_, 1, 1, 2)

        self.cmb_pers_symb = QComboBox()
        self.cmb_pers_symb.setEnabled(True)
        self.cmb_pers_symb.addItems(['RGB', 'CLASS', 'Intensity', 'Elevation', 'RGB + Intensity'])
        # self.cmb_pers_symb.addItems(['Intensity', 'RGB', 'CLASS', 'Elevation', 'Point Source ID', 'PSID + Intensity'])
        self.cmb_pers_symb.setCurrentText('CLASS')
        rr_ += 1
        gl_layer.addWidget(self.cmb_pers_symb, rr_, 1)

        rr_ += 1
        gl_layer.addItem(QSpacerItem(10, 10, QSizePolicy.Expanding, QSizePolicy.Expanding), rr_, 0)

        rr_ += 1
        gl_layer.addWidget(QLabel('Density (ppm):'), rr_, 1, 1, 2)

        lh_ = QHBoxLayout()
        self.le_max_dens = QLineEdit()
        self.le_max_dens.setEnabled(False)
        self.le_max_dens.setText('10.0')
        lh_.addWidget(self.le_max_dens)
        self.pb_set_dens = QPushButton('Get')
        self.pb_set_dens.setIcon(QIcon(os.path.join(plugin_path, f'icons/icon_session_stats.png')))
        self.pb_set_dens.setMaximumWidth(45)
        lh_.addWidget(self.pb_set_dens)
        rr_ += 1
        gl_layer.addLayout(lh_, rr_, 1, 1, 2)

        f_ = QFrame()
        f_.setFrameShape(QFrame.VLine)
        rr_ += 1
        gl_layer.addWidget(f_, 0, 0, rr_, 1)

        wd_view = QWidget()
        sp_ = QSizePolicy()
        sp_.setHorizontalPolicy(QSizePolicy.Expanding)
        sp_.setHorizontalStretch(255)
        sp_.setVerticalPolicy(QSizePolicy.Expanding)
        wd_view.setSizePolicy(sp_)
        gl_view = QGridLayout()
        gl_view.setContentsMargins(0, 0, 0, 0)
        wd_view.setLayout(gl_view)
        spt_left.addWidget(wd_view)

        f_ = QFrame()
        f_.setFrameShape(QFrame.VLine)
        gl_view.addWidget(f_, 0, 0, 2, 1)

        self.wdl_prof = ProfileView(self, 'Rec. Prof.')
        self.wdl_prof.resize(300, 150)
        self.wdl_prof.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        gl_view.addWidget(self.wdl_prof, 0, 1)

        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setLayout(gl_prof)
        lg_sa = QGridLayout()
        lg_sa.setContentsMargins(0, 0, 0, 0)
        lg_sa.setSpacing(0)
        lg_sa.addWidget(scroll_area)

        self.trigger_actions()
        return lg_sa

    def trigger_actions(self):
        self.btn_group.buttonClicked[int].connect(self.move_fp)
        self.pb_def_fp.clicked.connect(self.get_fp)
        self.chk_hand_width.toggled.connect(lambda is_chk: self.le_hand_width.setEnabled(is_chk))
        self.chk_hand_length.toggled.connect(lambda is_chk: self.le_hand_length.setEnabled(is_chk))
        self.mlcb_layer.layerChanged.connect(self.clear_dic_ax)
        self.pb_get_feature.clicked.connect(self.clear_dic_ax)
        self.chk_follow_line.toggled.connect(self.enable_follow_line)
        self.spb_point_size.valueChanged.connect(self.update_session_painter)
        self.cmb_pers_symb.currentTextChanged.connect(self.wdl_prof.update)
        self.pb_set_dens.clicked.connect(self.get_max_dens_fp)
        self.pb_get_feature.clicked.connect(self.select_tool)
        QgsProject.instance().layersAdded.connect(self.layer_added)
        QgsProject.instance().layersRemoved.connect(self.layer_removed)

    def config_rubber_bands(self):

        rb_ = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.LineGeometry)
        rb_.setColor(QColor(0, 0, 255, 90))
        rb_.setLineStyle(Qt.DotLine)
        rb_.setWidth(5)
        self.dic_layers_tools['track'] = {'rb': rb_}

        rb_ = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.PolygonGeometry)
        rb_.setWidth(2)
        rb_.setColor(QColor(0, 255, 255, 50))
        color_ = QColor(Qt.lightGray)
        color_.setAlpha(90)
        rb_.setFillColor(color_)
        self.dic_layers_tools['rec'] = {'rb': rb_}

        rb_ = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.LineGeometry)
        rb_.setColor(Qt.green)
        rb_.setLineStyle(Qt.DashLine)
        rb_.setWidth(2)
        self.dic_layers_tools['line'] = {'rb': rb_}

    def get_fp(self):
        print('get_fp')
        if 'maptool' not in self.dic_layers_tools['fp']:
            mt_ = SelectFootPrint(self.iface, parent=self)
            self.dic_layers_tools['fp']['maptool'] = mt_
        else:
            mt_ = self.dic_layers_tools['fp']['maptool']
        self.iface.mapCanvas().setMapTool(mt_)

    def select_fp(self, dic_):
        print('select_fp')
        self.dic_layers_tools['track']['rb'].reset()
        if 'line' in dic_:
            self.dic_layers_tools['line']['rb'].reset()
            self.dic_layers_tools['line']['rb'].setToGeometry(dic_['line'])
        if 'rec' in dic_:
            self.dic_layers_tools['rec']['rb'].reset()
            self.dic_layers_tools['rec']['rb'].setToGeometry(dic_['rec'])
        if 'identify' in dic_:
            stats_ = True if dic_['identify'] == 'stats' else False
            self.identify_cloud(stats=stats_)
        if 'reset' in dic_:
            self.dic_layers_tools['rec']['rb'].reset()
            self.dic_layers_tools['line']['rb'].reset()
            self.wdl_prof.p_track = None
            self.wdl_prof.update()

    def get_max_dens_fp(self):
        print('get_fp')
        if 'maptool' not in self.dic_layers_tools['fp_stats']:
            mt_ = SelectFootPrint(self.iface, parent=self, stats=True)
            self.dic_layers_tools['fp_stats']['maptool'] = mt_
        else:
            mt_ = self.dic_layers_tools['fp_stats']['maptool']
        self.iface.mapCanvas().setMapTool(mt_)

    def move_fp(self, bt_):
        sign_ = int(bt_ / 10)
        print('move_fp', bt_)
        if self.chk_follow_line.isChecked() and self.mlcb_layer.currentLayer():
            if not self.dic_cur_ax:
                if self.mlcb_layer.currentLayer().selectedFeatureCount():
                    feats_ = self.mlcb_layer.currentLayer().getSelectedFeatures()
                else:
                    feats_ = self.mlcb_layer.currentLayer().getFeatures()
                dic_data = {}
                for i, feat_ in enumerate(feats_):
                    id_ = feat_.id()
                    geom_ = feat_.geometry()
                    len_ = geom_.length()
                    dic_data.update({id_: len_})
                    # print(id_, len_)
                self.dic_cur_ax['dic_len'] = dic_data
                self.dic_cur_ax['id'] = list(dic_data)[0]

            v_ = float(self.le_interval.text())
            step_ = 0.0 if self.le_cur_step.text() == '-' else float(self.le_cur_step.text()) + sign_ * v_
            id_ = self.dic_cur_ax['id']
            if step_ <= 0 and self.le_cur_step.text() != '-' or self.dic_cur_ax['dic_len'][id_] <= step_:
                if float(self.le_cur_step.text()) == 0 or float(self.le_cur_step.text()) == self.dic_cur_ax['dic_len'][id_]:
                    print('next feature')
                    temp = list(self.dic_cur_ax['dic_len'])
                    try:
                        id_ = temp[temp.index(id_ + sign_)]
                        if step_ <= 0:
                            step_ = self.dic_cur_ax['dic_len'][id_]
                        else:
                            step_ = 0
                    except (ValueError, IndexError):
                        return
                else:
                    if step_ <= 0:
                        step_ = 0
                    else:
                        step_ = self.dic_cur_ax['dic_len'][id_]
            self.le_cur_step.setText(f'{step_:0.1f}')
            geom_ = self.mlcb_layer.currentLayer().getFeature(id_).geometry()
            pi_ = geom_.interpolate(step_).asPoint()
            az_ = math.degrees(geom_.interpolateAngle(step_))
            l_ = float(self.le_hand_length.text()) / 2
            p1 = calc_proj(pi_, az_ - 90, l_)
            p2 = calc_proj(pi_, az_ + 90, l_)
            line_ = QgsGeometry().fromPolylineXY([p1, p2])
            az_ += 90

        elif not self.dic_layers_tools['line']['rb'].asGeometry().isNull():
            geom_ = self.dic_layers_tools['line']['rb'].asGeometry()
            stt_point = QgsPoint(geom_.asPolyline()[0])
            end_point = QgsPoint(geom_.asPolyline()[-1])
            az_ = stt_point.azimuth(end_point)
            i_ = float(self.le_interval.text())
            p1 = calc_proj(stt_point, (90 * bt_/10) + az_, i_)
            p2 = calc_proj(end_point, (90 * bt_/10) + az_, i_)
            line_ = QgsGeometry().fromPolylineXY([p1, p2])
        else:
            return
        d_ = float(self.le_hand_width.text()) / 2
        pr1 = calc_proj(p1, az_ - 90, d_)
        pr2 = calc_proj(p2, az_ - 90, d_)
        pr3 = calc_proj(p2, az_ + 90, d_)
        pr4 = calc_proj(p1, az_ + 90, d_)
        pol_ = QgsGeometry.fromPolygonXY([[pr1, pr2, pr3, pr4,]])
        self.select_fp({'line': line_, 'rec': pol_, 'identify': True})

    def enable_follow_line(self, is_chk):
        self.mlcb_layer.setEnabled(is_chk)
        self.pb_get_feature.setEnabled(is_chk)
        self.le_cur_step.setEnabled(is_chk)

    def identify_cloud(self, stats=False):
        print('identify_cloud')
        list_rll = []

        pol_ = self.dic_layers_tools['rec']['rb'].asGeometry()
        list_ = pol_.asMultiPolygon()[0][0]
        lg_1 = QgsGeometry().fromPolyline([QgsPoint(list_[3]), QgsPoint(list_[0])])
        lg_2 = QgsGeometry().fromPolyline([QgsPoint(list_[3]), QgsPoint(list_[2])])
        list_rll.append([0, lg_1, lg_2])
        dic_ = {'rec': pol_, 'rll': list_rll}

        if not self.cloud_layer_catalog:
            self.vrt_cl_catalog()
        list_layer = []
        for feat_ in self.cloud_layer_catalog.getFeatures():
            if feat_.geometry().intersects(pol_):
                layer_name = feat_['nome_layer']
                list_layer.append(QgsProject.instance().mapLayersByName(layer_name)[0])

        if self.cloud_thread:
            self.cloud_thread.stop = True
        self.cloud_thread = CloudThread(main=self.main, parent=self, dic_=dic_, list_layer=list_layer, stats=stats)
        self.cloud_thread.start()

    def vrt_cl_catalog(self):
        print('vrt_cl_catalog')
        layers_ = QgsProject.instance().mapLayersByName('SV_clouds_tiles')
        ls_ = QgsProject.instance().mapLayers()
        if layers_:
            self.cloud_layer_catalog = layers_[0]
        else:
            canva_crs = QgsProject.instance().crs().authid()
            self.cloud_layer_catalog = QgsVectorLayer('Polygon?crs={}&index=yes'.format(canva_crs), 'SV_clouds_tiles', "memory")
            fields_catalog = QgsFields()
            fields_catalog.append(QgsField('nome_layer', QVariant.String))
            pr_ = self.cloud_layer_catalog.dataProvider()
            pr_.addAttributes(fields_catalog)
            self.cloud_layer_catalog.updateFields()

            style_path = os.path.join(plugin_path, 'styles', f'sv_clouds_tiles.qml')
            self.cloud_layer_catalog.loadNamedStyle(style_path)
            QgsProject.instance().addMapLayer(self.cloud_layer_catalog, False)
            self.cloud_node_group = QgsProject.instance().layerTreeRoot().findGroup('__CLOUD__')
            if not self.cloud_node_group:
                self.cloud_node_group = QgsProject.instance().layerTreeRoot().insertGroup(-1, '__CLOUD__')
            self.cloud_node_group.addLayer(self.cloud_layer_catalog)
        for ln_ in ls_:
            layer_ = ls_[ln_]
            self.layer_added([layer_])

        return

    def layer_added(self, slot_):
        if self.cloud_layer_catalog:
            pr_ = self.cloud_layer_catalog.dataProvider()
            self.cloud_layer_catalog.startEditing()
            for layer_ in slot_:
                if layer_.type() == 6:
                    rec_ = layer_.extent()
                    geom_ = QgsGeometry().fromRect(rec_)
                    feat_ = QgsFeature()
                    feat_.setGeometry(geom_)
                    feat_.setAttributes([layer_.name()])
                    pr_.addFeatures([feat_])
                    self.cloud_layer_catalog.updateExtents()
            self.cloud_layer_catalog.commitChanges()
            self.cloud_layer_catalog.triggerRepaint()

    def layer_removed(self, slot_):
        print('layer_removed', slot_)
        for layer_ in slot_:
            if layer_[:len('SV_clouds_tiles')] == 'SV_clouds_tiles':
                self.cloud_layer_catalog = None
                return
        if self.cloud_layer_catalog:
            self.cloud_layer_catalog.startEditing()
            for layer_ in slot_:
                for feat_cat in self.cloud_layer_catalog.getFeatures():
                    print(feat_cat['nome_layer'], layer_[:len(feat_cat['nome_layer'])])
                    if feat_cat['nome_layer'].replace('-', '_').replace('.', '_') == layer_[:len(feat_cat['nome_layer'])]:
                        self.cloud_layer_catalog.deleteFeature(feat_cat.id())
                        print('removed')
                        break
                self.cloud_layer_catalog.updateExtents()
            self.cloud_layer_catalog.commitChanges()
            self.cloud_layer_catalog.triggerRepaint()

    def set_track_rb(self, lt_=None):
        self.dic_layers_tools['track']['rb'].reset()
        if lt_:
            self.dic_layers_tools['track']['rb'].setToGeometry(lt_)

    def clear_dic_ax(self):
        self.le_cur_step.setText('-')
        self.dic_cur_ax = {}

    def update_session_painter(self, v_=None):
        print('update_session_paint', v_)
        if self.wdl_prof:
            self.wdl_prof.update()

    def select_tool(self):
        self.iface.actionSelect().trigger()


class SelectFootPrint(QgsMapToolIdentify):
    """ Select Photo on map """

    def __init__(self, iface, parent=None, stats=None):
        QgsMapToolIdentify.__init__(self, iface.mapCanvas())
        self.canvas = iface.mapCanvas()
        self.iface = iface
        self.parent = parent
        self.cursor = QCursor(QPixmap(["16 16 3 1",
                                       "      c None",
                                       ".     c #FF0000",
                                       "+     c #FFFFFF",
                                       "                ",
                                       "+               ",
                                       "+               ",
                                       "+          ..   ",
                                       "+         .  ...",
                                       "+         .     ",
                                       "+         .     ",
                                       "+        .      ",
                                       "+       .       ",
                                       "+     ..        ",
                                       "+   ..          ",
                                       "+  .            ",
                                       "+ .             ",
                                       "+.              ",
                                       "+               ",
                                       "++++++++++++++++"]))
        self.p1 = None
        self.p2 = None
        self.p3 = None
        self.stats = stats

    def activate(self):
        self.canvas.setCursor(self.cursor)

    def canvasPressEvent(self, evt_):
        print('canvasPressEvent')
        if not self.p1:
            print('set p1')
            self.parent.select_fp({'reset': True})
            self.p1 = self.toMapCoordinates(evt_.pos())
            if self.stats:
                self.geom_r = self.buffer_point()
                self.parent.select_fp({'rec': self.geom_r, 'identify': 'stats'})
                self.p1 = None

        elif self.p3:
            print('reset')
            self.parent.select_fp({'reset': True})
            self.p1 = self.toMapCoordinates(evt_.pos())
            self.p2 = None
            self.p3 = None

    def canvasMoveEvent(self, evt_):
        if self.p3:
            pass
        elif self.p2:
            d_ = self.geom_l.distance(QgsGeometry(QgsPoint(self.toMapCoordinates(evt_.pos()))))
            # self.geom_r = self.geom_l.buffer(d_, -1)
            self.geom_r = self.buffer_line(d_)
            self.parent.select_fp({'rec': self.geom_r})
        elif self.p1:
            self.geom_l = QgsGeometry().fromPolylineXY([self.p1, self.toMapCoordinates(evt_.pos())])
            self.parent.select_fp({'line':self.geom_l})

    def canvasReleaseEvent(self, evt_):
        print('canvasReleaseEvent', self.stats)
        if self.p2 and not self.parent.chk_hand_width.isChecked():
            print('set p3')
            self.p3 = self.toMapCoordinates(evt_.pos())
            w_ = self.geom_l.distance(QgsGeometry(QgsPoint(self.p3)))
            self.parent.le_hand_width.setText(f'{w_:0.1f}')
            # self.geom_r = self.geom_l.buffer(d_, -1)
            self.geom_r = self.buffer_line(w_)
            # self.geom_r = self.geom_l.buffer(d_, 5, QgsGeometry.CapFlat)
            self.parent.select_fp({'rec': self.geom_r, 'identify': True})
        elif self.p1 and self.toMapCoordinates(evt_.pos()) != self.p1:
            print('set p2')
            self.p2 = self.toMapCoordinates(evt_.pos())
            if self.parent.chk_hand_length.isChecked() and self.parent.le_hand_length.text():
                l_ = float(self.parent.le_hand_length.text())
                az_ = self.p1.azimuth(self.p2)
                self.p2 = calc_proj(self.p1, az_, l_)
            else:
                l_ = self.p1.distance(self.p2)
                self.parent.le_hand_length.setText(f'{l_:0.1f}')
            self.geom_l = QgsGeometry().fromPolylineXY([self.p1, self.p2])
            self.parent.select_fp({'line': self.geom_l})
            if self.parent.chk_hand_width.isChecked():
                w_ = float(self.parent.le_hand_width.text()) / 2
                self.geom_r = self.buffer_line(w_)
                self.parent.select_fp({'rec': self.geom_r, 'identify': True})
                self.p3 = True

        # self.parent.select_fp(self.pointXY)

    def buffer_line(self, d_):
        az_ = self.p1.azimuth(self.p2)
        return QgsGeometry.fromPolygonXY([[
            calc_proj(self.p1, az_ - 90, d_),
            calc_proj(self.p2, az_ - 90, d_),
            calc_proj(self.p2, az_ + 90, d_),
            calc_proj(self.p1, az_ + 90, d_),
            # calc_proj(self.p1, az_ - 90),
        ]])

    def buffer_point(self, az_=0, d_=1):
        p1_ = calc_proj(self.p1, az_ + 45, d_ / 2 * (2 ** 0.5))
        p2_ = calc_proj(p1_, az_ + 180 , d_)
        p3_ = calc_proj(p2_, az_ - 90, d_)
        p4_ = calc_proj(p3_, az_, d_)
        return QgsGeometry.fromPolygonXY([[p1_, p2_, p3_, p4_,]])


class ProfileView(QWidget):
    def __init__(self, parent, type=None):
        super().__init__(parent)
        self.parent = parent
        self.type = type
        self.mp_ = None
        self.setAttribute(Qt.WA_StyledBackground, True)
        self.setStyleSheet('background-color: lightgray;')
        self.setMouseTracking(True)
        self.x_value = 0
        self.y_value = 999999
        self.p1 = None
        self.p2 = None
        self.fps = 18
        self.list_spiral = []
        self.dic_t = {}
        self.dic_l = {}
        self.p_track = None

    def paintEvent(self, paint_event):
        print('paintEvent', dt.now())
        painter = QPainter(self)

        pen = QPen()
        ps_ = int(self.parent.spb_point_size.value())
        pen.setWidth(ps_)

        font_ = QFont()
        font_.setPixelSize(self.fps)
        painter.setFont(font_)

        painter.setRenderHint(QPainter.Antialiasing, True)
        if self.dic_t:
            for ln_ in self.dic_l:
                dic_c = self.dic_l[ln_]['color'] # class colors
                dic_d = self.dic_l[ln_]['pcs'] # point cloud data
                for coord_ in dic_d:
                    if not dic_d:
                        break
                    min_ = int(self.dic_t['I_min'])
                    max_ = int(self.dic_t['I_max'])
                    colourmap = matplotlib.cm.get_cmap('coolwarm')
                    normalize = matplotlib.colors.Normalize(0, self.height())
                    for dep_ in dic_d[coord_]:
                        if not dic_d:
                            break
                        d_ = dic_d[coord_][dep_]
                        c_ = d_['C']
                        # if c_ in dic_c:
                        if self.parent.cmb_pers_symb.currentText() == 'CLASS':
                            pen.setColor(dic_c[c_])
                        elif self.parent.cmb_pers_symb.currentText() == 'RGB':
                            pen.setColor(QColor(d_['R'], d_['G'], d_['B']))
                        elif (self.parent.cmb_pers_symb.currentText() == 'RGB + Intensity' or
                              self.parent.cmb_pers_symb.currentText() == 'Intensity'):
                            i_ = int(((d_['I'] - min_) / (max_ - min_)) * 255)
                            if self.parent.cmb_pers_symb.currentText() == 'RGB + Intensity':
                                pen.setColor(QColor(d_['R'], d_['G'], d_['B'], i_))
                            else:
                                pen.setColor(QColor(i_, i_, i_))
                        elif self.parent.cmb_pers_symb.currentText() == 'Elevation':
                            tup_ = colourmap(normalize(d_['Y']))
                            pen.setColor(QColor(int((1-tup_[0])*255), int((1-tup_[1])*255), int((1-tup_[2])*255)))
                            # 'Turbo'
                        pen.setCapStyle(Qt.FlatCap)
                        painter.setPen(pen)
                        painter.drawPoint(d_['X'], d_['Y'])

        if not self.mp_:
            self.mp_ = QPoint(int(self.width() / 2), self.height())
        if self.parent.z_ is not None and self.dic_t:
            y_ = int(self.height() * (1  - (self.parent.z_ - self.dic_t['Z_min']) / (self.dic_t['Z_max'] - self.dic_t['Z_min'])))
            pen.setColor(QColor(255, 255, 255))
            pen.setCapStyle(Qt.FlatCap)
            painter.setPen(pen)
            painter.drawLine(self.mp_.x() - 5, y_, self.mp_.x() + 5, y_)
            painter.drawText(self.mp_.x() + 15, y_, str(round(self.parent.z_, 3)))
        if self.p1 and self.p2 and self.dic_t:
            pen.setWidth(1)
            pen.setColor(QColor(255, 0, 165))
            pen.setCapStyle(Qt.FlatCap)
            painter.setPen(pen)
            yi, yf, x_yi, x_yf = [self.p1.y(), self.p2.y(), self.p1.x(), self.p2.x()] \
                if self.p1.y() < self.p2.y() else [self.p2.y(), self.p1.y(), self.p2.x(), self.p1.x()]
            yi = min(max(0, yi), self.height())
            yf = min(max(0, yf), self.height())
            x_yi = min(max(0, x_yi), self.width())
            x_yf = min(max(0, x_yf), self.width())
            painter.drawLine(x_yi, yf, x_yf, yf)
            dd = abs((x_yf - x_yi) * (self.dic_t['D_max'] - self.dic_t['D_min']) / self.width())
            painter.drawText(int((x_yi + x_yf) / 2), max(yf, self.fps), f' {round(dd, 3)}')

            pen.setColor(QColor(255, 165, 0))
            painter.setPen(pen)
            painter.drawLine(x_yi, yi, x_yi, yf)
            dz = (yf - yi) * (self.dic_t['Z_max'] - self.dic_t['Z_min']) / self.height()

            painter.setPen(pen)
            painter.drawText(min(x_yi, self.width() - self.fps * 3), max(self.fps, int((yi + yf) / 2)), f' {round(dz, 3)}')

        pen.setWidth(15)
        pen.setBrush(QColor(255, 0, 0, 100))
        pen.setCapStyle(Qt.RoundCap)
        painter.setPen(pen)
        painter.drawPoint(self.mp_)

        if self.p_track:
            pen.setWidth(5)
            pen.setStyle(Qt.DotLine)
            pen.setBrush(QColor(0, 0, 255, 90))

            painter.setPen(pen)
            painter.drawLine(self.p_track.x(), 0, self.p_track.x(), self.height())
        if self.dic_t:
            pen.setBrush(QColor(255, 255, 0))
            painter.setPen(pen)
            painter.drawText(0, 20, str(round(self.dic_t['Z_max'], 3)))
            painter.drawText(0, self.height(), str(round(self.dic_t['Z_min'], 3)))

            pen.setWidth(15)
            pen.setBrush(QColor(255, 0, 0, 100))
            painter.setPen(pen)
            z_ = self.dic_t['Z_min'] + (1 - self.mp_.y() / self.height()) * (self.dic_t['Z_max'] - self.dic_t['Z_min'])
            painter.drawText(self.mp_.x() + 15, self.mp_.y(), str(round(z_, 3)))

    def mouseReleaseEvent(self, evt_):
        if self.p1 == evt_.pos():
            if not self.dic_l:
                return
            self.mp_ = evt_.pos()
            self.x_value = (self.dic_t[f'D_min'] +
                            self.mp_.x() / self.width() * (self.dic_t[f'D_max'] - self.dic_t[f'D_min']))
            y_ = (self.dic_t['Z_min'] +
                            (1 - self.mp_.y() / self.height()) * (self.dic_t['Z_max'] - self.dic_t['Z_min']))
            if y_ < self.y_value:
                self.y_value = y_
            # if not self.list_spiral:
            #     self.list_spiral = self.calc_spiral()
        else:
            self.p2 = evt_.pos()
        self.update()

    def mousePressEvent(self, evt_):
        if self.p2:
            self.p2 = None
        self.p1 = evt_.pos()

    def mouseMoveEvent(self, evt_):
        if not self.dic_l:
            return
        if self.p1 and evt_.buttons() != Qt.NoButton:
            self.p2 = evt_.pos()
        if 'LL' in self.dic_t:
            if 'track' not in self.dic_t:
                ll_ = self.dic_t['LL'].asPolyline()
                pt0_ = QgsPoint(ll_[0])
                pt1_ = QgsPoint(ll_[1])
                az_ = pt0_.azimuth(pt1_)
                len_l = pt0_.distance(pt1_)
                self.dic_t['track'] = {'az': az_, 'len_l': len_l}
                lf_ = self.dic_t['LF'].asPolyline()
                pt2_ = QgsPoint(lf_[0])
                pt3_ = QgsPoint(lf_[1])
                len_f = pt2_.distance(pt3_)
                self.dic_t['track']['len_f'] = len_f
            else:
                az_ = self.dic_t['track']['az']
                len_l = self.dic_t['track']['len_l']
                len_f = self.dic_t['track']['len_f']
            perc_1 = evt_.pos().x() / self.width()
            perc_2 = self.dic_t['D_min'] + (self.dic_t['D_max'] - self.dic_t['D_min']) * perc_1
            pt2_ = self.dic_t['LF'].interpolate(perc_2).get()
            if not pt2_:
                return
            x_ = pt2_.x()
            y_ = pt2_.y()
            az1 = math.radians(az_)
            pt3_ =  QgsPoint(x_ + len_l * math.sin(az1), y_ + len_l * math.cos(az1))
            lg_1 = QgsGeometry().fromPolyline([pt2_, pt3_])
            self.p_track = evt_.pos()
            self.parent.set_track_rb(lg_1)
        self.update()

    def mouseDoubleClickEvent(self, a0):
        self.y_value = (self.dic_t['Z_min'] +
              (1 - self.mp_.y() / self.height()) * (self.dic_t['Z_max'] - self.dic_t['Z_min']))

    def reset(self):
        self.list_ = []
        self.dic_t = {}
        self.dic_d = {}
        self.x_value = 0
        self.y_value = 999999
        self.update()

    # @staticmethod
    # def calc_spiral(ws_=9):
    #     x = y = 0
    #     dx = 0
    #     dy = -1
    #     list_ = []
    #     for i in range(ws_ ** 2):
    #         if (-ws_ / 2 < x <= ws_ / 2) and (-ws_ / 2 < y <= ws_ / 2):
    #             list_.append((x, y))
    #         if x == y or (x < 0 and x == -y) or (x > 0 and x == 1 - y):
    #             dx, dy = -dy, dx
    #         x, y = x + dx, y + dy
    #     return list_

    def update_painter(self, dic_):
        print('update_painter')
        self.dic_l = dic_ # layer data and information
        self.update()


class CloudThread(QThread):
    sig_status = pyqtSignal(list, dict)

    def __init__(self, main, parent, dic_, list_layer=None, stats=False):
        QThread.__init__(self)

        self.main = main
        self.parent = parent
        self.list_ = []
        self.stats = stats
        self.pol_ =dic_['rec']
        self.dic_ = dic_
        self.list_layer = list_layer
        self.stop = False
        self.dic_threshold = None
        self.dic_class_color = {}
        self.dic_layer_pcs = {}
        self.dic_layer_quant = {}
        self.list_pc = []
        self.dic_default_class_color = {
            0: QColor(186, 186, 186), 1: QColor(170, 170, 170), 2: QColor(170, 85, 0), 3: QColor(0, 170, 170),
            4: QColor(85, 255, 85), 5: QColor(0, 170, 0), 6: QColor(255, 85, 85), 7: QColor(170, 0, 0),
            8: QColor(85, 85, 85), 9: QColor(85, 255, 255), 10: QColor(170, 0, 170), 11: QColor(0, 0, 0),
            13: QColor(255, 255, 85), 14: QColor(255, 255, 85), 15: QColor(255, 85, 255), 16: QColor(255, 255, 85),
            17: QColor(85, 85, 255), 18: QColor(100, 100, 100), 19: QColor(225, 89, 137), 20: QColor(190, 207, 80),
            21: QColor(152, 125, 183), 22: QColor(231, 113, 72), 23: QColor(183, 72, 75), 24: QColor(255, 158, 23),
            25: QColor(232, 113, 141), 26: QColor(141, 90, 153), 27: QColor(243, 166, 178), 28: QColor(114, 155, 111),
            29: QColor(213, 180, 60), 30: QColor(164, 113, 88), 31: QColor(133, 182, 111), }

    def run(self):
        if self.stats:
            dens_ = 0
            for layer_ in self.list_layer:
                quant_ = 0
                prov_ = layer_.dataProvider()
                max_err = 0.0001
                p_limit = 1
                while True:
                    p_limit *= 2
                    len_ = len(prov_.identify(maxErrorInMapCoords=max_err, extentGeometry=self.pol_, pointsLimit=p_limit))
                    if len_ <= quant_:
                        break
                    quant_ = len_
                dens_ += quant_
            self.parent.le_max_dens.setText(f'{dens_:0.1f}')
            return

        press_ = 0.001
        quant_ = 1000000

        for layer_ in self.list_layer:
            self.dic_class_color[layer_.name()] = {}
            if layer_.renderer().type() == 'classified':
                for cat_ in layer_.renderer().categories():
                    if cat_.renderState():
                        self.dic_class_color[layer_.name()].update({cat_.value(): cat_.color()})
            else:
                self.dic_class_color[layer_.name()] = self.dic_default_class_color

        self.dic_threshold = {
            'D_min': 99999.0,
            'D_max': 0.0,
            'P_min': 0.0,
            'P_max': 0.0,
            'Z_min': 99999.0,
            'Z_max': -99999.0,
            'I_min': 99999,
            'I_max': -99999,
        }
        for layer_ in self.list_layer:
            prov_ = layer_.dataProvider()
            layer_name = layer_.name()

            list_pc = prov_.identify(maxErrorInMapCoords=press_, extentGeometry=self.pol_, pointsLimit=quant_)

            self.dic_layer_pcs[layer_name] = list_pc


            if not list_pc:
                continue
            for pc_ in list_pc:
                if self.stop:
                    return
                if pc_['Classification'] not in self.dic_class_color[layer_name]:
                    continue
                p_l = QgsGeometry(QgsPoint(pc_['X'], pc_['Y']))
                for i, ll_, lf_  in self.dic_['rll']:
                    d_n = ll_.distance(p_l)

                    pc_['D'] = d_n
                    if d_n < self.dic_threshold['D_min']:
                        self.dic_threshold['D_min'] = d_n
                    elif d_n > self.dic_threshold['D_max']:
                        self.dic_threshold['D_max'] = d_n

                    dep_n = lf_.distance(p_l)
                    pc_['P'] = dep_n
                    if not self.dic_threshold['P_max']:
                        self.dic_threshold['P_max'] = lf_.length()

                z_n = pc_['Z']
                if z_n < self.dic_threshold['Z_min']:
                    self.dic_threshold['Z_min'] = z_n
                elif z_n > self.dic_threshold['Z_max']:
                    self.dic_threshold['Z_max'] = z_n

                i_n = pc_['Intensity']
                if i_n < self.dic_threshold['I_min']:
                    self.dic_threshold['I_min'] = i_n
                elif i_n > self.dic_threshold['I_max']:
                    self.dic_threshold['I_max'] = i_n

        self.calc_poins()

    def calc_poins(self):
        print('calc_poins')

        wdl_ = self.parent.wdl_prof
        h_ = wdl_.height()
        w_ = wdl_.width()
        dic_t = {}
        dic_t['LL'] = self.dic_['rll'][0][1]
        dic_t['LF'] = self.dic_['rll'][0][2]
        for tag_ in self.dic_threshold:
            dic_t[tag_] = self.dic_threshold[tag_]
        dic_aux = {}
        for tag_ in dic_t:
            dic_aux[tag_] = dic_t[tag_]
        wdl_.dic_t = dic_aux
        wdl_.y_value = 999999
        wdl_.mp_ = None

        dic_l = {}
        for layer_name in self.dic_layer_pcs:
            dic_l[layer_name] = {'color': self.dic_class_color[layer_name]}
            list_pc = self.dic_layer_pcs[layer_name]
            dic_0 = {}
            if not list_pc:
                continue
            for pc_ in list_pc:
                if self.stop:
                    return
                if not 'D' in pc_:
                    continue
                y_ = int(h_ * (1  - (pc_['Z'] - dic_t['Z_min']) / (dic_t['Z_max'] - dic_t['Z_min'])))
                x_ = int((pc_['D'] - dic_t['D_min']) * w_ / (dic_t['D_max'] - dic_t['D_min']))
                p_ = pc_['P']
                i_ = int((pc_['Intensity'] - dic_t['I_min']) * 255 / (dic_t['I_max'] - dic_t['I_min']))
                c_ = int(pc_['Classification'])
                r_ = int(pc_['Red'] / 256) if 'Red' in pc_ else 0
                g_ = int(pc_['Green'] / 256) if 'Green' in pc_ else 0
                b_ = int(pc_['Blue'] / 256) if 'Blue' in pc_ else 0
                e_ = round(pc_['X'], 4)
                n_ = round(pc_['Y'], 4)
    
                if f'{x_},{y_}' not in dic_0:
                    dic_0[f'{x_},{y_}'] = {f'{p_:.4f}':
                        {'X': x_, 'Y': y_, 'P': p_, 'I': i_, 'R': r_, 'G': g_, 'B': b_, 'C': c_, 'E': e_, 'N': n_}}
                else:
                    dic_0[f'{x_},{y_}'][f'{p_:.4f}'] = \
                        {'X': x_, 'Y': y_, 'P': p_, 'I': i_, 'R': r_, 'G': g_, 'B': b_, 'C': c_, 'E': e_, 'N': n_}
            dic_l[layer_name].update({'pcs': dic_0})
        self.parent.wdl_prof.update_painter(dic_l)

def calc_proj(p_, az_, d_):
    x_ = p_.x()
    y_ = p_.y()
    az1 = math.radians(az_)
    return QgsPointXY(x_ + d_ * math.sin(az1), y_ + d_ * math.cos(az1))


# QgsPointCloudClassifiedRenderer - layer_.renderer().type() = 'classified'
# QgsPointCloudAttributeByRampRenderer - layer_.renderer().type() = 'ramp'
#    layer_.renderer().attribute() = 'Intensity'
# QgsPointCloudRgbRenderer - layer_.renderer().type() = 'rgb'