# --------------------------------------------------------
#    mmqgis_library - mmqgis operation functions
#
#    begin                : 10 May 2010
#    copyright            : (c) 2010 by Michael Minn
#    email                : See michaelminn.com
#
#   MMQGIS is free software and is offered without guarantee
#   or warranty. You can redistribute it and/or modify it 
#   under the terms of version 2 of the GNU General Public 
#   License (GPL v2) as published by the Free Software 
#   Foundation (www.gnu.org).
# --------------------------------------------------------

import csv
import sys
import time
import math
import urllib
import os.path
import operator
import tempfile

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from math import *

# --------------------------------------------------------
#    MMQGIS Utility Functions
# --------------------------------------------------------

def mmqgis_find_layer(layer_name):
	# print "find_layer(" + str(layer_name) + ")"

	for name, search_layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
		if search_layer.name() == layer_name:
			return search_layer

	return None

def mmqgis_is_float(s):
	try:
		float(s)
		return True
	except:
		return False

# Cumbersome function to give backward compatibility before python 2.7

def format_float(value, separator, decimals):
	formatstring = ("%0." + unicode(int(decimals)) + "f")
	# print str(value) + ": " + formatstring
	string = formatstring % value
	intend = string.find('.')
	if intend < 0:
		intend = len(string)

	if separator and (intend > 3):
		start = intend % 3
		if start == 0:
			start = 3
		intstring = string[0:start]

		for x in range(start, intend, 3):
			intstring = intstring + separator + string[x:x+3]

		string = intstring + string[intend:]

	return string

def mmqgis_round_significant_digits(value, digits, comma):
	if value == 0:
		return unicode(value)

	sign = 1
	if value < 0:
		sign = -1

	decimals = 0
	magnitude = abs(value)
		
	if magnitude > 1:
		magnitude = round(magnitude, digits - int(round(math.log(magnitude, 10))) - 1)

	else:
		while (magnitude * pow(10, decimals + 1)) < pow(10, digits):
			decimals = decimals + 1

	if (magnitude >= 1000) and comma:
		return format(int(round(magnitude * sign)), ",d")
	else:
		return format(magnitude * sign, "." + unicode(decimals) + "f")
		
def mmqgis_gridify_points(hspacing, vspacing, points):
	# Align points to grid
	point_count = 0
	deleted_points = 0
	newpoints = []
	for point in points:
		point_count += 1
		newpoints.append(QgsPoint(round(point.x() / hspacing, 0) * hspacing, \
				    round(point.y() / vspacing, 0) * vspacing))

	# Delete overlapping points
	z = 0
	while z < (len(newpoints) - 2):
		if newpoints[z] == newpoints[z + 1]:
			newpoints.pop(z + 1)
			deleted_points += 1
		else:
			z += 1

	# Delete line points that go out and return to the same place
	z = 0
	while z < (len(newpoints) - 3):
		if newpoints[z] == newpoints[z + 2]:
			newpoints.pop(z + 1)
			newpoints.pop(z + 1)
			deleted_points += 2
			# Step back to catch arcs
			if (z > 0):
				z -= 1
		else:
			z += 1

	# Delete overlapping start/end points
	while (len(newpoints) > 1) and (newpoints[0] == newpoints[len(newpoints) - 1]):
		newpoints.pop(len(newpoints) - 1)
		deleted_points += 2
				
	return newpoints, point_count, deleted_points


def mmqgis_searchable_streetname(name):
	# Use common address abbreviations to reduce naming discrepancies and improve hit ratio
	# print "searchable_name(" + str(name) + ")"
	if not name:
		return ""

	# name = unicode(name).strip().lower()
	name = name.strip().lower()

	name = name.replace(".", "")
	name = name.replace(" street", " st")
	name = name.replace(" avenue", " av")
	name = name.replace(" plaza", " plz")
	name = name.replace(" drive", " dr")
	name = name.replace("saint ", "st ")
	name = name.replace("fort ", "ft ")
	name = name.replace(" ave", " av")

	name = name.replace("east", "e")
	name = name.replace("west", "w")
	name = name.replace("north", "n")
	name = name.replace("south", "s")
	name = name.replace("1st", "1")
	name = name.replace("2nd", "2")
	name = name.replace("3rd", "3")
	name = name.replace("4th", "4")
	name = name.replace("5th", "5")
	name = name.replace("6th", "6")
	name = name.replace("7th", "7")
	name = name.replace("8th", "8")
	name = name.replace("9th", "9")
	name = name.replace("0th", "0")
	name = name.replace("1th", "1")
	name = name.replace("2th", "2")
	name = name.replace("3th", "3")

	return name


def mmqgis_wkbtype_to_text(wkbtype):
	if wkbtype == QGis.WKBUnknown: return "Unknown"
	if wkbtype == QGis.WKBPoint: return "point"
	if wkbtype == QGis.WKBLineString: return "linestring"
	if wkbtype == QGis.WKBPolygon: return "polygon"
	if wkbtype == QGis.WKBMultiPoint: return "multipoint"
	if wkbtype == QGis.WKBMultiLineString: return "multilinestring"
	if wkbtype == QGis.WKBMultiPolygon: return "multipolygon"
	if wkbtype == QGis.WKBPoint25D: return "point 2.5d"
	if wkbtype == QGis.WKBLineString25D: return "linestring 2.5D"
	if wkbtype == QGis.WKBPolygon25D: return "polygon 2.5D"
	if wkbtype == QGis.WKBMultiPoint25D: return "multipoint 2.5D"
	if wkbtype == QGis.WKBMultiLineString25D: return "multilinestring 2.5D"
	if wkbtype == QGis.WKBMultiPolygon25D: return "multipolygon 2.5D"
	return "Unknown WKB " + unicode(wkbtype)

def mmqgis_status_message(qgis, message):
	qgis.mainWindow().statusBar().showMessage(message)

def mmqgis_completion_message(qgis, message):
	mmqgis_status_message(qgis, message)
	qgis.messageBar().pushMessage(message, 0, 3)

# --------------------------------------------------------
#    mmqgis_animate_columns - Create animations by
#		interpolating offsets from attributes
# --------------------------------------------------------

def mmqgis_animate_columns(qgis, layer_name, long_col, lat_col, outdir, frame_count):

	# Error Checks
	layer = mmqgis_find_layer(layer_name)
	if layer == None:
		return "Invalid map layer ID: " + unicode(map_layer_id)

	long_col_index = layer.dataProvider().fieldNameIndex(long_col)
	if (long_col_index < 0):
		return "Invalid longitude column index: " + unicode(long_col)

	lat_col_index = layer.dataProvider().fieldNameIndex(lat_col)
	if (lat_col_index < 0):
		return "Invalid latitude column: " + unicode(lat_col)

	if not os.path.isdir(outdir):
		return "Invalid output directory: " + unicode(outdir)

	if frame_count <= 0:
		return "Invalid number of frames specified: " + unicode(frame_count)


	# Initialize temporary shapefile

	tempdir = tempfile.mkdtemp()
	tempfilename = tempdir + "/mmqgis_animate.shp"
	tempcrs = layer.dataProvider().crs()
	if not tempcrs.isValid():
		tempcrs.createFromSrid(4326)
		print "Defaulting layer " + unicode(layer.id()) + " to CRS " + unicode(tempcrs.epsg())

	tempwriter = QgsVectorFileWriter(tempfilename, "utf-8",
		layer.dataProvider().fields(), layer.dataProvider().geometryType(), tempcrs)
	del tempwriter

	templayer = qgis.addVectorLayer(tempfilename, "animate", "ogr")

	if hasattr(templayer, 'isUsingRendererV2') and templayer.isUsingRendererV2():
		templayer.setRendererV2(layer.rendererV2().clone())
	else:
		templayer.setRenderer(layer.renderer().clone())

	templayer.enableLabels(layer.hasLabelsEnabled())

	qgis.legendInterface().setLayerVisible(layer, 0)


	# Iterate Frames

	for frame in range(frame_count + 1):
		mmqgis_status_message(qgis, "Rendering frame " + unicode(frame))

		# Read, move and rewrite features

		feature_ids = []
		#feature = QgsFeature()
		#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
		#layer.dataProvider().rewind()
		#while layer.dataProvider().nextFeature(feature):
		for feature in layer.dataProvider().getFeatures():
			attributes = feature.attributes()
			xoffset, valid = attributes[long_col_index].toDouble()
			yoffset, valid = attributes[lat_col_index].toDouble()
			xoffset = xoffset * frame / frame_count;
			yoffset = yoffset * frame / frame_count;

			newfeature = QgsFeature()
			newgeometry = feature.geometry()
			newgeometry.translate(xoffset, yoffset)
			newfeature.setGeometry(newgeometry)
			newfeature.setAttributes(attributes)

			if templayer.dataProvider().addFeatures([newfeature]):
				feature_ids.append(newfeature.id())

		# Write Frame

		# templayer.commitChanges()
		qgis.mapCanvas().refresh()
		framefile = outdir + "/frame" + format(frame, "06d") + ".png"
		qgis.mapCanvas().saveAsImage(framefile)

		# Delete features from temporary shapefile

		for feature_id in feature_ids:
			templayer.dataProvider().deleteFeatures([feature_id])

	# Clean up

	QgsMapLayerRegistry.instance().removeMapLayers([templayer.id()])
	QgsVectorFileWriter.deleteShapeFile(tempfilename)
	qgis.legendInterface().setLayerVisible(layer, 1)

	return None

# --------------------------------------------------------
#    mmqgis_animate_rows - Create animations by
#		displaying successive rows
# --------------------------------------------------------

def mmqgis_animate_rows(qgis, layer_names, cumulative, outdir):

	#print "mmqgis_animate_rows()"
	#for id in animate_layer_ids:
	#	print str(id)
	#print "cumulative = " + str(cumulative)
	#print outdir

	# Error Checks
	if not os.path.isdir(outdir):
		return "Invalid output directory: " + unicode(outdir)

	layers = []
	for layer_name in layer_names:
		layer = mmqgis_find_layer(layer_name)
		if layer == None:
			return "Invalid layer name: " + unicode(layer_name)
		layers.append(layer)

	frame_count = 0
	for layer in layers:
		if frame_count < layer.dataProvider().featureCount():
			frame_count = layer.dataProvider().featureCount()

	if frame_count <= 0:
		return "Invalid number of frames specified"


	# Feature ID arrays

	feature_ids = [None] * len(layers)
	for index in range(len(layers)):
		layer = layers[index]
		feature_ids[index] = []

		#feature = QgsFeature()
		#layer.dataProvider().select()
		#layer.dataProvider().rewind()
		#while layer.dataProvider().nextFeature(feature):

		for feature in layer.dataProvider().getFeatures():
			feature_ids[index].append(feature.id());
			# print feature.id()

	# Create temporary layers

	tempdir = tempfile.mkdtemp()
	tempnames = [None] * len(layers)
	templayers = [None] * len(layers)
	for layer_index, layer in enumerate(layers):
		tempnames[layer_index] = tempdir + "/mmqgis_animate" + unicode(layer_index) + ".shp"
		tempcrs = layer.dataProvider().crs()
		if not tempcrs.isValid():
			tempcrs.createFromSrid(4326)
			print "Defaulting layer " + unicode(animate_layer_ids[layer_index]) + \
				" to CRS " + unicode(tempcrs.epsg())

		tempwriter = QgsVectorFileWriter(tempnames[layer_index], "utf-8",
			layer.dataProvider().fields(), layer.dataProvider().geometryType(), tempcrs)
		del tempwriter

		templayers[layer_index] = qgis.addVectorLayer(tempnames[layer_index], "animate" + unicode(layer_index), "ogr")

		# Linux doesn't have V2, but Windoze doesn't have legacy renderer
		if hasattr(templayers[layer_index], 'isUsingRendererV2') and templayers[layer_index].isUsingRendererV2():
			templayers[layer_index].setRendererV2(layer.rendererV2().clone())
		else:
			templayers[layer_index].setRenderer(layer.renderer().clone())

		# There doesn't seem to be a way to directly copy labeling fields and attributes
		if layer.hasLabelsEnabled():
			templayers[layer_index].enableLabels(1)
			label = layer.label()
			templabel = templayers[layer_index].label()
			templabel.setFields(label.fields())

			try:
				templabel.setLabelField(0, templabel.fields().toList().index(label.labelField(0)))
				#for field_index, field in templabel.fields().toList():
				#	if field.name() == label.labelField(0):
				#		templabel.setLabelField(0, field_index)

				attributes = label.labelAttributes()
				tempattributes = templabel.labelAttributes()
				tempattributes.setFamily(attributes.family())
				tempattributes.setBold(attributes.bold())
				tempattributes.setItalic(attributes.italic())
				tempattributes.setUnderline(attributes.underline())
				tempattributes.setSize(attributes.size(), attributes.sizeType())
				tempattributes.setColor(attributes.color())
			except ValueError:
				pass

		qgis.legendInterface().setLayerVisible(layer, 0)

	# return "Testing"

	# Iterate frames

	for frame in range(int(frame_count + 1)):
		mmqgis_status_message(qgis, "Rendering frame " + unicode(frame))

		for layer_index, layer in enumerate(layers):
			if frame < layer.featureCount():
				feature_id = feature_ids[layer_index][frame]
				for feature in layer.dataProvider().getFeatures(QgsFeatureRequest().setFilterFid(feature_id)):
					#feature = QgsFeature()
					#feature = layer.dataProvider().featureAtId(featureid, feature, 1, \
					#	layers[layer_index].dataProvider().attributeIndexes())

					newfeature = QgsFeature()
					newfeature.setGeometry(feature.geometry())
					newfeature.setAttributes(feature.attributes())
					if templayers[layer_index].dataProvider().addFeatures([newfeature]):
						feature_ids[layer_index][frame] = newfeature.id()
					#templayers[layer_index].commitChanges()

		qgis.mapCanvas().refresh()
		framefile = outdir + "/frame" + format(frame, "06d") + ".png"
		qgis.mapCanvas().saveAsImage(framefile)

		if not cumulative:
			for layer_index in range(len(layers)):
				if frame < layers[layer_index].featureCount():
					feature = QgsFeature()
					featureid = feature_ids[layer_index][frame]
					templayers[layer_index].dataProvider().deleteFeatures([featureid])

	for layer_index in range(len(layers)):
		QgsMapLayerRegistry.instance().removeMapLayers([templayers[layer_index].id()])
		QgsVectorFileWriter.deleteShapeFile(tempnames[layer_index])
		qgis.legendInterface().setLayerVisible(layers[layer_index], 1)

	return None

# ----------------------------------------------------------
#    mmqgis_attribute_export - Export attributes to CSV file
# ----------------------------------------------------------

def mmqgis_attribute_export(qgis, outfilename, layername, attribute_names, field_delimiter, line_terminator):
	# Error checks

	if (not outfilename) or (len(outfilename) <= 0):
		return "No output CSV file given"
	
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Layer " + layername + " not found"

	# Find attribute indices
	attribute_indices = []
	if (not attribute_names) or (len(attribute_names) <= 0):
		attribute_names = []
		# print "fields: " + str(layer.dataProvider().fields())
		for index, field in enumerate(layer.dataProvider().fields()):
			attribute_indices.append(index)
			attribute_names.append(field.name())

	else:
		for x in range(0, len(attribute_names)):
			index = layer.dataProvider().fieldNameIndex(attribute_names[x])
			if index < 0:
				return "Layer " + layername + " has no attribute " + attribute_names[x]
			attribute_indices.append(index)

	# Create the CSV file
	try:
		outfile = open(outfilename, 'w')
    	except:
		return "Failure opening " + outfilename

	writer = csv.writer(outfile, delimiter = field_delimiter, lineterminator = line_terminator)
	writer.writerow(attribute_names) # header


	# Iterate through each feature in the source layer
	feature_count = layer.dataProvider().featureCount()

	#feature = QgsFeature()
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
        #while layer.dataProvider().nextFeature(feature):

	for index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (index % 50) == 0:
			qgis.mainWindow().statusBar().showMessage \
				("Exporting feature " + unicode(feature.id()) + " of " + unicode(feature_count))
		attributes = feature.attributes()

		row = []
		for column in attribute_indices:
			row.append(unicode(attributes[column].toString()).encode("utf-8"))

		writer.writerow(row)

	del writer

	mmqgis_completion_message(qgis, unicode(feature_count) + " records exported")

	return None

# --------------------------------------------------------
#    mmqgis_attribute_join - Join attributes from a CSV
#                            file to a shapefile
# --------------------------------------------------------

def mmqgis_attribute_join(qgis, layername, infilename, joinfield, targetfield, outfilename, notfoundname, addlayer):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Layer " + unicode(layername) + " not found"

	target_index = layer.fieldNameIndex(targetfield)
	if target_index < 0:
		return "Invalid join attribute " + unicode(targetfield)

	if len(infilename) <= 0:
		return "No input CSV file given"

	if len(outfilename) <= 0:
		return "No output shapefile name given"
		
	# Create a combined field list from the source layer and the CSV file header
	try:
		infile = open(infilename, 'r')
	except:
		return "Failure opening input file: " + unicode(infilename)
			
	try:
		dialect = csv.Sniffer().sniff(infile.read(2048))
	except:
		return "Bad CSV file (verify that your delimiters are consistent)" + unicode(infilename)

	infile.seek(0)
	reader = csv.reader(infile, dialect)

	# Build composite list of fields
	join_index = -1
	newfields = layer.dataProvider().fields()
	header = reader.next()
	for index in range(0, len(header)):
		fieldname = unicode(header[index], 'utf-8').strip()[0:9]
		if fieldname.lower() == joinfield[0:9].strip().lower():
			join_index = index
		else:
			if newfields.indexFromName(fieldname) >= 0:
				fieldname = fieldname[0:8].strip() + "2" # mitigates duplicate field names
			newfields.append(QgsField(fieldname, QVariant.String))

	if join_index < 0:
		return "Join field " + unicode(joinfield) + " not found in " + unicode(infilename)

	# Create the output shapefile
	if QFile(outfilename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(outfilename):
			return "Failure deleting existing shapefile: " + unicode(outfilename)

	#print newfields

	outfile = QgsVectorFileWriter(outfilename, "utf-8", newfields, 
		layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Read the CSV file data into memory
	csv_data = []
	csv_found = []
	for row in reader:
		csv_data.append(row)
		csv_found.append(0)

	del reader


	# Iterate through each feature in the source layer
	matched_count = 0
	feature_count = layer.dataProvider().featureCount()

	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (feature_index % 50) == 0:
			mmqgis_status_message(qgis, "Joining feature " + unicode(feature_index) + \
					" of " + unicode(feature_count) + " (" + unicode(matched_count) + " matched)")
		attributes = feature.attributes()

		if feature.geometry() == None:
			return "No geometry in layer: " + unicode(layername)

		# Key must be UTF-8 encoded to deal with UTF-8 encoded values from CSV file
		key = unicode(attributes[target_index].toString()).encode("utf-8").lower().strip()

		for row_index, row in enumerate(csv_data):
			if row[join_index].strip().lower() == key:
				# print key + " --------------"
				newattributes = []
				for value in attributes:
					newattributes.append(value)
					
				for combine_index, combine in enumerate(row):
					if combine_index != join_index:
						try:
							newattribute = unicode(combine, 'utf-8')
						except:
							return "CSV file does not appear to be UTF-8 encoded: " + unicode(infilename)
						newattributes.append(newattribute)

				newfeature = QgsFeature()
				newfeature.setAttributes(newattributes)
				newfeature.setGeometry(feature.geometry())
				outfile.addFeature(newfeature)
				matched_count += 1
				csv_found[row_index] += 1

	if matched_count <= 0:
		return "No matching records found"

	del outfile
	del infile

	# Write records that were not joined to the notfound file
	try:
		outfile = open(notfoundname, 'w')
	except:
		return "Failure opening not found file: " + unicode(notfoundname)

	else:
		writer = csv.writer(outfile, dialect)
		writer.writerow(header)
		for x in range(0, len(csv_data)):
			if not csv_found[x]:
				writer.writerow(csv_data[x])
		del writer
		del outfile
	
	if addlayer:	
		qgis.addVectorLayer(outfilename, os.path.basename(outfilename), "ogr")

	mmqgis_completion_message(qgis, unicode(matched_count) + " records joined from " + \
		unicode(feature_count) + " shape records and " + unicode(len(csv_data)) + " CSV file records")

	return None

# --------------------------------------------------------
#    mmqgis_buffers - Create buffers around shapes
# --------------------------------------------------------

def mmqgis_buffers(qgis, layername, radius, unit, buffer_edges, savename, addlayer):

	# Error checking
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Layer " + unicode(layername) + " not found"

	if len(savename) <= 0:
		return "No output filename given"

	if type(radius) == float:
		field_index = -1
		if radius <= 0:
			return "Invalid radius: " + unicode(radius)
	else:
		field_index = layer.fieldNameIndex(radius)
		if field_index < 0:
			return "Invalid radius attribute name: " + unicode(radius)

	# Create the output file
	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename
 
	wgs84 = QgsCoordinateReferenceSystem()
	wgs84.createFromProj4("+proj=longlat +datum=WGS84 +no_defs")
	transform = QgsCoordinateTransform(layer.dataProvider().crs(), wgs84)
	

	outfile = QgsVectorFileWriter(savename, "utf-8",
			layer.dataProvider().fields(), QGis.WKBPolygon, wgs84)

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Create buffers for each feature
	buffercount = 0
	featurecount = layer.dataProvider().featureCount();
	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		mmqgis_status_message(qgis, "Writing feature " + \
			unicode(feature.id()) + " of " + unicode(featurecount))

		if field_index < 0:
			feature_radius = radius
		else:
			attribute = feature.attributes()[field_index]
			feature_radius, isvalid = attribute.toDouble()
			if isvalid:
				feature_radius = float(feature_radius)
			else:
				feature_radius = 0.0

		if feature_radius > 0:
			# Buffer radii are always in meters
			if unit == "Feet":
				feature_radius = feature_radius / 3.2808399

			elif unit == "Miles":
				feature_radius = feature_radius * 1609.344

			elif unit == "Kilometers":
				feature_radius = feature_radius * 1000


			geometry = feature.geometry()
			geometry.transform(transform) # Needs to be WGS 84

			if (geometry.wkbType() == QGis.WKBPoint) or \
			   (geometry.wkbType() == QGis.WKBPoint25D):
				newgeometry = mmqgis_buffer_point(geometry.asPoint(), feature_radius, buffer_edges)

			else:
				newgeometry = mmqgis_buffer_geometry(geometry, feature_radius)

			newfeature = QgsFeature()
			newfeature.setGeometry(newgeometry)
			newfeature.setAttributes(feature.attributes())
			outfile.addFeature(newfeature)
			buffercount = buffercount + 1

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(buffercount) + " buffers created for " + \
		unicode(featurecount) + " features")

	return None


def mmqgis_buffer_geometry(geometry, meters):
	if meters <= 0:
		return None

	# To approximate meaningful meter distances independent of the original CRS,
	# the geometry is transformed to an azimuthal equidistant projection
	# with the center of the polygon as the origin. After buffer creation,
	# the buffer is transformed to WGS 84 and returned. While this may introduce
	# some deviation from the original CRS, buffering is assumed in practice
	# to be a fairly inexact operation that can tolerate such deviation

	wgs84 = QgsCoordinateReferenceSystem()
	wgs84.createFromProj4("+proj=longlat +datum=WGS84 +no_defs")

	proj4 = "+proj=aeqd +lat_0=" + str(geometry.centroid().asPoint().y()) + \
		" +lon_0=" + str(geometry.centroid().asPoint().x()) + \
		" +x_0=0 +y_0=0 +datum=WGS84 +units=m +no_defs"
	azimuthal_equidistant = QgsCoordinateReferenceSystem()
	azimuthal_equidistant.createFromProj4(proj4)
	
	transform = QgsCoordinateTransform(wgs84, azimuthal_equidistant)
	geometry.transform(transform)

	newgeometry = geometry.buffer(meters, 7)

	wgs84 = QgsCoordinateReferenceSystem()
	wgs84.createFromProj4("+proj=longlat +datum=WGS84 +no_defs")

	transform = QgsCoordinateTransform(azimuthal_equidistant, wgs84)
	newgeometry.transform(transform)

	return newgeometry

def mmqgis_buffer_point(point, meters, edges):
	if (meters <= 0) or (edges < 3):
		return None

	# Points are treated separately from other geometries so that discrete
	# edges can be supplied for non-circular buffers that are not supported
	# by the QgsGeometry.buffer() function

	wgs84 = QgsCoordinateReferenceSystem()
	wgs84.createFromProj4("+proj=longlat +datum=WGS84 +no_defs")

	proj4 = "+proj=aeqd +lat_0=" + str(point.y()) + " +lon_0=" + str(point.x()) + \
		" +x_0=0 +y_0=0 +datum=WGS84 +units=m +no_defs"
	azimuthal_equidistant = QgsCoordinateReferenceSystem()
	azimuthal_equidistant.createFromProj4(proj4)

	transform = QgsCoordinateTransform(azimuthal_equidistant, wgs84)

	polyline = []
	for edge in range(0, edges + 1):
		radians = float(edge) * 2.0 * math.pi / float(edges)
		x = math.sin(radians) * meters
		y = math.cos(radians) * meters
		newpoint = transform.transform(QgsPoint(x, y))
		polyline.append(newpoint)
		# print str(edge) + ") " + str(x) + ", " + str(y) + " = " + str(newpoint.x()) + ", " + str(newpoint.y())

	return QgsGeometry.fromPolygon([polyline])



# --------------------------------------------------------
#    mmqgis_color_ramp - Robust layer coloring
# --------------------------------------------------------

def mmqgis_interpolate_color(scale, lowcolor, midcolor, highcolor):
	if scale <= 0:
		return QColor((lowcolor >> 16) & 0xff, (lowcolor >> 8) & 0xff, lowcolor & 0xff)

	if scale >= 1:
		return QColor((highcolor >> 16) & 0xff, (highcolor >> 8) & 0xff, highcolor & 0xff)

	
	lowred = (lowcolor >> 16) & 0xff
	lowgreen = (lowcolor >> 8) & 0xff
	lowblue = lowcolor & 0xff

	midred = (midcolor >> 16) & 0xff
	midgreen = (midcolor >> 8) & 0xff
	midblue = midcolor & 0xff
		
	highred = (highcolor >> 16) & 0xff
	highgreen = (highcolor >> 8) & 0xff
	highblue = highcolor & 0xff
		
	if scale <= 0.5:
		scale = scale * 2
		red = int((lowred * (1.0 - scale)) + (midred * scale))
		green = int((lowgreen * (1.0 - scale)) + (midgreen * scale))
		blue = int((lowblue * (1.0 - scale)) + (midblue * scale))

	else:
		scale = (scale - 0.5) * 2
		red = int((midred * (1.0 - scale)) + (highred * scale))
		green = int((midgreen * (1.0 - scale)) + (highgreen * scale))
		blue = int((midblue * (1.0 - scale)) + (highblue * scale))

	# print str(scale) + " = " + str(red) + "," + str(green) + "," + str(blue)
	return QColor(red, green, blue)

def mmqgis_color_ramp_symbol(layer, symboltype, symbolsize, thickness, color, border_color):

	if (layer.dataProvider().geometryType() == QGis.WKBPoint) or \
	   (layer.dataProvider().geometryType() == QGis.WKBMultiPoint):
		# core/symbology-ng/qgsmarkersymbollayerv2.cpp
		# "circle", "square", "rectangle", "diamond", "pentagon", "triangle", "star", "arrow"
		#print unicode(symboltype) + ": " + unicode(color.name()) + ", " + \
		#	unicode(border_color.name()) + ", " + unicode(symbolsize)
		symbol_layer = QgsSimpleMarkerSymbolLayerV2(symboltype, color, border_color, float(symbolsize))
		#print unicode(type(symbol_layer))
		symbol = QgsMarkerSymbolV2()
		symbol.changeSymbolLayer(0, symbol_layer)

	elif (layer.dataProvider().geometryType() == QGis.WKBLineString) or \
	     (layer.dataProvider().geometryType() == QGis.WKBMultiLineString):
		symbol_layer = QgsSimpleLineSymbolLayerV2(color, thickness)
		symbol = QgsLineSymbolV2()
		symbol.changeSymbolLayer(0, symbol_layer)

	else: # polygon
		outline_pen = Qt.SolidLine
		if thickness <= 0:
			outline_pen = 0
		symbol_layer = QgsSimpleFillSymbolLayerV2(color, Qt.SolidPattern,
			border_color, outline_pen, float(thickness))
		symbol = QgsFillSymbolV2()
		symbol.changeSymbolLayer(0, symbol_layer)

	return symbol

def mmqgis_ramped_ranges(minimum, maximum, categories, ramptype):
	ranges = []
	for index in range(0, categories):
		if ramptype == "Logarithmic":
			lower = minimum + ((maximum - minimum) * log(index + 1) / log(categories + 1))
			upper = minimum + ((maximum - minimum) * log(index + 2) / log(categories + 1))

		elif ramptype == "Exponential":
			if index == 0:
				lower = 0
			else:
				lower = minimum + ((maximum - minimum) * exp(index - 1) / exp(categories - 1))
			upper = minimum + ((maximum - minimum) * exp(index) / exp(categories - 1))

		elif ramptype == "Square Root":
			lower = minimum + ((maximum - minimum) * sqrt(index) / sqrt(categories))
			upper = minimum + ((maximum - minimum) * sqrt(index + 1) / sqrt(categories))

		elif ramptype == "Squared":
			lower = minimum + ((maximum - minimum) * pow(index, 2) / pow(categories, 2))
			upper = minimum + ((maximum - minimum) * pow(index + 1, 2) / pow(categories, 2))

		else: # linear
			lower = minimum + ((maximum - minimum) * float(index) / float(categories))
			upper = minimum + ((maximum - minimum) * float(index + 1) / float(categories))

		ranges.append([lower, upper])

	return ranges

def mmqgis_legend_label(ranges):
	label = ""
	for index, value in enumerate(ranges):
		label = label + mmqgis_round_significant_digits(value, 4, True)

		# if value == 0:
		#	label = label + "0"

		#else:
		#	digits = -int(math.ceil(math.log(abs(value), 10)))
		#	if digits <= -4:
		#		label = label + format(int(round(value, digits + 4)), ",d")
		#	elif digits < 0:
		#		label = label + format(value, "." + str(digits + 4) + "f")
		#	else:
		#		label = label + format(value, "f")

		if (index + 1) < len(ranges):
			label = label + " - "

	return label


def mmqgis_set_color_ramp(qgis, layername, fieldname, ramptype, symboltype, symbolsize, 
		thickness, categories, lowcolor, midcolor, highcolor, outlinecolor):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Invalid layer name: " + layername

	# ------- RASTER -------
	if layer.type() == QgsMapLayer.RasterLayer:
		band_number = layer.bandNumber(fieldname)
		if band_number == 0:
			return "Invalid band name: " + fieldname

		if ramptype == "Discrete": # To be implemented
			ramptype = "Linear"
			categories = 100

		if ramptype == "Quantiles": # To be implemented
			ramptype = "Linear"

		if categories < 3:
			return "Invalid number of categories: " + str(categories)

		# print "Band #" + str(band_number)
		# print "Renderer: " + str(layer.renderer().type())

		minimum = layer.dataProvider().minimumValue(band_number)
		maximum = layer.dataProvider().maximumValue(band_number)
		# print "Range: " + str(minimum) + " - " + str(maximum)

		shades = []
		range_list = mmqgis_ramped_ranges(minimum, maximum, categories, ramptype)
		for index, ranges in enumerate(range_list):
			if index == 0:
				value = ranges[0]
			else:
				value = ranges[1]
			color = mmqgis_interpolate_color(float(index) / float(categories - 1), lowcolor, midcolor, highcolor)
			shades.append(QgsColorRampShader.ColorRampItem(value, color, mmqgis_legend_label([value])))

		ramp_shader = QgsColorRampShader(minimum, maximum)
		ramp_shader.setColorRampType(QgsColorRampShader.INTERPOLATED)
		ramp_shader.setColorRampItemList(shades)

		shader = QgsRasterShader(minimum, maximum)
		shader.setRasterShaderFunction(ramp_shader)
		renderer = QgsSingleBandPseudoColorRenderer(layer.dataProvider(), band_number, shader)
		layer.setRenderer(renderer)
		layer.triggerRepaint()
		qgis.legendInterface().refreshLayerSymbology(layer)

		return None

	# ------- VECTOR -------
	max_symbols = 128 # arbitrary limit

	field_index = layer.fieldNameIndex(fieldname)
	if (field_index < 0):
		return "Invalid field name: " + unicode(fieldname)
	field = layer.dataProvider().fields()[field_index]
	# print "field[" + str(field_index) + "] = " + field.name()

	if ramptype == "Discrete":
		# Get List of discrete values
		values = []
		for feature in layer.dataProvider().getFeatures(QgsFeatureRequest().setFlags(
		    QgsFeatureRequest.NoGeometry).setSubsetOfAttributes([field_index])):
			attribute = unicode(feature.attribute(fieldname).toString())
			values.append(attribute)
			# print unicode(feature.id()) + ") " + attribute

		values = set(values)
		if len(values) > max_symbols:
			return unicode(len(values)) + " discrete values exceeds " + unicode(max_symbols) + " limit"
		if len(values) < 2:
			return "Too few discrete values for color ramp"
		values = sorted(values)

		renderer = QgsCategorizedSymbolRendererV2(fieldname)
		for index, value in enumerate(values):
			color = mmqgis_interpolate_color(float(index) / float(len(values) - 1), lowcolor, midcolor, highcolor)
			border_color = QColor((outlinecolor >> 16) & 0xff, (outlinecolor >> 8) & 0xff, outlinecolor % 0xff)
			symbol = mmqgis_color_ramp_symbol(layer, symboltype, symbolsize, thickness, color, border_color)
			category = QgsRendererCategoryV2(value, symbol, unicode(value))
			renderer.addCategory(category)

	elif ramptype == "Quantiles":
		if (field.type() != QVariant.Int) and (field.type() != QVariant.Double):
			return "Only numeric fields can be used with quantiles: " + unicode(fieldname)

		if categories < 3:
			return "Too few categories: " + unicode(categories)

		# Get List of values
		values = []
		for feature in layer.dataProvider().getFeatures(QgsFeatureRequest().setFlags(
		    QgsFeatureRequest.NoGeometry).setSubsetOfAttributes([field_index])):
			attribute = unicode(feature.attribute(fieldname).toString())
			values.append(float(attribute))
			# print unicode(feature.id()) + ") " + attribute

		if len(values) < categories:
			return "Too few discrete values for " + str(categories) + " category color ramp"

		values = sorted(values)

		classes = []

		for index in range(0, categories):
			lower = float(values[index * len(values) / categories])
			upper = float(values[((index + 1) * len(values) / categories) - 1])

			color = mmqgis_interpolate_color(float(index) / float(categories - 1), lowcolor, midcolor, highcolor)
			border_color = QColor((outlinecolor >> 16) & 0xff, (outlinecolor >> 8) & 0xff, outlinecolor % 0xff)
			symbol = mmqgis_color_ramp_symbol(layer, symboltype, symbolsize, thickness, color, border_color)

			category = QgsRendererRangeV2(lower, upper, symbol, mmqgis_legend_label([lower, upper]))
			classes.append(category)

		renderer = QgsGraduatedSymbolRendererV2(fieldname, classes)

	else: # Range categories
		if (field.type() != QVariant.Int) and (field.type() != QVariant.Double):
			return "Only numeric fields can be used with quantiles: " + unicode(fieldname)

		if categories < 3:
			return "Too few categories: " + unicode(categories)

		minimum, mingood = layer.minimumValue(field_index).toDouble()
		maximum, maxgood = layer.maximumValue(field_index).toDouble()
		range_list = mmqgis_ramped_ranges(minimum, maximum, categories, ramptype)
		classes = []

		for index, ranges in enumerate(range_list):
			color = mmqgis_interpolate_color(float(index) / float(categories - 1), lowcolor, midcolor, highcolor)
			border_color = QColor((outlinecolor >> 16) & 0xff, (outlinecolor >> 8) & 0xff, outlinecolor % 0xff)
			symbol = mmqgis_color_ramp_symbol(layer, symboltype, symbolsize, thickness, color, border_color)

			category = QgsRendererRangeV2(ranges[0], ranges[1], symbol, 
				mmqgis_legend_label([ranges[0], ranges[1]]))
			classes.append(category)

		renderer = QgsGraduatedSymbolRendererV2(fieldname, classes)

	layer.setRendererV2(renderer)
	layer.triggerRepaint()
	qgis.legendInterface().refreshLayerSymbology(layer)

	return None

#	EXAMPLE WITH OLD RENDERER
#
#	renderer = QgsUniqueValueRenderer(layer.geometryType())
#	for index, value in enumerate(values):
#		symbol = QgsSymbol(layer.geometryType(), unicode(value), unicode(value))
#		color = mmqgis_interpolate_color(float(index) / float(len(values) - 1), lowcolor, midcolor, highcolor)
#		outline_color = QColor((outlinecolor >> 16) & 0xff, (outlinecolor >> 8) & 0xff, outlinecolor % 0xff)
#
#		print str(value) + " = " + unicode(color.name())
#		if (layer.dataProvider().geometryType() == QGis.WKBLineString) or \
#		   (layer.dataProvider().geometryType() == QGis.WKBMultiLineString):
#			symbol.setColor(color)
#		else:
#			symbol.setColor(outline_color)
#
#		symbol.setLineStyle(Qt.SolidLine)
#		symbol.setLineWidth(thickness)
#		symbol.setFillColor(color)
#		symbol.setFillStyle(Qt.SolidPattern)
#		symbol.setNamedPointSymbol(symboltype)
#		symbol.setPointSize(symbolsize)
#		symbol.setPointSizeUnits(False) 
#		# symbol.setScaleClassificationField( it->second->scaleClassificationField() );
#		# symbol.setRotationClassificationField( it->second->rotationClassificationField() );
#		# symbol.setCustomTexture("")
#
#		renderer.insertValue(unicode(value), symbol)
#
#	renderer.updateSymbolAttributes()
#	renderer.setClassificationField(field_index)
#	layer.setRenderer(renderer)
#	layer.triggerRepaint()
#	qgis.legendInterface().refreshLayerSymbology(layer)
#
#	return None


# ---------------------------------------------------------
#    mmqgis_delete_columns - Change text fields to numbers
# ---------------------------------------------------------

def mmqgis_delete_columns(qgis, layername, columns, savename, addlayer):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "No layer specified to modify: " + layername

	if len(savename) <= 0:
		return "No output filename given"

	if len(columns) <= 0:
		return "No columns specified for deletion"

	# Build dictionary of fields excluding fields flagged for deletion
	srcindex = []
	srcfields = QgsFields()
	destfields = QgsFields()
	for index, field in enumerate(layer.dataProvider().fields()):
		keep = 1
		for column in columns:
			if field.name() == column:
				keep = 0

		if keep:
			srcindex.append(index)
			srcfields.append(field)
			destfields.append(field)

	if len(destfields) <= 0:
		return "All columns being deleted"
 
	# Create the output file
	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", destfields,
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())


	# Write the features with modified attributes
	featurecount = layer.dataProvider().featureCount();
	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (feature_index % 50) == 0:
			mmqgis_status_message(qgis, "Writing feature " + \
				unicode(feature.id()) + " of " + unicode(featurecount))

		attributes = []
		for index, field in enumerate(srcfields):
			attributes.append(feature.attributes()[srcindex[index]])

		feature.setAttributes(attributes)
		outfile.addFeature(feature)

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(len(columns)) + " columns deleted and written to " + savename)

	return None

# --------------------------------------------------------
#    mmqgis_delete_duplicate_geometries - Save to shaperile
#			while removing duplicate shapes
# --------------------------------------------------------

def mmqgis_delete_duplicate_geometries(qgis, layername, savename, addlayer):

	# Initialization and error checking
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Invalid layer name: " + savename

	if len(savename) <= 0:
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(),
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Read geometries into an array
	# Have to save as WKT because saving geometries causes segfault 
	# when they are used with equal() later
	geometries = []

	for feature in layer.dataProvider().getFeatures():
		geometries.append(feature.geometry().exportToWkt())

	# NULL duplicate geometries
	for x in range(0, len(geometries) - 1):
		if geometries[x] != None:
			if (x % 20) == 0:
				mmqgis_status_message(qgis, "Checking feature " + unicode(x))

			for y in range(x + 1, len(geometries)):
				#print "Comparing " + str(x) + ", " + str(y)
				if geometries[x] == geometries[y]:
					#print "None " + str(x)
					geometries[y] = None

	# Write unique features to output
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
	#for x in range(0, len(geometries)):
	#	if layer.dataProvider().nextFeature(feature):
	#		if geometries[x] != None:
	#			writecount += 1
	#			outfile.addFeature(feature)

	writecount = 0
	for index, feature in enumerate(layer.dataProvider().getFeatures()):
		if geometries[index] != None:
			writecount += 1
			outfile.addFeature(feature)
				
	del outfile

	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(writecount) + " of " + \
		unicode(layer.dataProvider().featureCount()) + \
		" unique features written to " + savename)

	return None

# ---------------------------------------------------------
#    mmqgis_float_to_text - String format numeric fields
# ---------------------------------------------------------

def mmqgis_float_to_text(qgis, layername, attributes, separator, 
			decimals, prefix, suffix, savename, addlayer):

	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Project has no active vector layer to convert: " + layername

	if decimals < 0:
		return "Invalid number of decimals: " + unicode(decimals)

	if len(savename) <= 0:
		return "No output filename given"

	# Build dictionary of fields with selected fields for conversion to floating point
	changecount = 0
	fieldchanged = []
	destfields = QgsFields();
	for index, field in enumerate(layer.dataProvider().fields()):
		if field.name() in attributes:
			if (field.type() != QVariant.Int) and (field.type() != QVariant.Double):
				return "Cannot convert non-numeric field: " + unicode(field.name())
		
			changecount += 1
			fieldchanged.append(True)
			destfields.append(QgsField (field.name(), QVariant.String, field.typeName(), \
				20, 0, field.comment()))
		else:
			fieldchanged.append(False)
			destfields.append(QgsField (field.name(), field.type(), field.typeName(), \
				field.length(), field.precision(), field.comment()))

	if (changecount <= 0):
		return "No numeric fields selected for conversion"

	# Create the output file
	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", destfields,
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())


	# Write the features with modified attributes
	featurecount = layer.dataProvider().featureCount();
	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (feature_index % 50) == 0:
			mmqgis_status_message(qgis, "Writing feature " + \
				unicode(feature.id()) + " of " + unicode(featurecount))

		attributes = feature.attributes()
		for index, field in enumerate(layer.dataProvider().fields()):
			if fieldchanged[index]:
				floatvalue, test = attributes[index].toDouble()
				if not test:
					floatvalue = 0
				value = prefix + format_float(floatvalue, separator, decimals) + suffix
				attributes[index] = QVariant(value)

		feature.setAttributes(attributes)
		outfile.addFeature(feature)

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(changecount) + " numeric fields converted to text")

	return None

# --------------------------------------------------------------
#    mmqgis_geocode_google - Geocode CSV points from Google Maps
# --------------------------------------------------------------

def mmqgis_geocode_google(qgis, csvname, shapefilename, notfoundfile, keys, addlayer):
	# Read the CSV file header

	if (not csvname) or (len(csvname) <= 0):
		return "No CSV address file given"
	
	try:
		infile = open(csvname, 'r')
	except:
		return "Failure opening " + csvname

	try:
		dialect = csv.Sniffer().sniff(infile.read(2048))
	except:
		return "Failure reading " + unicode(csvname) + ": " + unicode(sys.exc_info()[1])


	fields = QgsFields()
	indices = []
	try:
		infile.seek(0)
		reader = csv.reader(infile, dialect)
		header = reader.next()
	except:
		return "Failure reading " + unicode(csvname) + ": " + unicode(sys.exc_info()[1])

	for x in range(0, len(header)):
		for y in range(0, len(keys)):
			if header[x] == keys[y]:
				indices.append(x)

		fieldname = header[x].strip()
		fields.append(QgsField(fieldname[0:9], QVariant.String))

	if (len(fields) <= 0) or (len(indices) <= 0):
		return "No valid location fields in " + csvname


	# Create the CSV file for ungeocoded records
	try:
		notfound = open(notfoundfile, 'w')
	except:
		return "Failure opening " + notfoundfile

	notwriter = csv.writer(notfound, dialect)
	notwriter.writerow(header)


	# Create the output shapefile
	if QFile(shapefilename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(shapefilename):
			return "Failure deleting existing shapefile: " + unicode(shapefilename)

	crs = QgsCoordinateReferenceSystem()
	crs.createFromSrid(4326)
	outfile = QgsVectorFileWriter(shapefilename, "utf-8", fields, QGis.WKBPoint, crs)

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Geocode and import
	recordcount = 0
	notfoundcount = 0
	for row in reader:
		time.sleep(0.5) # to avoid Google rate quota limits

		recordcount += 1	
		mmqgis_status_message(qgis, "Geocoding " + unicode(recordcount) + 
			" (" + unicode(notfoundcount) + " not found)")

		address = ""
		for x in indices:
			if x < len(row):
				# value = row[x].strip().replace(" ","+")
				try:
					# The unicode coversion throws an exception on non-utf-8 characters
					# However, urllib.quote() requires the encoded string or it throws an error
					utf8_test = unicode(row[x], "utf-8").strip()
					# print utf8_test
					value = urllib.quote(row[x].strip())
					# print value
				except:
					return "CSV file must be in UTF-8 encoding"

				if len(value) > 0:
					if x != indices[0]:
						address += "+"
					address += value

		if len(address) <= 0:
			notfoundcount += 1
			notwriter.writerow(row)
	
		else:
			url = "http://maps.googleapis.com/maps/api/geocode/xml?sensor=false&address=" + address
			# print url
			xml = urllib.urlopen(url).read()

			latstart = xml.find("<lat>")
			latend = xml.find("</lat>")
			longstart = xml.find("<lng>")
			longend = xml.find("</lng>")

			if (latstart > 0) and (latend > (latstart + 5)) and \
			   (longstart > 0) and (longend > (longstart + 5)):
				x = float(xml[longstart + 5:longend])
				y = float(xml[latstart + 5:latend])
				# print address + ": " + str(x) + ", " + str(y)

				attributes = []
				for z in range(0, len(header)):
					if z < len(row):
						attributes.append(QVariant(unicode(row[z], 'utf-8').strip()))

				#y = 40.714353
				#x = -74.005973 
				newfeature = QgsFeature()
				newfeature.setAttributes(attributes)
				geometry = QgsGeometry.fromPoint(QgsPoint(x, y))
				newfeature.setGeometry(geometry)
				outfile.addFeature(newfeature)

			else:
				notfoundcount += 1
				notwriter.writerow(row)
				# print xml

	del outfile
	del notfound

	if addlayer and (recordcount > notfoundcount) and (recordcount > 0):
		vlayer = qgis.addVectorLayer(shapefilename, os.path.basename(shapefilename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(recordcount - notfoundcount) + " of " + unicode(recordcount)
		+ " addresses geocoded with Google")

	return None


# --------------------------------------------------------
#    mmqgis_geocode_street_layer - Geocode addresses from street 
#			     address finder shapefile
# --------------------------------------------------------

def mmqgis_geocode_street_layer(qgis, layername, csvname, streetnamefield, numberfield, zipfield, \
	streetname, fromx, fromy, tox, toy, leftfrom, rightfrom, leftto, rightto, leftzip, rightzip, \
	setback, shapefilename, notfoundfile, addlayer):

	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Address layer not found: " + layername

	if len(csvname) <= 0:
		return "No CSV address file given"

	# Read the CSV file data into memory
	try:
		infile = open(csvname, 'r')
	except:
		return "Failure opening " + csvname

	try:
		dialect = csv.Sniffer().sniff(infile.read(2048))
	except:
		return "Bad CSV file (verify that your delimiters are consistent): " + csvname

	infile.seek(0)
	reader = csv.reader(infile, dialect)

	# Build attribute fields for geocoded address shapefile
	fields = QgsFields()
	header = reader.next()
	streetnamefield_index = -1
	numberfield_index = -1
	zipfield_index = -1
	for index, field in enumerate(header):
		try:
			field = unicode(field, "utf-8")
		except:
			return "CSV file must be in UTF-8 encoding"

		if field == streetnamefield:
			streetnamefield_index = index
		if field == numberfield:
			numberfield_index = index
		if (zipfield != None) and (field == zipfield):
			zipfield_index = index
		fields.append(QgsField(field[0:9].strip(), QVariant.String))

	fields.append(QgsField("Longitude", QVariant.Double, "real", 24, 16))
	fields.append(QgsField("Latitude", QVariant.Double, "real", 24, 16))

	if streetnamefield_index < 0:
		return "Invalid street name field: " + str(streetnamefield)
	if (numberfield_index < 0):
		return "Invalid street number field: " + str(numberfield)


	# Read CSV addresses into memory and convert to unicode
	addresses = []
	for row in reader:
		for index in range(0, len(row)):
			try:
				row[index] = unicode(row[index], "utf-8")
			except:
				return "CSV file must be in UTF-8 encoding"

		try:
			message = row[numberfield_index]
			row[numberfield_index] = int(row[numberfield_index])
		except:
			return "Invalid street address number: " + message

		addresses.append(row)

	del reader
	del infile

	# Create the output shapefile
	if QFile(shapefilename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(shapefilename):
			return "Failure deleting existing shapefile: " + shapefilename

	outfile = QgsVectorFileWriter(shapefilename, "utf-8", fields, QGis.WKBPoint, layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	fromx_attribute = None
	fromy_attribute = None
	tox_attribute = None
	toy_attribute = None
	if (fromx != "(street line start)") and (fromx != "(street line end)"):
		fromx_attribute = layer.dataProvider().fieldNameIndex(fromx)
	if (fromy != "(street line start)") and (fromy != "(street line end)"):
		fromy_attribute = layer.dataProvider().fieldNameIndex(fromy)
	if (tox != "(street line start)") and (tox != "(street line end)"):
		tox_attribute = layer.dataProvider().fieldNameIndex(tox)
	if (toy != "(street line start)") and (toy != "(street line end)"):
		toy_attribute = layer.dataProvider().fieldNameIndex(toy)

	streetname_attribute = layer.dataProvider().fieldNameIndex(streetname)
	leftfrom_attribute = layer.dataProvider().fieldNameIndex(leftfrom)
	rightfrom_attribute = layer.dataProvider().fieldNameIndex(rightfrom)
	leftto_attribute = layer.dataProvider().fieldNameIndex(leftto)
	rightto_attribute = layer.dataProvider().fieldNameIndex(rightto)

	leftzip_attribute = -1
	rightzip_attribute = -1
	if leftzip:
		leftzip_attribute = layer.dataProvider().fieldNameIndex(leftzip)
	if rightzip:
		rightzip_attribute = layer.dataProvider().fieldNameIndex(rightzip)

	# Iterate through each feature in the source layer
	matched_count = 0
	feature_count = layer.dataProvider().featureCount()
	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (feature.id() % 10) == 0:
			mmqgis_status_message(qgis, "Searching street " + \
				unicode(feature_index) + " of " + unicode(feature_count) + \
				" (" + unicode(matched_count) + " matched)")

		attributes = feature.attributes()
		feature_streetname = mmqgis_searchable_streetname(unicode(attributes[streetname_attribute].toString()))

		# Check each address against this feature
		for row_index, row in enumerate(addresses):
			street = mmqgis_searchable_streetname(row[streetnamefield_index].lower())
			# print "Compare " + str(feature_streetname) + " ?= " + str(street)

			if feature_streetname == street:
				# print "   Name match " + str(street) + ", feture " + str(feature.id())
				# print str(len(attributes)) + "," + str(leftto_attribute) + "," + str(leftfrom_attribute) + \
				#	"," + str(rightto_attribute) + "," + str(rightfrom_attribute)

				# Find range of street numbers on this feature
				(leftto_number, test) = attributes[leftto_attribute].toInt()
				(leftfrom_number, test) = attributes[leftfrom_attribute].toInt()
				(rightto_number, test) = attributes[rightto_attribute].toInt()
				(rightfrom_number, test) = attributes[rightfrom_attribute].toInt()
				if leftzip_attribute >= 0:
					leftzipcode = unicode(attributes[leftzip_attribute].toString())
				else:
					leftzipcode = None
				if rightzip_attribute >= 0:
					rightzipcode = unicode(attributes[leftzip_attribute].toString())
				else:
					rightzipcode = None
				
				number = row[numberfield_index]
				if zipfield_index >= 0:
					zipcode = row[zipfield_index]
				else:
					zipcode = None

				# Check address number
				if ((leftto_number >= leftfrom_number) \
				    and (number >= leftfrom_number) \
				    and (number <= leftto_number) \
				    and ((leftfrom_number % 2) == (number % 2)) \
				    and ((leftzipcode == None) or (zipcode == None) or (zipcode == leftzipcode))) or \
				    ((leftto_number < leftfrom_number) \
				    and (number >= leftto_number) \
				    and (number <= leftfrom_number) \
				    and ((leftfrom_number % 2) == (number % 2)) \
				    and ((leftzipcode == None) or (zipcode == None) or (zipcode == leftzipcode))) or \
				   ((rightto_number >= rightfrom_number) \
				    and (number >= rightfrom_number) \
				    and (number <= rightto_number) \
				    and ((rightfrom_number % 2) == (number % 2))
				    and ((rightzipcode == None) or (zipcode == None) or (zipcode == rightzipcode))) or \
				   ((rightto_number < rightfrom_number) \
				    and (number >= rightto_number) \
				    and (number <= rightfrom_number) \
				    and ((rightfrom_number % 2) == (number % 2))
				    and ((rightzipcode == None) or (zipcode == None) or (zipcode == rightzipcode))):

					# Find line start and end points
					geometry = feature.geometry()
					if (geometry.wkbType() == QGis.WKBLineString):
						line = geometry.asPolyline()
						fromx = line[0].x()
						fromy = line[0].y()
						tox = line[len(line) - 1].x()
						toy = line[len(line) - 1].y()

					elif (geometry.wkbType() == QGis.WKBMultiLineString):
						lines = geometry.asMultiPolyline()
						line = lines[0]
						fromx = line[0].x()
						fromy = line[0].y()
						line = lines[len(lines) - 1]
						tox = line[len(line) - 1].x()
						toy = line[len(line) - 1].y()

					else:
						return "Street layer must be a lines or multilines"

					# Use attribute values if specified
					if tox_attribute:					
						(tox, test) = attributes[tox_attribute].toDouble()
					if toy_attribute:
						(toy, test) = attributes[toy_attribute].toDouble()
					if fromx_attribute:
						(fromx, test) = attributes[fromx_attribute].toDouble()
					if fromy_attribute:
						(fromy, test) = attributes[fromy_attribute].toDouble()


					# Find percentage distance along street
					left = ((leftfrom_number % 2) == (number % 2))
					if left:
						if (leftfrom_number == leftto_number):
							ratio = 0.5
						else:
							ratio = float(number - leftfrom_number) \
								/ float(leftto_number - leftfrom_number)
					else:
						if (rightfrom_number == rightto_number):
							ratio = 0.5
						else:
							ratio = float(number - rightfrom_number) \
								/ float(rightto_number - rightfrom_number)

					# setback from corner
					angle = atan2(toy - fromy, tox - fromx)
					setback_fromx = fromx + (setback * cos(angle))
					setback_tox = tox - (setback * cos(angle))
					setback_fromy = fromy + (setback * sin(angle))
					setback_toy = toy - (setback * sin(angle))

					x = setback_fromx + ((setback_tox - setback_fromx) * ratio)
					y = setback_fromy + ((setback_toy - setback_fromy) * ratio)

					# setback from street center
					if left:
						y += (setback * cos(angle))
						x -= (setback * sin(angle))
					else:
						y -= (setback * cos(angle))
						x += (setback * sin(angle))

					# Create the output feature
					newattributes = []
					for field in row:
						newattributes.append(QVariant(field))

					newattributes.append(QVariant(x))
					newattributes.append(QVariant(y))

					newfeature = QgsFeature()
					newfeature.setAttributes(newattributes)
					geometry = QgsGeometry.fromPoint(QgsPoint(x, y))
					newfeature.setGeometry(geometry)
					outfile.addFeature(newfeature)
					matched_count += 1

					# Empty address so not searched further
					row[streetnamefield_index] = ""

	del outfile

	# Write records that were not joined to the notfound file
	try:
		outfile = open(notfoundfile, 'w')
	except:
		return "Failure opening " + notfoundfile
	else:
                # Encoding is forced to UTF-8 because CSV writer doesn't support Unicode
                writer = csv.writer(outfile, dialect)
                writer.writerow(header)
		for index, row in enumerate(addresses):
			if row[streetnamefield_index] > "":
				for x in range(0, len(row)):
					row[x] = unicode(row[x]).encode('utf-8')
				writer.writerow(row)
                del outfile


	if matched_count and addlayer:
		vlayer = qgis.addVectorLayer(shapefilename, os.path.basename(shapefilename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(matched_count) + " of " + unicode(len(addresses)) \
		+ " addresses geocoded from " + unicode(feature_count) + " street records")

	return None


# --------------------------------------------------------
#    mmqgis_geometry_convert - Convert geometries to
#		simpler types
# --------------------------------------------------------

def mmqgis_geometry_convert(qgis, layername, newtype, splitnodes, savename, addlayer):
	layer = mmqgis_find_layer(layername)

	if (layer == None) and (layer.type() != QgsMapLayer.VectorLayer):
		return "Invalid Vector Layer " + layername

	# Create output file
	if len(savename) <= 0:
		return "Invalid output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(), newtype, layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Iterate through each feature in the source layer
	feature_count = layer.dataProvider().featureCount()
	out_count = 0

        for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		# shapeid = unicode(feature.id()).strip()

		if (feature_index % 10) == 0:
			mmqgis_status_message(qgis, "Converting feature " + str(feature_index) \
				+ " of " + unicode(feature_count))

		if (feature.geometry().wkbType() == QGis.WKBPoint) or \
		   (feature.geometry().wkbType() == QGis.WKBPoint25D):

			if (newtype == QGis.WKBPoint):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(QgsGeometry.fromPoint(feature.geometry().asPoint()))
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			else:
				return "Invalid Conversion: " + mmqgis_wkbtype_to_text(feature.geometry().wkbType()) + \
					" to " + mmqgis_wkbtype_to_text(newtype)

		elif (feature.geometry().wkbType() == QGis.WKBLineString) or \
		     (feature.geometry().wkbType() == QGis.WKBLineString25D):

			if (newtype == QGis.WKBPoint) and splitnodes:
				polyline = feature.geometry().asPolyline()
				for point in polyline:
					newfeature = QgsFeature()
					newfeature.setAttributes(feature.attributes())
					newfeature.setGeometry(QgsGeometry.fromPoint(point))
					outfile.addFeature(newfeature)
					out_count = out_count + 1

			elif (newtype == QGis.WKBPoint):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry().centroid())
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			elif (newtype == QGis.WKBLineString):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry())
				outfile.addFeature(newfeature)
				out_count = out_count + 1
				
			else:
				return "Invalid Conversion: " + mmqgis_wkbtype_to_text(feature.geometry().wkbType()) + \
					" to " + mmqgis_wkbtype_to_text(newtype)

		elif (feature.geometry().wkbType() == QGis.WKBPolygon) or \
		     (feature.geometry().wkbType() == QGis.WKBPolygon25D):

			if (newtype == QGis.WKBPoint) and splitnodes:
				polygon = feature.geometry().asPolygon()
				for polyline in polygon:
					for point in polyline:
						newfeature = QgsFeature()
						newfeature.setAttributes(feature.attributes())
						newfeature.setGeometry(QgsGeometry.fromPoint(point))
						outfile.addFeature(newfeature)
						out_count = out_count + 1

			elif (newtype == QGis.WKBPoint):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry().centroid())
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			elif (newtype == QGis.WKBLineString):
				polygon = feature.geometry().asPolygon()
				for polyline in polygon:
					newfeature = QgsFeature()
					newfeature.setAttributes(feature.attributes())
					newfeature.setGeometry(QgsGeometry.fromPolyline(polyline))
					outfile.addFeature(newfeature)
					out_count = out_count + 1

			elif (newtype == QGis.WKBMultiLineString):
				linestrings = []
				polygon = feature.geometry().asPolygon()
				for polyline in polygon:
					linestrings.append(polyline)

				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(QgsGeometry.fromMultiPolyline(linestrings))
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			elif (newtype == QGis.WKBPolygon):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry())
				outfile.addFeature(newfeature)
				out_count = out_count + 1
				
			else:
				return "Invalid Conversion: " + mmqgis_wkbtype_to_text(feature.geometry().wkbType()) + \
					" to " + mmqgis_wkbtype_to_text(newtype)

		elif (feature.geometry().wkbType() == QGis.WKBMultiPoint) or \
		     (feature.geometry().wkbType() == QGis.WKBMultiPoint25D):

			if (newtype == QGis.WKBPoint) and splitnodes:
				points = feature.geometry().asMultiPoint()
				for point in points:
					newfeature = QgsFeature()
					newfeature.setAttributes(feature.attributes())
					newfeature.setGeometry(QgsGeometry.fromPoint(point))
					outfile.addFeature(newfeature)
					out_count = out_count + 1

			elif (newtype == QGis.WKBPoint):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry().centroid())
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			else:
				return "Invalid Conversion: " + mmqgis_wkbtype_to_text(feature.geometry().wkbType()) + \
					" to " + mmqgis_wkbtype_to_text(newtype)


		elif (feature.geometry().wkbType() == QGis.WKBMultiLineString) or \
		     (feature.geometry().wkbType() == QGis.WKBMultiLineString25D):

			if (newtype == QGis.WKBPoint) and splitnodes:
				polylines = feature.geometry().asMultiPolyline()
				for polyline in polylines:
					for point in polyline:
						newfeature = QgsFeature()
						newfeature.setAttributes(feature.attributes())
						newfeature.setGeometry(QgsGeometry.fromPoint(point))
						outfile.addFeature(newfeature)
						out_count = out_count + 1

			elif (newtype == QGis.WKBPoint):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry().centroid())
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			elif (newtype == QGis.WKBLineString):
				linestrings = feature.geometry().asMultiPolyline()
				for linestring in linestrings:
					newfeature = QgsFeature()
					newfeature.setAttributes(feature.attributes())
					newfeature.setGeometry(QgsGeometry.fromPolyline(linestring))
					outfile.addFeature(newfeature)
					out_count = out_count + 1

			elif (newtype == QGis.WKBMultiLineString):
				linestrings = feature.geometry().asMultiPolyline()
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(QgsGeometry.fromMultiPolyline(linestrings))
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			else:
				return "Invalid Conversion: " + mmqgis_wkbtype_to_text(feature.geometry().wkbType()) + \
					" to " + mmqgis_wkbtype_to_text(newtype)

		elif (feature.geometry().wkbType() == QGis.WKBMultiPolygon) or \
		     (feature.geometry().wkbType() == QGis.WKBMultiPolygon25D):

			if (newtype == QGis.WKBPoint) and splitnodes:
				polygons = feature.geometry().asMultiPolygon()
				for polygon in polygons:
					for polyline in polygon:
						for point in polyline:
							newfeature = QgsFeature()
							newfeature.setAttributes(feature.attributes())
							newfeature.setGeometry(QgsGeometry.fromPoint(point))
							outfile.addFeature(newfeature)
							out_count = out_count + 1
	
			elif (newtype == QGis.WKBPoint):
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(feature.geometry().centroid())
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			elif (newtype == QGis.WKBLineString):
				polygons = feature.geometry().asMultiPolygon()
				for polygon in polygons:
					for polyline in polygon:
						newfeature = QgsFeature()
						newfeature.setAttributes(feature.attributes())
						newfeature.setGeometry(QgsGeometry.fromPolyline(polyline))
						outfile.addFeature(newfeature)
						out_count = out_count + 1

			elif (newtype == QGis.WKBPolygon):
				polygons = feature.geometry().asMultiPolygon()
				for polygon in polygons:
					newfeature = QgsFeature()
					newfeature.setAttributes(feature.attributes())
					newfeature.setGeometry(QgsGeometry.fromPolygon(polygon))
					outfile.addFeature(newfeature)
					out_count = out_count + 1

			elif (newtype == QGis.WKBMultiLineString) or \
			     (newtype == QGis.WKBMultiPolygon):
				polygons = feature.geometry().asMultiPolygon()
				newfeature = QgsFeature()
				newfeature.setAttributes(feature.attributes())
				newfeature.setGeometry(QgsGeometry.fromMultiPolygon(polygons))
				outfile.addFeature(newfeature)
				out_count = out_count + 1

			else:
				return "Invalid Conversion: " + mmqgis_wkbtype_to_text(feature.geometry().wkbType()) + \
					" to " + mmqgis_wkbtype_to_text(newtype)
			
	del outfile

	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")

	mmqgis_completion_message(qgis, unicode(feature_count) + " features converted to " + unicode(out_count) + " features")

	return None

# --------------------------------------------------------
#    mmqgis_geometry_to_multipart - Convert singlepart 
#		to multipart geometries
# --------------------------------------------------------

def mmqgis_geometry_to_multipart(qgis, layername, mergefield, mergeattop, savename, addlayer):

	# Error checking
	layer = mmqgis_find_layer(layername)
	if (layer == None) and (layer.type() != QgsMapLayer.VectorLayer):
		return "Invalid Vector Layer " + layername

	if (layer.dataProvider().geometryType() == QGis.WKBPoint) or \
	   (layer.dataProvider().geometryType() == QGis.WKBPoint25D):
		newtype = QGis.WKBMultiPoint

	elif (layer.dataProvider().geometryType() == QGis.WKBLineString) or \
	     (layer.dataProvider().geometryType() == QGis.WKBLineString25D):
		newtype = QGis.WKBMultiLineString

	elif (layer.dataProvider().geometryType() == QGis.WKBPolygon) or \
	     (layer.dataProvider().geometryType() == QGis.WKBPolygon25D):
		newtype = QGis.WKBMultiPolygon

	else:
		return "Geometry is already multipart: " + mmqgis_wkbtype_to_text(layer.dataProvider().geometryType())

	merge_index = layer.dataProvider().fields().indexFromName(mergefield)
	if merge_index < 0:
		return "Invalid merge field: " + mergefield

	
	# Create output file
	if len(savename) <= 0:
		return "Invalid output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", 
		layer.dataProvider().fields(), newtype, layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Have to read features into memory because nested loops of getFeature() don't work
	feature_count = layer.dataProvider().featureCount()
        features = []
	for index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (index % 10) == 0:
			mmqgis_status_message(qgis, "Reading feature " + unicode(index) \
				+ " of " + unicode(feature_count))
		features.append(feature)
	

	# Iterate through each feature in the source layer
	merge_count = 0
	for x in range(0, len(features)):
		if (x % 10) == 0:
			mmqgis_status_message(qgis, "Converting feature " + str(x) \
				+ " of " + unicode(len(features)))

		if features[x] != None:
			attributes = features[x].attributes()
			key = unicode(attributes[merge_index].toString()).lower()
			# print "Processing " + unicode(x) + ": " + key

			newgeometry = []
			if newtype == QGis.WKBMultiPoint:
				if (feature.geometry().wkbType() == QGis.WKBPoint) or \
				   (feature.geometry().wkbType() == QGis.WKBPoint25D):
					newgeometry.append(features[x].geometry().asPoint())

				elif (feature.geometry().wkbType() == QGis.WKBMultiPoint) or \
				     (feature.geometry().wkbType() == QGis.WKBMultiPoint25D):
					for point in features[x].geometry().asMultiPoint():
						newgeometry.append(point)
				else:
					return "Invalid multipoint geometry type: " + \
						mmqgis_wkbtype_to_text(features[x].geometry().wkbType())

			elif newtype == QGis.WKBMultiLineString:
				# This is a workaround since shapefiles do not distinguish
				# between polylines and multipolylines - all polygons can have multiple
				# parts. QgsGeometry.wkbType() returns WKBLineString even if the 
				# geometry is WKBMultiLineString

				#if (feature.geometry().wkbType() == QGis.WKBLineString) or \
				#   (feature.geometry().wkbType() == QGis.WKBLineString25D):

				if len(features[x].geometry().asPolyline()) > 0:
					newgeometry.append(features[x].geometry().asPolyline())

				#elif (feature.geometry().wkbType() == QGis.WKBMultiLineString) or \
				#     (feature.geometry().wkbType() == QGis.WKBMultiLineString25D):

				elif len(features[x].geometry().asMultiPolyline()) > 0:
					for polyline in features[x].geometry().asMultiPolyline():
						newgeometry.append(polyline)
				else:
					return "Invalid multilinestring geometry type: " + \
						mmqgis_wkbtype_to_text(features[x].geometry().wkbType())

			else: # newtype == QGis.WKBMultiPolygon:
				# This is a workaround since shapefiles do not distinguish
				# between polygons and multipolygons - all polygons can have multiple
				# parts. QgsGeometry.wkbType() returns WKBPolygon even if the 
				# geometry is WKBMultiPolygon

				#if (feature.geometry().wkbType() == QGis.WKBPolygon) or \
				#   (feature.geometry().wkbType() == QGis.WKBPolygon25D):

				if len(features[x].geometry().asPolygon()) > 0:
					newgeometry.append(features[x].geometry().asPolygon())

				#elif (feature.geometry().wkbType() == QGis.WKBMultiPolygon) or \
				#     (feature.geometry().wkbType() == QGis.WKBMultiPolygon25D):

				elif len(features[x].geometry().asMultiPolygon()) > 0:
					for polygon in features[x].geometry().asMultiPolygon():
						newgeometry.append(polygon)
				else:
					return "Invalid multipolygon geometry type: " + \
						mmqgis_wkbtype_to_text(features[x].geometry().wkbType())

			for y in range(x + 1, len(features)):
				#print "   Comparing " + unicode(y)
				if (features[y] != None) and \
				   (unicode(features[y].attributes()[merge_index].toString()).lower() == key):
					#print "   Match " + unicode(y)

					if newtype == QGis.WKBMultiPoint:
						newgeometry.append(features[y].geometry().asPoint())

					elif newtype == QGis.WKBMultiLineString:
						newgeometry.append(features[y].geometry().asPolyline())

					else: # QGis.WKBMultiPolygon:
						newgeometry.append(features[y].geometry().asPolygon())
					
					if mergeattop == "Sum":
						for zindex, zvalue in enumerate(features[y].attributes()):
							if (zvalue.type() == QVariant.Int):
								xval, test = attributes[zindex].toInt()
								yval, test = features[y].attributes()[zindex].toInt()
								attributes[zindex] = QVariant(xval + yval)

							elif (zvalue.type() == QVariant.Double):
								xval, test = attributes[zindex].toDouble()
								yval, test = features[y].attributes()[zindex].toDouble()
								attributes[zindex] = QVariant(xval + yval)

							# print "      Sum " + unicode(zindex) + ": " + \
							#	unicode(attributes[zindex].typeName())

					features[y] = None
								
			newfeature = QgsFeature()
			newfeature.setAttributes(attributes)

			if newtype == QGis.WKBMultiPoint:
				newfeature.setGeometry(QgsGeometry.fromMultiPoint(newgeometry))

			elif newtype == QGis.WKBMultiLineString:
				newfeature.setGeometry(QgsGeometry.fromMultiPolyline(newgeometry))

			else: # WKBMultiPolygon:
				newfeature.setGeometry(QgsGeometry.fromMultiPolygon(newgeometry))

			outfile.addFeature(newfeature)
			merge_count = merge_count + 1

	del outfile

	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")

	mmqgis_completion_message(qgis, unicode(feature_count) + 
		" features merged to " + unicode(merge_count) + " features")

	return None



# --------------------------------------------------------
#    mmqgis_geometry_export_to_csv - Shape node dump to CSV
# --------------------------------------------------------

def mmqgis_geometry_export_to_csv(qgis, layername, node_filename, attribute_filename, field_delimiter, line_terminator):
	layer = mmqgis_find_layer(layername)

	if (layer == None) or (layer.type() != QgsMapLayer.VectorLayer):
		return "Invalid Vector Layer " + layername

	node_header = ["shapeid", "x", "y"]
	attribute_header = ["shapeid"]
	for index, field in enumerate(layer.dataProvider().fields()):
		if (layer.geometryType() == QGis.Point):
			node_header.append(field.name())
		else:
			attribute_header.append(field.name())

	try:
		nodefile = open(node_filename, 'w')
    	except:
		return "Failure opening " + node_filename

	node_writer = csv.writer(nodefile, delimiter = field_delimiter, 
		lineterminator = line_terminator, quoting=csv.QUOTE_NONNUMERIC)
	node_writer.writerow(node_header)

	if (layer.geometryType() != QGis.Point):
		try:
			attributefile = open(attribute_filename, 'w')
 	   	except:
			return "Failure opening " + attribute_filename

		attribute_writer = csv.writer(attributefile, delimiter = field_delimiter,
			lineterminator = line_terminator, quoting=csv.QUOTE_NONNUMERIC)
		attribute_writer.writerow(attribute_header)


	# Iterate through each feature in the source layer

	#feature = QgsFeature()
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
        #while layer.dataProvider().nextFeature(feature):

	feature_type = ""
	feature_count = layer.dataProvider().featureCount()
	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		feature_type = unicode(mmqgis_wkbtype_to_text(feature.geometry().wkbType()))
		# shapeid = unicode(feature.id()).strip()
		# print "Feature " + str(feature_index) + " = " + feature_type

		if (feature_index % 10) == 0:
			mmqgis_status_message(qgis, "Exporting feature " + unicode(feature_index) \
				+ " of " + unicode(feature_count))

		if (feature.geometry() == None):
			return "Cannot export layer with no shape data"

		elif (feature.geometry().wkbType() == QGis.WKBPoint) or \
		     (feature.geometry().wkbType() == QGis.WKBPoint25D):
			point = feature.geometry().asPoint()
			row = [ unicode(feature_index), unicode(point.x()), unicode(point.y()) ]
			for attindex, attribute in enumerate(feature.attributes()):
				row.append(unicode(attribute.toString()).encode("utf-8"))
			node_writer.writerow(row)

		elif (feature.geometry().wkbType() == QGis.WKBMultiPoint) or \
		     (feature.geometry().wkbType() == QGis.WKBMultiPoint25D):
			points = feature.geometry().asMultiPoint()
			for point_index, point in enumerate(points):
				shape_id = unicode(feature_index) + "." + unicode(point_index)
				row = [ shape_id, unicode(point.x()), unicode(point.y()) ]
				for attindex, attribute in enumerate(feature.attributes()):
					row.append(unicode(attribute.toString()).encode("utf-8"))
				node_writer.writerow(row)

		elif (feature.geometry().wkbType() == QGis.WKBLineString) or \
		     (feature.geometry().wkbType() == QGis.WKBLineString25D):
			polyline = feature.geometry().asPolyline()
			for point in polyline:
				# print "  Point " + str(point.x()) + ", " + str(point.y())
				row = [ unicode(feature_index), unicode(point.x()), unicode(point.y()) ]
				node_writer.writerow(row)

			row = [ feature_index ]
			for attindex, attribute in enumerate(feature.attributes()):
				row.append(attribute.toString())
			attribute_writer.writerow(row)

		elif (feature.geometry().wkbType() == QGis.WKBMultiLineString) or \
		     (feature.geometry().wkbType() == QGis.WKBMultiLineString25D):
			polylines = feature.geometry().asMultiPolyline()
			for polyline_index, polyline in enumerate(polylines):
				shape_id = unicode(feature_index) + "." + unicode(polyline_index)
				for point in polyline:
					# print "  Point " + str(point.x()) + ", " + str(point.y())
					row = [ shape_id, unicode(point.x()), unicode(point.y()) ]
					node_writer.writerow(row)

				row = [ shape_id ]
				for attindex, attribute in enumerate(feature.attributes()):
					row.append(attribute.toString())
				attribute_writer.writerow(row)

		elif (feature.geometry().wkbType() == QGis.WKBPolygon) or \
		     (feature.geometry().wkbType() == QGis.WKBPolygon25D):
			polygon = feature.geometry().asPolygon()
			for polyline in polygon:
				for point in polyline:
					row = [ unicode(feature_index), unicode(point.x()), unicode(point.y()) ]
					node_writer.writerow(row)

			row = [ feature_index ]
			for attindex, attribute in enumerate(feature.attributes()):
				row.append(unicode(attribute.toString()).encode("utf-8"))
			attribute_writer.writerow(row)

		elif (feature.geometry().wkbType() == QGis.WKBMultiPolygon) or \
		     (feature.geometry().wkbType() == QGis.WKBMultiPolygon25D):
			multipolygon = feature.geometry().asMultiPolygon()
			for polygon_index, polygon in enumerate(multipolygon):
				shape_id = unicode(feature_index) + "." + unicode(polygon_index)
				for polyline in polygon:
					for point in polyline:
						row = [ shape_id, unicode(point.x()), unicode(point.y()) ]
						node_writer.writerow(row)

				row = [ shape_id ]
				for attindex, attribute in enumerate(feature.attributes()):
					row.append(unicode(attribute.toString()).encode("utf-8"))
				attribute_writer.writerow(row)

		else:
			return "Unsupported geometry: " + unicode(mmqgis_wkbtype_to_text(feature.geometry().wkbType()))
			
	del nodefile
	if (layer.geometryType() != QGis.Point):
		del attributefile

	mmqgis_completion_message(qgis, unicode(feature_count) + " records exported (" + feature_type + ")")

	return None


# --------------------------------------------------------
#    mmqgis_geometry_import_from_csv - Shape node import from CSV
# --------------------------------------------------------

def mmqgis_geometry_import_from_csv(qgis, node_filename, long_colname, lat_colname, 
	shapeid_colname, geometry_type, shapefile_name, addlayer):
	try:
		infile = open(node_filename, 'r')
	except:
		return "Failure opening " + node_filename
			
	try:
		dialect = csv.Sniffer().sniff(infile.read(2048))
	except:
		return "Bad CSV file (verify that your delimiters are consistent): " + node_filename

	infile.seek(0)
	reader = csv.reader(infile, dialect)
	header = reader.next()

	lat_col = -1
	long_col = -1
	shapeid_col = -1
	for x in range(len(header)):
		# print header[x]
		if (header[x] == lat_colname):
			lat_col = x
		elif (header[x] == long_colname):
			long_col = x
		elif (header[x] == shapeid_colname):
			shapeid_col = x

	if (lat_col < 0):
		return "Invalid latitude column name: " + lat_colname

	if (long_col < 0):
		return "Invalid longitude column name: " + long_colname

	if (shapeid_col < 0):
		return "Invalid shape ID column name: " + shapeid_colname

	if (geometry_type == "Point"):
		wkb_type = QGis.WKBPoint

	elif (geometry_type == "Polyline"):
		wkb_type = QGis.WKBLineString

	elif (geometry_type == "Polygon"):
		wkb_type = QGis.WKBPolygon
	else:
		return "Invalid geometry type: " + geometry_type

	# Create the output shapefile
	if QFile(shapefile_name).exists():
		if not QgsVectorFileWriter.deleteShapeFile(shapefile_name):
			return "Failure deleting existing shapefile: " + shapefile_name

	if qgis.activeLayer() and qgis.activeLayer().dataProvider():
		crs = qgis.activeLayer().dataProvider().crs()
	else:
		crs = QgsCoordinateReferenceSystem()
		crs.createFromSrid(4326) # WGS 84

	fields = QgsFields()
	fields.append(QgsField(shapeid_colname, QVariant.String))
	if (geometry_type == "Point"):
		for x in range(len(header)):
			if ((x != lat_col) and (x != long_col) and (x != shapeid_col)):
				fields.append(QgsField(header[x], QVariant.String))

	outfile = QgsVectorFileWriter(shapefile_name, "utf-8", fields, wkb_type, crs)

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	polyline = []
	node_count = 0
	shape_count = 0
	current_shape_id = False
	reading = True
	while reading:
		try:
			row = reader.next()
		except:
			reading = False

		if reading and (len(row) > long_col) and (len(row) > lat_col) and (len(row) > shapeid_col) \
				and mmqgis_is_float(row[long_col]) and mmqgis_is_float(row[lat_col]):
			node_count += 1
			if (node_count % 10) == 0:
				mmqgis_status_message(qgis, "Importing node " + unicode(node_count))
			point = QgsPoint(float(row[long_col]), float(row[lat_col]))
		else:
			point = False

		if reading and (wkb_type != QGis.WKBPoint) and (row[shapeid_col] == current_shape_id):
			polyline.append(point)

		else:
			#print str(wkb_type) + ": " + str(current_shape_id)
			#print polyline

			bad_feature = False
			if wkb_type == QGis.WKBPoint:
				if point:
					geometry = QgsGeometry.fromPoint(point)
					current_shape_id = row[shapeid_col]
				else:
					bad_feature = True

			elif wkb_type == QGis.WKBLineString:
				if len(polyline) < 2:
					bad_feature = True
				else:
					geometry = QgsGeometry.fromPolyline(polyline)

			elif wkb_type == QGis.WKBPolygon:
				if len(polyline) < 3:
					bad_feature = True
				else:
					# polyline[len(polyline) - 1] = polyline[0] # must close polygons
					polygon = [ polyline ]
					geometry = QgsGeometry.fromPolygon(polygon)

			if not bad_feature:
				# attributes = QgsAttributes()
				attributes = [ QVariant(str(current_shape_id)) ]
				if (geometry_type == "Point"):
					for x in range(len(header)):
						if x >= len(row):
							attributes.append(QVariant(""))
						elif ((x != lat_col) and (x != long_col) and (x != shapeid_col)):
							attributes.append(QVariant(unicode(row[x], 'utf-8')))

				#print attributes
				newfeature = QgsFeature()
				newfeature.setAttributes(attributes)
				newfeature.setGeometry(geometry)
				outfile.addFeature(newfeature)
				shape_count += 1
	
			polyline = []
			if reading and point:
				current_shape_id = row[shapeid_col]
				polyline.append(point)

	del infile
	del outfile

	if addlayer:
		qgis.addVectorLayer(shapefile_name, os.path.basename(shapefile_name), "ogr")
		
	mmqgis_completion_message(qgis, "Loaded " + unicode(shape_count) + " shapes (" + unicode(node_count) + " nodes)")

	return None

# --------------------------------------------------------
#    mmqgis_grid - Grid shapefile creation
# --------------------------------------------------------

def mmqgis_grid(qgis, savename, hspacing, vspacing, width, height, originx, originy, gridtype, addlayer):
	if len(savename) <= 0:
		return "No output filename given"

	if (hspacing <= 0) or (vspacing <= 0):
		return "Invalid grid spacing: " + unicode(hspacing) + " / " + unicode(vspacing)
	
	if (width <= hspacing) or (width < vspacing):
		return "Invalid width / height: " + unicode(width) + " / " + unicode(height)
		
	fields = QgsFields()
	fields.append(QgsField("longitude", QVariant.Double, "real", 24, 16, "Longitude"))
	fields.append(QgsField("latitude", QVariant.Double, "real", 24, 16, "Latitude"))

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename


	if gridtype.find("polygon") >= 0:
		shapetype = QGis.WKBPolygon
	else:
		shapetype = QGis.WKBLineString

	# print gridtype + "," + str(shapetype)
		
	outfile = QgsVectorFileWriter(savename, "utf-8", fields, shapetype, QgsCoordinateReferenceSystem());

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	linecount = 0
	if gridtype == "Rectangle (line)":
		x = originx
		while x <= (originx + width):
			polyline = []
			geometry = QgsGeometry()
			feature = QgsFeature()
			
			y = originy
			while y <= (originy + height):
				polyline.append(QgsPoint(x, y))
				y = y + vspacing;

			feature.setGeometry(geometry.fromPolyline(polyline))
			feature.setAttributes([QVariant(x), QVariant(0)])
			outfile.addFeature(feature)
			linecount = linecount + 1
			x = x + hspacing;

		y = originy
		while y <= (originy + height):
			polyline = []
			geometry = QgsGeometry()
			feature = QgsFeature()
			
			x = originx
			while x <= (originx + width):
				polyline.append(QgsPoint(x, y))
				x = x + hspacing;

			feature.setGeometry(geometry.fromPolyline(polyline))
			feature.setAttributes([QVariant(0), QVariant(y)])
			outfile.addFeature(feature)
			linecount = linecount + 1
			y = y + vspacing;

	elif gridtype == "Rectangle (polygon)":
		x = originx
		while x < (originx + width):
			y = originy
			while y < (originy + height):
				polyline = []
				polyline.append(QgsPoint(x, y))
				polyline.append(QgsPoint(x + hspacing, y))
				polyline.append(QgsPoint(x + hspacing, y + vspacing))
				polyline.append(QgsPoint(x, y + vspacing))

				geometry = QgsGeometry()
				feature = QgsFeature()
				feature.setGeometry(geometry.fromPolygon([polyline]))
				feature.setAttributes([
					QVariant(x + (hspacing / 2.0)),
					QVariant(y + (vspacing / 2.0)) ])
				outfile.addFeature(feature)
				linecount = linecount + 1
				y = y + vspacing;

			x = x + hspacing

	elif gridtype == "Diamond (polygon)":
		x = originx
		colnum = 0
		while x < (originx + width):
			if (colnum % 2) == 0:
				y = originy
			else:
				y = originy + (vspacing / 2.0)

			while y < (originy + height):
				polyline = []
				polyline.append(QgsPoint(x + (hspacing / 2.0), y))
				polyline.append(QgsPoint(x + hspacing, y + (vspacing / 2.0)))
				polyline.append(QgsPoint(x + (hspacing / 2.0), y + vspacing))
				polyline.append(QgsPoint(x, y + (vspacing / 2.0)))

				geometry = QgsGeometry()
				feature = QgsFeature()
				feature.setGeometry(geometry.fromPolygon([polyline]))
				feature.setAttributes([
					QVariant(x + (hspacing / 2.0)),
					QVariant(y + (vspacing / 2.0))])
				outfile.addFeature(feature)
				linecount = linecount + 1
				y = y + vspacing;

			x = x + (hspacing / 2.0)
			colnum = colnum + 1

	elif gridtype == "Hexagon (polygon)":
		# To preserve symmetry, hspacing is fixed relative to vspacing
		xvertexlo = 0.288675134594813 * vspacing;
		xvertexhi = 0.577350269189626 * vspacing;
		hspacing = xvertexlo + xvertexhi

		x = originx + xvertexhi
		# print str(x) + ", " + str(originx + width)

		colnum = 0
		while x < (originx + width):
			if (colnum % 2) == 0:
				y = originy + (vspacing / 2.0)
			else:
				y = originy + vspacing

			# print str(x) + "," + str(y)

			while y < (originy + height):
				polyline = []
				polyline.append(QgsPoint(x + xvertexhi, y))
				polyline.append(QgsPoint(x + xvertexlo, y + (vspacing / 2.0)))
				polyline.append(QgsPoint(x - xvertexlo, y + (vspacing / 2.0)))
				polyline.append(QgsPoint(x - xvertexhi, y))
				polyline.append(QgsPoint(x - xvertexlo, y - (vspacing / 2.0)))
				polyline.append(QgsPoint(x + xvertexlo, y - (vspacing / 2.0)))

				geometry = QgsGeometry()
				feature = QgsFeature()
				feature.setGeometry(geometry.fromPolygon([polyline]))
				feature.setAttributes([QVariant(x), QVariant(y)])
				outfile.addFeature(feature)
				linecount = linecount + 1
				y = y + vspacing;

			x = x + hspacing
			colnum = colnum + 1

	del outfile

	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(linecount) + " feature grid shapefile created")

	return None

# --------------------------------------------------------
#    mmqgis_gridify - Snap shape verticies to grid
# --------------------------------------------------------

def mmqgis_gridify_layer(qgis, layername, hspacing, vspacing, savename, addlayer):
	layer = mmqgis_find_layer(layername)
	if not layer:
		return "Project has no active vector layer to gridify"
	
	if (hspacing <= 0) or (vspacing <= 0):
		return "Invalid grid spacing: " + unicode(hspacing) + "/" + unicode(vspacing)

	if len(savename) <= 0:
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(),
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	point_count = 0
	deleted_points = 0

	feature_count = layer.dataProvider().featureCount()

	#feature = QgsFeature()
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
        #while layer.dataProvider().nextFeature(feature):

	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (feature_index % 50) == 0:
			mmqgis_status_message(qgis, "Gridifying feature " + unicode(feature_index))

		geometry = feature.geometry()

		if (geometry.wkbType() == QGis.WKBPoint) or \
		   (geometry.wkbType() == QGis.WKBPoint25D):
			points, added, deleted = mmqgis_gridify_points(hspacing, vspacing, [geometry.asPoint()])
			geometry = geometry.fromPoint(points[0])
			point_count += added
			deleted_points += deleted

		elif (geometry.wkbType() == QGis.WKBLineString) or \
		     (geometry.wkbType() == QGis.WKBLineString25D):
			#print "LineString"
			polyline, added, deleted = mmqgis_gridify_points(hspacing, vspacing, geometry.asPolyline())
			if len(polyline) < 2:
				geometry = None
			else:
				geometry = geometry.fromPolyline(polyline)
			point_count += added
			deleted_points += deleted

		elif (geometry.wkbType() == QGis.WKBPolygon) or \
		     (geometry.wkbType() == QGis.WKBPolygon25D):
			newpolygon = []
			for polyline in geometry.asPolygon():
				newpolyline, added, deleted = mmqgis_gridify_points(hspacing, vspacing, polyline)
				point_count += added
				deleted_points += deleted

				if len(newpolyline) > 1:
					newpolygon.append(newpolyline)

			if len(newpolygon) <= 0:
				geometry = None
			else:
				geometry = geometry.fromPolygon(newpolygon)

		elif (geometry.wkbType() == QGis.WKBMultiPoint) or \
		     (geometry.wkbType() == QGis.WKBMultiPoint25D):
			newmultipoints = []
			for index, point in enumerate(geometry.asMultiPoint()):
				# print unicode(index) + ": " + unicode(type(point))
				gridded, added, deleted = mmqgis_gridify_points(hspacing, vspacing, [ point ])
				# append() causes fail in fromMultiPoint(), extend() doesn't
				newmultipoints.extend(gridded)
				point_count += added
				deleted_points += deleted

			geometry = geometry.fromMultiPoint(newmultipoints)

		elif (geometry.wkbType() == QGis.WKBMultiLineString) or \
		     (geometry.wkbType() == QGis.WKBMultiLineString25D):
			#print "MultiLineString"
			newmultipolyline = []
			for polyline in geometry.asMultiPolyline():
				newpolyline, added, deleted = mmqgis_gridify_points(hspacing, vspacing, polyline)
				if len(newpolyline) > 1:
					newmultipolyline.append(newpolyline)
				point_count += added
				deleted_points += deleted

			if len(newmultipolyline) <= 0:
				geometry = None
			else:
				geometry = geometry.fromMultiPolyline(newmultipolyline)


		elif (geometry.wkbType() == QGis.WKBMultiPolygon) or \
		     (geometry.wkbType() == QGis.WKBMultiPolygon25D):
			#print "MultiPolygon"
			newmultipolygon = []
			for polygon in geometry.asMultiPolygon():
				newpolygon = []
				for polyline in polygon:
					newpolyline, added, deleted = mmqgis_gridify_points(hspacing, vspacing, polyline)

					if len(newpolyline) > 2:
						newpolygon.append(newpolyline)

					point_count += added
					deleted_points += deleted

				if len(newpolygon) > 0:
					newmultipolygon.append(newpolygon)

			if len(newmultipolygon) <= 0:
				geometry = None
			else:
				geometry = geometry.fromMultiPolygon(newmultipolygon)

		else:
			return "Unknown geometry type " + mmqgis_wkbtype_to_text(geometry.wkbType()) + \
				" on feature " + unicode(feature_index)

		# print "Closing feature"
	
		if geometry != None:
			out_feature = QgsFeature()
			out_feature.setGeometry(geometry)
			out_feature.setAttributes(feature.attributes())
			outfile.addFeature(out_feature)

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
			
	mmqgis_completion_message(qgis, "Gridified shapefile created (" + \
		unicode(deleted_points) + " of " + unicode(point_count) + " points deleted)")

	return None


# --------------------------------------------------------
#    mmqgis_hub_distance - Create shapefile of distances
#			   from points to nearest hub
# --------------------------------------------------------

class mmqgis_hub:
	def __init__(self, point, newname):
		self.point = point
		self.name = newname


def mmqgis_hub_distance(qgis, sourcename, destname, nameattributename, units, addlines, savename, addlayer):

	# Error checks
	sourcelayer = mmqgis_find_layer(sourcename)
	if (sourcelayer == None) or (sourcelayer.featureCount() <= 0):
		return "Origin Layer " + sourcename + " not found"

	hubslayer = mmqgis_find_layer(destname)
	if (hubslayer == None) or (hubslayer.featureCount() <= 0):
		return "Hub layer " + destname + " not found"

	if sourcename == destname:
		return "Same layer given for both hubs and spokes"

	nameindex = hubslayer.dataProvider().fieldNameIndex(nameattributename)
	if nameindex < 0:
		return "Invalid name attribute: " + nameattributename

	outputtype = QGis.WKBPoint
	if addlines:
		outputtype = QGis.WKBLineString

	# Create output file
	if len(savename) <= 0:
		return "Invalid output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename


	outfields = sourcelayer.dataProvider().fields()
	outfields.append(QgsField(QString("HubName"), QVariant.String))
	outfields.append(QgsField(QString("HubDist"), QVariant.Double))

	outfile = QgsVectorFileWriter(savename, "utf-8", outfields, outputtype, sourcelayer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())


	# Create array of hubs in memory
	#feature = QgsFeature()
	#hubslayer.dataProvider().select(hubslayer.dataProvider().attributeIndexes())
	#hubslayer.dataProvider().rewind()
	#while hubslayer.dataProvider().nextFeature(feature):

	hubs = []
	for index, feature in enumerate(hubslayer.dataProvider().getFeatures()):
		if (index % 20) == 0:
			mmqgis_status_message(qgis, "Reading hub " + unicode(feature.id()))
		hubs.append(mmqgis_hub(feature.geometry().boundingBox().center(), \
				feature.attributes()[nameindex].toString()))

	del hubslayer

	# Scan source points, find nearest hub, and write to output file
	#feature = QgsFeature()
	#sourcelayer.dataProvider().select(sourcelayer.dataProvider().attributeIndexes())
	#sourcelayer.dataProvider().rewind()
	#while sourcelayer.dataProvider().nextFeature(feature):

	writecount = 0
	for feature in sourcelayer.dataProvider().getFeatures():
		source = feature.geometry().boundingBox().center()
		distance = QgsDistanceArea()
		distance.setSourceCrs(sourcelayer.dataProvider().crs().srsid())
		# distance.setProjectionsEnabled(1)

		closest = hubs[0]
		hubdist = distance.measureLine(source, closest.point)

		#print unicode(feature.attributes()[0])
		for hub in hubs:
			thisdist = distance.measureLine(source, hub.point)
			if thisdist < hubdist:
				closest = hub
				hubdist = thisdist

		attributes = feature.attributes()
		attributes.append(QVariant(closest.name))
		if units == "Feet":
			hubdist = hubdist * 3.2808399
		elif units == "Miles":
			hubdist = hubdist * 0.000621371192
		elif units == "Kilometers":
			hubdist = hubdist / 1000
		elif units != "Meters":
                	hubdist = sqrt(pow(source.x() - closest.point.x(), 2.0) + pow(source.y() - closest.point.y(), 2.0))

		#print str(hubdist) + " " + units

		attributes.append(QVariant(hubdist))

		outfeature = QgsFeature()
		outfeature.setAttributes(attributes)

		if outputtype == QGis.WKBPoint:
			geometry = QgsGeometry()
			outfeature.setGeometry(geometry.fromPoint(source))
		else:
			polyline = []
			polyline.append(source)
			polyline.append(closest.point)
			geometry = QgsGeometry()
			outfeature.setGeometry(geometry.fromPolyline(polyline))

		outfile.addFeature(outfeature)

		writecount += 1
		if (writecount % 50) == 0:
			mmqgis_status_message(qgis, "Writing feature " + unicode(writecount) +\
				" of " + unicode(sourcelayer.dataProvider().featureCount()))

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
			
	mmqgis_completion_message(qgis, unicode(writecount) + " node hub distance file created")

	return None

# --------------------------------------------------------
#    mmqgis_hub_lines - Create shapefile of lines from
#			spoke points to matching hubs
# --------------------------------------------------------


def mmqgis_hub_lines(qgis, hubname, hubattr, spokename, spokeattr, savename, addlayer):

	# Find layers
	if hubname == spokename:
		return "Same layer given for both hubs and spokes"

	hublayer = mmqgis_find_layer(hubname)
	if (hublayer == None) or (hublayer.featureCount() <= 0):
		return "Hub layer " + hubname + " not found"

	spokelayer = mmqgis_find_layer(spokename)
	if spokelayer == None:
		return "Spoke Point Layer " + spokename + " not found"

	# Find Hub ID attribute indices
	hubindex = hublayer.dataProvider().fieldNameIndex(hubattr)
	if hubindex < 0:
		return "Invalid name attribute: " + hubattr

	spokeindex = spokelayer.dataProvider().fieldNameIndex(spokeattr)
	if spokeindex < 0:
		return "Invalid name attribute: " + spokeattr

	# Create output file
	if len(savename) <= 0:
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfields = spokelayer.dataProvider().fields()

	outfile = QgsVectorFileWriter(savename, "utf-8", outfields, QGis.WKBLineString, spokelayer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Scan spoke points
	linecount = 0
	for spokepoint in spokelayer.dataProvider().getFeatures():
		spokex = spokepoint.geometry().boundingBox().center().x()
		spokey = spokepoint.geometry().boundingBox().center().y()
		spokeid = unicode(spokepoint.attributes()[spokeindex].toString())
		mmqgis_status_message(qgis, "Reading spoke " + unicode(spokepoint.id()))
		#print "Spoke " + str(spokex) + ", " + str(spokey)

		# Scan hub points to find first matching hub
		for hubpoint in hublayer.dataProvider().getFeatures():
			hubid = unicode(hubpoint.attributes()[hubindex].toString())
			if hubid == spokeid:
				hubx = hubpoint.geometry().boundingBox().center().x()
				huby = hubpoint.geometry().boundingBox().center().y()
				#print "   Hub " + str(hubx) + ", " + str(huby)

				# Write line to the output file
				outfeature = QgsFeature()
				outfeature.setAttributes(spokepoint.attributes())

				polyline = []
				polyline.append(QgsPoint(spokex, spokey))
				polyline.append(QgsPoint(hubx, huby))
				geometry = QgsGeometry()
				outfeature.setGeometry(geometry.fromPolyline(polyline))
				outfile.addFeature(outfeature)
				linecount = linecount + 1
				break

	del spokelayer
	del hublayer
	del outfile

	if linecount <= 0:
		return "No spoke/hub matches found to create lines"

	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")

	mmqgis_completion_message(qgis, unicode(linecount) + " hub/spoke lines written")

	return None

# ----------------------------------------------------------
#    mmqgis_kml_export - Export attributes to KML file
#			 suitable for display in Google Maps
# ----------------------------------------------------------

def mmqgis_kml_export(qgis, layername, nameattribute, description, separator, outfilename, addlayer):
	layer = mmqgis_find_layer(layername)
	if not layer:
		return "Layer not found: " + layername

	if (separator == None) or (len(separator) <= 0):
		return "Invalid description field separator"

	nameindex = layer.dataProvider().fieldNameIndex(nameattribute)
	if nameindex < 0:
		return "Invalid name attribute: " + nameattribute

	descindices = []
	for fieldname in description:
		index = layer.dataProvider().fieldNameIndex(fieldname)
		if index < 0:
			return "Invalid description attribute: " + fieldname
		descindices.append(index)


	# Create output file
	try:
		outfile = open(outfilename, 'w')
		# outfile = sys.stdout
    	except:
		return "Failure opening " + outfilename

	outfile.write('<?xml version="1.0" encoding="UTF-8"?>\n')
	outfile.write('<kml xmlns="http://earth.google.com/kml/2.2">\n')
	outfile.write('<Document>\n')
	outfile.write('<name>' + unicode(layername) + '</name>\n')
	#  <description><![CDATA[Test description]]></description>

	# Build stylesheet
	stylecount = 0
	if layer.isUsingRendererV2():
		renderer = layer.rendererV2()
		symbolcount = len(renderer.symbols())
		for index, symbol in enumerate(renderer.symbols()):
			outfile.write('<Style id="style' + unicode(index + 1) + '">\n')

			if symbol.type() == QgsSymbolV2.Fill:
				outfile.write('\t<LineStyle>\n')
				outfile.write('\t\t<color>40000000</color>\n')
				outfile.write('\t\t<width>3</width>\n')
				outfile.write('\t</LineStyle>\n')

				# KML colors are AABBGGRR
				color = (int(round(symbol.alpha() * 255)) << 24) + (symbol.color().blue() << 16) + \
					(symbol.color().green() << 8) + symbol.color().red()
				#print "Color " + unicode(symbol.alpha()) + ", " + unicode(symbol.color().blue()) + \
				#	", " + unicode(symbol.color().green()) + ", " + unicode(symbol.color().red())

				outfile.write('\t<PolyStyle>\n')
				outfile.write('\t\t<color>' + format(color, '08x') + '</color>\n')
				outfile.write('\t\t<fill>1</fill>\n')
				outfile.write('\t\t<outline>1</outline>\n')
				outfile.write('\t</PolyStyle>\n')

			elif symbol.type() == QgsSymbolV2.Line:
				# KML colors are AABBGGRR
				color = (int(round(symbol.alpha() * 255)) << 24) + (symbol.color().blue() << 16) + \
					(symbol.color().green() << 8) + symbol.color().red()
				outfile.write('\t<LineStyle>\n')
				outfile.write('\t\t<color>' + format(color, '08x') + '</color>\n')
				outfile.write('\t\t<width>5</width>\n')
				outfile.write('\t</LineStyle>\n')

			else: # Marker
				icon = mmqgis_kml_icon(symbol.color().red(), symbol.color().green(), symbol.color().blue())
				outfile.write('\t<IconStyle>\n')
				outfile.write('\t\t<Icon>\n')
				outfile.write('\t\t\t<href>' + icon + '</href>\n')
				outfile.write('\t\t</Icon>\n')
				outfile.write('\t</IconStyle>\n')

			# print unicode(index) + ") " + unicode(symbol.color().name())

			outfile.write('</Style>\n')

	else: # Legacy renderer
		renderer = layer.renderer()
		symbolcount = len(renderer.symbols())
		for index, symbol in enumerate(renderer.symbols()):
			outfile.write('<Style id="style' + unicode(index + 1) + '">\n')

			if symbol.type() == QGis.Polygon:
				outfile.write('\t<LineStyle>\n')
				outfile.write('\t\t<color>#40000000</color>\n')
				outfile.write('\t\t<width>3</width>\n')
				outfile.write('\t</LineStyle>\n')
				
				# KML colors are AABBGGRR
				color = (symbol.fillColor().alpha() << 24) + (symbol.fillColor().blue() << 16) + \
					(symbol.fillColor().green() << 8) + symbol.fillColor().red()
				outfile.write('\t<PolyStyle>\n')
				outfile.write('\t\t<color>' + format(color, '08x') + '</color>\n')
				outfile.write('\t\t<fill>1</fill>\n')
				outfile.write('\t\t<outline>1</outline>\n')
				outfile.write('\t</PolyStyle>\n')

			elif symbol.type() == QGis.Line:
				# KML colors are AABBGGRR
				color = (symbol.color().alpha() << 24) + (symbol.color().blue() << 16) + \
					(symbol.color().green() << 8) + symbol.color().red()
				outfile.write('\t<LineStyle>\n')
				outfile.write('\t\t<color>' + format(color, '08x') + '</color>\n')
				outfile.write('\t\t<width>5</width>\n')
				outfile.write('\t</LineStyle>\n')

			else: # Marker
				icon = mmqgis_kml_icon(symbol.fillColor().red(), 
					symbol.fillColor().green(), symbol.fillColor().blue())
				outfile.write('\t<IconStyle>\n')
				outfile.write('\t\t<Icon>\n')
				outfile.write('\t\t\t<href>' + icon + '</href>\n')
				outfile.write('\t\t</Icon>\n')
				outfile.write('\t</IconStyle>\n')

			outfile.write('</Style>\n')

		#if type(renderer) == QgsCategorizedSymbolRendererV2:
		#	print "Categorized on " + unicode(renderer.classAttribute())
		#	stylecount = len(renderer.categories())
		#	for index, category in enumerate(renderer.categories()):
		#		print "  Category " + unicode(index) + " = " + unicode(category.value().toString())
		#		symbol = category.symbol()
		#		# print unicode(symbol.dump())
		#		for layerindex in range(0, symbol.symbolLayerCount()):
		#			symbollayer = symbol.symbolLayer(layerindex)
		#			print "    Layer " + unicode(layerindex) + ": " + \
		#				unicode(symbollayer.type()) + ": " + unicode(symbollayer.color().name())

		#elif type(renderer) == QgsGraduatedSymbolRendererV2:
		#	print "Graduated on " + unicode(renderer.classAttribute())
		#	stylecount = len(renderer.ranges())
		#	for index, symbolrange in enumerate(renderer.ranges())
		#		print "  Range " + unicode(index) + ": " + unicode(symbolrange.lowerValue()) + \
		#			", " + unicode(symbolrange.upperValue())
		#		symbol = symbolrange.symbol()
		#		for layerindex in range(0, symbol.symbolLayerCount()):
		#			symbollayer = symbol.symbolLayer(layerindex)
		#			print "    Layer " + unicode(layerindex) + ": " + \
		#				unicode(symbollayer.type()) + ": " + unicode(symbollayer.color().name())
		#		
		#elif type(renderer) == QgsSingleSymbolRendererV2:
		#	print "Single symbol"
		#	symbol = renderer.symbol()
		#	symbolcount = 1
		#	for layerindex in range(0, symbol.symbolLayerCount()):
		#		symbollayer = symbol.symbolLayer(layerindex)
		#		print "    Layer " + unicode(layerindex) + ": " + \
		#			unicode(symbollayer.type()) + ": " + unicode(symbollayer.color().name())

		#else:
		#	print unicode(type(renderer))
		
		# fieldindex = layer.dataProvider().fieldNameIndex(renderer.classAttribute())
		#print "Attribute: " + unicode(renderer.classAttribute())

	# Transform projection to WGS84 long/lat
	wgs84 = QgsCoordinateReferenceSystem()
	wgs84.createFromProj4("+proj=longlat +datum=WGS84 +no_defs")
	transform = QgsCoordinateTransform(layer.dataProvider().crs(), wgs84)

	# Write features to KML
	featurecount = 0
	for featureindex, feature in enumerate(layer.dataProvider().getFeatures()):

		# Find style for feature
		style = '#style0'
		if layer.isUsingRendererV2():
			for symbolsindex, featuresymbol in enumerate(renderer.symbolsForFeature(feature)):
				for renderindex, rendersymbol in enumerate(renderer.symbols()):
					if featuresymbol.dump() == rendersymbol.dump():
						style = '#style' + unicode(renderindex + 1)

		else: # legacy symbol renderer
			featuresymbol = renderer.symbolForFeature(feature)
			for renderindex, rendersymbol in enumerate(renderer.symbols()):
				if (featuresymbol.pen() == rendersymbol.pen()) and \
				   (featuresymbol.brush() == rendersymbol.brush()) and \
				   (featuresymbol.color() == rendersymbol.color()):
					style = '#style' + unicode(renderindex + 1)


		# Build name/description strings for feature
		name = unicode(feature.attributes()[nameindex].toString())
		description = ""
		if (separator != 'Paragraphs') and (separator != 'Field Names'):
			description = "<p>"

		for index, fieldindex in enumerate(descindices):
			if separator == 'Paragraphs':
				description = description + "<p>"
			elif separator == 'Field Names':
				description = description + "<p>" + \
					unicode(layer.dataProvider().fields()[fieldindex].name()) + ": "

			attribute = unicode(feature.attributes()[fieldindex].toString())
			attribute = attribute.replace('[', '\\[')
			attribute = attribute.replace(']', '\\]')
			description = description + attribute

			if (separator == 'Paragraphs') or (separator == 'Field Names'):
				description = description + "</p>"
			elif index < (len(descindices) - 1):
				if separator == 'Commas':
					description = description + ','
				else:
					description = description + unicode(separator)

		if (separator != 'Paragraphs') and (separator != 'Field Names'):
			description = description + "</p>"
		

		# KML always in WGS 84 long/lat
		geometry = feature.geometry()
		geometry.transform(transform)

		# Write features
		if (geometry.wkbType() == QGis.WKBPoint) or \
		   (geometry.wkbType() == QGis.WKBPoint25D):
			mmqgis_kml_write_point(geometry.asPoint(), name, description, style, outfile)
			featurecount = featurecount + 1

		elif (geometry.wkbType() == QGis.WKBMultiPoint) or \
		     (geometry.wkbType() == QGis.WKBMultiPoint25D):
			for point in geometry.asMultiPoint():
				mmqgis_kml_write_point(point, name, description, style, outfile)
				featurecount = featurecount + 1

		elif (geometry.wkbType() == QGis.WKBLineString) or \
		     (geometry.wkbType() == QGis.WKBLineString25D):
			mmqgis_kml_write_line(geometry.asPolyline(), name, description, style, outfile)
			featurecount = featurecount + 1

		elif (geometry.wkbType() == QGis.WKBMultiLineString) or \
		     (geometry.wkbType() == QGis.WKBMultiLineString25D):
			for line in geometry.asMultiPolyline():
				mmqgis_kml_write_line(line, name, description, style, outfile)
				featurecount = featurecount + 1

		elif (geometry.wkbType() == QGis.WKBPolygon) or \
		     (geometry.wkbType() == QGis.WKBPolygon25D):
			mmqgis_kml_write_polygon(geometry.asPolygon(), name, description, style, outfile)
			featurecount = featurecount + 1

		elif (geometry.wkbType() == QGis.WKBMultiPolygon) or \
		     (geometry.wkbType() == QGis.WKBMultiPolygon25D):
			for polygon in geometry.asMultiPolygon():
				mmqgis_kml_write_polygon(polygon, name, description, style, outfile)
				featurecount = featurecount + 1

	outfile.write('</Document>\n')
	outfile.write('</kml>')
	outfile.close()

	mmqgis_completion_message(qgis, unicode(featurecount) + " features exported to KML")

	return None

def mmqgis_kml_icon(red, green, blue):
	# Placemarks in Google (tm) maps are images referenced with a URL
	# These are the standard placemark icons used in Google maps
	# red = (color & 0xff0000) >> 16
	# green = (color & 0xff00) >> 8
	# blue = (color & 0xff)
	threshold = (min(red, green, blue) + max(red, green, blue)) / 2
	composite = 0
	if red >= threshold:
		composite = composite + 4
	if green >= threshold:
		composite = composite + 2
	if blue >= threshold:
		composite = composite + 1

	# print "rgb(" + unicode(red) + "," + unicode(green) + "," + unicode(blue) + ") = " + unicode(composite)

	if composite == 0: # black
        	return 'http://maps.gstatic.com/mapfiles/ms2/micons/blue-dot.png'
	elif composite == 1: # blue
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/blue-dot.png'
	elif composite == 2: # green
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/green-dot.png'
	elif composite == 3: # cyan
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/ltblue-dot.png'
	elif composite == 4: # red
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/red-dot.png'
	elif composite == 5: # magenta
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/pink-dot.png'
	elif composite == 6: # yellow
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/yellow-dot.png'
	else: # 7: white
		return 'http://maps.gstatic.com/mapfiles/ms2/micons/purple-dot.png'

def mmqgis_kml_write_point(point, name, description, style, outfile):
	outfile.write('<Placemark>\n')
	outfile.write('<name>' + name + '</name>\n')
	outfile.write('<description><![CDATA[' + description + ']]></description>\n')
	outfile.write('<styleUrl>' + style + '</styleUrl>\n')
	outfile.write('\t<Point>\n')
	outfile.write('\t\t<coordinates>' + unicode(point.x()) + ',' + unicode(point.y()) + ',0.00000</coordinates>\n')
	outfile.write('\t</Point>\n')
	outfile.write('</Placemark>\n')

def mmqgis_kml_write_line(line, name, description, style, outfile):
	outfile.write('<Placemark>\n')
	outfile.write('<name>' + name + '</name>\n')
	outfile.write('<description><![CDATA[' + description + ']]></description>\n')
	outfile.write('<styleUrl>' + style + '</styleUrl>\n')
	outfile.write('\t<LineString>\n')
	outfile.write('\t\t<tessellate>1</tessellate>\n')
	outfile.write('\t\t<coordinates>\n')
	for point in line:
		outfile.write('\t\t\t' + unicode(point.x()) + ',' + unicode(point.y()) + ',0.00000\n')
	outfile.write('\t\t</coordinates>\n')
	outfile.write('\t</LineString>\n')
	outfile.write('</Placemark>\n')

def mmqgis_kml_write_polygon(polygon, name, description, style, outfile):
	outfile.write('<Placemark>\n')
	outfile.write('<name>' + name + '</name>\n')
	outfile.write('<description><![CDATA[' + description + ']]></description>\n')
	outfile.write('<styleUrl>' + style + '</styleUrl>\n')
	outfile.write('\t<Polygon>\n')
	outfile.write('\t\t<outerBoundaryIs>\n')
	outfile.write('\t\t\t<LinearRing>\n')
	outfile.write('\t\t\t\t<tessellate>1</tessellate>\n')
	outfile.write('\t\t\t\t<coordinates>\n')
	for line in polygon:
		for point in line:
			outfile.write('\t\t\t\t\t' + unicode(point.x()) + ',' + unicode(point.y()) + ',0.00000\n')
	outfile.write('\t\t\t\t</coordinates>\n')
	outfile.write('\t\t\t</LinearRing>\n')
	outfile.write('\t\t</outerBoundaryIs>\n')
	outfile.write('\t</Polygon>\n')
	outfile.write('</Placemark>\n')



# --------------------------------------------------------
#    mmqgis_label - Create single label points for
#		    single- or multi-feature items
# --------------------------------------------------------

class mmqgis_label():
	def __init__(self, name, attribute_list, bounding_box):
		self.name = name
		self.attributes = attribute_list
		self.bounding_box = bounding_box

def mmqgis_label_point(qgis, layername, labelattributename, savename, addlayer):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Invalid layer name " . layername

	labelindex = layer.dataProvider().fieldNameIndex(labelattributename)
	if labelindex < 0:
		return "Invalid label field name: " + labelattributename

	# print  "labelindex = " + str(labelindex)

	if len(savename) <= 0:
		return "No output filename given"

	# Open file (delete any existing)
	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(),
				QGis.WKBPoint, layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Build dictionary of items, averaging center for multi-feature items
	label_features = {}
	readcount = 0
	feature_count = layer.featureCount()

	for feature in layer.dataProvider().getFeatures():
		key = unicode(feature.attributes()[labelindex].toString())
		# print "Feature: " + str(feature.id()) + " = " + key
		if not label_features.has_key(key):
			label_features[key] = mmqgis_label(key, feature.attributes(), feature.geometry().boundingBox())
		else:
			label_features[key].bounding_box.combineExtentWith(feature.geometry().boundingBox())

		readcount += 1
		if not (readcount % 10):
			mmqgis_status_message(qgis,  \
				"Reading feature " + unicode(readcount) + " of " + unicode(feature_count))



	# Sort keys so features appear in alphabetical order
	keys = label_features.keys()
	keys.sort()

	# Calculate points and write them to the output file
	writecount = 0
	for key in keys:
		label_point = label_features[key]

		feature = QgsFeature()
		feature.setAttributes(label_features[key].attributes)
		feature.setGeometry(QgsGeometry.fromPoint(label_features[key].bounding_box.center()))

		if not outfile.addFeature(feature):
			return "Failure writing feature to shapefile"

		writecount += 1
		if not (writecount % 50):
			mmqgis_status_message(qgis,  \
				"Writing feature " + unicode(writecount) + " of " + unicode(len(label_features)))

	del outfile

	if addlayer:
		newlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")

		newsymbol = QgsSymbolV2.defaultSymbol(newlayer.geometryType())
		newsymbol.setAlpha(0)
		renderer = QgsSingleSymbolRendererV2(newsymbol)
		newlayer.setRendererV2(renderer)

		settings = QgsPalLayerSettings()
		settings.readFromLayer(newlayer)
		settings.enabled = True
		settings.fieldIndex = labelindex
		settings.fieldName = QString(labelattributename)
		settings.placement = QgsPalLayerSettings.OverPoint
		settings.bufferColor = QColor(255, 255, 255)
		settings.bufferSize = 1.0
		settings.textColor = QColor(0, 0, 0)
		settings.textFont = QFont("Sans Serif", 10)
		settings.fontSizeInMapUnits = False
		settings.writeToLayer(newlayer)

		# Deprecated labeling configuration
		# newlayer.enableLabels(True)
		# newlayer.label().setLabelField(0, labelindex)
		# newlayer.label().setBufferColor(QColor(255, 255, 255))
		# newlayer.label().setSize(12.0, QgsLabelAttributes.PointUnits)

		# for index, field in enumerate(newlayer.dataProvider().fields()):
		#	print unicode(index) + ": " + unicode(field.name())
		# print unicode(newlayer.label().labelAttributes()) + ": " + unicode(newlayer.label().labelField(0))

		newlayer.triggerRepaint()
		qgis.legendInterface().refreshLayerSymbology(newlayer)

		#labelfields = QgsFields()
		#labelfields.append(newlayer.dataProvider().fields()[labelindex])
		#labelindex = layer.dataProvider().fieldNameIndex(labelattributename)
		# print unicode(layer.label())

	mmqgis_completion_message(qgis, unicode(writecount) + " label shapefile created from " + layername)

	return None

# --------------------------------------------------------
#    mmqgis_merge - Merge layers to single shapefile
# --------------------------------------------------------

def mmqgis_merge(qgis, layernames, savename, addlayer):
	fields = QgsFields()
	layers = []
	totalfeaturecount = 0

	for x in range(0, len(layernames)):
		layername = layernames[x]
		layer = mmqgis_find_layer(layername)
		if layer == None:
			return "Layer " + layername + " not found"

		# Verify that all layers are the same type (point, polygon, etc)
		if (len(layers) > 0):
			if (layer.dataProvider().geometryType() != layers[0].dataProvider().geometryType()):
				return "Merged layers must all be same type of geometry (" + \
					mmqgis_wkbtype_to_text(layer.dataProvider().geometryType()) + " != " + \
					mmqgis_wkbtype_to_text(layers[0].dataProvider().geometryType()) + ")"

			#if (layer.dataProvider().crs() != layers[0].dataProvider().crs()):
			#	QMessageBox.critical(qgis.mainWindow(), 
			#		"Merge Layers", "Merged layers must all have same coordinate system")
			#	return None
				
		layers.append(layer)
		totalfeaturecount += layer.featureCount()

		# Add any fields not in the composite field list
		for sindex, sfield in enumerate(layer.dataProvider().fields()):
			found = None
			for dfield in fields:
				if (dfield.name() == sfield.name()) and (dfield.type() == sfield.type()):
					found = dfield

			if not found:
				fields.append(sfield)
			
	if (len(layers) <= 0):
		return "No layers given to merge"
	
	# Create the output shapefile
	if len(savename) <= 0:
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", fields,
		layers[0].dataProvider().geometryType(), layers[0].dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	# Copy layer features to output file
	featurecount = 0
	for layer in layers:
		#feature = QgsFeature()
		#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
		#layer.dataProvider().rewind()
                #while layer.dataProvider().nextFeature(feature):

		for feature in layer.dataProvider().getFeatures():
			sattributes = feature.attributes()
			dattributes = []
			for dindex, dfield in enumerate(fields):
				dattribute = QVariant(dfield.type())
				for sindex, sfield in enumerate(layer.dataProvider().fields()):
					if (sfield.name() == dfield.name()) and (sfield.type() == dfield.type()):
						dattribute = sattributes[sindex]
						break
				dattributes.append(dattribute)

			#for dindex, dfield in dattributes.iteritems():
			#	print layer.name() + " (" + str(dindex) + ") " + str(dfield.toString())

			feature.setAttributes(dattributes)
			outfile.addFeature(feature)
			featurecount += 1
			if (featurecount % 50) == 0:
				mmqgis_status_message(qgis, "Writing feature " + \
					unicode(featurecount) + " of " + unicode(totalfeaturecount))

	del outfile

	# Add the merged layer to the project
	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")

	mmqgis_completion_message(qgis, unicode(featurecount) + " records exported")

	return None

# ----------------------------------------------------------
#    mmqgis_select - Select features by attribute
# ----------------------------------------------------------

def mmqgis_select(qgis, layername, selectattributename, comparisonvalue, comparisonname, savename, addlayer):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Project has no active vector layer to select from"

	selectindex = layer.dataProvider().fieldNameIndex(selectattributename)
	if selectindex < 0:
		return "Invalid select field name: " + selectattributename

	# print  "selectindex = " + str(selectindex) + " " + comparisonname + " " + comparisonvalue

	if (not comparisonvalue) or (len(comparisonvalue) <= 0):
		return "No comparison value given"

	if (not savename) or (len(savename) <= 0):
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(),
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	#feature = QgsFeature()
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
	#while layer.dataProvider().nextFeature(feature):

	readcount = 0
	writecount = 0
	for index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (comparisonname == 'begins with') or (comparisonname == 'contains') or \
		   (feature.attributes()[selectindex].type() == QVariant.String):
			x = unicode(feature.attributes()[selectindex].toString())
			y = unicode(comparisonvalue)
		else:
			# print feature.attributes()[selectindex].typeName()
			try:
				x = float(feature.attributes()[selectindex].toString())
				y = float(comparisonvalue)
			except:
				return "Non-numeric value used in comparison with numeric field: " + unicode(comparisonvalue)

		match = False
		if (comparisonname == '=='):
			match = (x == y)
		elif (comparisonname == '!='):
			match = (x != y)
		elif (comparisonname == '>'):
			match = (x > y)
		elif (comparisonname == '>='):
			match = (x >= y)
		elif (comparisonname == '<'):
			match = (x < y)
		elif (comparisonname == '<='):
			match = (x <= y)
		elif (comparisonname == 'begins with'):
			match = x.startswith(y)
		elif (comparisonname == 'contains'):
			match = (x.find(y) >= 0)
	
		readcount += 1
		if (match):
			outfile.addFeature(feature)
			writecount += 1

		if (index % 20) == 0:
			mmqgis_status_message(qgis, "Scanning feature " + \
				unicode(readcount) + " of " + unicode(layer.dataProvider().featureCount()) + \
				"(" + unicode(writecount) + " selected)")

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, "Selected " + unicode(writecount) + " features to " + savename)

	return None

# --------------------------------------------------------
#    mmqgis_sort - Sort shapefile by attribute
# --------------------------------------------------------

def mmqgis_sort(qgis, layername, sortattributename, savename, direction, addlayer):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Project has no active vector layer to sort"

	sortindex = layer.dataProvider().fieldNameIndex(sortattributename)
	if sortindex < 0:
		return "Invalid sort field name: " + sortattributename
	
	# print  "sortindex = " + str(sortindex)

	if len(savename) <= 0:
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(),
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	#feature = QgsFeature()
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
	#while layer.dataProvider().nextFeature(feature):

	table = []
	for index, feature in enumerate(layer.dataProvider().getFeatures()):
		featuretype = feature.attributes()[sortindex].type()
		if (featuretype == QVariant.Int):
			record = feature.id(), feature.attributes()[sortindex].toInt()
		elif (featuretype == QVariant.Double):
			record = feature.id(), feature.attributes()[sortindex].toDouble()
		else:
			record = feature.id(), unicode(feature.attributes()[sortindex].toString())

		if (index % 50) == 0:
			mmqgis_status_message(qgis, "Reading feature " + unicode(feature.id()))
		table.append(record)

	if (direction.lower() == "descending"):
		table.sort(key = operator.itemgetter(1), reverse=True)
	else:
		table.sort(key = operator.itemgetter(1))

	writecount = 0
	for index, record in enumerate(table):
		feature = QgsFeature()
		layer.featureAtId(record[0], feature)
		outfile.addFeature(feature)
		writecount += 1

		if (index % 50) == 0:
			mmqgis_status_message(qgis, "Writing feature " + unicode(writecount) +\
				" of " + unicode(len(table)))

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, "Sorted shapefile created from " + layername)

	return None

# ----------------------------------------------------------
#    mmqgis_spatial_join - Spatial Join
# ----------------------------------------------------------

def mmqgis_spatial_join(qgis, targetname, spatialop, joinname, fields, fieldop, outfilename, addlayer):
	target = mmqgis_find_layer(targetname)
	if target == None:
		return "Invalid target layer name: " + targetname

	join = mmqgis_find_layer(joinname)
	if join == None:
		return "Invalid join layer name: " + joinname

	transform = None
	if target.dataProvider().crs() != join.dataProvider().crs():
		transform = QgsCoordinateTransform(join.dataProvider().crs(), target.dataProvider().crs())


	# Build composite field list
	target_fields = []
	newfields = QgsFields()
	for index, field in enumerate(target.dataProvider().fields()):
		if field.name() in fields:
			newfields.append(field)
			target_fields.append(index)

	join_fields = []
	for index, field in enumerate(join.dataProvider().fields()):
		if field.name() in fields:
			# INT fields converted to DOUBLE to avoid overflow and rounding errors
			if field.type() == QVariant.Int:
				field.setType(QVariant.Double)
			newfields.append(field)
			join_fields.append(index)

	newfields.append(QgsField("COUNT", QVariant.Int))

	# Open file (delete any existing)
	if QFile(outfilename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(outfilename):
			return "Failure deleting existing shapefile: " + outfilename

	outfile = QgsVectorFileWriter(outfilename, "utf-8", newfields,
		target.dataProvider().geometryType(), target.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())


	# Interate through target features
	target_count = 0
	feature_count = target.featureCount()
	for target_index, target_feature in enumerate(target.dataProvider().getFeatures()):
		if (target_index % 10) == 0:
			mmqgis_status_message(qgis, "Joining feature " + unicode(target_index) + \
				" of " + unicode(feature_count))

		target_geometry = target_feature.geometry()

		attributes = []
		for index, copyindex in enumerate(target_fields):
			attributes.append(target_feature.attributes()[copyindex])

		for index, copyindex in enumerate(join_fields):
			attributes.append(QVariant(join.dataProvider().fields()[copyindex].type()))

		attributes.append(QVariant(0)) # Join count

		# Iterate through join features
		join_count = 0
		for join_index, join_feature in enumerate(join.dataProvider().getFeatures()):
			join_geometry = join_feature.geometry()
			if transform:
				join_geometry.transform(transform)
			
			if ((spatialop == 'Intersects') and (target_geometry.intersects(join_geometry))) or \
			   ((spatialop == 'Within') and (target_geometry.within(join_geometry))) or \
			   ((spatialop == 'Contains') and (target_geometry.contains(join_geometry))):

				join_count = join_count + 1
				for index, att_index in enumerate(join_fields):
					dest_index = len(target_fields) + index
					attribute = join_feature.attributes()[att_index]

					# Since strings cannot be mathematically combined,
					# non-scalar attributes are always the first encountered
					if (fieldop == "First") or (attributes[dest_index].type() != QVariant.Double):
						if (join_count == 1):
							attributes[dest_index] = attribute

					else:
						ratio = 1.0
						if fieldop == "Proportional Sum":
							total_area = join_geometry.area()
							intersect_area = target_geometry.intersection(join_geometry).area()
							if (total_area > 0):
								ratio = intersect_area / total_area

						if join_count <= 1:
							target_value = 0
						else:
							target_value, test = attributes[dest_index].toDouble()
						join_value, test = join_feature.attributes()[att_index].toDouble()
						attributes[dest_index] = QVariant(target_value + (ratio * join_value))

						# print unicode(target_index) + ":" + unicode(join_index) + ") " + \
						#	unicode(target_value) + " + " + unicode(join_value) + " * " + \
						#	unicode(ratio)
						

		# Divide sums to get averages
		if (fieldop == "Average") and (join_count > 0):
			for index in range(len(target_fields), len(target_fields) + len(join_fields)):
				if (attributes[index].type() == QVariant.Double):
					summation, test = attributes[index].toDouble()
					attributes[index] = QVariant(summation / float(join_count))

		# Counter
		attributes[len(target_fields) + len(join_fields)] = QVariant(join_count)

		# Add the feature
		# if join_count > 0:
		target_count = target_count + 1
		newfeature = QgsFeature()
		newfeature.setGeometry(target_feature.geometry())
		newfeature.setAttributes(attributes)
		# print unicode(target_count) + ") " + unicode(attributes[0].toString())
		if not outfile.addFeature(newfeature):
			return "Failure writing feature to shapefile"

	del outfile

	if addlayer:
		qgis.addVectorLayer(outfilename, os.path.basename(outfilename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(target_count) + " features joined")

	return None

# ---------------------------------------------------------
#    mmqgis_text_to_float - Change text fields to numbers
# ---------------------------------------------------------

def mmqgis_text_to_float(qgis, layername, attributes, savename, addlayer):
	layer = mmqgis_find_layer(layername)
	if layer == None:
		return "Project has no active vector layer to convert: " + layername

	if len(savename) <= 0:
		return "No output filename given"

	# Build list of fields with selected fields changed to floating point
	changecount = 0
	destfields = QgsFields()
	fieldchanged = []
	for index, field in enumerate(layer.dataProvider().fields()):
		if (field.name() in attributes) and ((field.type() == QVariant.String) or (field.type() == QVariant.Int)):
			fieldchanged.append(True)
			# Arbitrary floating point length/precision 14.6 = nnnnnnnnnnnnnn.dddddd
			destfields.append(QgsField (field.name(), QVariant.Double, field.typeName(), \
				14, 6, field.comment()))
		else:
			changecount += 1
			fieldchanged.append(False)
			destfields.append(QgsField (field.name(), field.type(), field.typeName(), \
				field.length(), field.precision(), field.comment()))

	if (changecount <= 0):
		return "No string or integer fields selected for conversion to floating point"


	# Create the output file
	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", destfields,
			layer.dataProvider().geometryType(), layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())


	# Write the features with modified attributes
	featurecount = layer.dataProvider().featureCount();
	for feature_index, feature in enumerate(layer.dataProvider().getFeatures()):
		if (feature_index % 50) == 0:
			mmqgis_status_message(qgis, "Writing feature " + \
				unicode(feature.id()) + " of " + unicode(featurecount))

		attributes = feature.attributes()
		for index, field in enumerate(layer.dataProvider().fields()):
			if fieldchanged[index]:
				string = unicode(attributes[index].toString())
				multiplier = 1.0
				if string.find("%") >= 0:
					multiplier = 1 / 100.0
					string = string.replace("%", "")
				if string.find(",") >= 0:
					string = string.replace(",", "")

				try:	
					value = float(string) * multiplier
				except:
					value = 0
						
				attributes[index] = QVariant(value)

		feature.setAttributes(attributes)
		outfile.addFeature(feature)

	del outfile

	if addlayer:
		vlayer = qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")
		
	mmqgis_completion_message(qgis, unicode(changecount) + " text fields converted to numeric")

	return None


# --------------------------------------------------------
#    mmqgis_voronoi - Voronoi diagram creation
# --------------------------------------------------------

def mmqgis_voronoi_diagram(qgis, sourcelayer, savename, addlayer):
	layer = mmqgis_find_layer(sourcelayer)
	if layer == None:
		return "Layer " + sourcename + " not found"
	
	if len(savename) <= 0:
		return "No output filename given"

	if QFile(savename).exists():
		if not QgsVectorFileWriter.deleteShapeFile(savename):
			return "Failure deleting existing shapefile: " + savename

	outfile = QgsVectorFileWriter(savename, "utf-8", layer.dataProvider().fields(), \
			QGis.WKBPolygon, layer.dataProvider().crs())

	if (outfile.hasError() != QgsVectorFileWriter.NoError):
		return "Failure creating output shapefile: " + unicode(outfile.errorMessage())

	points = []
	xmin = 0
	xmax = 0
	ymin = 0
	ymax = 0

	#feature = QgsFeature()
	#layer.dataProvider().select(layer.dataProvider().attributeIndexes())
	#layer.dataProvider().rewind()
	#while layer.dataProvider().nextFeature(feature):

	for feature in layer.dataProvider().getFeatures():
		# Re-read by feature ID because nextFeature() doesn't always seem to read attributes
		# layer.featureAtId(feature.id(), feature)
		geometry = feature.geometry()
		mmqgis_status_message(qgis, "Reading feature " + unicode(feature.id()))
		# print str(feature.id()) + ": " + str(geometry.wkbType())
		if geometry.wkbType() == QGis.WKBPoint:
			points.append( (geometry.asPoint().x(), geometry.asPoint().y(), feature.attributes()) )
			if (len(points) <= 1) or (xmin > geometry.asPoint().x()):
				xmin = geometry.asPoint().x()
			if (len(points) <= 1) or (xmax < geometry.asPoint().x()):
				xmax = geometry.asPoint().x()
			if (len(points) <= 1) or (ymin > geometry.asPoint().y()):
				ymin = geometry.asPoint().y()
			if (len(points) <= 1) or (ymax < geometry.asPoint().y()):
				ymax = geometry.asPoint().y()

	if (len(points) < 3):
		return "Too few points to create diagram"

	for center in points:
	# for center in [ points[17] ]:
		# print "\nCenter, " + str(center[0]) + ", " + str(center[1])
		mmqgis_status_message(qgis, "Processing point " + \
			unicode(center[0]) + ", " + unicode(center[1]))

		# Borders are tangents to midpoints between all neighbors
		tangents = []
		for neighbor in points:
			border = mmqgis_voronoi_line((center[0] + neighbor[0]) / 2.0, (center[1] + neighbor[1]) / 2.0)
			if ((neighbor[0] != center[0]) or (neighbor[1] != center[1])):
				tangents.append(border)

		# Add edge intersections to clip to extent of points
		offset = (xmax - xmin) * 0.01
		tangents.append(mmqgis_voronoi_line(xmax + offset, center[1]))
		tangents.append(mmqgis_voronoi_line(center[0], ymax + offset))
		tangents.append(mmqgis_voronoi_line(xmin - offset, center[1]))
		tangents.append(mmqgis_voronoi_line(center[0], ymin - offset))
		#print "Extent x = " + str(xmax) + " -> " + str(xmin) + ", y = " + str(ymax) + " -> " + str(ymin)

		# Find vector distance and angle to border from center point
		for scan in range(0, len(tangents)):
			run = tangents[scan].x - center[0]
			rise = tangents[scan].y - center[1]
			tangents[scan].distance = sqrt((run * run) + (rise * rise))
			if (tangents[scan].distance <= 0):
				tangents[scan].angle = 0
			elif (tangents[scan].y >= center[1]):
				tangents[scan].angle = acos(run / tangents[scan].distance)
			elif (tangents[scan].y < center[1]):
				tangents[scan].angle = (2 * pi) - acos(run / tangents[scan].distance)
			elif (tangents[scan].x > center[0]):
				tangents[scan].angle = pi / 2.0
			else:
				tangents[scan].angle = 3 * pi / 4

			#print "  Tangent, " + str(tangents[scan].x) + ", " + str(tangents[scan].y) + \
			#	", angle " + str(tangents[scan].angle * 180 / pi) + ", distance " + \
			#	str(tangents[scan].distance)


		# Find the closest line - guaranteed to be a border
		closest = -1
		for scan in range(0, len(tangents)):
			if ((closest == -1) or (tangents[scan].distance < tangents[closest].distance)):
				closest = scan

		# Use closest as the first border
		border = mmqgis_voronoi_line(tangents[closest].x, tangents[closest].y)
		border.angle = tangents[closest].angle
		border.distance = tangents[closest].distance
		borders = [ border ]

		#print "  Border 0) " + str(closest) + " of " + str(len(tangents)) + ", " \
		#	+ str(border.x) + ", " + str(border.y) \
		#	+ ", (angle " + str(border.angle * 180 / pi) + ", distance " \
		#	+ str(border.distance) + ")"

		# Work around the tangents in a CCW circle
		circling = 1
		while circling:
			next = -1
			scan = 0
			while (scan < len(tangents)):
				anglebetween = tangents[scan].angle - borders[len(borders) - 1].angle
				if (anglebetween < 0):
					anglebetween += (2 * pi)
				elif (anglebetween > (2 * pi)):
					anglebetween -= (2 * pi)

				#print "    Scanning " + str(scan) + " of " + str(len(borders)) + \
				#	", " + str(tangents[scan].x) + ", " + str(tangents[scan].y) + \
				#	", angle " + str(tangents[scan].angle * 180 / pi) + \
				#	", anglebetween " + str(anglebetween * 180 / pi)

				# If border intersects to the left
				if (anglebetween < pi) and (anglebetween > 0):
					# A typo here with a reversed slash cost 8/13/2009 debugging
					tangents[scan].iangle = atan2( (tangents[scan].distance / 
						borders[len(borders) - 1].distance) \
						- cos(anglebetween), sin(anglebetween))
					tangents[scan].idistance = borders[len(borders) - 1].distance \
						/ cos(tangents[scan].iangle)

					tangents[scan].iangle += borders[len(borders) - 1].angle

					# If the rightmost intersection so far, it's a candidate for next border
					if (next < 0) or (tangents[scan].iangle < tangents[next].iangle):
						# print "      Take idistance " + str(tangents[scan].idistance)
						next = scan

				scan += 1

			# iangle/distance are for intersection of border with next border
			borders[len(borders) - 1].iangle = tangents[next].iangle
			borders[len(borders) - 1].idistance = tangents[next].idistance

			# Stop circling if back to the beginning
			if (borders[0].x == tangents[next].x) and (borders[0].y == tangents[next].y):
				circling = 0

			else:
				# Add the next border
				border = mmqgis_voronoi_line(tangents[next].x, tangents[next].y)
				border.angle = tangents[next].angle
				border.distance = tangents[next].distance
				border.iangle = tangents[next].iangle
				border.idistance = tangents[next].idistance
				borders.append(border)
				#print "  Border " + str(len(borders) - 1) + \
				#	") " + str(next) + ", " + str(border.x) + \
				#	", " + str(border.y) + ", angle " + str(border.angle * 180 / pi) +\
				#	", iangle " + str(border.iangle * 180 / pi) +\
				#	", idistance " + str(border.idistance) + "\n"

			# Remove the border from the list so not repeated
			tangents.pop(next)
			if (len(tangents) <= 0):
				circling = 0

		if len(borders) >= 3:
			polygon = []
			for border in borders:
				ix = center[0] + (border.idistance * cos(border.iangle))
				iy = center[1] + (border.idistance * sin(border.iangle))
				#print "  Node, " + str(ix) + ", " + str(iy) + \
				#	", angle " + str(border.angle * 180 / pi) + \
				#	", iangle " + str(border.iangle * 180 / pi) + \
				#	", idistance " + str(border.idistance) + ", from " \
				#	+ str(border.x) + ", " + str(border.y)
				polygon.append(QgsPoint(ix, iy))

			# attributes = { 0:QVariant(center[0]), 1:QVariant(center[1]) }

			geometry = QgsGeometry.fromPolygon([ polygon ])
			feature = QgsFeature()
			feature.setGeometry(geometry)
			feature.setAttributes(center[2])
			outfile.addFeature(feature)
				
	del outfile

	if addlayer:
		qgis.addVectorLayer(savename, os.path.basename(savename), "ogr")

	mmqgis_completion_message(qgis, "Created " + unicode(len(points)) + " polygon Voronoi diagram")

	return None

class mmqgis_voronoi_line:
	def __init__(self, x, y):
		self.x = x
		self.y = y
		self.angle = 0
		self.distance = 0

	def list(self, title):
		print title + ", " + unicode(self.x) + ", " + unicode(self.y) + \
			", angle " + unicode(self.angle * 180 / pi) + ", distance " + unicode(self.distance)

	def angleval(self):
		return self.angle


