# --------------------------------------------------------
#    mmqgis_dialogs - Dialog classes for mmqgis
#
#    begin                : 10 May 2010
#    copyright            : (c) 2009 - 2012 by Michael Minn
#    email                : See michaelminn.com
#
#   MMQGIS is free software and is offered without guarantee
#   or warranty. You can redistribute it and/or modify it 
#   under the terms of version 2 of the GNU General Public 
#   License (GPL v2) as published by the Free Software 
#   Foundation (www.gnu.org).
# --------------------------------------------------------

import csv
import os.path
import operator

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *

from mmqgis_library import *

import sys
sys.path.append(os.path.dirname(os.path.abspath(__file__)) + "/forms")

# --------------------------------------------------------
#    mmqgis_animate_columns - Create animations by
#		interpolating offsets from attributes
# --------------------------------------------------------

from mmqgis_animate_columns_form import *

class mmqgis_animate_columns_dialog(QDialog, Ui_mmqgis_animate_columns_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)

		QObject.connect(self.browseoutfile, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for layer in iface.mapCanvas().layers():
			self.layername.addItem(layer.name())

		self.set_column_options()

		#for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
		#	if layer.type() == QgsMapLayer.VectorLayer:
		#		self.sourcelayer.addItem(layer.name())

		#self.attfilename.setText(os.getcwd() + "/attributes.csv")
		#self.nodefilename.setText(os.getcwd() + "/nodes.csv")

	def set_column_options(self):
		self.latoffsetcol.clear()
		self.longoffsetcol.clear()

		if self.layername.currentIndex() >= 0:
			#print str(self.layername.currentIndex())
			layer = self.iface.mapCanvas().layer(self.layername.currentIndex())
			if layer.type() == QgsMapLayer.VectorLayer:
				for index, field in layer.dataProvider().fields().iteritems():
					#print "fields: " + str(index) + str(field.type())
					if field.type() in [ QVariant.Int, QVariant.Double ]:
						self.latoffsetcol.addItem(field.name())
						self.longoffsetcol.addItem(field.name())
						if ("lat" in field.name().toLower()) and ("off" in field.name().toLower()):
							self.latoffsetcol.setCurrentIndex(self.latoffsetcol.count() - 1)
						if ("lon" in field.name().toLower()) and ("off" in field.name().toLower()):
							self.longoffsetcol.setCurrentIndex(self.longoffsetcol.count() - 1)
						

        def browse_outfile(self):
		newname = QFileDialog.getExistingDirectory(None, QString.fromLocal8Bit("Output Frames Directory"),
			self.outdirname.displayText())

		if newname != None:
                	self.outdirname.setText(QString(newname))

	def run(self):
		layer = unicode(self.layername.currentText())
		latcol = unicode(self.latoffsetcol.currentText())
		longcol = unicode(self.longoffsetcol.currentText())
		outdir = unicode(self.outdirname.displayText())
		frame_count = int(self.durationframes.displayText())

		message = mmqgis_animate_columns(self.iface, layer, longcol, latcol, outdir, frame_count)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Animate Columns", message)

# --------------------------------------------------------
#    mmqgis_animate_rows - Create animations by
#		displaying successive rows
# --------------------------------------------------------

from mmqgis_animate_rows_form import *

class mmqgis_animate_rows_dialog(QDialog, Ui_mmqgis_animate_rows_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)

		QObject.connect(self.browseoutfile, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for layer in iface.mapCanvas().layers():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.layernames.addItem(layer.name())


        def browse_outfile(self):
		newname = QFileDialog.getExistingDirectory(None, QString.fromLocal8Bit("Output Frames Directory"),
			self.outdirname.displayText())

		if newname != None:
                	self.outdirname.setText(QString(newname))

	def run(self):
		layers = []
		for x in range(0, self.layernames.count()):
			if self.layernames.item(x).isSelected():
				layers.append(self.layernames.item(x).text())

		cumulative = self.cumulative.isChecked()
		outdir = unicode(self.outdirname.displayText())

		message = mmqgis_animate_rows(self.iface, layers, cumulative, outdir)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Animate Rows", message)



# --------------------------------------------------------
#    mmqgis_attribute_join - Join attributes from a CSV
#                            file to a shapefile
# --------------------------------------------------------

from mmqgis_attribute_join_form import *

class mmqgis_attribute_join_dialog(QDialog, Ui_mmqgis_attribute_join_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse_infile, SIGNAL("clicked()"), self.browse_infiles)
		QObject.connect(self.browse_outfile, SIGNAL("clicked()"), self.browse_outfiles)
		QObject.connect(self.browse_notfound, SIGNAL("clicked()"), self.browse_notfoundfiles)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.joinlayer.addItem(layer.name())

		QObject.connect(self.joinlayer, SIGNAL("currentIndexChanged(QString)"), self.set_join_attributes)
		self.set_join_attributes()

		self.outfilename.setText(os.getcwd() + "/temp.shp")
		self.notfoundfilename.setText(os.getcwd() + "/notfound.csv")

        def browse_infiles(self):
		newname = QFileDialog.getOpenFileName(None, QString.fromLocal8Bit("Input CSV File"), 
			self.infilename.displayText(), "*.csv")

                if newname:
			try:
				infile = open(newname, 'r')
				dialect = csv.Sniffer().sniff(infile.read(1024))
			except:
				QMessageBox.information(self.iface.mainWindow(), "Input CSV File", \
					"File inaccessible or in unrecognized CSV format " + newname)
				return

			infile.seek(0)
			reader = csv.reader(infile, dialect)
			header = reader.next()
			del reader
			del infile

			self.csvfilefield.clear()
			for field in header:
				self.csvfilefield.addItem(QString(field))

			self.infilename.setText(QString(newname))

        def browse_outfiles(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.outfilename.displayText(), "*.shp")
                if newname != None:
                	self.outfilename.setText(QString(newname))

        def browse_notfoundfiles(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Unjoined CSV File"), 
			self.notfoundfilename.displayText(), "*.csv")
                if newname != None:
                	self.notfoundfilename.setText(QString(newname))

	def set_join_attributes(self):
		self.joinlayerattribute.clear()
		layername = self.joinlayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.joinlayerattribute.addItem(field.name())

	def run(self):
		layername = unicode(self.joinlayer.currentText())
		joinfield = unicode(self.csvfilefield.currentText())
		joinattribute = unicode(self.joinlayerattribute.currentText())
		infilename = unicode(self.infilename.displayText())
		outfilename = unicode(self.outfilename.displayText())
		notfoundname = unicode(self.notfoundfilename.displayText()).strip()

		message = mmqgis_attribute_join(self.iface, layername, infilename, joinfield, \
			joinattribute, outfilename, notfoundname, 1)
		if message != None:
			QMessageBox.critical(self.iface.mainWindow(), "Attribute Join", message)

# ----------------------------------------------------------
#    mmqgis_attribute_export - Export attributes to CSV file
# ----------------------------------------------------------

from mmqgis_attribute_export_form import *

class mmqgis_attribute_export_dialog(QDialog, Ui_mmqgis_attribute_export_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browseoutfile, SIGNAL("clicked()"), self.browse_outfiles)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_attributes)
		self.set_attributes()

		self.outfilename.setText(os.getcwd() + "/temp.csv")

        def browse_outfiles(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output CSV File"), 
			self.outfilename.displayText(), "*.csv")
                if newname != None:
                	self.outfilename.setText(QString(newname))

	def set_attributes(self):
		self.attributes.clear()
		layername = self.sourcelayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.attributes.addItem(field.name())

		#if (layername[0] != '/') and (layername[0] != '\\'):
		#	self.outfilename.setText(os.getcwd() + "/" + layername + ".csv")
		#else:
		#	self.outfilename.setText(layername + ".csv")

	def run(self):
		layername = unicode(self.sourcelayer.currentText())
		outfilename = unicode(self.outfilename.displayText())
		delimiter = ","
		if unicode(self.delimiter.currentText()) == "(bar)":
			delimiter = "|"
		elif unicode(self.delimiter.currentText()) == "(space)":
			delimiter = " "

		lineterminator = "\r\n"
		if unicode(self.lineterminator.currentText()) == "LF":
			lineterminator = "\n"

		# Compile a list and header of selected attributes
		attribute_names = []
		for x in range(0, self.attributes.count()):
			list_item = self.attributes.item(x)
			if list_item.isSelected():
				attribute_names.append(unicode(list_item.text()))

		#if len(attribute_names) <= 0:
		#	QMessageBox.critical(self.iface.mainWindow(), "Attribute Export", "No attributes selected")
		#	return

		message = mmqgis_attribute_export(self.iface, outfilename, layername, \
			attribute_names, delimiter, lineterminator)

		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Attribute Export", message)


# --------------------------------------------------------
#    mmqgis_color_map - Robust layer coloring
# --------------------------------------------------------

from mmqgis_color_map_form import *

class mmqgis_color_map_dialog(QDialog, Ui_mmqgis_color_map_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.lowcolor, SIGNAL("clicked()"), self.select_color)
		QObject.connect(self.midcolor, SIGNAL("clicked()"), self.select_color)
		QObject.connect(self.highcolor, SIGNAL("clicked()"), self.select_color)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			self.layername.addItem(layer.name())
		
		self.preset.addItem("Red - Green")
		self.preset.addItem("Green - Red")
		self.preset.addItem("Blue - Brown")
		self.preset.addItem("Brown - Blue")
		self.preset.addItem("Purple - Green")
		self.preset.addItem("Green - Purple")
		QObject.connect(self.preset, SIGNAL("currentIndexChanged(QString)"), self.set_color_preset)
		self.preset.setCurrentIndex(0)

		QObject.connect(self.layername, SIGNAL("currentIndexChanged(QString)"), self.set_band_names)
		self.set_band_names()

		QObject.connect(self.bandname, SIGNAL("currentIndexChanged(QString)"), self.set_band_range)
		self.set_band_range()

        def select_color(self):
		button = self.sender()
		button_color = button.palette().color(QPalette.Normal, QPalette.Button)
		button_color = QColorDialog.getColor(button_color)
		print button_color
		if button_color.isValid():
			button.setStyleSheet("* { background-color: " + button_color.name() + " }");

	def set_color_preset(self):
		name = unicode(self.preset.currentText())
		if name == "Red - Green":
			self.lowcolor.setStyleSheet("* { background-color: #800000 }");
			self.midcolor.setStyleSheet("* { background-color: #808000 }");
			self.highcolor.setStyleSheet("* { background-color: #008000 }");
		elif name == "Green - Red":
			self.highcolor.setStyleSheet("* { background-color: #800000 }");
			self.midcolor.setStyleSheet("* { background-color: #808000 }");
			self.lowcolor.setStyleSheet("* { background-color: #008000 }");
		elif name == "Blue - Brown":
			self.lowcolor.setStyleSheet("* { background-color: #6e462d }");
			self.midcolor.setStyleSheet("* { background-color: #c9c865 }");
			self.highcolor.setStyleSheet("* { background-color: #346666 }");
		elif name == "Brown - Blue":
			self.highcolor.setStyleSheet("* { background-color: #6e462d }");
			self.midcolor.setStyleSheet("* { background-color: #c9c865 }");
			self.lowcolor.setStyleSheet("* { background-color: #346666 }");
		elif name == "Purple - Green":
			self.lowcolor.setStyleSheet("* { background-color: #5f309c }");
			self.midcolor.setStyleSheet("* { background-color: #fcfbbd }");
			self.highcolor.setStyleSheet("* { background-color: #147a0b }");
		elif name == "Green - Purple":
			self.highcolor.setStyleSheet("* { background-color: #5f309c }");
			self.midcolor.setStyleSheet("* { background-color: #fcfbbd }");
			self.lowcolor.setStyleSheet("* { background-color: #147a0b }");
				
	def set_band_names(self):
		self.bandname.clear()
		layer = mmqgis_find_layer(unicode(self.layername.currentText()))
		if layer != None:
			if layer.type() == QgsMapLayer.RasterLayer:
				self.steps.setText("256");
				for x in range(1, layer.bandCount() + 1):
					self.bandname.addItem(layer.bandName(x))

			else:
				self.steps.setText("5");
				for index, field in layer.dataProvider().fields().iteritems():
					if ((field.type() == QVariant.Double) or
					    (field.type() == QVariant.Int) or
					    (field.type() == QVariant.UInt)):
						self.bandname.addItem(field.name())

	def set_band_range(self):
		minimum = 0
		maximum = 100
		layer = mmqgis_find_layer(unicode(self.layername.currentText()))
		if layer != None:
			if layer.type() == QgsMapLayer.RasterLayer:
				for index in range(1, (layer.bandCount() + 1)):
					if layer.bandName(index) == self.bandname.currentText():
						minimum = layer.minimumValue(index)
						maximum = layer.maximumValue(index)

			else: # vector
				minimum, maximum, valid = mmqgis_layer_attribute_bounds(layer, unicode(self.bandname.currentText()))
		
		self.lowvalue.setText(unicode(minimum))
		self.midvalue.setText(unicode((minimum + maximum) / 2))
		self.highvalue.setText(unicode(maximum))
						

	def run(self):
		#print self.lowvalue.text()
		#print self.midvalue.text()
		#print self.highvalue.text()
		message = mmqgis_set_color_map(self.iface, unicode(self.layername.currentText()),
			unicode(self.bandname.currentText()), 
			float(unicode(self.lowvalue.text())),
			float(unicode(self.midvalue.text())), 
			float(unicode(self.highvalue.text())),
			int(self.steps.text()),
			int(self.lowcolor.palette().color(QPalette.Normal, QPalette.Button).rgb()),
			int(self.midcolor.palette().color(QPalette.Normal, QPalette.Button).rgb()),
			int(self.highcolor.palette().color(QPalette.Normal, QPalette.Button).rgb()))

		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Color Map", message)

# --------------------------------------------------------
#    mmqgis_geometry_convert - Convert geometries to
#		simpler types
# --------------------------------------------------------

from mmqgis_geometry_convert_form import *

class mmqgis_geometry_convert_dialog(QDialog, Ui_mmqgis_geometry_convert_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_geometry_types)

		self.set_geometry_types()

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_geometry_types(self):
		newtypes = []
		layername = self.sourcelayer.currentText()
		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			# print layername + " =? " + layer.name() + ": " + str(layer.dataProvider().geometryType())

			if layer.name() == layername:
				if layer.dataProvider().geometryType() == QGis.WKBPoint:
					self.oldgeometry.setText("Point (no conversions available)")

				elif layer.dataProvider().geometryType() == QGis.WKBLineString:
					self.oldgeometry.setText("Linestring")
					newtypes = ["Centroids", "Nodes"]

				elif layer.dataProvider().geometryType() == QGis.WKBPolygon:
					self.oldgeometry.setText("Polygon")
					newtypes = ["Centroids", "Nodes", "Multilinestrings"]

				elif layer.dataProvider().geometryType() == QGis.WKBMultiPoint:
					self.oldgeometry.setText("Multipoint")
					newtypes = ["Centroids", "Nodes"]

				elif layer.dataProvider().geometryType() == QGis.WKBMultiLineString:
					self.oldgeometry.setText("Multilinestring")
					newtypes = ["Centroids", "Nodes", "Linestrings"]

				elif layer.dataProvider().geometryType() == QGis.WKBMultiPolygon:
					self.oldgeometry.setText("Multilinestring")
					newtypes = ["Centroids", "Nodes", "Multilinestrings", "Polygons"] 

				elif layer.dataProvider().geometryType() == QGis.WKBPoint25D:
					self.oldgeometry.setText("Point 2.5D")
					newtypes = ["Point"]

				elif layer.dataProvider().geometryType() == QGis.WKBLineString25D:
					self.oldgeometry.setText("Linestring 2.5D")
					newtypes = ["Centroids", "Nodes", "Linestrings"]

				elif layer.dataProvider().geometryType() == QGis.WKBPolygon25D:
					self.oldgeometry.setText("Polygon 2.5D")
					newtypes = ["Centroids", "Nodes", "Multilinestrings", "Polygons"] 

				elif layer.dataProvider().geometryType() == QGis.WKBMultiPoint25D:
					self.oldgeometry.setText("Multipoint 2.5D")
					newtypes = ["Point"]

				elif layer.dataProvider().geometryType() == QGis.WKBMultiLineString25D:
					self.oldgeometry.setText("Multilinestring 2.5D")
					newtypes = ["Centroids", "Nodes", "Linestrings", "Multilinestrings"]

				elif layer.dataProvider().geometryType() == QGis.WKBMultiPolygon25D:
					self.oldgeometry.setText("Multipolygon 2.5D")
					newtypes = ["Centroids", "Nodes", "Multilinestrings", "Polygons", "Multipolygons"]

		self.newgeometry.clear()
		for newtype in newtypes:
			self.newgeometry.addItem(newtype)

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		layername = unicode(self.sourcelayer.currentText())
		splitnodes = 0
		if self.newgeometry.currentText() == "Points":
			newtype = QGis.WKBPoint
		elif self.newgeometry.currentText() == "Centroids":
			newtype = QGis.WKBPoint
		elif self.newgeometry.currentText() == "Nodes":
			newtype = QGis.WKBPoint
			splitnodes = 1
		elif self.newgeometry.currentText() == "Linestrings":
			newtype = QGis.WKBLineString
		elif self.newgeometry.currentText() == "Multilinestrings":
			newtype = QGis.WKBMultiLineString
		elif self.newgeometry.currentText() == "Polygons":
			newtype = QGis.WKBPolygon
		else:
			newtype = QGis.WKBPoint

		message = mmqgis_geometry_convert(self.iface, layername, newtype, splitnodes, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Label", message)

# ---------------------------------------------------------------
#    mmqgis_delete_columns - Delete columns and save to shapefile
# ---------------------------------------------------------------

from mmqgis_delete_columns_form import *

class mmqgis_delete_columns_dialog(QDialog, Ui_mmqgis_delete_columns_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_fieldnames)

		self.set_fieldnames()

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_fieldnames(self):
		self.fieldnames.clear()
		layername = self.sourcelayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.fieldnames.addItem(field.name())

	def run(self):
		layername = unicode(self.sourcelayer.currentText())
		savename = unicode(self.filename.displayText()).strip()

		attributes = []
		for x in range(0, self.fieldnames.count()):
			if self.fieldnames.item(x).isSelected():
				attributes.append(self.fieldnames.item(x).text())

		message = mmqgis_delete_columns(self.iface, layername, attributes, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Delete Columns", message)


# --------------------------------------------------------
#    mmqgis_delete_duplicate_geometries - Save to shaperile
#			while removing duplicate shapes
# --------------------------------------------------------

from mmqgis_delete_duplicate_form import *

class mmqgis_delete_duplicate_dialog(QDialog, Ui_mmqgis_delete_duplicate_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		layername = unicode(self.sourcelayer.currentText())

		message = mmqgis_delete_duplicate_geometries(self.iface, layername, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Delete Duplicate Geometries", message)

# ---------------------------------------------------------
#    mmqgis_float_to_text - Change text fields to numbers
# ---------------------------------------------------------

from mmqgis_float_to_text_form import *

class mmqgis_float_to_text_dialog(QDialog, Ui_mmqgis_float_to_text_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_fieldnames)

		self.set_fieldnames()

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_fieldnames(self):
		self.fieldnames.clear()
		layername = self.sourcelayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					if (field.type() == QVariant.Double) or (field.type() == QVariant.Int):
						self.fieldnames.addItem(field.name())
						self.fieldnames.item(self.fieldnames.count() - 1).setSelected(1)

	def run(self):
		layername = unicode(self.sourcelayer.currentText())
		savename = unicode(self.filename.displayText()).strip()

		if unicode(self.separator.currentText()) == "Comma":
			separator = ','
		elif unicode(self.separator.currentText()) == "Space":
			separator = ' '
		else:
			separator = None

		decimals = self.decimals.currentIndex()
		prefix = unicode(self.prefix.text())
		suffix = unicode(self.suffix.text())

		attributes = []
		for x in range(0, self.fieldnames.count()):
			if self.fieldnames.item(x).isSelected():
				attributes.append(self.fieldnames.item(x).text())

		message = mmqgis_float_to_text(self.iface, layername, attributes, separator, 
			decimals, prefix, suffix, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Float to Text", message)



# --------------------------------------------------------
#    mmqgis_geometry_export - Import geometries from a
#			CSV files of nodes and attributes
#			into a shapefile
# --------------------------------------------------------

from mmqgis_geometry_export_form import *

class mmqgis_geometry_export_dialog(QDialog, Ui_mmqgis_geometry_export_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.nodebrowse, SIGNAL("clicked()"), self.browse_nodes)
		QObject.connect(self.attbrowse, SIGNAL("clicked()"), self.browse_attributes)
		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.check_layer_type)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		self.attfilename.setText(os.getcwd() + "/attributes.csv")
		self.nodefilename.setText(os.getcwd() + "/nodes.csv")

        def browse_nodes(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Nodes CSV File"), 
			self.nodefilename.displayText(), "*.csv")

		if newname != None:
                	self.nodefilename.setText(QString(newname))

        def browse_attributes(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Nodes CSV File"), 
			self.attfilename.displayText(), "*.csv")

		if newname != None:
                	self.attfilename.setText(QString(newname))

	def check_layer_type(self):
		layer = mmqgis_find_layer(self.sourcelayer.currentText())
		if (layer != None) and (layer.type() == QgsMapLayer.VectorLayer):
			if (layer.geometryType() == QGis.Point):
				self.attbrowse.setEnabled(False)
				self.attfilename.setEnabled(False)
			else:
				self.attbrowse.setEnabled(True)
				self.attfilename.setEnabled(True)

	def run(self):
		delimiter = ","
		if unicode(self.delimiter.currentText()) == "(bar)":
			delimiter = "|"
		elif unicode(self.delimiter.currentText()) == "(space)":
			delimiter = " "

		lineterminator = "\r\n"
		if unicode(self.lineterminator.currentText()) == "LF":
			lineterminator = "\n"

		sourcelayer = self.sourcelayer.currentText()
		nodefilename = self.nodefilename.displayText()
		attributefilename = self.attfilename.displayText()

		message = mmqgis_geometry_export_to_csv(self.iface, sourcelayer, nodefilename, 
			attributefilename, delimiter, lineterminator)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Geometry Export", message)


# --------------------------------------------------------
#    mmqgis_geometry_import - Import geometries from a
#			CSV files of nodes and attributes
#			into a shapefile
# --------------------------------------------------------

from mmqgis_geometry_import_form import *

class mmqgis_geometry_import_dialog(QDialog, Ui_mmqgis_geometry_import_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.nodebrowse, SIGNAL("clicked()"), self.browse_nodes)
		QObject.connect(self.outfilebrowse, SIGNAL("clicked()"), self.browse_shapefile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		self.geometrytype.addItem(QString("Point"))
		self.geometrytype.addItem(QString("Polyline"))
		self.geometrytype.addItem(QString("Polygon"))

		self.nodefilename.setText(os.getcwd() + "/nodes.csv")
		self.outfilename.setText(os.getcwd() + "/geometry.shp")

	def browse_nodes(self):
		newname = QFileDialog.getOpenFileName(None, QString.fromLocal8Bit("Input Nodes CSV File"), 
			self.nodefilename.displayText(), "*.csv")

                if not newname:
			return

		header = mmqgis_read_csv_header(self.iface, newname)
		if not header:
			return

		self.longcol.clear()
		self.latcol.clear()
		self.shapeidcol.clear()
		for field in header:
			self.longcol.addItem(QString(field))
			self.latcol.addItem(QString(field))
			self.shapeidcol.addItem(QString(field))

		for x in range(0, len(header)):
			if (header[x].lower() == "shapeid") or (header[x].lower() == 'shape_id'):
				self.shapeidcol.setCurrentIndex(x)
				break

		for x in range(0, len(header)):
			if (header[x].find("x") >= 0) or (header[x].find("X") >= 0) or (header[x].lower()[0:3] == 'lon'):
				self.longcol.setCurrentIndex(x)
				break

		for x in range(0, len(header)):
			if (header[x].find("y") >= 0) or (header[x].find("Y") >= 0) or (header[x].lower()[0:3] == 'lat'):
				self.latcol.setCurrentIndex(x)
				break


		#for x in range(self.shapeidcol.count()):
		#	if (unicode(self.shapeidcol.itemText(x)).lower() == "shapeid") or \
		#	   (unicode(self.shapeidcol.itemText(x)).lower() == "shape_id"):
		#		self.shapeidcol.setCurrentIndex(x)

		#for x in range(self.longcol.count()):
		#	if (unicode(self.longcol.itemText(x)).lower() == "x") or \
		#	   (unicode(self.longcol.itemText(x)).lower()[0:3] == "lon"):
		#		self.longcol.setCurrentIndex(x)

		#for x in range(self.latcol.count()):
		#	if (unicode(self.latcol.itemText(x)).lower() == "x") or \
		#	   (unicode(self.latcol.itemText(x)).lower()[0:3] == "lon"):
		#		self.latcol.setCurrentIndex(x)

		self.nodefilename.setText(newname)
		shapename = newname
		shapename = shapename.replace(".csv", ".shp")
		shapename = shapename.replace(".CSV", ".shp")
		if shapename == newname:
			shapename = unicode(shapename) + ".shp"
		self.outfilename.setText(shapename)


        def browse_shapefile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.outfilename.displayText(), "*.shp")

                if newname != None:
                	self.outfilename.setText(QString(newname))

	def run(self):
		message = mmqgis_geometry_import_from_csv(self.iface, unicode(self.nodefilename.displayText()), 
			unicode(self.longcol.currentText()), unicode(self.latcol.currentText()),
			unicode(self.shapeidcol.currentText()), unicode(self.geometrytype.currentText()),
			unicode(self.outfilename.displayText()), 1)

		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Geometry Import", message)


# --------------------------------------------------------
#    mmqgis_geocode_google - Geocode using Google Maps
# --------------------------------------------------------

from mmqgis_geocode_google_form import *

#pyqt4-dev-tools
#designer

class mmqgis_geocode_google_dialog(QDialog, Ui_mmqgis_geocode_google_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse_infile, SIGNAL("clicked()"), self.browse_infile_dialog)
		QObject.connect(self.browse_shapefile, SIGNAL("clicked()"), self.browse_shapefile_dialog)
		QObject.connect(self.browse_notfound, SIGNAL("clicked()"), self.browse_notfound_dialog)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

        def browse_infile_dialog(self):
		newname = QFileDialog.getOpenFileName(None, QString.fromLocal8Bit("Address CSV Input File"), 
			self.infilename.displayText(), "*.csv")

                if newname:
			try:
				infile = open(newname, 'r')
				dialect = csv.Sniffer().sniff(infile.read(1024))
			except:
				QMessageBox.critical(self.iface.mainWindow(), "Geocode Google", \
					newname + " inaccessible or in unrecognized CSV format: " + unicode(sys.exc_info()[1]))
				return

			if len(newname) > 4:
				prefix = newname[:len(newname) - 4]
				self.shapefilename.setText(prefix + ".shp")
				self.notfoundfilename.setText(prefix + "_notfound.csv")
			else:
				self.shapefilename.setText(os.getcwd() + "/geocoded.shp")
				self.notfoundfilename.setText(os.getcwd() + "/notfound.csv")

			infile.seek(0)
			try:
				reader = csv.reader(infile, dialect)
				header = reader.next()
			except:
				QMessageBox.critical(self.iface.mainWindow(), "Geocode Google", \
					newname + " inaccessible or in unrecognized CSV format: " + unicode(sys.exc_info()[1]))
				return

			del reader

			combolist = [self.addressfield, self.cityfield, self.statefield, self.countryfield]
			for box in combolist:
				box.clear()
				box.addItem(QString("(none)"))
				box.setCurrentIndex(0)
				
			for index in range(0, len(header)):
				field = header[index]
				for box in combolist:
					box.addItem(QString(field))

				if field.lower().find("addr") >= 0:
					self.addressfield.setCurrentIndex(index + 1)
				if field.lower().find("street") >= 0:
					self.addressfield.setCurrentIndex(index + 1)
				if field.lower().find("city") >= 0:
					self.cityfield.setCurrentIndex(index + 1)
				if field.lower().find("state") >= 0:
					self.statefield.setCurrentIndex(index + 1)
				if field.lower() == "st":
					self.statefield.setCurrentIndex(index + 1)
				if field.lower().find("province") >= 0:
					self.statefield.setCurrentIndex(index + 1)
				if field.lower().find("country") >= 0:
					self.countryfield.setCurrentIndex(index + 1)

                	self.infilename.setText(QString(newname))

        def browse_notfound_dialog(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Not Found List Output File"), 
			self.notfoundfilename.displayText(), "*.csv")
                if newname != None:
                	self.notfoundfilename.setText(QString(newname))

        def browse_shapefile_dialog(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.shapefilename.displayText(), "*.shp")
                if newname != None:
                	self.shapefilename.setText(QString(newname))

	def run(self):
		csvname = unicode(self.infilename.displayText()).strip()
		shapefilename = unicode(self.shapefilename.displayText())
		notfoundfile = self.notfoundfilename.displayText()

		fields = [unicode(self.addressfield.currentText()).strip(),
			  unicode(self.cityfield.currentText()).strip(),
			  unicode(self.statefield.currentText()).strip(),
			  unicode(self.countryfield.currentText()).strip()]
	
		for x in range(0, len(fields)):
			if fields[x] == "(none)":
				fields[x] = ""

		# print csvname + "," + "," + shapefilename
		message = mmqgis_geocode_google(self.iface, csvname, shapefilename, notfoundfile, fields, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Geocode Goodle", message)

# --------------------------------------------------------
#    mmqgis_grid - Grid creation plugin
# --------------------------------------------------------

from mmqgis_grid_form import *

class mmqgis_grid_dialog(QDialog, Ui_mmqgis_grid_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.hspacing, SIGNAL("textEdited(QString)"), self.hspacing_changed)
		QObject.connect(self.vspacing, SIGNAL("textEdited(QString)"), self.vspacing_changed)
		QObject.connect(self.gridtype, SIGNAL("currentIndexChanged(QString)"), self.gridtype_changed)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		self.crs = QgsCoordinateReferenceSystem()
		self.crs.createFromProj4("+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs")
		extent = QgsRectangle (-10.0, -10.0, 10.0, 20.0)

		# Done as exception handler to make backward compatible from "crs" functions new to qgis 1.7 and 2.0
		# if (self.iface.mapCanvas() != None) and (self.iface.mapCanvas().mapRenderer() != None):
		try:
			self.crs = self.iface.mapCanvas().mapRenderer().destinationCrs()
			extent = self.iface.mapCanvas().mapRenderer().extent()

		#elif self.iface.activeLayer() != None:
		except:
			try:
				extent = self.iface.activeLayer().extent()
				self.crs = self.iface.activeLayer().crs()
			except:
				extent = extent
		# print self.crs.toProj4()

		spacing = 1
		if extent.width() > 10000:
			spacing = 1000
		elif extent.width() > 1000:
			spacing = 100
		elif extent.width() > 100:
			spacing = 10

		self.xtype.setCurrentIndex(1);
		self.ytype.setCurrentIndex(1);
		self.xvalue.setText(unicode((extent.xMinimum() + extent.xMaximum()) / 2))
		self.yvalue.setText(unicode((extent.yMinimum() + extent.yMaximum()) / 2))
		#self.width.setText(unicode(round(extent.width() / spacing) * spacing))
		#self.height.setText(unicode(round(extent.height() / spacing) * spacing))
		self.width.setText(unicode(extent.width()))
		self.height.setText(unicode(extent.height()))
		self.hspacing.setText(unicode(spacing))
		self.vspacing.setText(unicode(spacing))

		self.gridtype.addItem(QString("Rectangle (line)"))
		self.gridtype.addItem(QString("Rectangle (polygon)"))
		self.gridtype.addItem(QString("Diamond (polygon)"))
		self.gridtype.addItem(QString("Hexagon (polygon)"))
		self.gridtype.setCurrentIndex(0)

		self.filename.setText(os.getcwd() + "/grid.shp")
		

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def vspacing_changed(self, text):
		# Hexagonal grid must maintain fixed aspect ratio to make sense
		if unicode(self.gridtype.currentText()) == "Hexagon (polygon)":
			spacing = float(text)
			self.hspacing.setText(unicode(spacing * 0.866025403784439))

	def hspacing_changed(self, text):
		if unicode(self.gridtype.currentText()) == "Hexagon (polygon)":
			spacing = float(text)
			self.vspacing.setText(unicode(spacing / 0.866025))

	def gridtype_changed(self, text):
		if text == "Hexagon (polygon)":
			spacing = float(self.vspacing.displayText())
			self.hspacing.setText(unicode(spacing * 0.866025))

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		hspacing = float(self.hspacing.displayText())
		vspacing = float(self.vspacing.displayText())
		width = float(self.width.displayText())
		height = float(self.height.displayText())

		originx = float(self.xvalue.displayText())
		if (str(self.xtype.currentText()) == "Center X") and (hspacing != 0):
			originx = originx - (round(width / 2.0 / hspacing) * hspacing)
		elif str(self.xtype.currentText()) == "Right X":
			originx = originx - width

		originy = float(self.yvalue.displayText())
		if (str(self.ytype.currentText()) == "Middle Y") and (vspacing != 0):
			originy = originy - (round(height / 2.0 / vspacing) * vspacing)
		elif str(self.ytype.currentText()) == "Top Y":
			originy = originy - height

		gridtype = unicode(self.gridtype.currentText())

		message = mmqgis_grid(self.iface, savename, hspacing, vspacing, width, height, originx, originy, gridtype, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Grid", message)



# --------------------------------------------------------
#    mmqgis_gridify - Snap shape verticies to grid
# --------------------------------------------------------

from mmqgis_gridify_form import *

class mmqgis_gridify_dialog(QDialog, Ui_mmqgis_gridify_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(int)"), self.layer_changed)
		
		self.filename.setText(os.getcwd() + "/temp.shp")
		

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def layer_changed(self):
		layer = None
		layername = unicode(self.sourcelayer.currentText())
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				layer = searchlayer

		extent = QgsRectangle (-10.0, -10.0, 10.0, 20.0)
		if (layer):
			extent = layer.extent()

		#if (self.iface.mapCanvas() != None) and (self.iface.mapCanvas().mapRenderer() != None):
		#	extent = self.iface.mapCanvas().mapRenderer().extent()
		#elif self.iface.activeLayer():
		#	extent = self.iface.activeLayer().extent()

		self.hspacing.setText(unicode(extent.width() / 200))
		self.vspacing.setText(unicode(extent.height() / 200))

	def run(self):
		layername = unicode(self.sourcelayer.currentText()).strip()
		hspacing = float(self.hspacing.displayText())
		vspacing = float(self.hspacing.displayText())
		savename = unicode(self.filename.displayText()).strip()

		message = mmqgis_gridify_layer(self.iface, layername, hspacing, vspacing, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Gridify", message)


# --------------------------------------------------------
#    mmqgis_hub_distance - Create shapefile of distances
#			   from points to nearest hub
# --------------------------------------------------------

from mmqgis_hub_distance_form import *

class mmqgis_hub_distance_dialog(QDialog, Ui_mmqgis_hub_distance_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayerbox.addItem(layer.name())
				self.hubslayerbox.addItem(layer.name())

		if self.hubslayerbox.count() > 1:
			self.hubslayerbox.setCurrentIndex(1)

		QObject.connect(self.hubslayerbox, SIGNAL("currentIndexChanged(QString)"), self.set_name_attributes)

		self.set_name_attributes()

		self.outputtype.addItem("Point")
		self.outputtype.addItem("Line to Hub")

		self.measurement.addItem("Meters")
		self.measurement.addItem("Feet")
		self.measurement.addItem("Miles")
		self.measurement.addItem("Kilometers")
		self.measurement.addItem("Layer Units")
	
		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_name_attributes(self):
		self.nameattributebox.clear()
		layername = self.hubslayerbox.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.nameattributebox.addItem(field.name())

	def run(self):
		sourcename = unicode(self.sourcelayerbox.currentText())
		destname = unicode(self.hubslayerbox.currentText())
		nameattributename = unicode(self.nameattributebox.currentText())
		units = unicode(self.measurement.currentText())
		addlines = (self.outputtype.currentText() == "Line to Hub")
		savename = unicode(self.filename.displayText()).strip()
			
		message = mmqgis_hub_distance(self.iface, sourcename, destname, nameattributename, units, addlines, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Hub Distance", message)


# --------------------------------------------------------
#    mmqgis_hub_lines - Create shapefile of lines from
#			spoke points to matching hubs
# --------------------------------------------------------

from mmqgis_hub_lines_form import *

class mmqgis_hub_lines_dialog(QDialog, Ui_mmqgis_hub_lines_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.hublayer.addItem(layer.name())
				self.spokelayer.addItem(layer.name())

		QObject.connect(self.hublayer, SIGNAL("currentIndexChanged(QString)"), self.set_hub_attributes)
		QObject.connect(self.spokelayer, SIGNAL("currentIndexChanged(QString)"), self.set_spoke_attributes)

		self.set_hub_attributes(self.hublayer.currentText())
		self.set_spoke_attributes(self.spokelayer.currentText())

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_hub_attributes(self, layername):
		self.hubid.clear()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.hubid.addItem(field.name())

	def set_spoke_attributes(self, layername):
		self.spokehubid.clear()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.spokehubid.addItem(field.name())

	def run(self):
		hubname = unicode(self.hublayer.currentText())
		hubattr = unicode(self.hubid.currentText())
		spokename = unicode(self.spokelayer.currentText())
		spokeattr = unicode(self.spokehubid.currentText())
		savename = unicode(self.filename.displayText()).strip()
			
		message = mmqgis_hub_lines(self.iface, hubname, hubattr, spokename, spokeattr, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Hub Lines", message)


# --------------------------------------------------------
#    mmqgis_label - Create single label points for
#		    single- or multi-feature items
# --------------------------------------------------------

from mmqgis_label_form import *

class mmqgis_label_dialog(QDialog, Ui_mmqgis_label_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_label_attributes)

		self.set_label_attributes()

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_label_attributes(self):
		self.labelattribute.clear()
		layername = self.sourcelayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.labelattribute.addItem(field.name())

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		layername = unicode(self.sourcelayer.currentText())
		labelattributename = self.labelattribute.currentText()

		message = mmqgis_label_point(self.iface, layername, labelattributename, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Label", message)


# --------------------------------------------------------
#    mmqgis_merge - Merge layers to single shapefile
# --------------------------------------------------------

from mmqgis_merge_form import *

class mmqgis_merge_dialog(QDialog, Ui_mmqgis_merge_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browseoutfile, SIGNAL("clicked()"), self.browse_outfiles)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		self.sourcelayers.clear()
		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayers.addItem(QString(layer.name()))
				self.sourcelayers.item(self.sourcelayers.count() - 1).setSelected(1)

		self.outfilename.setText(os.getcwd() + "/temp.shp")

        def browse_outfiles(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.outfilename.displayText(), "*.shp")
                if newname != None:
                	self.outfilename.setText(QString(newname))

	def run(self):
		layernames = []
		for x in range(0, self.sourcelayers.count()):
			if self.sourcelayers.item(x).isSelected():
				layernames.append(unicode(self.sourcelayers.item(x).text()))

		savename = unicode(self.outfilename.displayText()).strip()

		message = mmqgis_merge(self.iface, layernames, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Merge", message)


# ----------------------------------------------------------
#    mmqgis_select - Select features by attribute
# ----------------------------------------------------------

from mmqgis_select_form import *

class mmqgis_select_dialog(QDialog, Ui_mmqgis_select_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_select_attributes)
		self.set_select_attributes()

		comparisons = ['==', '!=', '>', '>=', '<', '<=', 'begins with', 'contains']
		for x in comparisons:
			self.comparison.addItem(x)

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_select_attributes(self):
		self.selectattribute.clear()
		layername = self.sourcelayer.currentText()
		for name, selectlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if selectlayer.name() == layername:
				for index, field in selectlayer.dataProvider().fields().iteritems():
					self.selectattribute.addItem(field.name())

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		layername = unicode(self.sourcelayer.currentText())
		comparisonname = self.comparison.currentText()
		comparisonvalue = unicode(self.value.displayText()).strip()
		selectattributename = self.selectattribute.currentText()

		message = mmqgis_select(self.iface, layername, selectattributename, \
			comparisonvalue, comparisonname, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Select", message)


# --------------------------------------------------------
#    mmqgis_sort - Sort shapefile by attribute
# --------------------------------------------------------

from mmqgis_sort_form import *

class mmqgis_sort_dialog(QDialog, Ui_mmqgis_sort_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())
		#QObject.connect(self.sourcelayer, SIGNAL("editTextChanged(QString)"), self.set_sort_attributes)
		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_sort_attributes)

		self.set_sort_attributes()

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_sort_attributes(self):
		self.sortattribute.clear()
		layername = self.sourcelayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.sortattribute.addItem(field.name())

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		layername = unicode(self.sourcelayer.currentText())
		direction = unicode(self.direction.currentText())
		sortattributename = self.sortattribute.currentText()

		message = mmqgis_sort(self.iface, layername, sortattributename, savename, direction, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Sort", message)


# --------------------------------------------------------
#    mmqgis_geocode_street_layer - Geocode addresses from street 
#			     address finder shapefile
# --------------------------------------------------------

from mmqgis_geocode_street_layer_form import *

class mmqgis_geocode_street_layer_dialog(QDialog, Ui_mmqgis_geocode_street_layer_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse_infile, SIGNAL("clicked()"), self.browse_infile_dialog)
		QObject.connect(self.browse_shapefile, SIGNAL("clicked()"), self.browse_shapefile_dialog)
		QObject.connect(self.browse_notfound, SIGNAL("clicked()"), self.browse_notfound_dialog)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.streetlayer.addItem(layer.name())

		QObject.connect(self.streetlayer, SIGNAL("currentIndexChanged(int)"), self.set_layer_attributes)
		#QObject.connect(self.streetlayer, SIGNAL("activated(int)"), self.set_layer_attributes)

		self.set_layer_attributes(0)

		self.shapefilename.setText(os.getcwd() + "/temp.shp")
		self.notfoundfilename.setText(os.getcwd() + "/notfound.csv")

        def browse_infile_dialog(self):
		newname = QFileDialog.getOpenFileName(None, QString.fromLocal8Bit("Address CSV Input File"), 
			self.infilename.displayText(), "*.csv")

                if newname:
			try:
				infile = open(newname, 'r')
				dialect = csv.Sniffer().sniff(infile.read(1024))
			except:
				QMessageBox.critical(self.iface.mainWindow(), "Input CSV File", \
					"File inaccessible or in unrecognized CSV format " + newname)
				return

			infile.seek(0)
			reader = csv.reader(infile, dialect)
			header = reader.next()
			del reader

			# Add attributes to street and number
			self.streetnamefield.clear()
			self.numberfield.clear()
			for field in header:
				self.streetnamefield.addItem(QString(field))
				self.numberfield.addItem(QString(field))

			for x in range(0, len(header)):
				if header[x].strip().lower().find("street") >= 0:
					self.streetnamefield.setCurrentIndex(x)

				if header[x].strip().lower().find("number") >= 0:
					self.numberfield.setCurrentIndex(x)

                	self.infilename.setText(QString(newname))

        def browse_notfound_dialog(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Not Found List Output File"), 
			self.notfoundfilename.displayText(), "*.csv")
                if newname != None:
                	self.notfoundfilename.setText(QString(newname))

        def browse_shapefile_dialog(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.shapefilename.displayText(), "*.shp")
                if newname != None:
                	self.shapefilename.setText(QString(newname))

	def set_layer_attributes(self, index):
		# index parameter required for currentIndexChanged() signal and is not used
		layername = unicode(self.streetlayer.currentText())
		layer = mmqgis_find_layer(layername)

		if not layer:
			print "Layer not found " + layername
			return

		if (layer.dataProvider().crs().mapUnits() == QGis.Feet):
                	self.setback.setText("60")
		elif (layer.dataProvider().crs().mapUnits() == QGis.Meters):
                	self.setback.setText("4")
		elif (layer.dataProvider().crs().mapUnits() == QGis.Degrees):
			meters = 4 * cos(layer.extent().center().y() * pi / 180) * 6378137.0
                	self.setback.setText(unicode(meters))
		else:
                	self.setback.setText("4")

		self.streetname.clear()
		self.fromx.clear()
		self.fromy.clear()
		self.tox.clear()
		self.toy.clear()
		self.leftfrom.clear()
		self.leftto.clear()
		self.rightfrom.clear()
		self.rightto.clear()

		# From/To options to use line geometries for X/Y coordinates
		# Assumes order of line vertices in shapefile is consistent

		self.fromx.addItem("(street line start)")
		self.fromx.addItem("(street line end)")
		self.fromy.addItem("(street line start)")
		self.fromy.addItem("(street line end)")
		self.tox.addItem("(street line start)")
		self.tox.addItem("(street line end)")
		self.toy.addItem("(street line start)")
		self.toy.addItem("(street line end)")

		self.fromx.setCurrentIndex(0)
		self.fromy.setCurrentIndex(0)
		self.tox.setCurrentIndex(1)
		self.toy.setCurrentIndex(1)

		# Add all attributes to lists

		for index, field in layer.dataProvider().fields().iteritems():
			self.streetname.addItem(unicode(field.name()))
			self.fromx.addItem(unicode(field.name()))
			self.fromy.addItem(unicode(field.name()))
			self.tox.addItem(unicode(field.name()))
			self.toy.addItem(unicode(field.name()))
			self.leftfrom.addItem(unicode(field.name()))
			self.leftto.addItem(unicode(field.name()))
			self.rightfrom.addItem(unicode(field.name()))
			self.rightto.addItem(unicode(field.name()))


		# Select different parameters based on guesses from attribute names

		for index, field in layer.dataProvider().fields().iteritems():
			if unicode(field.name()).lower().find("name") >= 0:
				self.streetname.setCurrentIndex(index)
			elif unicode(field.name()).lower() == "street":
				self.streetname.setCurrentIndex(index)
			elif unicode(field.name()).lower() == "xfrom":
				self.fromx.setCurrentIndex(index + 2)
			elif unicode(field.name()).lower() == "yfrom":
				self.fromy.setCurrentIndex(index + 2)
			elif unicode(field.name()).lower() == "xto":
				self.tox.setCurrentIndex(index + 2)
			elif unicode(field.name()).lower() == "yto":
				self.toy.setCurrentIndex(index + 2)
			elif unicode(field.name()).lower() == "fromleft":
				self.leftfrom.setCurrentIndex(index)
			elif unicode(field.name()).lower() == "fromright":
				self.rightfrom.setCurrentIndex(index)
			elif unicode(field.name()).lower() == "toleft":
				self.leftto.setCurrentIndex(index)
			elif unicode(field.name()).lower() == "toright":
				self.rightto.setCurrentIndex(index)

	def run(self):
		layername = unicode(self.streetlayer.currentText())
		csvname = unicode(self.infilename.displayText()).strip()
		streetnamefield = unicode(self.streetnamefield.currentText()).strip()
		numberfield = unicode(self.numberfield.currentText()).strip()
		shapefilename = unicode(self.shapefilename.displayText())
		streetname = unicode(self.streetname.currentText())
		fromx = unicode(self.fromx.currentText())
		fromy = unicode(self.fromy.currentText())
		tox = unicode(self.tox.currentText())
		toy = unicode(self.toy.currentText())
		leftfrom = unicode(self.leftfrom.currentText())
		rightfrom = unicode(self.rightfrom.currentText())
		leftto = unicode(self.leftto.currentText())
		rightto = unicode(self.rightto.currentText())
		setback = float(self.setback.displayText())
		notfoundfile = self.notfoundfilename.displayText()
		# addlayer = self.addtoproject.isChecked()

		message = mmqgis_geocode_street_layer(self.iface, layername, csvname, streetnamefield, 
			numberfield, shapefilename, streetname, fromx, fromy, tox, toy, leftfrom, rightfrom, 
			leftto, rightto, setback, notfoundfile, 1)

		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Geocode", message)

# ---------------------------------------------------------
#    mmqgis_text_to_float - Change text fields to numbers
# ---------------------------------------------------------

from mmqgis_text_to_float_form import *

class mmqgis_text_to_float_dialog(QDialog, Ui_mmqgis_text_to_float_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		QObject.connect(self.sourcelayer, SIGNAL("currentIndexChanged(QString)"), self.set_fieldnames)

		self.set_fieldnames()

		self.filename.setText(os.getcwd() + "/temp.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def set_fieldnames(self):
		self.fieldnames.clear()
		layername = self.sourcelayer.currentText()
		for name, searchlayer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if searchlayer.name() == layername:
				for index, field in searchlayer.dataProvider().fields().iteritems():
					self.fieldnames.addItem(field.name())

				for index, field in searchlayer.dataProvider().fields().iteritems():
					if (field.type() == QVariant.String):
						self.fieldnames.item(index).setSelected(1)

	def run(self):
		layername = unicode(self.sourcelayer.currentText())
		savename = unicode(self.filename.displayText()).strip()
		# addlayer = self.addtoproject.isChecked()

		attributes = []
		for x in range(0, self.fieldnames.count()):
			if self.fieldnames.item(x).isSelected():
				attributes.append(self.fieldnames.item(x).text())

		message = mmqgis_text_to_float(self.iface, layername, attributes, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Text to Float", message)


# --------------------------------------------------------
#    mmqgis_voronoi - Voronoi diagram creation
# --------------------------------------------------------

from mmqgis_voronoi_form import *

class mmqgis_voronoi_dialog(QDialog, Ui_mmqgis_voronoi_form):
	def __init__(self, iface):
		QDialog.__init__(self)
		self.iface = iface
		self.setupUi(self)
		QObject.connect(self.browse, SIGNAL("clicked()"), self.browse_outfile)
        	QtCore.QObject.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.run)

		for name, layer in QgsMapLayerRegistry.instance().mapLayers().iteritems():
			if layer.type() == QgsMapLayer.VectorLayer:
				self.sourcelayer.addItem(layer.name())

		self.filename.setText(os.getcwd() + "/voronoi.shp")

        def browse_outfile(self):
		newname = QFileDialog.getSaveFileName(None, QString.fromLocal8Bit("Output Shapefile"), 
			self.filename.displayText(), "*.shp")
                if newname != None:
                	self.filename.setText(QString(newname))

	def run(self):
		savename = unicode(self.filename.displayText()).strip()
		layer = unicode(self.sourcelayer.currentText())

		message = mmqgis_voronoi_diagram(self.iface, layer, savename, 1)
		if message <> None:
			QMessageBox.critical(self.iface.mainWindow(), "Voronoi", message)


# --------------------------------------------------------
#    mmqgis_read_csv_header()
#
# Reads the header (column names) from a CSV file
# --------------------------------------------------------

def mmqgis_read_csv_header(qgis, filename):
	try:
		infile = open(filename, 'r')
	except:
		QMessageBox.information(qgis.mainWindow(), "Input CSV File", "Failure opening " + filename)
		return None

	try:
		dialect = csv.Sniffer().sniff(infile.read(2048))
	except:
		QMessageBox.information(qgis.mainWindow(), "Input CSV File", 
			"Bad CSV file - verify that your delimiters are consistent");
		return None

	infile.seek(0)
	reader = csv.reader(infile, dialect)
		
	header = reader.next()
	del reader
	del infile

	if len(header) <= 0:
		QMessageBox.information(qgis.mainWindow(), "Input CSV File", 
			filename + " does not appear to be a CSV file")
		return None

	return header

