# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MIDAS
                                 A QGIS plugin
 Memory-Efficient I/O-Improved Drainage Analysis System
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-26
        copyright            : (C) 2025 by Huidae Cho
        email                : grass4u@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Huidae Cho'
__date__ = '2025-07-26'
__copyright__ = '(C) 2025 by Huidae Cho'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsMessageLog,
                       QgsProject,
                       QgsProcessing,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterString,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterNumber,
                       QgsProcessingException,
                       QgsRasterFileWriter,
                       QgsVectorFileWriter,
                       QgsRasterLayer,
                       QgsRasterPipe,
                       QgsVectorLayer,
                       QgsCoordinateTransformContext,
                       QgsCoordinateReferenceSystem,
                       QgsWkbTypes,
                       QgsFields,
                       QgsField)
from PyQt5.QtCore import QMetaType
from .utils import run_command_stream_output


class MESHEDAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    dir_ = "dir"
    format_ = "format"
    encoding = "encoding"
    outlets = "outlets"
    idcol = "idcol"
    wsheds = "wsheds"
    use_lessmem = "use_lessmem"
    compress_output = "compress_output"
    hier = "hier"
    nprocs = "nprocs"

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.dir_,
                self.tr("Input flow direction raster")
            )
        )

        param = QgsProcessingParameterString(
                self.format_,
                self.tr("Input flow direction encoding"),
                "power2"
        )
        param.setMetadata({
            'widget_wrapper': {
                'value_hints': [
                    "degree",
                    "45degree",
                    "power2",
                    "taudem"
                ]}
            }
        )
        self.addParameter(param)

        self.addParameter(
            QgsProcessingParameterString(
                self.encoding,
                self.tr("Custom flow direction encoding (a comma-separated list of eight numbers for E,SE,S,SW,W,NW,N,NE)"),
                optional=True
            ),
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.outlets,
                self.tr("Input outlet points vector"),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.idcol,
                self.tr("Input field for outlet IDs"),
                parentLayerParameterName=self.outlets
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.wsheds,
                self.tr("Output watersheds GeoTIFF"),
                "GeoTIFF files (*.tif)"
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.use_lessmem,
                self.tr("Use less memory")
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.compress_output,
                self.tr("Compress output GeoTIFF")
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.hier,
                self.tr("Output subwatershed hierarchy CSV"),
                "CSV files (*.csv)",
                optional=True,
                createByDefault=False
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.nprocs,
                self.tr("Number of threads for parallel computing (0: use OpenMP default; >0: use nprocs; <0: use MAX-nprocs)"),
                QgsProcessingParameterNumber.Integer,
                0,
                minValue=0
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        dir_rast = self.parameterAsRasterLayer(parameters, self.dir_, context)
        format_ = self.parameterAsString(parameters, self.format_, context)
        encoding = self.parameterAsString(parameters, self.encoding, context)
        outlets_vect = self.parameterAsVectorLayer(parameters, self.outlets, context)
        idcol = self.parameterAsString(parameters, self.idcol, context)
        wsheds_path = self.parameterAsFileOutput(parameters, self.wsheds, context)
        use_lessmem = self.parameterAsBoolean(parameters, self.use_lessmem, context)
        compress_output = self.parameterAsBoolean(parameters, self.compress_output, context)
        hier_path = self.parameterAsFileOutput(parameters, self.hier, context)
        nprocs = self.parameterAsString(parameters, self.nprocs, context)

        # XXX: not sure how QGIS passes a raster layer name in GeoPackage;
        # GDALOpenEx() understands GPKG:file.gpkg:layer_name; for now, just
        # pass it as is
        #dir_path = dir_rast.dataProvider().dataSourceUri()
        dir_path = dir_rast.source()

        # QGIS passes a vector layer name in GeoPackage as
        # file.gpkg|layername=name
        # XXX: other special cases to handle? selected features?
        #outlets_path = outlets_vect.dataProvider().dataSourceUri()
        outlets_path = outlets_vect.source()
        outlets_layer = None
        if "|layername=" in outlets_path:
            items = outlets_path.split("|layername=")
            outlets_path = items[0]
            outlets_layer = items[1]

        if format_ == "custom":
            if not encoding:
                raise QgsProcessingException(f"Encoding not specified for custom format")
        else:
            encoding = format_

        cmd = ["meshed", dir_path, outlets_path, idcol, wsheds_path, "-e", encoding, "-t", nprocs]

        if outlets_layer:
            cmd.extend(["-o", outlets_layer])

        if use_lessmem:
            cmd.extend(["-m"])
        if compress_output:
            cmd.extend(["-c"])

        if hier_path:
            cmd.extend(["-h", hier_path])

        QgsMessageLog.logMessage(str(cmd))

        run_command_stream_output(cmd, feedback)
        #result = subprocess.run(cmd, capture_output=True)
        #if result.returncode != 0:
        #    raise QgsProcessingException(f"Executable failed: {result.stderr.decode()}")

        ret = {}
        project = context.project()
        wsheds_rast = QgsRasterLayer(f"{wsheds_path}", "Watersheds", "gdal")
        if not wsheds_rast.isValid():
            feedback.reportError("Failed to load wsheds layer!")
            ret[self.wsheds] = None
        else:
            project.addMapLayer(wsheds_rast)
            ret[self.wsheds] = wsheds_rast.source()

        QgsMessageLog.logMessage(str(ret))

        return ret

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Delineate watersheds'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return MESHEDAlgorithm()
