# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MGWR
                                 A QGIS plugin
 A QGIS plugin for Multiscale Geographically Weighted Regression (MGWR)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-05-05
        copyright            : (C) 2024 by Huaying Gu & Yi Wu
        email                : aaron.yiwu15@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Huaying Gu & Yi Wu'
__date__ = '2024-05-05'
__copyright__ = '(C) 2024 by Huaying Gu & Yi Wu'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterField,
                       QgsProcessingParameterString,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingMultiStepFeedback,
                       QgsFields,
                       QgsField,
                       QgsFeature,
                       QgsWkbTypes,
                       QgsProcessingUtils)

import os
import inspect
import platform
import pandas as pd
from mgwr.sel_bw import Sel_BW
from mgwr.gwr import MGWR

class MGWRAlgorithm(QgsProcessingAlgorithm):
    """
    This is an algorithm that takes a vector layer and
    sakdlfjas;ljf;askldf

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'
    # load to qgis directly as sink layer
    SINK_LAYER = 'SINK_LAYER'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        # add the input vector features source.
        # the input vector feature must be points or polygons.
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                name = 'source_layer',
                description = 'Input layer',
                types = [QgsProcessing.TypeVector],
                defaultValue = None,
                optional = False
            )
        )

        # load dependent field from source layer
        self.addParameter(
            QgsProcessingParameterField(
                name = 'dependent_field',
                description = 'Select one dependent field', 
                type=QgsProcessingParameterField.Any, 
                parentLayerParameterName='source_layer', 
                allowMultiple=False, 
                optional = False
            )
        )

        # load explanatory fields from source layer
        self.addParameter(
            QgsProcessingParameterField(
                name = 'explanatory_field',
                description = 'Select one or multiple explanatory fields', 
                type=QgsProcessingParameterField.Any, 
                parentLayerParameterName='source_layer', 
                allowMultiple=True, 
                optional = False
            )
        )
                
        # select spatial neighborhood type
        self.addParameter(
            QgsProcessingParameterEnum(
                name = 'neighborhood_type',
                description = 'Select Neighborhood type',
                options = ['Adaptive','Fixed'],
                allowMultiple = False, 
                defaultValue = 'Adaptive',
                optional = False
                
            )
        )
        
        # select spatial kernel function
        self.addParameter(
            QgsProcessingParameterEnum(
                name = 'kernel_function',
                description = 'Select spatial kernel function',
                options = ['Bisquare','Gaussian', 'Exponential'],
                allowMultiple = False, 
                defaultValue = 'Bisquare',
                optional = False
                
            )
        )

        # select bandwidth searching criterion
        self.addParameter(
            QgsProcessingParameterEnum(
                name = 'bandwidth_searching_criterion',
                description = 'Select criterion for bandwidth searching',
                options = ['AICc', 'AIC','BIC','CV'],
                allowMultiple = False, 
                defaultValue = 'AICc',
                optional = False
                
            )
        )
        
        # input minimum bandwidth value: single value for all covariates
        # or setting specific value for each covariate including intersept
        self.addParameter(
            QgsProcessingParameterString(
                name = 'multi_bw_min',
                description = 'Min Bandwidth for each covariate (seperated by space or one value for all)',
                multiLine = False, 
                optional = True, 
            )
        )

        # input maximum bandwidth value: single value for all covariates
        # or setting specific value for each covariate including intersept
        self.addParameter(
            QgsProcessingParameterString(
                name = 'multi_bw_max',
                description = 'Max Bandwidth for each covariate (seperated by space or one value for all)',
                multiLine = False, 
                optional = True, 
            )
        )

        # output the mgwr result summary
        self.addParameter(
            QgsProcessingParameterFileDestination(
                name = 'output_summary', 
                description = 'Summary File', 
                fileFilter='TXT Files (*.txt)', 
                defaultValue = None,
                optional = True
            )
        )
        
        # output the feature layer as sink to qgis directly
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.SINK_LAYER, 
                description = 'Output',
                type = QgsProcessing.TypeVectorAnyGeometry,
                defaultValue = None,
                optional = True
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # ************************************************************************************#
        # *********************************do some prep work**********************************#
        # ************************************************************************************#   
   
        # Use a multi-step feedback, so that individual child algorithm progress reports are
        # adjusted for the overall progress through the model
        feedback = QgsProcessingMultiStepFeedback(2, feedback)
        
        # temporary: parameter to save the output layer: SINK_LAYER
        results = {}
        
        # temporary: parameter to save result explanatory fields
        sink_result_name_explanatory_field = []

        # get the input layer as source
        input_featuresource = self.parameterAsSource(parameters, 'source_layer', context)
        if input_featuresource is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, 'source_layer'))
        
        # set the output mgwr result summary     
        txt = self.parameterAsFileOutput(parameters, 'output_summary', context)
        
        # get features of input layer 
        features = input_featuresource.getFeatures()

        # Judge spherical coordinates (long-lat) 
        spherical = input_featuresource.sourceCrs().isGeographic()
        #feedback.pushInfo('spherical is ' + str(spherical))

        # get the features fieldname of input layer
        fieldnames = [field.name() for field in input_featuresource.fields()]
        
        # set the list as the processing variables
        layer_attributes_attr = []  # temporary: parameter to save attributes of input layer features
        layer_attributes_attr_geometry = [] # temporary: parameter to save geometry of input layer features
        location_x = [] # temporary: parameter to save the x coordinate
        location_y = [] # temporary: parameter to save the y coordinate
        g_coords = [] # temporary: parameter to save the [x y] coordinates
        g_y = []    # temporary: parameter to save the dependent field
        g_X= []     # temporary: parameter to save the explanatory fields
        
        
        # get the attributes and geometry info from input layer
        for ft in features:
            layer_attributes_attr.append(ft.attributes())
            layer_attributes_attr_geometry.append(ft.geometry())

        
        # ************************************************************************************#
        # *********************************start processing data******************************#
        # ************************************************************************************# 

        # Convert the layer_attributes_attr array to GeoDataFrame format        
        layer_attributes = pd.DataFrame(layer_attributes_attr)
        
        # rename the layer_attributes table fields
        for i in range(len(fieldnames)):
            layer_attributes.rename(columns={i: fieldnames[i]}, inplace=True)
        
        # Get the geometry type of the currently selected input layer
        # geometry type: 0 for point, 2 for polygon
        wkbtype = input_featuresource.wkbType()
        geomtype = QgsWkbTypes.geometryType(wkbtype)
        self.geomtype = geomtype

        # Judge the type of input layer
        # If it is point type, read it directly or calculate the x y coordination
        if  geomtype == QgsWkbTypes.PointGeometry:            
            if {'X', 'Y'}.issubset(layer_attributes.columns):
                feedback.pushInfo('X Y exist, directly read')
                feedback.pushInfo('processing...wait a second...')                
                location_x = layer_attributes['X']
                location_y = layer_attributes['Y']            
            else:
                feedback.pushInfo('X Y not exist, need to calculate')
                feedback.pushInfo('processing...wait a second...')
                features = input_featuresource.getFeatures()
                for ft in features:
                    location_x.append(ft.geometry().asPoint()[0])
                    location_y.append(ft.geometry().asPoint()[1]) 
                location_x = pd.Series(location_x)
                location_y = pd.Series(location_y)
        # If it is a Polygon, calculate the centroid
        elif geomtype == QgsWkbTypes.PolygonGeometry:
            features = input_featuresource.getFeatures()
            for ft in features:
                location_x.append(ft.geometry().centroid().asPoint()[0])
                location_y.append(ft.geometry().centroid().asPoint()[1])
               
        else:
            feedback.pushInfo('The input layer type is invalid. It should be Point or Polygon.')

        g_coords = list(zip(location_x, location_y))

        g_y = layer_attributes[parameters['dependent_field']].values.reshape((-1,1))        
        g_X = layer_attributes[parameters['explanatory_field']].values

        # Standardize variables
        g_X = (g_X - g_X.mean(axis=0)) / g_X.std(axis=0)
        g_y = g_y.reshape((-1,1))
        g_y = (g_y - g_y.mean(axis=0)) / g_y.std(axis=0)

        # ************************************************************************************#
        # *********************************Calibrate MGWR model********************************#
        # ************************************************************************************# 

        # Issue: Parallel computing fails in Windows system. 1 for single cell. -1 for unlimited cells
        var_core_number = {'Windows':1, 'Linux': -1, 'Darwin': -1} 

        # initialize some parameters of the mgwr model
        var_neighborhood_type = [False,True]
        var_kernel_function = ['bisquare', 'gaussian', 'exponential']
        var_bandwidth_searching_criterion = ['AICc', 'AIC','BIC','CV']
        
        # get the input parameters from users
        core_number = var_core_number[platform.system()]
        neighborhood_type = var_neighborhood_type[parameters['neighborhood_type']]
        kernel_function = var_kernel_function[parameters['kernel_function']]
        bandwidth_searching_criterion = var_bandwidth_searching_criterion[parameters['bandwidth_searching_criterion']]

        # get the min and max bandwidth value
        if parameters['multi_bw_min'] == '':
            multi_bw_min = [None]
        else:
            multi_bw_min = [float(i) for i in parameters['multi_bw_min'].split()]
        
        if parameters['multi_bw_max'] == '':
            multi_bw_max = [None]
        else:
            multi_bw_max = [float(i) for i in parameters['multi_bw_max'].split()]

        # bandwidth selection
        mgwr_selector = Sel_BW(g_coords, g_y, g_X, kernel=kernel_function, fixed=neighborhood_type, 
                                   multi=True, spherical=spherical, n_jobs=core_number)
            
        mgwr_bw = mgwr_selector.search(multi_bw_min=multi_bw_min, multi_bw_max=multi_bw_max,
                                        criterion=bandwidth_searching_criterion)

        # get mgwr result
        mgwr_results = MGWR(g_coords, g_y, g_X, mgwr_selector, kernel=kernel_function, 
                            fixed=neighborhood_type, spherical=spherical, n_jobs=core_number).fit()

        # test the variables getting from above
        #feedback.pushInfo('model_type: ' + str(model_type))
        feedback.pushInfo('fixed: ' + str(neighborhood_type))
        feedback.pushInfo('kernel_function: ' + str(kernel_function))
        feedback.pushInfo('bandwidth_searching_criterion: ' + str(bandwidth_searching_criterion))
        feedback.pushInfo('bandwidth is: ' + str(mgwr_bw))
        feedback.pushInfo('dependent field is: ' + str(parameters['dependent_field']))


        # ************************************************************************************#
        # *************************************output to txt file*****************************#
        # ************************************************************************************#

        # Prepare MGWR results for mapping
        layer_attributes['#intercept'] = mgwr_results.params[:,0]
        for i in range(len(parameters['explanatory_field'])):
            result_name_explanatory_field = '#'+ parameters['explanatory_field'][i]
            sink_result_name_explanatory_field.append(result_name_explanatory_field)
            feedback.pushInfo('explanatory field is: ' + str(parameters['explanatory_field'][i]))
            layer_attributes[result_name_explanatory_field] = mgwr_results.params[:,i+1]
        
        layer_attributes['geometry'] = layer_attributes_attr_geometry    

        # save summary as a txt file
        mgwr_results_summary_str = mgwr_results.summary(as_str=True)
        with open(txt, 'w') as f:
            f.write(mgwr_results_summary_str)
        f.close()
        #feedback.pushInfo(mgwr_results_summary_str)


        # ************************************************************************************#
        # *************************************output to qgis*********************************#
        # ************************************************************************************#

        # 1. Defined a variable, type is qgsfields 
        outFields = QgsFields()

        # 2. Defined the fields: the attributes fields of original layer
        for field in input_featuresource.fields():
            outFields.append(field)

        # 3.  Defined the fields: the attributes fields of mgwr results
        outFields.append(QgsField('#intercept', QVariant.Double))      
        for i in range(len(parameters['explanatory_field'])):
            outFields.append(QgsField('#'+ parameters['explanatory_field'][i], QVariant.Double))
        
        # 4. Create the output sink with the previously defined fields: outfields
        (sink, dest_id) = self.parameterAsSink(parameters, self.SINK_LAYER, context,
                                            outFields, input_featuresource.wkbType(), input_featuresource.sourceCrs())

        # 5. Save all values to sink layer
        total = 100.0 / input_featuresource.featureCount() if input_featuresource.featureCount() else 0
        features2 = input_featuresource.getFeatures()        
        for current, feature in enumerate(features2):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            
            #create a new feature object
            feat = QgsFeature()

            #set fields of the feature
            feat.setFields(outFields)

            # set the attribute values 
            # Add the original attributes to the corresponding field column
            for i in range(len(fieldnames)):
                feat[fieldnames[i]] = feature[fieldnames[i]]                

            # Add the result of mgwr to the corresponding field column
            feat['#intercept'] = float(layer_attributes['#intercept'][current])
            for i in range(len(sink_result_name_explanatory_field)):
                feat[sink_result_name_explanatory_field[i]] = float(layer_attributes[sink_result_name_explanatory_field[i]][current])

            # set the geometry info
            feat.setGeometry(feature.geometry())

            #add feature to sink
            sink.addFeature(feat, QgsFeatureSink.FastInsert) 
            
            # back to dest_id
            results[self.SINK_LAYER] = dest_id

            feedback.setProgress(int(current * total))

        # to get hold of the layer in post processing:
        self.dest_id=dest_id

        return results

    def postProcessAlgorithm(self, context, feedback):
        retval = super().postProcessAlgorithm(context, feedback)
        output = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)
        if  self.geomtype == QgsWkbTypes.PointGeometry: 
            style_path = os.path.join(os.path.dirname(__file__), 'layer_style', 'mgwr_point.qml')
        elif self.geomtype == QgsWkbTypes.PolygonGeometry:
            style_path = os.path.join(os.path.dirname(__file__), 'layer_style', 'mgwr_poly.qml')
        else:
            pass

        output.loadNamedStyle(style_path)
        output.triggerRepaint()

        return {self.SINK_LAYER: self.dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Multiscale Geographically Weighted Regression'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)
    
    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'mgwr.png')))
        return icon

    def createInstance(self):
        return MGWRAlgorithm()
    
    def shortHelpString(self):
        return '''<html><body><h2>Algorithm Description</h2>
<p>Multiscale Geographically Weighted Regression (MGWR) is a spatial regression technique used in various fields like geography, urban planning, and ecology. It's an extension of a method called Geographically Weighted Regression (GWR). It process by fitting a regression equation to every feature in the dataset. Generally, MGWR is a powerful tool for analyzing spatially varying relationships while accounting for processes happening at different geographical scales.</p>
<h3>Input Layer</h3>
<p>Select the input layer containing all the fields used to build local regression model.</p>
<h3>Input Parameters</h3>
<p>Select only one dependent variable and one or multiple explanatory variables (a.k.a. independent variables).</p>
<p style="padding-left: 40px;"><strong>Dependent Variable</strong>: The field containing values dependent on explanatory variable(s) in regression model</p>
<p style="padding-left: 40px;"><strong>Explanatory Variable</strong>: The field(s) representing independent explanatory variable(s) in regression model.</p>
<h3>Neighbourhood Type</h3>
<p>Select the kernel neighbourhood type, either adaptive or fixed, usually depending on how observations distribute in the region of interest.</p>
<p style="padding-left: 40px;"><strong>Adaptive </strong>(default): The spatial context is a specified number of nearest neighbors. Where feature distribution is dense, the spatial context is smaller, while where feature distribution is sparse, the spatial context is larger.</p>
<p style="padding-left: 40px;"><strong>Fixed</strong>: The spatial context used to solve each local regression analysis is a fixed distance. Fixed kernel type is suggested when observations distribute evenly in the research region.</p>
<h3>Spatial Kernel Function</h3>
<p>Select one kernel function to calculate weight matrix for each observation.</p>
<p style="padding-left: 40px;"><strong>Bisquare </strong>(default)</p>
<p style="padding-left: 40px;"><strong>Gaussian</strong></p>
<p style="padding-left: 40px;"><strong>Exponential</strong></p>
<p><em><strong>Note:</strong>A potential issue with the Gaussian and Exponential kernel functions is that all observations retain non-zero weight, regardless of their distance from the calibration location. This means that even faraway observations can remain influential for moderate-to-large bandwidth parameters.</em></p>
<p><em>To avoid the issue above, the default kernel function is set to bi-square kernel. Moreover, bi-square kernel has a much more intuitive interpretation, where the bandwidth parameter is the distance or number of nearest neighbors away in space so that the remaining observations will not affect in searching optimal bandwidth parameters.</em></p>
<h3>Bandwidth Searching Criterion</h3>
<p>Specify how the extent of the kernel should be determined.</p>
<p style="padding-left: 40px;"><strong>AICc </strong>(default): Corrected Akaike information criterion is an improved version of AIC, as well as the mostly suggested one, since it penalizes smaller bandwidths that result in more complex models that consume more degrees of freedom.</p>
<p style="padding-left: 40px;"><strong>AIC</strong>: Akaike information criterion estimates the quality of each model, relative to each of the other models.</p>
<p style="padding-left: 40px;"><strong>BIC</strong>: Bayesian information criterion shares the same methodology with AIC, but performs better at model selection.</p>
<p style="padding-left: 40px;"><strong>CV</strong>: Cross validation criterion compares the model residuals based on different bandwidths and chooses the optimal solution.</p>
<h3><strong>Minimum and Maximum Bandwidth</strong> (optional)</h3>
<p>Specify the minimum and maximum bandwidth searching range for MGWR calibration.</p>
<p style="padding-left: 40px;">Minimum or maximum bandwidth can be set as one uniform value for all variables.</p>
<p style="padding-left: 40px;">Minimum or maximum bandwidth can be set to different value for each variable specifically. The format is a list of space-separated values,where the number of values must match the total number of explanatory variables plus one (for the intercept).</p>
<p align="right">Plugin Authors: Huaying Gu & Yi Wu</p>
<p align="right">Algorithm Version: 1.1</p><body><html>'''
