from qgis.core import QgsSettings
from qgis.PyQt.QtCore import QCoreApplication, pyqtSlot
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QCheckBox, QMenu, QToolButton, QWidgetAction

from ..compat import IS_QT6, MenuButtonPopup
from .maptools import MeterMapTool
from .settings import SettingsDialog


class MeterToolButton(QToolButton):
    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.setPopupMode(MenuButtonPopup)
        self.cfg = QgsSettings()
        self.setupUI()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("MeterToolButton", message)

    def setupUI(self):
        mainMenu = QMenu(self)

        self.meterMapTool = MeterMapTool(self.iface)
        self.meterMapTool.deactivated.connect(self.on_meterMapTool_deactivated)
        self.measureAction = QAction(
            QIcon(":images/themes/default/mActionMeasureBearing.svg"),
            self.tr("measure distance azimuth"),
            self,
        )
        self.measureAction.setCheckable(True)
        self.measureAction.toggled.connect(self.on_meterAction_toggled)
        self.setDefaultAction(self.measureAction)
        self.meterMapTool.setAction(self.measureAction)

        style = "QCheckBox {margin: 6px;}"
        useMultiple = QCheckBox(self.tr("Use Multiple"))
        useMultiple.setStyleSheet(style)
        useMultiple.setChecked(self.cfg.value("Meter/useMultiple", 0, int))

        keepInCanvas = QCheckBox(self.tr("Keep in mapcanvas"))
        keepInCanvas.setStyleSheet(style)
        keepInCanvas.setChecked(self.cfg.value("Meter/keepInCanvas", 0, int))

        if IS_QT6:
            useMultiple.checkStateChanged.connect(self.on_useMultiple_changed)
            keepInCanvas.checkStateChanged.connect(self.on_keepInCanvas_changed)
        else:
            useMultiple.stateChanged.connect(self.on_useMultiple_changed)
            keepInCanvas.stateChanged.connect(self.on_keepInCanvas_changed)

        self.settingsDialog = SettingsDialog(self.iface)
        settingAction = QAction(
            QIcon(":images/themes/default/console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        settingAction.triggered.connect(self.on_settingsAction_triggered)

        clearAction = QAction(
            QIcon(":images/themes/default/console/iconClearConsole.svg"),
            self.tr("Clear Mesures"),
            mainMenu,
        )
        clearAction.triggered.connect(self.on_clearAction_triggered)

        copyAction = QAction(
            QIcon(":images/themes/default/mActionEditCopy.svg"),
            self.tr("Copy Mesures"),
            mainMenu,
        )
        copyAction.triggered.connect(self.on_copyAction_triggered)

        useMultipleAction = QWidgetAction(mainMenu)
        useMultipleAction.setDefaultWidget(useMultiple)
        keepInCanvasAction = QWidgetAction(mainMenu)
        keepInCanvasAction.setDefaultWidget(keepInCanvas)
        mainMenu.addSeparator()
        mainMenu.addAction(useMultipleAction)
        mainMenu.addSeparator()
        mainMenu.addAction(keepInCanvasAction)
        mainMenu.addSeparator()
        mainMenu.addAction(settingAction)
        mainMenu.addSeparator()
        mainMenu.addAction(clearAction)
        mainMenu.addSeparator()
        mainMenu.addAction(copyAction)
        self.setMenu(mainMenu)

    @pyqtSlot(bool)
    def on_meterAction_toggled(self, checked):
        if checked:
            self.canvas.setMapTool(self.meterMapTool)
        else:
            self.canvas.unsetMapTool(self.meterMapTool)
            self.settingsDialog.close()
            self.iface.actionPan().trigger()

    @pyqtSlot()
    def on_settingsAction_triggered(self):
        self.settingsDialog.show()

    def on_useMultiple_changed(self, state):
        self.cfg.setValue("Meter/useMultiple", state)

    def on_keepInCanvas_changed(self, state):
        self.cfg.setValue("Meter/keepInCanvas", state)

    @pyqtSlot()
    def on_clearAction_triggered(self):
        self.meterMapTool.clearMeasures()

    @pyqtSlot()
    def on_copyAction_triggered(self):
        self.meterMapTool.copyMeasures()

    def clear(self):
        self.meterMapTool.clearMeasures()
        self.settingsDialog.close()

    def on_meterMapTool_deactivated(self):
        # uncheck will trigger toggled signal
        if self.measureAction.isChecked():
            self.measureAction.setChecked(False)
