import math
from pathlib import Path

from qgis.core import (
    Qgis,
    QgsAnnotationMarkerItem,
    QgsAnnotationPointTextItem,
    QgsDistanceArea,
    QgsGeometry,
    QgsGeometryUtilsBase,
    QgsMarkerSymbol,
    QgsPoint,
    QgsPointXY,
    QgsProject,
    QgsSettings,
    QgsWkbTypes,
)
from qgis.gui import QgsMapToolEmitPoint, QgsRubberBand, QgsVertexMarker
from qgis.PyQt.QtCore import QCoreApplication, Qt, pyqtSignal
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import QApplication

from ..compat import QT_CROSSCURSOR, QT_LEFT_BUTTON, QT_RIGHT_BUTTON
from .utils import loadTextFormat


class MeterMapTool(QgsMapToolEmitPoint):
    deactivated = pyqtSignal()

    def __init__(self, iface):
        self.iface = iface
        self.canvas = iface.mapCanvas()
        QgsMapToolEmitPoint.__init__(self, self.canvas)
        self.canvasClicked.connect(self.onCanvasClicked)
        self.measures = []
        self.reset()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("MeterMapTool", message)

    def reset(self):
        self.point1 = None
        self.point2 = None
        self.isEmittingPoint = False
        keepInCanvas = QgsSettings().value("Meter/keepInCanvas", 0, int)
        if not keepInCanvas:
            self.clearMeasures()

    def clearMeasures(self):
        if not self.measures:
            return

        for meter in self.measures:
            meter.clear()
        self.measures.clear()

    def copyMeasures(self):
        if not self.measures:
            return

        results: list[str] = []
        for meter in self.measures:
            status = "\t".join(map(str, meter.status.values()))
            results.append(status)

        clipboard = QApplication.clipboard()
        clipboard.setText("\n".join(results))

        msg = self.tr("measurements copied to clipboard")
        pushMsg = f"{len(results)} {msg}"
        self.iface.messageBar().pushMessage(pushMsg, level=Qgis.Info, duration=2)

    def onCanvasClicked(self, point: QgsPointXY, button: Qt.MouseButton):
        """LeftButton: draw start or end marker
        RightButton: clear current measurement
        """
        useMultiple = QgsSettings().value("Meter/useMultiple", 0, int)
        if button == QT_LEFT_BUTTON:
            if self.isEmittingPoint:
                self.point2 = point
                meter: Meter = self.measures[-1]
                meter.drawEndMarker(self.point1, self.point2)
                self.isEmittingPoint = False
            else:
                self.point1 = point
                self.isEmittingPoint = True
                if not useMultiple:
                    self.clearMeasures()
                meter = Meter(self.canvas)
                meter.drawStartMarker(point)
                self.measures.append(meter)
            return

        if button == QT_RIGHT_BUTTON:
            self.isEmittingPoint = False
            if self.measures:
                meter: Meter = self.measures[-1]
                meter.clear()

    def canvasMoveEvent(self, e):
        """draw line distance and azimuth in map canvas"""
        if not self.isEmittingPoint:
            return

        self.point2 = self.toMapCoordinates(e.pos())
        meter: Meter = self.measures[-1]
        meter.drawLineLabel(self.point1, self.point2)

    def activate(self):
        self.reset()
        self.canvas.setCursor(QT_CROSSCURSOR)

    def deactivate(self):
        self.reset()
        self.deactivated.emit()


class Meter:
    def __init__(self, canvas):
        self.canvas = canvas
        self.geoCalculator = QgsDistanceArea()
        self.geoCalculator.setEllipsoid("WGS84")

        self.cfg = QgsSettings()
        self.unit = self.cfg.value("Meter/unit", "auto", str)
        self.useMultiple = self.cfg.value("Meter/useMultiple", 0, int)
        lineColor = QColor(self.cfg.value("Meter/lineColor", "#0000ff"))
        lineWidth = self.cfg.value("Meter/lineWidth", 2, int)
        formatFile = Path(__file__).parent.joinpath("label_format.xml")
        self.labelFormat = loadTextFormat(formatFile)

        self.rubberBand = QgsRubberBand(self.canvas, QgsWkbTypes.LineGeometry)
        self.rubberBand.setColor(lineColor)
        self.rubberBand.setWidth(lineWidth)

        markerSize = 4 if lineWidth < 4 else lineWidth
        self.startMarker = QgsVertexMarker(self.canvas)
        self.startMarker.setIconType(QgsVertexMarker.ICON_CIRCLE)
        self.startMarker.setIconSize(markerSize)
        self.startMarker.setPenWidth(markerSize)
        self.startMarker.setColor(lineColor)
        self.startMarker.setFillColor(lineColor)

        outlineWidth = 0.5 if lineWidth < 4 else 1
        self.arrowSymbol = QgsMarkerSymbol.createSimple(
            {
                "name": "arrowhead",
                "color": lineColor.name(),
                "size": lineWidth,
                "outline_color": lineColor.name(),
                "outline_width": outlineWidth,
            }
        )

        self.annotationLayer = QgsProject.instance().mainAnnotationLayer()
        self.labelId = None
        self.endMarkerId = None
        self.labelOffset = self.canvas.mapUnitsPerPixel() * (lineWidth + 1)

        self.status = {
            "point1": None,
            "point2": None,
            "distance": None,
            "azimuth": None,
        }

    def clear(self):
        self.canvas.scene().removeItem(self.startMarker)
        self.rubberBand.reset(QgsWkbTypes.LineGeometry)
        self.annotationLayer.removeItem(self.labelId)
        self.annotationLayer.removeItem(self.endMarkerId)
        self.status = None

    def drawStartMarker(self, point: QgsPointXY):
        self.startMarker.setCenter(point)
        self.status["point1"] = f"{point.x():.6f},{point.y():.6f}"

    def drawEndMarker(self, point1: QgsPointXY, point2: QgsPointXY):
        self.endMarker = QgsAnnotationMarkerItem(QgsPoint())
        self.endMarker.setSymbol(self.arrowSymbol)
        self.endMarker.setGeometry(QgsPoint(point2.x(), point2.y()))
        self.arrowSymbol.setAngle(self.azimuth - 90)
        self.endMarkerId = self.annotationLayer.addItem(self.endMarker)

        # maybe stop moving point is not the end point
        self.drawLineLabel(point1, point2)
        self.status["point2"] = f"{point2.x():.6f},{point2.y():.6f}"
        self.status["distance"] = round(self.length)
        self.status["azimuth"] = self.azimuth

    def drawLineLabel(self, point1: QgsPointXY, point2: QgsPointXY):
        x1, y1 = point1.x(), point1.y()
        x2, y2 = point2.x(), point2.y()
        if x1 == x2 and y1 == y2:
            self.length = 0
            self.azimuth = 0
            return

        # draw line
        line = QgsGeometry.fromPolyline([QgsPoint(x1, y1), QgsPoint(x2, y2)])
        self.rubberBand.setToGeometry(line, None)

        self.length = self.geoCalculator.measureLength(line)
        lineAngle = QgsGeometryUtilsBase.lineAngle(x1, y1, x2, y2)
        self.azimuth = round(math.degrees(lineAngle))

        match self.unit:
            case "auto":
                if self.length < 1000:
                    distance, unit = round(self.length), "m"
                else:
                    distance, unit = round(self.length / 1000, 3), "km"
            case "m":
                distance, unit = round(self.length), "m"
            case "km":
                distance, unit = round(self.length / 1000, 3), "km"
            case _:
                distance, unit = round(self.length), "m"

        if self.azimuth > 180:
            offset = self.labelOffset
            labelAngle = self.azimuth - 180 - 90
        else:
            offset = -self.labelOffset
            labelAngle = self.azimuth - 90

        x, y = QgsGeometryUtilsBase.perpendicularOffsetPointAlongSegment(
            x1, y1, x2, y2, 0.5, offset
        )

        # draw label
        self.annotationLayer.removeItem(self.labelId)
        label = QgsAnnotationPointTextItem(None, QgsPointXY())
        self.labelId = self.annotationLayer.addItem(label)
        label.setPoint(QgsPointXY(x, y))
        label.setText(f"{distance}{unit}, {self.azimuth}°")
        label.setAngle(labelAngle)
        label.setFormat(self.labelFormat)
