from pathlib import Path

from qgis.core import QgsSettings
from qgis.gui import QgsColorDialog, QgsTextFormatDialog
from qgis.PyQt.QtCore import QCoreApplication, pyqtSignal
from qgis.PyQt.QtGui import QColor, QIcon, QPalette
from qgis.PyQt.QtWidgets import (
    QComboBox,
    QDialog,
    QFormLayout,
    QFrame,
    QLabel,
    QPushButton,
    QSpinBox,
)

from ..compat import IS_QT6
from .utils import loadTextFormat, saveTextFormat


class SettingsDialog(QDialog):
    def __init__(self, iface):
        self.iface = iface
        parent = iface.mainWindow()
        super().__init__(parent)
        self.setWindowTitle(self.tr("Meter Settings"))
        self.cfg = QgsSettings()
        self.setupUI()

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("SettingsDialog", message)

    def setupUI(self):
        oldUnit = self.cfg.value("Meter/unit", "auto", str)
        self.distUnit = QComboBox()
        self.distUnit.addItem(self.tr("auto"), "auto")
        self.distUnit.addItem(self.tr("m"), "m")
        self.distUnit.addItem(self.tr("km"), "km")
        unitIndex = self.distUnit.findData(oldUnit)
        self.distUnit.setCurrentIndex(unitIndex)
        self.distUnit.currentIndexChanged.connect(self.setDistanceUnit)

        oldLineColor = QColor(self.cfg.value("Meter/lineColor", "#0000ff"))
        self.lineColor = ColorLabel(oldLineColor)
        self.lineColor.clicked.connect(self.setLineColor)

        oldLineWidth = self.cfg.value("Meter/lineWidth", 2, int)
        self.lineWidth = QSpinBox()
        self.lineWidth.setRange(1, 10)
        self.lineWidth.setSingleStep(1)
        self.lineWidth.setValue(oldLineWidth)
        self.lineWidth.valueChanged.connect(self.setLineWidth)

        icon = QIcon(":images/themes/default/propertyicons/symbology.svg")
        self.labelFormatBtn = QPushButton(icon, None, self)
        self.labelFormatBtn.clicked.connect(self.showTextFormatDialog)

        layout = QFormLayout()
        layout.addRow(self.tr("Unit"), self.distUnit)
        layout.addRow(self.tr("Line width"), self.lineWidth)
        layout.addRow(self.tr("Line color"), self.lineColor)
        layout.addRow(self.tr("Label format"), self.labelFormatBtn)
        self.setLayout(layout)

    def setDistanceUnit(self):
        unit = self.distUnit.currentData()
        QgsSettings().setValue("Meter/unit", unit)

    def setLineColor(self):
        oldColor = QColor(self.cfg.value("Meter/lineColor", "#0000ff"))
        newColor = QgsColorDialog.getColor(oldColor, self)

        if newColor.isValid():
            self.cfg.setValue("Meter/lineColor", newColor)
            self.lineColor.setPalette(QPalette(newColor))
            self.lineColor.setAutoFillBackground(True)

    def setLineWidth(self, width):
        self.cfg.setValue("Meter/lineWidth", width)

    def showTextFormatDialog(self):
        formatFile = Path(__file__).parent.joinpath("label_format.xml")
        oldFormat = loadTextFormat(formatFile)
        dialog = QgsTextFormatDialog(oldFormat, self.iface.mapCanvas())

        result = dialog.exec() if IS_QT6 else dialog.exec_()
        if result == 1:
            newFormat = dialog.format()
            saveTextFormat(newFormat, formatFile)

    def showEvent(self, event):
        position = self.cfg.value("Meter/position", None)
        if position:
            self.move(position)

    def closeEvent(self, event):
        self.cfg.setValue("Meter/position", self.pos())


class ColorLabel(QLabel):
    clicked = pyqtSignal()

    def __init__(self, color=QColor("#0000ff"), parent=None):
        super(ColorLabel, self).__init__(parent)

        if IS_QT6:
            frameStyle = QFrame.Shadow.Sunken | QFrame.Shape.Panel
        else:
            frameStyle = QFrame.Sunken | QFrame.Panel

        self.setFrameStyle(frameStyle)
        self.setPalette(QPalette(color))
        self.setAutoFillBackground(True)

    def mousePressEvent(self, event):
        self.clicked.emit()
