# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SourceManager
                                 A QGIS plugin
 Tools for management of Data on mesh (Telemac, Uhaina)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-24
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Artelia/BRGM/ISL
        email                : a@a
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import time

from contextlib import suppress

from processing.algs.gdal.GdalUtils import GdalUtils
from qgis.core import (
    NULL,
    QgsCoordinateTransform,
    QgsCoordinateTransformContext,
    QgsField,
    QgsFields,
    QgsGeometry,
    QgsMapLayerProxyModel,
    QgsMapLayerType,
    QgsMesh,
    QgsPointXY,
    QgsVectorFileWriter,
    QgsVectorLayer,
)
from qgis.gui import QgsVertexMarker
from qgis.PyQt import uic
from qgis.PyQt.QtCore import QVariant, pyqtSignal
from qgis.PyQt.QtGui import QColor, QStandardItem, QStandardItemModel
from qgis.PyQt.QtWidgets import QFileDialog

from ..mesh_tools_dockwidget import MeshToolsDockWidget
from .create_shp_dlg import dlg_create_shapefile
from .dialog_for_CAS import DialogForCAS
from .MeshUtils import MeshUtils

FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), "..", "ui", "source_manager.ui"))


class SourceManager(MeshToolsDockWidget, FORM_CLASS):
    closingTool = pyqtSignal()

    def __init__(self, parent=None):
        super(SourceManager, self).__init__(parent)
        self.setupUi(self)
        self.prt = parent
        self.path_icon = os.path.join(os.path.dirname(__file__), "..", "icons/")
        self.file_src_style = None
        self.ctrl_signal_blocked = False

        self.lay_mesh = None
        self.lay_mesh_xform = None
        self.lay_src = None
        self.lay_src_xform = None

        self.native_mesh = None
        self.vertices = None
        self.hl_vertices = []

        self.src_type = None

        self.src_flds = [
            ("srcId", QVariant.Int, None),
            (self.tr("Name"), QVariant.String, self.txt_name),
            (self.tr("Note"), QVariant.String, self.txt_note),
        ]

        self.is_opening = True

        self.mdl_lay_src = QStandardItemModel()

        self.cb_lay_src.setModel(self.mdl_lay_src)

        self.project.layersAdded.connect(self.addLayers)
        self.project.layersRemoved.connect(self.removeLayers)

        self.cb_lay_mesh.setFilters(QgsMapLayerProxyModel.MeshLayer)
        self.cb_lay_mesh.layerChanged.connect(self.mesh_lay_changed)
        self.cb_type_src.currentIndexChanged.connect(self.src_type_changed)
        self.cb_lay_src.currentIndexChanged.connect(self.src_lay_changed)

        self.btn_new_src_file.clicked.connect(self.new_file)
        self.btn_hl_vertices.clicked.connect(self.highlight_vertices)

        for fld in self.src_flds:
            ctrl = fld[2]
            if ctrl is not None:
                ctrl.textChanged.connect(self.ctrl_edited)

        self.src_type_changed()
        self.mesh_lay_changed()

        if self.mdl_lay_src.rowCount() == 0:
            self.src_lay_changed()

        self.is_opening = False

    def addLayers(self, layers):
        for lay in layers:
            self.analyse_layer(lay)

    def removeLayers(self, layers):
        for r in range(self.mdl_lay_src.rowCount() - 1, -1, -1):
            if self.mdl_lay_src.item(r, 0).data(32) in layers:
                self.mdl_lay_src.takeRow(r)

    def analyse_project_layers(self):
        for lay in self.project.mapLayers().values():
            self.analyse_layer(lay)

    def analyse_layer(self, lay):
        if not lay.type() == QgsMapLayerType.VectorLayer:
            return

        if lay.geometryType() != self.src_type:
            return

        flds = [f.name() for f in lay.fields()]
        if all(elem in flds for elem in [fc[0] for fc in self.src_flds]):
            itm = QStandardItem()
            itm.setData(lay.name(), 0)
            itm.setData(lay.id(), 32)
            self.mdl_lay_src.appendRow(itm)
            self.mdl_lay_src.sort(0)

    def valid_mesh_src(self):
        if self.lay_mesh_xform is not None and self.lay_src_xform is not None:
            self.btn_hl_vertices.setEnabled(True)
            self.btn_export.setEnabled(True)
            self.gb_cur_src.setEnabled(True)
        else:
            self.btn_hl_vertices.setEnabled(False)
            self.btn_export.setEnabled(False)
            self.gb_cur_src.setEnabled(False)
        self.cur_src_changed()

    def clean(self):
        self.project.layersAdded.disconnect(self.addLayers)
        self.project.layersRemoved.disconnect(self.removeLayers)

        self.lay_mesh.crsChanged.disconnect(self.cur_mesh_changed)
        self.lay_src.crsChanged.disconnect(self.src_lay_changed)

        self.cb_lay_mesh.layerChanged.disconnect(self.mesh_lay_changed)
        self.cb_lay_src.currentIndexChanged.disconnect(self.src_lay_changed)

        self.clear_hl_vertices()

    ######################################################################################
    #                                                                                    #
    #                                     MESH LAYER                                     #
    #                                                                                    #
    ######################################################################################

    def mesh_lay_changed(self):
        with suppress(AttributeError, RuntimeError, TypeError):
            self.lay_mesh.crsChanged.disconnect()

        self.lay_mesh = self.cb_lay_mesh.currentLayer()

        if not self.lay_mesh:
            self.lay_mesh = None
            self.native_mesh = None
            self.vertices = None
        self.cur_mesh_changed()

    def cur_mesh_changed(self):
        if self.lay_mesh is None:
            return

        self.writeInfo(self.tr("Current mesh changed : {}").format(self.lay_mesh.name()))
        self.native_mesh = QgsMesh()
        self.lay_mesh.dataProvider().populateMesh(self.native_mesh)

        self.lay_mesh.crsChanged.connect(self.cur_mesh_changed)
        if self.lay_mesh.crs().isValid():
            self.lay_mesh_xform = QgsCoordinateTransform(
                self.lay_mesh.crs(), self.canvas.mapSettings().destinationCrs(), self.project
            )
        else:
            self.lay_mesh_xform = None
            self.valid_mesh_src()
            self.writeError(self.tr("Mesh CRS is not valid."))
            return

        self.valid_mesh_src()

        self.writeInfo(self.tr("Creation of vertices spatial index..."))
        t0 = time.time()
        self.vertices = MeshUtils.createVerticesSpatialIndex(self.native_mesh, self.lay_mesh_xform)
        self.writeInfo(self.tr("Vertices spatial index created in {} sec.").format(round(time.time() - t0, 1)))

    ######################################################################################
    #                                                                                    #
    #                                    SOURCE LAYER                                    #
    #                                                                                    #
    ######################################################################################

    def new_file(self):
        srs_mesh = None
        path = ""
        if self.lay_mesh:
            srs_mesh = self.lay_mesh.crs()
        dlg = dlg_create_shapefile(self.tr("source"), srs_mesh, self)
        dlg.setWindowModality(2)
        if dlg.exec_():
            path, crs = dlg.cur_shp, dlg.cur_crs
        if path:
            self.create_new_shp(path, crs)

    def create_new_shp(self, path, crs):
        layerFields = QgsFields()
        for fld in self.src_flds:
            layerFields.append(QgsField(fld[0], fld[1]))

        if self.src_type == 0:  # Individual source (point)
            tmp_lay = QgsVectorLayer(f"Point?crs={crs.authid()}", "", "memory")
        elif self.src_type == 2:  # Region (area)
            tmp_lay = QgsVectorLayer(f"Polygon?crs={crs.authid()}", "", "memory")
        pr = tmp_lay.dataProvider()
        pr.addAttributes(layerFields)
        tmp_lay.updateFields()

        layerDriver = GdalUtils.getVectorDriverFromFileName(path)

        options = QgsVectorFileWriter.SaveVectorOptions()
        options.driverName = layerDriver
        options.fileEncoding = "utf-8"
        QgsVectorFileWriter.writeAsVectorFormatV2(
            layer=tmp_lay,
            fileName=path,
            transformContext=QgsCoordinateTransformContext(),
            options=options,
        )

        shp_lay = QgsVectorLayer(path, os.path.basename(path).rsplit(".", 1)[0], "ogr")
        shp_lay.setCrs(crs)
        shp_lay.loadNamedStyle(self.file_src_style)
        shp_lay.saveDefaultStyle()
        self.project.addMapLayer(shp_lay)
        self.cb_lay_src.setCurrentIndex(self.cb_lay_src.findData(shp_lay.id(), 32))

    def src_type_changed(self):
        if self.cb_type_src.currentIndex() == 0:
            src_type = 0
        elif self.cb_type_src.currentIndex() == 1:
            src_type = 2

        if self.src_type == src_type:
            return

        self.src_type = src_type

        with suppress(AttributeError, RuntimeError, TypeError):
            self.btn_export.clicked.disconnect()

        if self.src_type == 0:  # Individual source (point)
            self.btn_hl_vertices.hide()
            self.file_src_style = os.path.join(os.path.dirname(__file__), "..", "styles/source_point.qml")
            self.btn_export.clicked.connect(self.showTxtForCAS)
        elif self.src_type == 2:  # Region (area)
            self.btn_hl_vertices.show()
            self.file_src_style = os.path.join(os.path.dirname(__file__), "..", "styles/source_polygon.qml")
            self.btn_export.clicked.connect(self.export)

        self.mdl_lay_src.clear()
        self.analyse_project_layers()
        self.src_lay_changed()

    def src_lay_changed(self):
        with suppress(AttributeError, RuntimeError, TypeError):
            self.lay_src.crsChanged.disconnect()

        lay_id = self.cb_lay_src.currentData(32)
        self.cur_src_id = None
        if lay_id:
            self.lay_src = self.project.mapLayer(lay_id)
            self.lay_src.selectionChanged.connect(self.cur_src_changed)
            self.lay_src.editingStarted.connect(self.cur_src_changed)
            self.lay_src.editingStopped.connect(self.cur_src_changed)
            self.lay_src.crsChanged.connect(self.src_lay_changed)

            if self.lay_src.crs().isValid():
                self.lay_src_xform = QgsCoordinateTransform(
                    self.lay_src.crs(), self.canvas.mapSettings().destinationCrs(), self.project
                )
            else:
                self.lay_src_xform = None
                self.writeWarning(self.tr("Source layer CRS is not valid."))
        else:
            self.lay_src = None
            self.lay_src_xform = None
        self.valid_mesh_src()

    def cur_src_changed(self):
        if self.lay_src:
            if self.lay_src.selectedFeatureCount() == 1:
                self.cur_src_id = self.lay_src.selectedFeatureIds()[0]
            else:
                self.cur_src_id = None
        else:
            self.cur_src_id = None
        self.display_src_info()

    def display_src_info(self):
        if self.lay_src:
            if self.cur_src_id is None:
                if self.lay_src.selectedFeatureCount() == 0:
                    self.gb_cur_src.setTitle(self.tr("No source selected"))
                else:
                    self.gb_cur_src.setTitle(self.tr("More than one source selected"))
                self.clear_info()
                self.gb_cur_src.setEnabled(False)
            else:
                self.gb_cur_src.setTitle(self.tr("Selected source informations"))
                self.fill_info()
                self.gb_cur_src.setEnabled(not self.lay_src.isEditable())
        else:
            self.gb_cur_src.setTitle(self.tr("No source layer selected"))
            self.clear_info()
            self.gb_cur_src.setEnabled(False)

    def clear_info(self):
        self.ctrl_signal_blocked = True
        for fld in self.src_flds:
            ctrl = fld[2]
            if ctrl is not None:
                ctrl.setText("")
        self.ctrl_signal_blocked = False

    def fill_info(self):
        self.ctrl_signal_blocked = True
        ft = self.lay_src.getFeature(self.cur_src_id)
        for fld in self.src_flds:
            if fld[2] is not None:
                fld[2].setText(str(ft[fld[0]]))
        self.ctrl_signal_blocked = False

    def ctrl_edited(self):
        if not self.ctrl_signal_blocked:
            if self.cur_src_id is not None:
                ctrl = self.sender()
                val = ctrl.text()

                field_idx = None
                for idx in range(len(self.src_flds)):
                    if ctrl == self.src_flds[idx][2]:
                        field_name = self.src_flds[idx][0]
                        field_idx = self.lay_src.fields().indexFromName(field_name)
                        break

                if field_idx is not None:
                    attrs = {field_idx: val}
                    self.lay_src.dataProvider().changeAttributeValues({self.cur_src_id: attrs})
                    self.lay_src.commitChanges()

    def clear_hl_vertices(self):
        for marker in self.hl_vertices:
            self.canvas.scene().removeItem(marker)
        self.hl_vertices = []

    def highlight_vertices(self):
        if self.src_type == 0:  # Should not happen
            return

        if self.hl_vertices:
            self.clear_hl_vertices()
            return

        for feat in self.lay_src.getFeatures():
            polygon = feat.geometry().asQPolygonF()
            self.lay_src_xform.transformPolygon(polygon)
            vertices = MeshUtils.findVerticesInGeometry(
                self.native_mesh,
                self.vertices,
                self.lay_mesh_xform,
                QgsGeometry.fromQPolygonF(polygon),
            )
            if vertices:
                for vertex in vertices:
                    marker = QgsVertexMarker(self.canvas)
                    marker.setCenter(QgsPointXY(vertex))
                    marker.setColor(QColor(255, 255, 255))
                    marker.setFillColor(QColor(72, 123, 182))
                    marker.setPenWidth(2)
                    marker.setIconType(QgsVertexMarker.ICON_CIRCLE)
                    marker.setIconSize(10)
                    marker.show()
                    self.hl_vertices.append(marker)

    ######################################################################################
    #                                                                                    #
    #                                       EXPORT                                       #
    #                                                                                    #
    ######################################################################################

    def showTxtForCAS(self):
        dlg = DialogForCAS()
        dlg.setText(self.tr("Copy the text below in your CAS file."))

        if self.lay_src.featureCount() > 20:
            dlg.appendCodeText(self.tr("MAXIMUM NUMBER OF SOURCES = {}\n").format(self.lay_src.featureCount()))

        xs = []
        ys = []
        for feat in self.lay_src.getFeatures():
            point = feat.geometry().asPoint()
            point = self.lay_src_xform.transform(point)
            point = self.lay_mesh_xform.transform(point, QgsCoordinateTransform.ReverseTransform)
            xs.append(str(round(point.x(), 4)))
            ys.append(str(round(point.y(), 4)))

        txt = "{} = {}\n".format(self.tr("ABSCISSAE OF SOURCES"), ";".join(xs))
        dlg.appendCodeText(txt)
        txt = "{} = {}".format(self.tr("ORDINATES OF SOURCES"), ";".join(ys))
        dlg.appendCodeText(txt)

        dlg.exec()

    def export(self):
        if not self.lay_src:
            return

        src_file_name, _ = QFileDialog.getSaveFileName(
            self, self.tr("Region source file"), "", self.tr("Text File (*.txt)")
        )

        if not src_file_name:
            return

        dlg = DialogForCAS()
        dlg.setText(self.tr("Copy the text below in your CAS file."))
        dlg.appendCodeText(self.tr("SOURCE REGIONS DATA FILE = '{}'\n").format(os.path.basename(src_file_name)))

        nameField = self.src_flds[1][0]
        maxLenGeom = -1
        with open(src_file_name, "w", encoding="utf-8") as src_file:
            for feat in self.lay_src.getFeatures():
                featId = feat.id() + 1
                src_file.write("#\n")
                if feat[nameField] != NULL:
                    scr_name = feat[nameField]
                else:
                    scr_name = self.tr("SOURCE REGION {}").format(featId)
                src_file.write("{} {}\n".format(self.tr("# COORDINATES AT"), scr_name))
                src_file.write("#\n")
                src_file.write(f"X({featId})\tY({featId})\n")

                geom = feat.geometry().asMultiPolygon()[0][0]
                maxLenGeom = max(len(geom), maxLenGeom)

                for point in geom:
                    point = self.lay_src_xform.transform(point)
                    point = self.lay_mesh_xform.transform(point, QgsCoordinateTransform.ReverseTransform)
                    src_file.write(f"{round(point.x(), 4)}\t{round(point.y(), 4)}\n")
            src_file.write("#")

        if self.lay_src.featureCount() > 20:
            dlg.appendCodeText(self.tr("MAXIMUM NUMBER OF SOURCES = {}\n").format(self.lay_src.featureCount()))
        if maxLenGeom > 10:
            dlg.appendCodeText(self.tr("MAXIMUM NUMBER OF POINTS FOR SOURCES REGIONS = {}\n").format(maxLenGeom))
        dlg.exec()
