# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MeshQuality
                                 A QGIS plugin
 Tools for management of Data on mesh (Telemac, Uhaina)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-24
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Artelia/BRGM/ISL
        email                : a@a
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import math
import os

from qgis.core import QgsCoordinateTransform, QgsMapLayerProxyModel, QgsMesh, QgsPointXY
from qgis.gui import QgsVertexMarker
from qgis.PyQt import uic
from qgis.PyQt.QtGui import QColor

from ..mesh_tools_dockwidget import MeshToolsDockWidget
from .MeshUtils import MeshUtils, showWaitCursor

FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), "..", "ui", "mesh_quality.ui"))


class MeshQuality(MeshToolsDockWidget, FORM_CLASS):
    def __init__(self, parent=None):
        super(MeshQuality, self).__init__(parent)
        self.setupUi(self)
        self.prt = parent
        self.path_icon = os.path.join(os.path.dirname(__file__), "..", "icons/")

        self.lay_mesh = self.mMapLayerComboBox.currentLayer()
        self.native_mesh = None
        self.native_mesh_faces_count = None
        self.xform = None

        self.markers = []

        self.mMapLayerComboBox.setFilters(QgsMapLayerProxyModel.MeshLayer)
        self.mMapLayerComboBox.layerChanged.connect(self.mesh_lay_changed)

        self.mesh_lay_changed()

        self.mBtnAnalyse.clicked.connect(self.analyse_mesh)
        self.mBtnReset.clicked.connect(self.resetVertexMarker)

        self.mDsbAngle1.setClearValue(self.mDsbAngle1.value())
        self.mDsbAngle2.setClearValue(self.mDsbAngle2.value())
        self.mDsbAngle3.setClearValue(self.mDsbAngle3.value())
        self.mDsbSize.setClearValue(self.mDsbSize.value())
        self.mDsbArea.setClearValue(self.mDsbArea.value())
        self.mDsbNeighbors.setClearValue(self.mDsbNeighbors.value())

    def clean(self):
        self.resetVertexMarker()

    ######################################################################################
    #                                                                                    #
    #                                     MESH LAYER                                     #
    #                                                                                    #
    ######################################################################################

    def mesh_lay_changed(self):
        self.lay_mesh = self.mMapLayerComboBox.currentLayer()

        if self.lay_mesh:
            self.writeInfo(self.tr("Current mesh changed : {}").format(self.lay_mesh.name()))
            self.xform = QgsCoordinateTransform(
                self.lay_mesh.crs(),
                self.canvas.mapSettings().destinationCrs(),
                self.project,
            )
        else:
            self.lay_mesh = None
            self.native_mesh = None
            self.xform = None
        self.cur_mesh_changed()

    def cur_mesh_changed(self):
        if self.lay_mesh:
            self.native_mesh = QgsMesh()
            self.lay_mesh.dataProvider().populateMesh(self.native_mesh)

    @showWaitCursor
    def analyse_mesh(self, clicked):  # Add dummy arg to respect Qt signature
        if not self.lay_mesh:
            self.writeError(self.tr("No mesh selected"))

        wasInEditMode = False
        if self.lay_mesh.isEditable():
            wasInEditMode = True
            self.lay_mesh.commitFrameEditing(self.xform, False)
            self.cur_mesh_changed()

        self.resetVertexMarker()

        self.native_mesh_faces_count = self.native_mesh.faceCount()

        for index in range(self.native_mesh_faces_count):
            triangle = MeshUtils.faceToTriangle(self.native_mesh, index)
            centroid = self.xform.transform(QgsPointXY(triangle.centroid()))

            if self.mChkEquilateralness.isChecked():
                if checkAngles(triangle.angles(), self.mDsbAngle1.value()):
                    self.addVertexMarker(centroid, "bad_angle_1")
                elif checkAngles(triangle.angles(), self.mDsbAngle2.value()):
                    self.addVertexMarker(centroid, "bad_angle_2")
                elif checkAngles(triangle.angles(), self.mDsbAngle3.value()):
                    self.addVertexMarker(centroid, "bad_angle_3")

            if self.mChkSize.isChecked():
                if (
                    self.mCbSize.currentIndex() == 0
                    and any(length < self.mDsbSize.value() for length in triangle.lengths())
                ) or (self.mCbSize.currentIndex() == 1 and triangle.inscribedRadius() * 2 < self.mDsbSize.value()):
                    self.addVertexMarker(centroid, "bad_size")
                if triangle.area() < self.mDsbArea.value():
                    self.addVertexMarker(centroid, "bad_area")

        if self.mChkNeighbors.isChecked():
            verticesNeighbors = MeshUtils.computeNeighbors(self.native_mesh)
            for vertexId, neighbors in verticesNeighbors.items():
                if len(neighbors) > self.mDsbNeighbors.value():
                    point = self.xform.transform(QgsPointXY(self.native_mesh.vertex(vertexId)))
                    self.addVertexMarker(point, "too_much_neighbors")

        if self.markers:
            self.mBtnReset.setEnabled(True)
            self.showVertexMarker()

        if wasInEditMode:
            self.lay_mesh.startFrameEditing(self.xform)

    def addVertexMarker(self, point, check):
        marker = QgsVertexMarker(self.canvas)
        marker.setCenter(QgsPointXY(point))
        # Set default marker config
        marker.setPenWidth(2)
        marker.setIconSize(10)
        marker.setColor(QColor(255, 0, 0))

        if check == "bad_angle_1":
            marker.setIconType(QgsVertexMarker.ICON_X)
        elif check == "bad_angle_2":
            marker.setColor(QColor(255, 255, 0))
            marker.setIconType(QgsVertexMarker.ICON_X)
        elif check == "bad_angle_3":
            marker.setColor(QColor(0, 0, 255))
            marker.setIconType(QgsVertexMarker.ICON_X)
        elif check == "bad_size":
            marker.setIconType(QgsVertexMarker.ICON_BOX)
        elif check == "bad_area":
            marker.setIconType(QgsVertexMarker.ICON_TRIANGLE)
        elif type == "too_much_neighbors":
            marker.setColor(QColor(255, 0, 255))
            marker.setIconType(QgsVertexMarker.ICON_RHOMBUS)

        marker.hide()

        self.markers.append(marker)

    def showVertexMarker(self, markerId=None):
        if markerId:
            self.markers[markerId].show()
        else:
            for marker in self.markers:
                marker.show()

    def hideVertexMarker(self, markerId=None):
        if markerId:
            self.markers[markerId].hide()
        else:
            for marker in self.markers:
                marker.hide()

    def resetVertexMarker(self):
        if self.markers:
            for marker in self.markers:
                marker.hide()
                self.canvas.scene().removeItem(marker)
        self.markers = []
        self.mBtnReset.setEnabled(False)


def checkAngles(angles, checkValue):
    angles = [math.degrees(a) for a in angles]

    # We need to trick because of a strange behavior of QgsTriangle.angles()
    # Sum all angles and check if it's lower than 179 (case of rounding issue...)
    # https://lists.osgeo.org/pipermail/qgis-developer/2021-December/064349.html
    if sum(angles) < 179:
        angles[angles.index(min(angles))] += 90

    return any(a < checkValue for a in angles)
