# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MeshUtils
                                 A QGIS plugin
 Tools for management of Data on mesh (Telemac, Uhaina)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-24
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Artelia/BRGM/ISL
        email                : a@a
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from typing import Optional

from qgis.core import (
    QgsCoordinateTransform,
    QgsFeature,
    QgsGeometry,
    QgsLineString,
    QgsMesh,
    QgsMeshDatasetIndex,
    QgsMeshLayer,
    QgsPointXY,
    QgsSpatialIndex,
    QgsTriangle,
)
from qgis.PyQt.QtCore import QCoreApplication


class MeshUtils:
    @staticmethod
    def pointInMesh(mesh: QgsMeshLayer, point: QgsPointXY) -> bool:
        return not mesh.snapOnElement(QgsMesh.Face, point, 0).isEmpty()

    @staticmethod
    def findNearestVertex(
        mesh: QgsMeshLayer, vertexSpatialIndex: QgsSpatialIndex, point: QgsPointXY, xform: QgsCoordinateTransform
    ):
        error = None
        vertexIndex = None

        x_point = xform.transform(point)

        if not MeshUtils.pointInMesh(mesh, point):
            error = MeshUtils.tr("The point is not within the mesh.")
            return vertexIndex, error

        vertexIndex = vertexSpatialIndex.nearestNeighbor(x_point, 1)[0]

        return vertexIndex + 1, error

    @staticmethod
    def n1n2FromFeature(
        mesh: QgsMeshLayer, vertexSpatialIndex: QgsSpatialIndex, feat: QgsFeature, xform: QgsCoordinateTransform
    ):
        error = None
        n = [None, None]

        points = feat.geometry().asMultiPolyline()
        for p in (0, -1):
            point = points[p][p]

            n[p], err = MeshUtils.findNearestVertex(mesh, vertexSpatialIndex, point, xform)

            if err is not None:
                error = err

        return n, error

    @staticmethod
    def zFromN(mesh: QgsMeshLayer, meshDatasetIndex: QgsMeshDatasetIndex, n: int):
        dset_val = mesh.dataProvider().datasetValues(meshDatasetIndex, n, 1)

        return round(dset_val.value(0).scalar(), 2)

    @staticmethod
    def xyFromN(nativeMesh: QgsMesh, n: int, mesh_xorm: QgsCoordinateTransform, culv_xform: QgsCoordinateTransform):
        error = None
        point = None

        if n < 1:
            error = MeshUtils.tr("N should a strict positive integer.")
            return point, error

        point = QgsPointXY(nativeMesh.vertex(n - 1))

        # It's safer to transform mesh point to canvas CRS then reverse transform
        # to culvert layer CRS.
        if mesh_xorm is not None and culv_xform is not None:
            point = mesh_xorm.transform(point)
            point = culv_xform.transform(point, QgsCoordinateTransform.ReverseTransform)

        return point, error

    @staticmethod
    def createVerticesSpatialIndex(nativeMesh: QgsMesh, xform: Optional[QgsCoordinateTransform] = None):
        spindex = QgsSpatialIndex()

        count = nativeMesh.vertexCount()
        offset = 0
        batch_size = 10
        while offset < count:
            lst_ft = list()
            iterations = min(batch_size, count - offset)
            for i in range(iterations):
                ft = QgsFeature()
                vertex = QgsPointXY(nativeMesh.vertex(offset + i))
                if xform is not None:
                    vertex = xform.transform(vertex)
                ft.setGeometry(QgsGeometry.fromPointXY(vertex))
                ft.setId(offset + i)
                lst_ft.append(ft)
            spindex.addFeatures(lst_ft)
            offset += iterations

        return spindex

    @staticmethod
    def faceToTriangle(nativeMesh: QgsMesh, faceIdx: int):
        face = nativeMesh.face(faceIdx)
        points = [nativeMesh.vertex(v) for v in face]
        triangle = QgsTriangle()
        triangle.setExteriorRing(QgsLineString(points))
        return triangle

    @classmethod
    def tr(cls, message, context=""):
        if context == "":
            context = cls.__name__
        return QCoreApplication.translate(context, message)
