# -*- coding: utf-8 -*-

"""
/***************************************************************************
 medSens
                                 A QGIS plugin
 MedSens Plugin
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-11-19
        copyright            : (C) 2018 by Matteo Ghetta
        email                : matteo.ghetta@faunalia.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Matteo Ghetta'
__date__ = '2018-11-19'
__copyright__ = '(C) 2018 by Matteo Ghetta'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication, QDate, Qt, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsField,
                       QgsFeatureRequest,
                       QgsProcessingAlgorithm,
                       QgsProcessingException,
                       NULL,
                       QgsDistanceArea,
                       QgsUnitTypes,
                       QgsFeature,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterString,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingLayerPostProcessorInterface,
                       QgsVectorLayer
                       )

from processing.gui.wrappers import WidgetWrapper
from qgis.PyQt.QtWidgets import QDateEdit
from datetime import datetime
from collections import defaultdict
import os
import processing

pluginPath = os.path.dirname(__file__)


class MyLayerPostProcessor(QgsProcessingLayerPostProcessorInterface):

    instance = None

    def postProcessLayer(self, layer, context, feedback):
        if not isinstance(layer, QgsVectorLayer):
            return

        style_path = os.path.join(
            os.path.dirname(__file__),
            'styles',
            'MedSensTot_color_legend.qml')

        layer.loadNamedStyle(style_path)

    # Hack to work around sip bug!
    @staticmethod
    def create() -> 'MyLayerPostProcessor':

        MyLayerPostProcessor.instance = MyLayerPostProcessor()
        return MyLayerPostProcessor.instance


class DateTimeWidget(WidgetWrapper):
    """
    QDateEdit widget wrapper with calendar pop up
    """

    def createWidget(self):
        """
        Create the Wrapper as QDateEdit and set a default date with calendar view
        """
        self.combo = QDateEdit()
        self.combo.setCalendarPopup(True)
        self.combo.setDisplayFormat('yyyy-MM-dd')

        # get the current date and set the default day to the combo
        today = QDate(2000, 1, 1)
        self.combo.setDate(today)

        return self.combo

    def value(self):
        """
        Grab the date and return it as ISOTime (e.g. 2018-11-30T12:00:00)
        """
        date_chosen = self.combo.date()

        return date_chosen.toString('yyyy-MM-dd')


class DateTimeWidget2(WidgetWrapper):
    """
    QDateEdit widget wrapper with calendar pop up
    """

    def createWidget(self):
        """
        Create the Wrapper as QDateEdit and set a default date with calendar view
        """
        self.combo2 = QDateEdit()
        self.combo2.setCalendarPopup(True)
        self.combo2.setDisplayFormat('yyyy-MM-dd')

        # get the current date and set the default day to the combo
        today = QDate.currentDate()
        self.combo2.setDate(today)

        return self.combo2

    def value(self):
        """
        Grab the date and return it as ISOTime (e.g. 2018-11-30T12:00:00)
        """
        date_chosen2 = self.combo2.date()

        return date_chosen2.toString('yyyy-MM-dd')


class medSensAlgorithm(QgsProcessingAlgorithm):

    # Define class (algorithm variables)

    INPOINT = 'INPOINT'
    INIDATE = 'INIDATE'
    ENDDATE = 'ENDDATE'
    POLYGON = 'POLYGON'
    OUTPOLYGON = 'OUTPOLYGON'

    def name(self):
        return 'Get MedSens index'

    def displayName(self):
        return self.tr(self.name())

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'MedSens'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return medSensAlgorithm()

    def tags(self):
        return self.tr("medsens, index, biology, marine").split(",")

    def icon(self):
        return QIcon(os.path.join(pluginPath, "icon.png"))

    def shortHelpString(self):
        return self.tr(
            '''
        Calculates <b>MedSens</b> indices in selected areas and chosen time period.
        '''
        )

    def helpUrl(self):
        return "https://www.reefcheckmed.org/english/underwater-monitoring-protocol/medsens-index/"

    def initAlgorithm(self, config):

        # Input point layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPOINT,
                self.tr('MedSens data'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # Input polygon layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.POLYGON,
                self.tr('Polygon layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # init date
        param_ini = QgsProcessingParameterString(
            self.INIDATE,
            self.tr('Initial Date')
        )
        param_ini.setMetadata({
            'widget_wrapper': {
                'class': DateTimeWidget
            }
        })

        # add the date to tue UI
        self.addParameter(param_ini)

        # end date
        param_end = QgsProcessingParameterString(
            self.ENDDATE,
            self.tr('Final Date')
        )
        param_end.setMetadata({
            'widget_wrapper': {
                'class': DateTimeWidget2
            }
        })

        # add the date to tue UI
        self.addParameter(param_end)

        # add the final sink
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPOLYGON,
                self.tr('MedSens classification')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        source = self.parameterAsSource(
            parameters,
            self.INPOINT,
            context
        )

        if source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPOINT))

        sampling_areas = self.parameterAsSource(
            parameters,
            self.POLYGON,
            context
        )

        if sampling_areas is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPOINT))

        # get QgsArea and Distance Objects
        da = QgsDistanceArea()
        da.setSourceCrs(sampling_areas.sourceCrs(), context.transformContext())
        da.setEllipsoid(context.project().ellipsoid())

        # Add fields to output polygon layer
        poly_fields = sampling_areas.fields()
        poly_fields.append(QgsField("MedSensTot", QVariant.Double))
        poly_fields.append(QgsField("MedSensPhy", QVariant.Double))
        poly_fields.append(QgsField("MedSensChe", QVariant.Double))
        poly_fields.append(QgsField("MedSensBio", QVariant.Double))
        poly_fields.append(QgsField("observers", QVariant.String))
        poly_fields.append(QgsField("observations", QVariant.Int))
        poly_fields.append(QgsField("taxa", QVariant.Int))
        poly_fields.append(QgsField(
            name="area_km2", type=QVariant.Double, typeName="double", len=5, prec=2))

        # define the sink layer
        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPOLYGON,
            context,
            poly_fields,
            sampling_areas.wkbType(),
            sampling_areas.sourceCrs()
        )

        # get the init date
        inidate = self.parameterAsString(
            parameters,
            self.INIDATE,
            context
        )

        # get the end date
        enddate = self.parameterAsString(
            parameters,
            self.ENDDATE,
            context
        )

        # parse dates (init and end) as dates and not time
        parsed_ini = datetime.strptime(
            parameters['INIDATE'], '%Y-%m-%d').date()
        parsed_end = datetime.strptime(
            parameters['ENDDATE'], '%Y-%m-%d').date()

        # FeatureRequest to speed up the loop
        area_extent = sampling_areas.sourceExtent()
        request = QgsFeatureRequest().setFilterRect(area_extent)

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / sampling_areas.featureCount() if sampling_areas.featureCount() else 0
        total_features = sampling_areas.featureCount()

        # get reference of features of sampling areas
        features = sampling_areas.getFeatures()

        # init empty MedSens indexes as defaultdict
        dd = defaultdict(dict)

        # start the algorithm logic by iterating on polygon areas
        for current, poly_feature in enumerate(features):

            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            feedback.setProgressText(
                'Polygon {} of {} is processed'.format(current + 1, total_features))
            feedback.setProgress(int(current * total))

            # get point geometry
            poly_geom = poly_feature.geometry()

            # get unique divers, observations and taxa, etc as keys in dictionary
            dd[poly_feature.id()]['observers'] = []
            dd[poly_feature.id()]['observations'] = []
            dd[poly_feature.id()]['taxa'] = []
            dd[poly_feature.id()]['ab'] = []
            dd[poly_feature.id()]['MedSensTot'] = []
            dd[poly_feature.id()]['MedSensPhy'] = []
            dd[poly_feature.id()]['MedSensChe'] = []
            dd[poly_feature.id()]['MedSensBio'] = []

            # polygon areas in square km
            dd[poly_feature.id()]['area'] = da.convertAreaMeasurement(
                da.measureArea(poly_geom), QgsUnitTypes.AreaSquareKilometers)

            for ii, point_feature in enumerate(source.getFeatures(request)):

                # filter date time and return it as date and not datetime
                dt = datetime.strptime(
                    point_feature["date"], '%d/%m/%Y %H:%M:%S').date()

                # filter only date intervals
                if dt > parsed_ini and dt < parsed_end:

                    point_geom = point_feature.geometry()

                    # check polygon and points intersection
                    if point_geom.intersects(poly_geom):

                        # append list of divers (observers) for each polygon area
                        if point_feature["source"] not in dd[poly_feature.id()]['observers']:
                            dd[poly_feature.id()]['observers'].append(
                                point_feature["source"])

                        # append list of observations for each polygon area
                        if point_feature.id() not in dd[poly_feature.id()]['observations']:
                            dd[poly_feature.id()]['observations'].append(
                                point_feature.id())

                        # append list of taxon for each polygon area
                        if point_feature["taxon"] not in dd[poly_feature.id()]['taxa']:
                            dd[poly_feature.id()]['taxa'].append(
                                point_feature["taxon"])

                        # get reference of polygon features as QgsFeature
                        dd[poly_feature.id()]['feature'] = poly_feature

                        # check the area of each polygon and continue only if the area is greater than 0.08 sqkm
                        if da.convertAreaMeasurement(da.measureArea(poly_geom), QgsUnitTypes.AreaSquareKilometers) > 0.08:

                            # append values as list for each polygon
                            if point_feature["ab"] != NULL:
                                dd[poly_feature.id()]['ab'].append(
                                    point_feature["ab"])

                            if point_feature["MSVtot"] != NULL:
                                dd[poly_feature.id()]['MedSensTot'].append(
                                    point_feature["MSVtot"] * point_feature["ab"])

                            if point_feature["MSVphy"] != NULL:
                                dd[poly_feature.id()]['MedSensPhy'].append(
                                    point_feature["MSVphy"] * point_feature["ab"])

                            if point_feature["MSVchem"] != NULL:
                                dd[poly_feature.id()]['MedSensChe'].append(
                                    point_feature["MSVchem"] * point_feature["ab"])

                            if point_feature["MSVbio"] != NULL:
                                dd[poly_feature.id()]['MedSensBio'].append(
                                    point_feature["MSVbio"] * point_feature["ab"])

        # loop in the defaultdict and create the QgsFeatures with geometry and attributes
        for k, v in dd.items():

            # create QgsFeature and fill it with dictionary id QgsFeature geometry
            out_feature = QgsFeature()
            try:
                out_feature.setGeometry(dd[k]['feature'].geometry())

                # Fill the QgsFeature with the attributes
                attrs = dd[k]['feature'].attributes()

                # extent the attributes with the calculated ones
                # check constraint of min observers, observation and taxa
                if len(dd[k]['observers']) >= 3 and len(dd[k]['observations']) >= 20 and len(dd[k]['taxa']) >= 10:
                    try:
                        attrs.extend(
                            [
                                sum(dd[k]['MedSensTot']) / sum(dd[k]['ab']),
                                sum(dd[k]['MedSensPhy']) / sum(dd[k]['ab']),
                                sum(dd[k]['MedSensChe']) / sum(dd[k]['ab']),
                                sum(dd[k]['MedSensBio']) / sum(dd[k]['ab']),
                                len(dd[k]['observers']),
                                len(dd[k]['observations']),
                                len(dd[k]['taxa']),
                                dd[k]['area']
                            ]
                        )
                    except:
                        pass
                else:
                        attrs.extend(
                            [
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                len(dd[k]['observers']),
                                len(dd[k]['observations']),
                                len(dd[k]['taxa']),
                                dd[k]['area']

                            ]
                        )

                # set finally the attributes to the QgsFeature
                out_feature.setAttributes(attrs)

                # Add a feature in the sink
                sink.addFeature(out_feature, QgsFeatureSink.FastInsert)
            except:
                pass

        if context.willLoadLayerOnCompletion(dest_id):
            context.layerToLoadOnCompletionDetails(
                dest_id).setPostProcessor(MyLayerPostProcessor.create())

        # return the final polygon with the style
        return {self.OUTPOLYGON: dest_id}

    def checkParameterValues(self, parameters, context):
        '''
        Perform some upstream checks on the sampling points quality
        Checks:
            - Initial date is not after Ending date
        '''

        # check the initial date versus the end date
        if parameters['INIDATE'] > parameters['ENDDATE']:
            return False, self.tr("Initial Date is <b>after</b> End Date")

        return super(medSensAlgorithm, self).checkParameterValues(parameters, context)

