"""
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import (
    qgsfunction,
    Qgis,
    QgsGeometry,
    QgsCircle,
)


from math import tau, pi, sqrt

# try to make this decorator work
# def set_invalid_geometry_type_error(func):
#     def wrapper(geometry: QgsGeometry, parent):
#         if geometry.type() != Qgis.GeometryType.Polygon:
#             parent.setEvalErrorString("Only polygon geometry are supported")
#         func(geometry)
#     return wrapper

FOUR_PI = 4 * pi  # Caching, slight optimization

if Qgis.QGIS_VERSION_INT > 34200:
    from qgis.core import QgsGeos

    def maximum_inscribed_circle(geometry: QgsGeometry) -> QgsGeometry:
        geos_geo = QgsGeos(geometry.get())
        epsilon = 0.00000001
        line_string = geos_geo.maximumInscribedCircle(epsilon)[0]

        return QgsGeometry(
            QgsCircle.fromCenterPoint(
                line_string.pointN(0), line_string.pointN(-1)
            ).toPolygon()
        )

    @qgsfunction(group="Compactness", referenced_columns=[])
    def compactness_skew(geometry: QgsGeometry, parent):
        """
        Calculate a skew compactness

        <p> A skew compactness compares the area of the maximum inscribed circle (A_mic) to the area of the minimum bounding circle (A_mbc). </p>

        <p>Can be written as:</p>
        <p>
        Skew = A_mic / A_mbc
        </p>

        <ul>
            <li> Where <b>A_mic</b> is the area of the maximum inscribed circle of the geometry </li>
            <li> Where <b>A_mbc</b> is the area of the minimum bounding circle of the geometry </li>
        </ul>

        Scores range from 0 to 1, where 0 is the least compact and 1 is the most compact.

        <h4>Syntax</h4>
        <p><b>compactness_skew</b>( <i>geometry</i> )</p>

        <h4>Arguments</h4>
        <p><i>geometry</i>: a polygon geometry</p>

        <h4>Example usage</h4>
        <ul>
        <li><b>compactness_skew</b>( $geometry )  &rarr; [0;1]</li>
        <li><b>compactness_skew</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 2, 0 0))') ) &rarr; 0,2</li>
        </ul>
        """
        if geometry.type() != Qgis.GeometryType.Polygon:
            parent.setEvalErrorString(
                "Only polygon geometry are supported for function `compactness_skew`"
            )
            return

        # mic = maximum inscribed circle
        A_mic = maximum_inscribed_circle(geometry).area()
        # mbc = minimal bounding circle
        A_mbc = geometry.minimalEnclosingCircle()[0].area()

        return A_mic / A_mbc


@qgsfunction(group="Compactness", referenced_columns=[])
def compactness_pp(geometry: QgsGeometry, parent):
    """
    Calculate a  <a href="https://en.wikipedia.org/wiki/Polsby%E2%80%93Popper_test">Polsby-Popper(PP) compactness</a> score

    <p>
    The Polsby-Popper (polsby & Popper, 1991) is the ratio of the area(A) of the geometry to the area of a circle whose circumference is equal to the perimeter(P) of the geometry .
    </p>

    <p>Can be written as:</p>
    <p>
    4 * pi * (A / P*P )
    </p>

    <ul>
      <li> Where <b>A</b> is the area of the geometry </li>
      <li> Where <b>P</b> is the perimeter of the geometry </li>
    </ul>

    a score of 0 indicating complete lack of compactness and a score of 1 indicating maximal compactness. Only a perfectly round geometry will reach a Polsby–Popper score of 1.

    <h4>Syntax</h4>
    <p><b>compactness_pp</b>( <i>geometry</i> )</p>

    <h4>Arguments</h4>
    <p><i>geometry</i>: a polygon geometry</p>

    <h4>Example usage</h4>
    <ul>
      <li><b>compactness_pp</b>( $geometry )  &rarr; [0;1]</li>
      <li><b>compactness_pp</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 2, 0 0))') ) &rarr; 0,698131&hellip;</li>
    </ul>
    """
    if geometry.type() != Qgis.GeometryType.Polygon:
        parent.setEvalErrorString(
            "Only polygon geometry are supported for function `compactness_pp`"
        )
        return

    A = geometry.area()
    P = geometry.length()

    return (FOUR_PI * A) / (P * P)


@qgsfunction(group="Compactness", referenced_columns=[])
def compactness_schwartz(geometry: QgsGeometry, parent):
    """
    Calculate a schwartzberg compactness

    <p>Can be written as:</p>
    <p>
    S = 1 / (P / (2pi * sqrt(A / pi)))
    </p>
    <ul>
      <li> Where <b>A</b> is the area of the geometry </li>
      <li> Where <b>P</b> is the perimeter of the geometry </li>
    </ul>


    Scores range from 0 to 1, where 0 is the least compact and 1 is the most compact.

    <h4>Syntax</h4>
    <p><b>compactness_schwartz</b>( <i>geometry</i> )</p>

    <h4>Arguments</h4>
    <p><i>geometry</i>: a polygon geometry</p>

    <h4>Example usage</h4>
    <ul>
      <li><b>compactness_schwartz</b>( $geometry )  &rarr; [0;1]</li>
      <li><b>compactness_schwartz</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 2, 0 0))') ) &rarr; 0,8355&hellip;</li>
    </ul>
    """
    if geometry.type() != Qgis.GeometryType.Polygon:
        parent.setEvalErrorString(
            "Only polygon geometry are supported for function `compactness_schwartz`"
        )
        return

    A = geometry.area()
    P = geometry.length()

    return 1 / (P / (tau * sqrt(A / pi)))


@qgsfunction(group="Compactness", referenced_columns=[])
def compactness_reock(geometry: QgsGeometry, parent):
    """
    Calculate a reock compactness

    <p> A reock compactness is the ratio between the area (A) of the geometry to the area of the minimum bounding circle (A_mbc)</p>

    <p>Can be written as:</p>
    <p>
    reock = A / A_mbc
    </p>
    <ul>
      <li> Where <b>A</b> is the area of the geometry </li>
      <li> Where <b>A_mbc</b> is the area of the minimum bounding circle of the geometry </li>
    </ul>

    Scores range from 0 to 1, where 0 is the least compact and 1 is the most compact.

    <h4>Syntax</h4>
    <p><b>compactness_reock</b>( <i>geometry</i> )</p>

    <h4>Arguments</h4>
    <p><i>geometry</i>: a polygon geometry</p>

    <h4>Example usage</h4>
    <ul>
      <li><b>compactness_reock</b>( $geometry )  &rarr; [0;1]</li>
      <li><b>compactness_reock</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 2, 0 0))') ) &rarr; 0,51189&hellip;</li>
    </ul>
    """
    if geometry.type() != Qgis.GeometryType.Polygon:
        parent.setEvalErrorString(
            "Only polygon geometry are supported for function `compactness_reock`"
        )
        return

    A = geometry.area()
    A_mbc = geometry.minimalEnclosingCircle()[0].area()

    return A / A_mbc


@qgsfunction(group="Compactness", referenced_columns=[])
def compactness_box_reock(geometry: QgsGeometry, parent):
    """
    Calculate a box reock compactness

    <p> A box reock compactness is the ratio between the area (A) of the geometry to the area of the minimum bounding box (A_bbox)</p>

    <p>Can be written as:</p>
    <p>
    box_reock = A / A_bbox
    </p>
    <ul>
      <li> Where <b>A</b> is the area of the geometry </li>
      <li> Where <b>A_bbox</b> is the area of the minimum bounding rectangle of the geometry </li>
    </ul>

    Scores range from 0 to 1, where 0 is the least compact and 1 is the most compact.

    <h4>Syntax</h4>
    <p><b>compactness_box_reock</b>( <i>geometry</i> )</p>

    <h4>Arguments</h4>
    <p><i>geometry</i>: a polygon geometry</p>

    <h4>Example usage</h4>
    <ul>
      <li><b>compactness_box_reock</b>( $geometry )  &rarr; [0;1]</li>
      <li><b>compactness_box_reock</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 2, 0 0))') ) &rarr; 1</li>
      <li><b>compactness_box_reock</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 0))') ) &rarr; 0.5</li>
    </ul>
    """
    if geometry.type() != Qgis.GeometryType.Polygon:
        parent.setEvalErrorString(
            "Only polygon geometry are supported for function `compactness_box_reock`"
        )
        return

    A = geometry.area()
    A_bbox = geometry.boundingBox().area()

    return A / A_bbox


@qgsfunction(group="Compactness", referenced_columns=[])
def compactness_lw(geometry: QgsGeometry, parent):
    """
    Calculate a Length-Width Compactness

    <p> Length-Width compactness compares the width and the length of a geometry using its bouding box</p> 
    <p>Can be written as:</p>
    <p>
    lw =  W_bbox / L_bbox
    </p>
    <ul>
      <li> Where <b>W_bbox</b> is the shorter side of the bounding box of the geometry </li>
      <li> Where <b>L_bbox</b> the longer side of the bounding box of the geometry </li>
    </ul>

    Scores range from 0 to 1, where 0 is the least compact and 1 is the most compact.

    <h4>Syntax</h4>
    <p><b>compactness_lw</b>( <i>geometry</i> )</p>

    <h4>Arguments</h4>
    <p><i>geometry</i>: a polygon geometry</p>

    <h4>Example usage</h4>
    <ul>
      <li><b>compactness_lw</b>( $geometry )  &rarr; [0;1]</li>
      <li><b>compactness_lw</b>( geom_from_wkt('POLYGON((0 0, 4 0, 4 2, 0 2, 0 0))') ) &rarr; 0.5</li>
    </ul>
    """
    if geometry.type() != Qgis.GeometryType.Polygon:
        parent.setEvalErrorString(
            "Only polygon geometry are supported for function `compactness_lw`"
        )
        return

    bbox = geometry.boundingBox()

    # We define width as the shortest side
    width = min(bbox.height(), bbox.width())
    length = max(bbox.height(), bbox.width())

    return width / length
