# -*- coding: utf-8 -*-
"""
/***************************************************************************
 MDI-DE Metadata Search
                                 A QGIS plugin
 mdi-de metadata search module
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 12.2024
        git sha              : $Format:%H$
        copyright            : (C) 2024 by terrestris GmbH & Co. KG
        email                : info@terrestris.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import Qt, QSettings, QTranslator, QCoreApplication, QUrl
from qgis.PyQt.QtGui import QIcon, QPixmap, QDesktopServices
from qgis.PyQt.QtWidgets import *
# some things for doing http requests
from qgis.core import Qgis, QgsNetworkAccessManager, QgsMessageLog, QgsSettings, QgsBrowserModel
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .mdi_de_metadata_search_dialog import MDIDEMetadataSearchDialog
import os.path
# some libs for json parsing under python3
import math
from qgis.utils import OverrideCursor
from owslib.util import Authentication
from mdi_de_metadatasearch.search.search_backend import get_catalog_service
from mdi_de_metadatasearch.search.catalogtype import CatalogType
from mdi_de_metadatasearch.resource_column import ResourceColumn
from mdi_de_metadatasearch.record.recordtype import RecordType


class MDIDEMetadataSearch:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'mdi_de_metadata_search_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&MDI-DE Metadata Search')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # read all settings from qgis settings
        self.settings = QgsSettings()

        # read actual browser model
        self.browser_model = QgsBrowserModel()

        self.remote_resource_list = [{
            'id': 'mdi-de-csw',
            'title': 'MDI-DE (CSW 2.0.2)',
            'url': 'https://nokis.mdi-de-dienste.org/csw',
            'type': CatalogType.OGC_CSW_202,
            'metadataURL': 'https://nokis.mdi-de-dienste.org/trefferanzeige?docuuid=',
            'maxrecords': 10
        }, {
            'id': 'gdi-de-csw',
            'title': 'gdi-de (CSW 2.0.2)',
            'url': 'https://gdk.gdi-de.org/gdi-de/srv/ger/csw',
            'type': CatalogType.OGC_CSW_202,
            'metadataURL': 'https://gdk.gdi-de.org/gdi-de/srv/ger/catalog.search#/metadata/'
        }, {
            'id': 'emodnet-csw',
            'title': 'emodnet (CSW 2.0.2)',
            'url': 'https://emodnet.ec.europa.eu/geonetwork/srv/eng/csw',
            'type': CatalogType.OGC_CSW_202,
            'metadataURL': 'https://emodnet.ec.europa.eu/geonetwork/srv/eng/catalog.search#/metadata/'
        }, {
            'id': 'test',
            'title': 'MDI-DE TEST (CSW 3.0.0)',
            'url': 'https://nokis.mdi-de-dienste.org/pycsw262/csw',
            'type': CatalogType.OGC_CSW_300,
            'maxrecords': 10
        }]

        #network access
        self.na_manager = QgsNetworkAccessManager.instance()

        self.catalog = None
        self.catalog_url = None
        self.catalog_username = None
        self.catalog_password = None
        self.catalog_type = None
        self.startfrom = 1
        self.constraints = []
        self.timeout = 10
        self.disable_ssl_verification = True

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('MDIDEMetadataSearch', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(os.path.dirname(__file__), "mdi_logo_grey_square_cropped.png")
        self.add_action(
            icon_path,
            text=self.tr(u'MDI-DE Metadata search'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(
                self.tr(u'&MDI-DE Metadata Search'),
                action)
            self.iface.removeToolBarIcon(action)


    def show_loader_img(self):
        """
        Should be used to show the loading of an external resource - does not work at the moment
        """
        self.dlg.labelSearchAnimation.setText(self.tr("Searching..."))
        pass


    def hide_loader_img(self):
        """
        Should be used to hide the loading of an external resource - does not work at the moment
        """
        self.dlg.labelSearchAnimation.setText(self.tr("Ready"))
        pass

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = MDIDEMetadataSearchDialog()
            # search_catalogues
            # important - events should only be added once - otherwise we will go into trouble!
            self.dlg.pushButton.clicked.connect(lambda: self.start_search(page=1))
            # add logo
            logo_path = os.path.join(os.path.dirname(__file__), "LOGO_END_lang_blue.png")
            if logo_path:
                # build
                pixmap = QPixmap(logo_path)
                # draw preview
                self.dlg.labelLogo.setPixmap(pixmap.scaled(self.dlg.labelLogo.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation))
                self.dlg.labelLogo.mousePressEvent = self.open_project_link
            else:
                QgsMessageLog.logMessage("An error occured while try to open logo", 'MDI-DE Metadata Search',
                                         level=Qgis.Critical)
            # add link to github for help
            help_icon_path = os.path.join(os.path.dirname(__file__), "questionmark.png")
            pixmap.load(help_icon_path)
            self.dlg.labelHelp.setPixmap(pixmap.scaled(self.dlg.labelHelp.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation))
            self.dlg.labelHelp.setText('<a href="https://gitlab.opencode.de/bfn/open-access/qgis-ms#">' +
                                       self.tr("Help") + '</a>')
            self.dlg.labelHelp.setOpenExternalLinks(True)
            self.populate_resource_list()
            self.dlg.comboBoxRemoteCsw.currentIndexChanged.connect(lambda: self.activate_region_filters())
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def open_project_link(self, event):
        if event.button() == Qt.LeftButton:
            QDesktopServices.openUrl(QUrl("https://projekt.mdi-de.org/"))

    def populate_resource_list(self):
        for catalogue in self.remote_resource_list:
            self.dlg.comboBoxRemoteCsw.addItem(catalogue.get('title'), catalogue.get('id'))
            self.dlg.comboBoxRemoteCsw.setEnabled(True)

    def activate_region_filters(self):
        selected_resource = [r for r in self.remote_resource_list if r.get('id') == self.dlg.comboBoxRemoteCsw.currentData()][0]
        if selected_resource.get('id') != 'mdi-de-csw':
            self.dlg.cbOstsee.setChecked(False)
            self.dlg.cbNordsee.setChecked(False)
            self.dlg.cbOstsee.setEnabled(False)
            self.dlg.cbNordsee.setEnabled(False)
        else: 
            self.dlg.cbOstsee.setEnabled(True)
            self.dlg.cbNordsee.setEnabled(True)

    def reset_form(self, reset_input=False):
        if reset_input:
            self.dlg.textEditSearchText.clear()
        self.dlg.numberOfAllPagesLabel.setText('0')
        self.dlg.numberOfPageLabel.setText('0')
        self.dlg.numberOfAllResultsLabel.setText('0')
        self.dlg.treeWidgetResource.clear()
        self.dlg.textBrowserResourceAbstract.clear()
        self.dlg.pushButtonLoad.setEnabled(False)
        self.dlg.pushButtonPageBack.setEnabled(False)
        self.dlg.pushButtonPageForward.setEnabled(False)
        # very important thing:
        try:
            self.dlg.treeWidgetResource.itemClicked.disconnect()
        except:
            pass


    def reset_resource_view(self, reset_input=False):
        if reset_input:
            self.dlg.textEditSearchText.clear()
        self.dlg.labelResourceType.setText('')
        self.dlg.labelResourceId.setText('')
        self.dlg.labelPreview.setText(self.tr('Preview...'))
        self.dlg.labelOrga.setText('')
        self.dlg.labelDate.setText('')
        self.dlg.labelAccessUrl.setText('')
        self.dlg.labelMetadata.setText('')
        self.dlg.textBrowserResourceAbstract.clear()
        self.dlg.tableWidgetExtent.clearContents()
        self.dlg.treeWidgetRestrictions.clear()


    def alter_pager(self, actual_page=1, number_of_all_pages=1):
        enable_forward = actual_page < number_of_all_pages
        enable_back = actual_page > 1
        self.dlg.numberOfPageLabel.setText(str(actual_page))
        self.dlg.numberOfAllPagesLabel.setText(str(number_of_all_pages))
        self.dlg.pushButtonPageBack.setEnabled(enable_back)
        self.dlg.pushButtonPageForward.setEnabled(enable_forward)
        self.dlg.pushButtonPageForward.disconnect()
        self.dlg.pushButtonPageBack.disconnect()
        self.dlg.pushButtonPageForward.clicked.connect(lambda: self.start_search(page=(actual_page + 1)))
        self.dlg.pushButtonPageBack.clicked.connect(lambda: self.start_search(page=(actual_page - 1)))


    def get_number_of_all_pages(self, number_of_results, results_per_page):
        return math.ceil(int(number_of_results) / results_per_page)


    def start_search(self, page=1):
        self.show_loader_img()
        self.reset_form(False)
        self.reset_resource_view()
        self.searchMetadata(page)
        self.update_resource_tree()

        number_of_results = self.catalog.matches if self.catalog is not None else 0
        results_per_page = self.maxrecords
        number_of_all_pages = self.get_number_of_all_pages(number_of_results, results_per_page)
        self.dlg.numberOfAllResultsLabel.setText(str(number_of_results))
        self.alter_pager(page, number_of_all_pages)

        # TODO check if we should compute search time
        # search_time = search_result.md.genTime
        # if search_time:
        #     QgsMessageLog.logMessage("Used seconds: " + str(search_time), 'MDI-DE Metadata Search',
        #                              level=Qgis.Info)
        #     self.dlg.labelSearchAnimation.setText("(" + str(round(search_time, 3)) + " " + self.tr("seconds") + ")")
        self.hide_loader_img()

    def get_selected_resource(self):
        return [r for r in self.remote_resource_list if r.get('id') == self.dlg.comboBoxRemoteCsw.currentData()][0]

    def compute_startfrom(self, page=1):
        return ((page - 1) * self.maxrecords) + 1
    
    def get_type_filters(self):
        # get checkbox values
        type_filters = {
            RecordType.MAP.value: self.dlg.cbKarte.isChecked(),
            RecordType.TILE.value: self.dlg.cbKachel.isChecked(),
            RecordType.DATASET.value: self.dlg.cbDatensatz.isChecked(),
            RecordType.DOWNLOAD.value: self.dlg.cbDownload.isChecked(),
            RecordType.APPLICATION.value: self.dlg.cbAnwendung.isChecked(),
            RecordType.SERIES.value: self.dlg.cbSerie.isChecked()
        }

        return type_filters
    
    def get_region_filters(self):
        region_filters = {
            "BalticSea": self.dlg.cbOstsee.isChecked(),
            "NorthSea": self.dlg.cbNordsee.isChecked()
        }
        return region_filters

    def searchMetadata(self, page=1):
        """execute search"""
        self.catalog = None
        self.constraints = []

        selected_resource = self.get_selected_resource()
        self.catalog_url = selected_resource.get('url')
        self.catalog_type = selected_resource.get('type')

        self.maxrecords = 20 if selected_resource.get('maxrecords') is None else selected_resource.get('maxrecords')

        self.startfrom = self.compute_startfrom(page)

        type_filters = self.get_type_filters()
        region_filters = self.get_region_filters()

        keywords = self.dlg.textEditSearchText.text()

        self._init_catalog()

        if not self.catalog:
            return

        try:
            with OverrideCursor(Qt.CursorShape.WaitCursor):
                self.catalog.query_records(
                    keywords, self.maxrecords, self.startfrom, type_filters, region_filters
                )
        except Exception as err:
            msg = self.tr("Search error: {0}").format(err)
            QMessageBox.warning(
                self.iface.mainWindow(), self.tr("Search error"), msg
            )
            return


    def _init_catalog(self):
        """convenience function to init catalog wrapper"""

        auth = None

        if self.disable_ssl_verification:
            try:
                auth = Authentication(verify=False)
            except NameError:
                pass

        # connect to the server
        with OverrideCursor(Qt.CursorShape.WaitCursor):
            try:
                self.catalog = get_catalog_service(
                    self.catalog_url,
                    catalog_type=self.catalog_type,
                    timeout=self.timeout,
                    username=self.catalog_username or None,
                    password=self.catalog_password or None,
                    auth=auth,
                    metadataUrl = self.get_selected_resource().get('metadataURL')
                )
            except Exception as err:
                msg = self.tr("Error connecting to service: {0}").format(err)
                QMessageBox.warning(self.iface.mainWindow(), self.tr("CSW Connection error"), msg)

    def update_resource_tree(self):
        """display search results"""
        self.dlg.treeWidgetResource.clear()

        if self.catalog is None or self.catalog.matches == 0:
            node = QTreeWidgetItem()
            node.setText(0, self.tr("No results"))
            self.dlg.treeWidgetResource.addTopLevelItem(node)
            return

        parent_nodes = {
            RecordType.MAP.value: None,
            RecordType.TILE.value: None,
            RecordType.DATASET.value: None,
            RecordType.DOWNLOAD.value: None,
            RecordType.APPLICATION.value: None,
            RecordType.SERIES.value: None,
            RecordType.OTHER.value: None,
        }
    
        for rec in self.catalog.records():
            item = QTreeWidgetItem()
            category_type = rec.get_category_type()
            if category_type:
                item.setData(ResourceColumn.TYPE.value, 0, category_type.value)
            if rec.get_title():
                item.setText(0, rec.get_title())
                item.setToolTip(0, rec.get_title())
                item.setData(ResourceColumn.TITLE.value, 0, rec.get_title())
            if rec.get_identifier():
                item.setData(ResourceColumn.IDENTIFIER.value, 0, rec.get_identifier())
            if parent_nodes[category_type.value] is None:
                parent_node = QTreeWidgetItem()
                parent_node.setText(0, QCoreApplication.translate('MDIDEMetadataSearchDialogBase', category_type.value))
                parent_nodes[category_type.value] = parent_node
            parent_nodes[category_type.value].addChild(item)

        self.dlg.treeWidgetResource.addTopLevelItems([node for _, node in parent_nodes.items() if node is not None])
        self.dlg.treeWidgetResource.expandAll()
        self.dlg.treeWidgetResource.itemClicked.connect(self.on_clicked_record)

    def render_record(self, item):
        item_id = item.data(ResourceColumn.IDENTIFIER.value, 0)
        record = self.catalog.get_record(item_id)
        record.render(self.dlg, self.iface)

    def on_clicked_record(self, item):
        self.render_record(item)
