###############################################################################
#
# CSW Client
# ---------------------------------------------------------
# QGIS Catalog Service client.
#
# Copyright (C) 2024 Tom Kralidis (tomkralidis@gmail.com)
# modified by terrestris GmbH & Co. KG (2024)
#
# This source is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# This code is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
###############################################################################

import warnings

import owslib
from owslib.fes import BBox, PropertyIsLike

from owslib.namespaces import Namespaces

# from MetaSearch.util import log_message

with warnings.catch_warnings():
    warnings.filterwarnings("ignore", category=ResourceWarning)
    warnings.filterwarnings("ignore", category=ImportWarning)
    from owslib.csw import CatalogueServiceWeb  # spellok

from qgis.core import Qgis

if owslib.__version__ < "0.25":
    OWSLIB_OAREC_SUPPORTED = False
else:
    OWSLIB_OAREC_SUPPORTED = True

from mdi_de_metadatasearch.search.catalogtype import CatalogType
from mdi_de_metadatasearch.search.csw202 import CSW202Search
from mdi_de_metadatasearch.search.csw300 import CSW300Search
from mdi_de_metadatasearch.search.oarecsearch import OARecSearch

def get_catalog_service(url, catalog_type, timeout, username, password, auth=None, metadataUrl=None):
    if catalog_type in [None, CatalogType.OGC_CSW_202]:
        #log_message("CSW endpoint detected", Qgis.MessageLevel.Info)
        return CSW202Search(url, timeout, username, password, auth, metadataUrl)
    elif catalog_type == CatalogType.OGC_API_RECORDS:
        #log_message("OARec endpoint detected", Qgis.MessageLevel.Info)
        if not OWSLIB_OAREC_SUPPORTED:
            raise ValueError("OGC API - Records requires OWSLib 0.25 or above")
        return OARecSearch(url, timeout, auth, metadataUrl)
    elif catalog_type == CatalogType.OGC_CSW_300:
        return CSW300Search(url, timeout, username, password, auth, metadataUrl)


def bbox_list_to_dict(bbox):
    if isinstance(bbox, list):
        dict_ = {"minx": bbox[0], "maxx": bbox[2], "miny": bbox[1], "maxy": bbox[3]}
    else:
        dict_ = {
            "minx": bbox.minx,
            "maxx": bbox.maxx,
            "miny": bbox.miny,
            "maxy": bbox.maxy,
        }
    return dict_
