# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Matti
                                 A QGIS plugin
 A plugin to estimate soybean maturity
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-05-20
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Nathaniel Burner
        email                : nburner@uga.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QDateTime, QDate, QVariant
from qgis.PyQt.QtGui import QIcon, QStandardItem, QStandardItemModel
from qgis.PyQt.QtWidgets import QAction, QListWidgetItem, QTableWidgetItem
from qgis.core import QgsProject, Qgis, QgsGeometry, QgsWkbTypes, QgsMapLayer, QgsApplication, QgsRasterLayer, NULL, \
    QgsVectorLayer, QgsFeatureRequest, QgsField, QgsProcessingFeedback, QgsProcessingContext
from qgis.analysis import QgsRasterCalculator, QgsRasterCalculatorEntry, QgsZonalStatistics
from qgis.gui import QgsFileWidget
from datetime import datetime, date, timedelta
from qgis import processing
import pandas as pd
import statsmodels.api as sm
from scipy.interpolate import interp1d

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .matti_dialog import MattiDialog
import os.path
import pandas as pd
import piecewise_regression
import scipy.sparse.linalg as sp
import numpy as np
import time
import plotly.express as px


class Matti:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'Matti_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Matti')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Matti', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/matti/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Estimate maturity'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Matti'),
                action)
            self.iface.removeToolBarIcon(action)

    def populate_field_list_widget(self):
        if self.dlg.shpCmboBox.currentIndex() == 0:
            self.dlg.exptCmboBox.clear()
            self.dlg.plotCmboBox.clear()
            return

        # Get selected vector layer
        selected_layer_name = self.dlg.shpCmboBox.currentText()
        if not selected_layer_name:
            return

        # Filter only vector layers with the selected name
        vector_layers = [
            layer for layer in QgsProject.instance().mapLayersByName(selected_layer_name)
            if isinstance(layer, QgsVectorLayer)
        ]

        if not vector_layers:
            self.iface.messageBar().pushMessage("Error", f"No vector layer found with name: {selected_layer_name}.",
                                                level=Qgis.Critical)
            return

        selected_layer = vector_layers[0]  # Select the first vector layer

        fields = selected_layer.fields()
        field_names = [field.name() for field in fields]

        field_names.sort()

        # Clear existing items in list widget
        self.dlg.exptCmboBox.clear()
        self.dlg.plotCmboBox.clear()

        self.dlg.exptCmboBox.addItem("")
        self.dlg.plotCmboBox.addItem("")

        for field_name in field_names:
            self.dlg.exptCmboBox.addItem(field_name)
            self.dlg.plotCmboBox.addItem(field_name)

    def validate_dates(self):
        imgDt = self.dlg.imgDt.date()
        dayzeroDt = self.dlg.dayzeroDt.date()

        if imgDt.year() != dayzeroDt.year():
            self.dlg.dtWarnLbl.setText("Warning: Selected date and Image Date have different years")
            self.dlg.dtWarnLbl.setStyleSheet("color: red;")
        elif imgDt < dayzeroDt:
            self.dlg.dtWarnLbl.setText("Warning: Selected date is after Image Date")
            self.dlg.dtWarnLbl.setStyleSheet("color: red;")
        else:
            self.dlg.dtWarnLbl.clear()

    def validate_col_boxes(self):
        expt_value = self.dlg.exptCmboBox.currentText()
        plot_value = self.dlg.plotCmboBox.currentText()

        if expt_value and plot_value and expt_value == plot_value:
            self.dlg.colWarnLbl.setText("Columns should not match")
            self.dlg.colWarnLbl.setStyleSheet("color: red;")
        else:
            self.dlg.colWarnLbl.clear()

    def reset_values(self):
        """Reset the dialog values to their defaults."""
        self.dlg.imgDt.setDateTime(QDateTime.currentDateTime())
        img_year = QDate.currentDate().year()
        aug_31 = QDate(img_year, 8, 31)
        aug_31_dt = QDateTime(aug_31)
        self.dlg.dayzeroDt.setDateTime(aug_31_dt)
        self.dlg.dtWarnLbl.clear()
        self.dlg.shpCmboBox.setCurrentIndex(0)
        self.dlg.rasterCmboBox.setCurrentIndex(0)
        self.dlg.exptCmboBox.setCurrentIndex(0)
        self.dlg.plotCmboBox.setCurrentIndex(0)
        self.dlg.threshSpin.setValue(0.01)
        self.dlg.csvFile.setFilePath("")
        self.dlg.colWarnLbl.setText("")
        self.dlg.gliKeep.setChecked(False)
        self.dlg.lastFlight.setChecked(False)
        self.dlg.summaryBox.setChecked(True)
        self.dlg.byDateBtn.setChecked(True)
        self.dlg.estBox.setChecked(True)

    def enable_run_button(self):
        """Enable runBtn when all proper selections have been made"""
        if (self.dlg.rasterCmboBox.currentIndex() != 0 and
                self.dlg.shpCmboBox.currentIndex() != 0 and
                self.dlg.exptCmboBox.currentIndex() != 0 and
                self.dlg.plotCmboBox.currentIndex() != 0 and
                self.dlg.plotCmboBox.currentText() != self.dlg.exptCmboBox.currentText()):
            self.dlg.runBtn.setEnabled(True)
        else:
            self.dlg.runBtn.setEnabled(False)

    def format_date(self, qdate):
        date_obj = datetime(qdate.year(), qdate.month(), qdate.day())
        return date_obj.strftime("%m-%d-%y")

    def generate_schedule(self):

        # Clear existing data in table
        self.dlg.matTbl.clear()

        # Convert checked days to indices
        days = {
            "Mon": 0,
            "Tue": 1,
            "Wed": 2,
            "Thu": 3,
            "Fri": 4,
            "Sat": 5,
            "Sun": 6
        }

        start_date = self.dlg.dayzeroDt.date().toPyDate()

        # Extract breaks
        breaks = []
        for i in range(self.dlg.wdaysCmboBox.model().rowCount()):
            item = self.dlg.wdaysCmboBox.model().item(i)
            if item.checkState() == Qt.Checked:
                breaks.append(days.get(item.text()))

        months = self.dlg.lengthSpinBox.value()
        skip_first_break = self.dlg.skipCheckBox.isChecked()

        end_date = start_date + pd.DateOffset(months=months)
        dates = pd.date_range(start=start_date, end=end_date, freq="D")

        # Initialize date blocks
        date_blocks = []
        db = 1

        for date in dates:
            weekday = date.weekday()

            if weekday in breaks:
                if skip_first_break:
                    date_blocks.append(db)
                    skip_first_break = False
                elif len(date_blocks) > 0:
                    db += 1
                    date_blocks.append(db)
                else:
                    date_blocks.append(db)
            else:
                date_blocks.append(db)

        # Create data frame
        mat_sched = pd.DataFrame({"DATE": dates, "Maturity": date_blocks})
        mat_sched["DAY"] = (
                    mat_sched["DATE"] - pd.Timestamp(start_date)).dt.days  # Add column for days after start date

        # Summary of results
        schedule_summary = mat_sched.groupby("Maturity").agg(Start=('DATE', 'min'), End=('DATE', 'max')).reset_index()[
            ['Start', 'End', 'Maturity']]

        schedule_summary['Start'] = schedule_summary['Start'].apply(lambda x: x.strftime('%b') + ' ' + str(x.day))
        schedule_summary['End'] = schedule_summary['End'].apply(lambda x: x.strftime('%b') + ' ' + str(x.day))

        # Set column headers
        self.dlg.matTbl.setColumnCount(len(schedule_summary.columns))
        self.dlg.matTbl.setHorizontalHeaderLabels(schedule_summary.columns)

        # Column sizes
        self.dlg.matTbl.setColumnWidth(0, 100)  # Set width of the first column
        self.dlg.matTbl.setColumnWidth(1, 100)  # Set width of the second column
        self.dlg.matTbl.setColumnWidth(2, 100)

        # Set row count
        self.dlg.matTbl.setRowCount(schedule_summary.shape[0])

        # Insert data into the table
        for row in range(schedule_summary.shape[0]):
            for col in range(schedule_summary.shape[1]):
                item = QTableWidgetItem(str(schedule_summary.iat[row, col]))
                item.setTextAlignment(Qt.AlignCenter)
                self.dlg.matTbl.setItem(row, col, item)

        # Resize table to fit the content
        # self.dlg.matTbl.resizeColumnsToContents()
        self.dlg.matTbl.resizeRowsToContents()

        return mat_sched

    def main(self):
        start_time = time.time()

        raster_layer_name = self.dlg.rasterCmboBox.currentText()
        vector_layer_name = self.dlg.shpCmboBox.currentText()

        csv_file_path = self.dlg.csvFile.filePath()

        # Add csv suffix if it does not already contain it
        if not csv_file_path.endswith('.csv'):
            csv_file_path = self.dlg.csvFile.filePath() + ".csv"

        # Import other values
        imgDt = self.dlg.imgDt.date()
        dayzeroDt = self.dlg.dayzeroDt.date()
        threshold = self.dlg.threshSpin.value()
        image_date_str = self.format_date(imgDt)
        day_zero_str = self.format_date(dayzeroDt)

        if not raster_layer_name:
            self.iface.messageBar().pushMessage("Error", "Please select a raster layer.", level=Qgis.Critical)
            return

        if not vector_layer_name:
            self.iface.messageBar().pushMessage("Error", "Please select a vector layer.", level=Qgis.Critical)
            return

        # Find raster layer by name and type
        raster_layers = [
            layer for layer in QgsProject.instance().mapLayersByName(raster_layer_name)
            if isinstance(layer, QgsRasterLayer)
        ]

        if not raster_layers:
            self.iface.messageBar().pushMessage("Error", f"No raster layer found with name: {raster_layer_name}.",
                                                level=Qgis.Critical)
            return

        raster_layer = raster_layers[0]

        if raster_layer.bandCount() < 3:
            self.iface.messageBar().pushMessage("Error", "Selected raster does not have at least 3 bands.",
                                                level=Qgis.Critical)
            return

        # Find vector layer by name and type
        vector_layers = [
            layer for layer in QgsProject.instance().mapLayersByName(vector_layer_name)
            if isinstance(layer, QgsVectorLayer)
        ]
        if not vector_layers:
            self.iface.messageBar().pushMessage("Error", f"No vector layer found with name: {vector_layer_name}.",
                                                level=Qgis.Critical)
            return

        vector_layer = vector_layers[0]

        # raster_layer = QgsProject.instance().mapLayersByName(raster_layer_name)[0]
        # vector_layer = QgsProject.instance().mapLayersByName(vector_layer_name)[0]
        #
        # if not isinstance(raster_layer, QgsRasterLayer):
        #     self.iface.messageBar().pushMessage("Error", "Selected layer is not a valid raster.", level=Qgis.Critical)
        #     return


        # Clip raster by vector to save computation time

        output_folder = self.dlg.outputFolder.filePath()
        formatted_date = self.format_date(self.dlg.imgDt.date())
        clipped_raster_path = os.path.join(output_folder, f"{formatted_date}_{QgsProject.instance().baseName()}_clipped.tif")

        params = {'INPUT': raster_layer,
                  'MASK': vector_layer,
                  'OUTPUT': clipped_raster_path}

        feedback = QgsProcessingFeedback()
        processing.run("gdal:cliprasterbymasklayer", params, feedback=feedback)

        # Load the clipped raster
        clipped_raster_layer = QgsRasterLayer(clipped_raster_path, f"{formatted_date}_{QgsProject.instance().baseName()}_clipped")
        QgsProject.instance().addMapLayer(clipped_raster_layer)

        if not clipped_raster_layer.isValid():
            self.iface.messageBar().pushMessage("Error", "Clipped raster is not valid.", level=Qgis.Critical)
            return

        # GLI calculation

        r_entry = QgsRasterCalculatorEntry()
        r_entry.ref = f'{clipped_raster_layer.name()}@1'
        r_entry.raster = clipped_raster_layer
        r_entry.bandNumber = 1

        g_entry = QgsRasterCalculatorEntry()
        g_entry.ref = f'{clipped_raster_layer.name()}@2'
        g_entry.raster = clipped_raster_layer
        g_entry.bandNumber = 2

        b_entry = QgsRasterCalculatorEntry()
        b_entry.ref = f'{clipped_raster_layer.name()}@3'
        b_entry.raster = clipped_raster_layer
        b_entry.bandNumber = 3

        expression = f'(2 * "{g_entry.ref}" - "{r_entry.ref}" - "{b_entry.ref}") / (2 * "{g_entry.ref}" + "{r_entry.ref}" + "{b_entry.ref}")'

        output_file_path = os.path.join(output_folder, f"{formatted_date}_{QgsProject.instance().baseName()}_GLI.tif")

        calculator = QgsRasterCalculator(expression, output_file_path, 'GTiff', raster_layer.extent(),
                                         raster_layer.width(), raster_layer.height(), [r_entry, g_entry, b_entry])
        result = calculator.processCalculation()

        if result != QgsRasterCalculator.Success:
            self.iface.messageBar().pushMessage("Error", "Raster calculation failed.", level=Qgis.Critical)
            return

        output_raster_layer = QgsRasterLayer(output_file_path,
                                             f"{formatted_date}_{QgsProject.instance().baseName()}_GLI")
        if not output_raster_layer.isValid():
            self.iface.messageBar().pushMessage("Error", "Failed to load the resulting raster.", level=Qgis.Critical)
            return

        QgsProject.instance().addMapLayer(output_raster_layer)

        self.iface.messageBar().pushMessage("Success", "GLI calculation completed.",
                                            level=Qgis.Success)

        # Remove clipped raster
        clipped_layer = QgsProject.instance().mapLayersByName(f"{formatted_date}_{QgsProject.instance().baseName()}_clipped")

        # Remove the layer from the project
        if clipped_layer:
            QgsProject.instance().removeMapLayer(clipped_layer[0].id())

        time.sleep(0.5)

        # Check if the file is still open or in use
        if os.path.exists(clipped_raster_path):
            try:
                os.remove(clipped_raster_path)
            except PermissionError as e:
                print(f"Error: {e}. The file might still be in use.")

        # Remove aux.xml file
        clipped_aux_path = clipped_raster_path + ".aux.xml"
        if os.path.exists(clipped_aux_path):
            try:
                os.remove(clipped_aux_path)
            except PermissionError as e:
                print(f"Error: {e}. The .aux.xml file might still be in use.")


        # Calculate mean value for each feature in the vector layer using QgsZonalStatistics
        stats = QgsZonalStatistics(vector_layer, output_raster_layer, stats=QgsZonalStatistics.Mean)
        stats.calculateStatistics(None)

        # Extracting EXPT and PLOT columns
        expt_col = self.dlg.exptCmboBox.currentText()
        plot_col = self.dlg.plotCmboBox.currentText()

        columns = [expt_col, plot_col, "mean"]

        values_dict = {column: [] for column in columns}

        for feature in vector_layer.getFeatures():
            for column in columns:
                value = feature.attribute(column)
                if value == NULL:
                    values_dict[column].append(np.NaN)
                else:
                    values_dict[column].append(value)

        # Calculate days between current date and day zero
        imgDt_date = date(imgDt.year(), imgDt.month(), imgDt.day())
        dayzeroDt_date = date(dayzeroDt.year(), dayzeroDt.month(), dayzeroDt.day())
        j_day = (imgDt_date - dayzeroDt_date).days

        # Remove GLI field from vector
        vector_layer.startEditing()
        mean_field_index = vector_layer.fields().indexOf("mean")
        if mean_field_index != -1:
            vector_layer.deleteAttribute(mean_field_index)
            vector_layer.updateFields()
        vector_layer.commitChanges()

        # Create a DataFrame from the dictionary
        df = pd.DataFrame(values_dict)
        df['DATE'] = imgDt_date
        df['DAY'] = j_day

        if os.path.exists(csv_file_path):
            prev_csv = pd.read_csv(csv_file_path)
            try:
                prev_csv['DATE'] = pd.to_datetime(prev_csv['DATE'], format="%Y-%m-%d").dt.date
            except ValueError:
                prev_csv['DATE'] = pd.to_datetime(prev_csv['DATE'], format="%m/%d/%Y").dt.date
            df = pd.concat([prev_csv, df])
            df.to_csv(csv_file_path, index=False)
        else:
            df.to_csv(csv_file_path, index=False)

        self.iface.messageBar().pushMessage("Success", "Zonal Statistics calculated.",
                                            level=Qgis.Success)

        # Remove gli raster
        if not self.dlg.gliKeep.isChecked():
            gli_layer = QgsProject.instance().mapLayersByName(
                f"{formatted_date}_{QgsProject.instance().baseName()}_GLI")

            # Remove the layer from the project
            if gli_layer:
                QgsProject.instance().removeMapLayer(gli_layer[0].id())

            time.sleep(0.5)

            # Check if the file is still open or in use
            if os.path.exists(output_file_path):
                try:
                    os.remove(output_file_path)
                except PermissionError as e:
                    print(f"Error: {e}. The file might still be in use.")

        # Begin Maturity estimates
        byDB = self.dlg.byDBBtn.isChecked()
        est_day = self.dlg.estBox.isChecked()
        last_flight = self.dlg.lastFlight.isChecked()
        trigger = 0.1

        if byDB:
            bp_min_2 = 4
            bp_min_1 = 3
            mat_sched = self.generate_schedule()
            mat_sched_dict = mat_sched.set_index('DAY')['Maturity'].to_dict()

            extent = 5  # window for acceptable maturity estimates
        else:
            bp_min_2 = 7
            bp_min_1 = 4

            extent = 3  # window for acceptable maturity estimates

        filtered = df[df['DATE'] <= imgDt_date]

        # Check if Maturity column exists, add if it doesn't
        if 'Maturity' not in [field.name() for field in vector_layer.fields()]:
            vector_layer.startEditing()

            if byDB:
                vector_layer.addAttribute(QgsField("Maturity", QVariant.Int))

            else:
                vector_layer.addAttribute(QgsField("Maturity", QVariant.String))
                vector_layer.addAttribute(QgsField("Maturity_d", QVariant.Int))

        if est_day and 'Rec_d' not in [field.name() for field in vector_layer.fields()]:
            vector_layer.addAttribute(QgsField("Rec_d", QVariant.String))


            vector_layer.updateFields()
            vector_layer.commitChanges()

        # Subset features without a value in "Maturity" field and extract expt_col and plot_col values
        expression = f'"Maturity" IS NULL'
        request = QgsFeatureRequest().setFilterExpression(expression)

        subset_values = [(feature[expt_col], feature[plot_col]) for feature in vector_layer.getFeatures(request)]

        subset_df = pd.DataFrame(subset_values, columns=[expt_col, plot_col])

        # Ensure value is string
        filtered[expt_col] = filtered[expt_col].astype(str)
        filtered[plot_col] = filtered[plot_col].astype(str)
        subset_df[expt_col] = subset_df[expt_col].astype(str)
        subset_df[plot_col] = subset_df[plot_col].astype(str)

        vi_subset = filtered.merge(subset_df, on=[expt_col, plot_col], how='inner')

        # Preprocess dates
        vi_subset['DATE'] = pd.to_datetime(vi_subset['DATE']).dt.date
        vi_subset = vi_subset.sort_values(by=[expt_col, plot_col, 'DATE'])

        out = {}

        for expt, plot in vi_subset[[expt_col, plot_col]].drop_duplicates().itertuples(index=False):

            plot_subset = vi_subset[(vi_subset[expt_col] == expt) & (vi_subset[plot_col] == plot)]

            # Sort by date
            # plot_subset = plot_subset.sort_values(by='DATE')

            gli_values = plot_subset['mean'].values
            dates = pd.to_datetime(plot_subset['DATE']).dt.date.values
            days = plot_subset['DAY'].values

            gli_len = len(gli_values)
            max_days = max(days)
            min_days = min(days)

            # Check if any GLI values are below the threshold
            if any(gli <= threshold for gli in gli_values):

                # Use first date if first observation is below threshold
                if gli_len == 1:

                    if byDB:
                        # db_est = mat_sched.loc[mat_sched['DAY'] == int(days[0]), 'Maturity'].values[0]

                        db_est = mat_sched_dict.get(days[0])
                        out[(expt, plot)] = {'Maturity': int(db_est)}
                    else:
                        out[(expt, plot)] = {'Maturity': dates[0],
                                             'Maturity_d': int(days[0])}
                    continue

                # Manually regress if only two points
                elif gli_len == 2:
                    slope = (gli_values[1] - gli_values[0]) / (days[1] - days[0])
                    y_int = gli_values[0] - slope * days[0]

                    mat_est = int(np.round((threshold - y_int) / slope))

                    if byDB:
                        db_est = mat_sched_dict.get(mat_est)
                        out[(expt, plot)] = {'Maturity': db_est}
                    else:
                        out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                             'Maturity_d': mat_est}
                    continue

                elif gli_len <= 4 and not byDB: # Linear regression if 4 points of less
                    reg = np.polyfit(days, gli_values, 1)
                    slope, y_int = reg

                    if slope >= 0: # Use last 2 points if slope is positive
                        slope = (gli_values[-1]- gli_values[-2]) / (days[-1] - days[-2])
                        y_int = gli_values[-2] - slope * days[-2]

                    mat_est = int(np.round((threshold - y_int) / slope))

                    if abs(mat_est - max_days) <= extent:
                        mat_est = round(mat_est)

                        out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                             'Maturity_d': mat_est}
                        continue

                    elif last_flight:
                        if min_days <= mat_est <= max_days + 7:

                            mat_est = round(mat_est)

                            out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                                 'Maturity_d': mat_est}

                            continue

                        else:
                            mat_est = np.nan

                            out[(expt, plot)] = {'Maturity': pd.NaT,
                                                 'Maturity_d': mat_est}

                            continue

                    else:
                        continue

                # Loess line
                else:
                    loess = sm.nonparametric.lowess
                    gli_loess = loess(gli_values, days)

                    fitted = gli_loess[:,1]

                    interp_funct = interp1d(fitted, days)

                    try:
                        mat_est = int(np.round(interp_funct(threshold)))

                        if byDB:
                            db_est = mat_sched_dict.get(mat_est)
                            out[(expt, plot)] = {'Maturity': int(db_est)}
                        else:
                            out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                                 'Maturity_d': mat_est}

                        continue

                    except ValueError:
                        pass

            # Low data point check
            if gli_len < bp_min_1 and all(gli > threshold for gli in gli_values):
                if last_flight:
                    mat_est = np.nan

                    if byDB:
                        out[(expt, plot)] = {'Maturity': mat_est}
                    else:
                        out[(expt, plot)] = {'Maturity': pd.NaT,
                                             'Maturity_d': mat_est}

                    continue
                else:
                    continue

            # Segmented regression
            if any(gli <= trigger for gli in gli_values):

                # 2 BP
                if gli_len >= bp_min_2:
                    seg = piecewise_regression.Fit(days, gli_values, n_breakpoints=2, n_boot=50, tolerance=0.01, min_distance_to_edge=0.1)

                    # Try one breakpoint if 2 point model doesn't converge
                    if not seg.get_params()['converged']:
                        seg = piecewise_regression.Fit(days, gli_values, n_breakpoints=1, n_boot=50, tolerance=0.01, min_distance_to_edge=0.1)

                        # Only run linear regression if last flight and nothing else converges
                        if not seg.get_params()['converged']:

                            reg = np.polyfit(days, gli_values, 1)
                            slope, y_int = reg

                            mat_est = int(np.round((threshold - y_int) / slope))

                            if abs(mat_est - max_days) <= extent:
                                mat_est = round(mat_est)

                                if byDB:
                                    db_est = mat_sched_dict.get(mat_est)
                                    out[(expt, plot)] = {'Maturity': int(db_est)}
                                else:
                                    out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                                         'Maturity_d': mat_est}

                                continue

                            elif last_flight:
                                if min_days <= mat_est <= max_days + 7:
                                    mat_est = round(mat_est)

                                    if byDB:
                                        db_est = mat_sched_dict.get(mat_est)
                                        out[(expt, plot)] = {'Maturity': int(db_est)}
                                    else:
                                        out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                                             'Maturity_d': mat_est}

                                    continue
                                else:
                                    mat_est = np.nan

                                    if byDB:
                                        out[(expt, plot)] = {'Maturity': mat_est}
                                    else:
                                        out[(expt, plot)] = {'Maturity': pd.NaT,
                                                             'Maturity_d': mat_est}

                                    continue
                            else:
                                continue

                    # If 2 BP, check that the 2nd BP isn't unusually high
                    else:
                        breakpoints = np.array([value for key, value in seg.get_params().items() if 'breakpoint' in key])

                        if seg.predict([breakpoints[1]]).item() > 0.1:
                            seg = piecewise_regression.Fit(days, gli_values, n_breakpoints=1, n_boot=50, tolerance=0.01, min_distance_to_edge=0.1) # Try 1 BP model

                            # If 1 BP doesn't work, revert to 2 BP
                            if not seg.get_params()['converged']:
                                seg = piecewise_regression.Fit(days, gli_values, n_breakpoints=2, n_boot=50, tolerance=0.01, min_distance_to_edge=0.1)

                # 1 BP
                elif gli_len >= bp_min_1:
                    seg = piecewise_regression.Fit(days, gli_values, n_breakpoints=1,n_boot=50, tolerance=0.01, min_distance_to_edge=0.1)

                    if not seg.get_params()['converged']:

                        reg = np.polyfit(days, gli_values, 1)
                        slope, y_int = reg

                        mat_est = int(np.round((threshold - y_int) / slope))

                        if abs(mat_est - max_days) <= extent or last_flight:

                            if byDB:
                                db_est = mat_sched_dict.get(mat_est)
                                out[(expt, plot)] = {'Maturity': int(db_est)}
                            else:
                                out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                                     'Maturity_d': mat_est}
                            continue

                        else:
                            continue

                # If less than BP_min_1 and last flight
                elif last_flight:

                    mat_est = np.nan

                    if byDB:
                        out[(expt, plot)] = {'Maturity': mat_est}
                    else:
                        out[(expt, plot)] = {'Maturity': pd.NaT,
                                             'Maturity_d': mat_est}
                    continue
                else:
                    continue

                # Continue with seg model

                # Extract slopes and breakpoints
                slopes = np.array([value for key, value in seg.get_params().items() if 'alpha' in key])
                breakpoints = np.array([value for key, value in seg.get_params().items() if 'breakpoint' in key])
                end_pts = np.append(breakpoints, max_days)

                # Calculate intercepts for each slope
                intercepts = []
                for i, slope in enumerate(slopes):
                    if i == 2:  # Use the 2nd breakpoint if there's a 3rd slope
                        intercepts.append(seg.predict([breakpoints[1]]).item() - slope * breakpoints[1])
                    else:  # Use 1st breakpoint for first 2 slopes
                        intercepts.append(seg.predict([breakpoints[0]]).item() - slope * breakpoints[0])

                filtered_slopes = []
                filtered_int = []
                filtered_bps = []

                # Identify negative slopes and those that end below trigger threshold
                for i, slope in enumerate(slopes):
                    if slope < 0:
                        y_last = seg.predict([end_pts[i]]).item()
                        if y_last <= trigger:
                            filtered_slopes.append(slope)
                            filtered_int.append(intercepts[i])
                            filtered_bps.append(end_pts[i])

                slope = np.array(filtered_slopes)
                y_int = np.array(filtered_int)
                breakpoints = np.array(filtered_bps)

                # If no negative slopes, skip
                if len(slope) == 0:
                    if last_flight:
                        mat_est = np.nan
                        if byDB:
                            out[(expt, plot)] = {'Maturity': mat_est}
                        else:
                            out[(expt, plot)] = {'Maturity': pd.NaT,
                                                 'Maturity_d': mat_est}

                        continue
                    else:
                        continue

                mat_ests = (threshold - y_int) / slope # calculate ests from each valid slope
                diffs = abs(mat_ests - max_days) # difference between estimate and current date

                if diffs[-1] <= extent: # Check to see if last slope can provide valid est
                    mat_est = mat_ests[-1]
                else: # Otherwise use estimate closest to current date
                    mat_est = mat_ests[np.argmin(diffs)]

                if last_flight: # Use estimate closest to end of each slope if last flight
                    diffs = abs(mat_ests - breakpoints)
                    mat_est = mat_ests[np.argmin(diffs)]

                # Check that est is within 3 days of current date
                if mat_est - max_days <= extent:
                    mat_est = round(mat_est)

                    if byDB:
                        db_est = mat_sched_dict.get(mat_est)
                        out[(expt, plot)] = {'Maturity': int(db_est)}
                    else:
                        out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                             'Maturity_d': mat_est}

                    continue

                # If last flight, check that estimate is at least reasonable
                elif last_flight:
                    if min_days <= mat_est <= max_days + 7:
                        mat_est = round(mat_est)

                        if byDB:
                            db_est = mat_sched_dict.get(mat_est)
                            out[(expt, plot)] = {'Maturity': int(db_est)}
                        else:
                            out[(expt, plot)] = {'Maturity': (dayzeroDt_date + timedelta(days=mat_est)),
                                                 'Maturity_d': mat_est}

                        continue
                    else:
                        mat_est = np.nan

                        if byDB:
                            out[(expt, plot)] = {'Maturity': mat_est}
                        else:
                            out[(expt, plot)] = {'Maturity': pd.NaT,
                                                 'Maturity_d': mat_est}
                        continue
                # Skip if invalid initial estimate and not last flight
                else:
                    continue

            elif last_flight:
                mat_est = np.nan

                if byDB:
                    out[(expt, plot)] = {'Maturity': mat_est}
                else:
                    out[(expt, plot)] = {'Maturity': pd.NaT,
                                         'Maturity_d': mat_est}
                continue

            else:
                continue

        # Add estimates to shapefile

        maturity_index = vector_layer.fields().indexFromName("Maturity")

        if not byDB:
            maturity_d_index = vector_layer.fields().indexFromName("Maturity_d")

        if est_day:
            rec_d_index = vector_layer.fields().indexFromName("Rec_d")

        vector_layer.startEditing()

        # Counter for new estimates
        new_mat = 0

        for feature in vector_layer.getFeatures():
            expt_val = str(feature[expt_col])
            plot_val = str(feature[plot_col])

            if (expt_val, plot_val) in out:
                new_mat += 1

                # Retrieve Maturity and Maturity_d values
                if byDB:
                    maturity_val = out[(expt_val, plot_val)]['Maturity']

                else:
                    try:
                        maturity_val = out[(expt_val, plot_val)]['Maturity'].strftime('%m/%d/%Y')
                    except ValueError:
                        maturity_val = pd.NaT

                    maturity_d_val = out[(expt_val, plot_val)]['Maturity_d']

                # Update the feature attributes
                if byDB:
                    vector_layer.changeAttributeValue(feature.id(), maturity_index, maturity_val)
                    if est_day:
                        vector_layer.changeAttributeValue(feature.id(), rec_d_index, imgDt_date.strftime('%m/%d/%Y'))
                else:
                    vector_layer.changeAttributeValue(feature.id(), maturity_index, maturity_val)
                    vector_layer.changeAttributeValue(feature.id(), maturity_d_index, maturity_d_val)
                    if est_day:
                        vector_layer.changeAttributeValue(feature.id(), rec_d_index, imgDt_date.strftime('%m/%d/%Y'))

        vector_layer.commitChanges()
        self.iface.messageBar().pushMessage("Info",
                                            f"New Estimates: {new_mat}; Remaining Plots: {len(subset_df) - new_mat}",
                                            level=Qgis.Info, duration=0)

        # Produce summary graph
        if self.dlg.summaryBox.isChecked():

            # Get list of features
            if byDB:
                vec_list = [(feature[expt_col], feature[plot_col], feature['Maturity']) for feature in vector_layer.getFeatures()]

                vec_df = pd.DataFrame(vec_list, columns=[expt_col, plot_col, 'Maturity'])

                # Ensure 'Maturity_d' column has np.nan for NULL values
                vec_df['Maturity'] = pd.to_numeric(vec_df['Maturity'], errors='coerce').astype('Int64')

            else:
                vec_list = [(feature[expt_col], feature[plot_col], feature['Maturity_d']) for feature in vector_layer.getFeatures()]

                vec_df = pd.DataFrame(vec_list, columns=[expt_col, plot_col, 'Maturity_d'])

                # Ensure 'Maturity_d' column has np.nan for NULL values
                vec_df['Maturity_d'] = pd.to_numeric(vec_df['Maturity_d'], errors='coerce').astype('Int64')


            # Count total plots by each EXPT
            n_plots = vec_df.groupby(expt_col)[plot_col].count().reset_index()
            n_plots.columns = [expt_col, 'n_plots']


            # Count plots without maturity values
            if byDB:
                n_mat = vec_df[vec_df['Maturity'].notnull()].groupby(expt_col)[plot_col].count().reset_index()

            else:
                n_mat = vec_df[vec_df['Maturity_d'].notnull()].groupby(expt_col)[plot_col].count().reset_index()

            n_mat.columns = [expt_col, 'n_mat']

            # Merge data frames
            merged = pd.merge(n_plots, n_mat, on = expt_col, how='left').fillna(0)
            merged['p_mat'] = merged['n_mat']/merged['n_plots']
            merged['p_mat'] = np.round(merged['p_mat'], 4)
            merged = merged.sort_values(by='p_mat', ascending=False)

            fig = px.bar(merged, y='p_mat', x=expt_col, orientation='v', color='n_plots',
                         color_continuous_scale=px.colors.sequential.Viridis,
                         labels={expt_col: 'Experiment', 'p_mat': 'Prop. of Plots with Maturity Estimates',
                                 'n_plots': 'Total Plots'},
                         title=f"{imgDt_date.strftime('%b %#d, %Y')} - {QgsProject.instance().baseName()} Maturity Status",
                         template='simple_white')

            fig.update_layout(title_x=0.5, yaxis_range = [0,1])
            fig.add_annotation(
                showarrow=False,
                text="Made with Matti, a QGIS plugin",
                font=dict(size=10),
                xref='x domain',
                x=1,
                yref='y domain',
                y=-0.25
            )
            fig.show()

        seconds = round(time.time() - start_time)
        minutes = round(seconds / 60, 1)

        if minutes == 1:
            self.iface.messageBar().pushMessage("Success", f"Maturity estimates completed in {minutes} minute",
                                                level=Qgis.Success, duration=0)
        else:
            self.iface.messageBar().pushMessage("Success", f"Maturity estimates completed in {minutes} minutes",
                                                level=Qgis.Success, duration=0)

    def run(self):
        """Run method that performs all the real work"""

        def add_item_with_icon(combo_box, icon, text):
            item = QStandardItem()
            item.setIcon(icon)
            item.setText(text)
            combo_box.model().appendRow(item)

        def update_label():
            current_year = QDate.currentDate().year()

            if self.dlg.byDateBtn.isChecked():
                self.dlg.label_7.setText("Day Zero")

                # Default to August 31 of the current year
                aug_31 = QDate(current_year, 8, 31)
                aug_31_dt = QDateTime(aug_31)
                self.dlg.dayzeroDt.setDateTime(aug_31_dt)

                # Hide additional objects
                self.dlg.label_10.setVisible(False)
                self.dlg.wdaysCmboBox.setVisible(False)
                self.dlg.wdayWarnLbl.setVisible(False)
                self.dlg.label_11.setVisible(False)
                self.dlg.lengthSpinBox.setVisible(False)
                self.dlg.skipCheckBox.setVisible(False)
                self.dlg.matTbl.setVisible(False)

            elif self.dlg.byDBBtn.isChecked():
                self.dlg.label_7.setText("First Day")

                # Default to September 1 of the current year
                sept_1 = QDate(current_year, 9, 1)
                sept_1_dt = QDateTime(sept_1)
                self.dlg.dayzeroDt.setDateTime(sept_1_dt)

                # Reset selected breaks
                for i in range(self.dlg.wdaysCmboBox.model().rowCount()):
                    item = self.dlg.wdaysCmboBox.model().item(i)
                    item.setCheckState(Qt.Unchecked)

                # Uncheck skip box
                self.dlg.skipCheckBox.setCheckState(Qt.Unchecked)

                # Reset months
                self.dlg.lengthSpinBox.setValue(3)

                # Show additional objects
                self.dlg.label_10.setVisible(True)
                self.dlg.wdaysCmboBox.setVisible(True)
                self.dlg.wdayWarnLbl.setVisible(True)
                self.dlg.label_11.setVisible(True)
                self.dlg.lengthSpinBox.setVisible(True)
                self.dlg.skipCheckBox.setVisible(True)

        def update_wday_warning():
            checked_count = 0

            # Iterate over day options
            for i in range(self.dlg.wdaysCmboBox.model().rowCount()):
                item = self.dlg.wdaysCmboBox.model().item(i)
                if item.checkState() == Qt.Checked:
                    checked_count += 1

            # Update label based on count
            if checked_count == 0:
                self.dlg.wdayWarnLbl.setText("Please select at least one day")
                self.dlg.wdayWarnLbl.setStyleSheet("color: red;")
                self.dlg.matTbl.setVisible(False)
            else:
                self.dlg.wdayWarnLbl.setText("")
                self.generate_schedule()
                self.dlg.matTbl.setVisible(True)

        # generate schedule used to be here


        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = MattiDialog()

            # Default to current date
            self.dlg.imgDt.setDateTime(QtCore.QDateTime.currentDateTime())
            current_year = QDate.currentDate().year()

            # Reset column warning
            self.dlg.colWarnLbl.setText("")

            aug_31 = QDate(current_year, 8, 31)
            aug_31_dt = QDateTime(aug_31)
            self.dlg.dayzeroDt.setDateTime(aug_31_dt)

            # Connect the signal when vector file is selected
            self.dlg.shpCmboBox.currentIndexChanged.connect(self.populate_field_list_widget)

            # Connect combo box index change signals to validation functions
            self.dlg.rasterCmboBox.currentIndexChanged.connect(self.enable_run_button)
            self.dlg.shpCmboBox.currentIndexChanged.connect(self.enable_run_button)
            self.dlg.exptCmboBox.currentIndexChanged.connect(self.enable_run_button)
            self.dlg.plotCmboBox.currentIndexChanged.connect(self.enable_run_button)

            # Connect date change signal to validation method
            self.dlg.imgDt.dateChanged.connect(self.validate_dates)
            self.dlg.dayzeroDt.dateChanged.connect(self.validate_dates)

            # Connect the rejected signal to the reset_values slot
            self.dlg.rejected.connect(self.reset_values)

            # Connect the close button to reject the dialog
            self.dlg.closeBtn.clicked.connect(self.dlg.reject)

            # Set the default path to the folder where the current QGIS project is located
            project_path = QgsProject.instance().fileName()
            if project_path:
                project_dir = os.path.dirname(project_path)
                self.dlg.outputFolder.setDefaultRoot(project_dir)
                self.dlg.outputFolder.setFilePath(project_dir)

                self.dlg.csvFile.setDefaultRoot(project_dir)

            # Disable the Run button initially
            self.dlg.runBtn.setEnabled(False)

            # Connect the runBtn clicked signal to execute dialog
            self.dlg.runBtn.clicked.connect(self.main)

            # Connect column boxes to warning box
            self.dlg.exptCmboBox.currentIndexChanged.connect(self.validate_col_boxes)
            self.dlg.plotCmboBox.currentIndexChanged.connect(self.validate_col_boxes)

            # Connect radio buttons to label_7
            self.dlg.byDateBtn.toggled.connect(update_label)
            self.dlg.byDBBtn.toggled.connect(update_label)

            # Connect weekdays combo box to label
            self.dlg.wdaysCmboBox.model().itemChanged.connect(update_wday_warning)
            self.dlg.wdaysCmboBox.model().itemChanged.connect(self.generate_schedule)

            # Connect plant row objects to generate_schedule
            if self.dlg.byDateBtn.isChecked():
                self.dlg.dayzeroDt.dateChanged.connect(update_wday_warning)
                self.dlg.skipCheckBox.toggled.connect(update_wday_warning)
                self.dlg.lengthSpinBox.valueChanged.connect(update_wday_warning)

            # Populate plant row schedule day list
            wdays = ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"]

            for day in wdays:
                item = QStandardItem(day)
                item.setCheckable(True)
                self.dlg.wdaysCmboBox.model().appendRow(item)

            # Initial checks
            update_wday_warning()
            update_label()

        self.validate_dates()

        # Get raster icon
        raster_icon = QgsApplication.getThemeIcon('/mIconRaster.svg')
        vector_icon = QgsApplication.getThemeIcon('/mIconVector.svg')
        csv_icon = QgsApplication.getThemeIcon('/mIconTableLayer.svg')

        # Fetch loaded rasters
        self.dlg.rasterCmboBox.clear()
        raster_model = QStandardItemModel(self.dlg.rasterCmboBox)
        self.dlg.rasterCmboBox.setModel(raster_model)
        self.dlg.rasterCmboBox.addItem("")

        raster_layer_names = []
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.RasterLayer:
                raster_layer_names.append(layer.name())

        raster_layer_names.sort()

        for name in raster_layer_names:
            add_item_with_icon(self.dlg.rasterCmboBox, raster_icon, name)

        # Fetch loaded shapefiles
        self.dlg.shpCmboBox.clear()
        vector_model = QStandardItemModel(self.dlg.shpCmboBox)
        self.dlg.shpCmboBox.setModel(vector_model)
        self.dlg.shpCmboBox.addItem("")

        vector_layer_names = []
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.VectorLayer and layer.wkbType() in (
                    QgsWkbTypes.Polygon, QgsWkbTypes.MultiPolygon):
                vector_layer_names.append(layer.name())

        # Sort the vector layer names alphabetically
        vector_layer_names.sort()

        # Add sorted vector layer names to the combo box
        for name in vector_layer_names:
            add_item_with_icon(self.dlg.shpCmboBox, vector_icon, name)



        # show the dialog
        self.dlg.show()
