import subprocess
import os


def run_mass(
        input_file,
        output_file,
        dim=2,
        method=3,
        n_x=300,
        k_type_x=1,
        b_x_ratio=1.0,
        n_y=300,
        k_type_y=1,
        b_y_ratio=1.0
):
    """
    Call mass_kdv.exe to generate product kernel density.

    Parameters:
    - input_file: path to the input data file
    - output_file: path to the output file
    - dim: data dimensionality (default: 2)
    - method: algorithm method
        0: SCAN
        1: SLAM
        2: MASS_CR
        3: MASS_OPT
        4: RQS_kd
        5: RQS_range
    - n_x, n_y: discretization resolution in x and y directions
    - k_type_x, k_type_y: kernel function type
        1: Epanechnikov
        2: Triangular
        3: Uniform
    - b_x_ratio, b_y_ratio: bandwidth ratios
    """

    # Get the absolute path of the exe file
    exe_path = os.path.join(os.path.dirname(__file__), "./bin/mass_pkdv.exe")
    exe_path = os.path.abspath(exe_path)

    # Check whether the exe exists
    if not os.path.exists(exe_path):
        raise FileNotFoundError(f"mass_kdv.exe not found at: {exe_path}")

    # Check whether the input file exists
    if not os.path.exists(input_file):
        raise FileNotFoundError(f"Input file not found: {input_file}")

    # Construct command-line arguments
    cmd = [
        exe_path,
        input_file,
        output_file,
        str(dim),
        str(method),
        str(n_x),
        str(k_type_x),
        str(b_x_ratio),
        str(n_y),
        str(k_type_y),
        str(b_y_ratio)
    ]

    # Execute the exe
    try:
        subprocess.run(cmd, check=True)
        print(f"mass_kdv finished successfully. Output saved to: {output_file}")
    except subprocess.CalledProcessError as e:
        print("Error running mass_kdv:", e)


# ==============================
# Command-line invocation
# ==============================
if __name__ == "__main__":
    # Default example data
    input_file = os.path.join(os.path.dirname(__file__), "./data/UK_LD_Crime")
    input_file = os.path.abspath(input_file)
    output_file = os.path.join(os.path.dirname(__file__), "./data/result.txt")
    output_file = os.path.abspath(output_file)

    run_mass(input_file, output_file)